/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * term.c ---   terminal i/o, system independent parts
 * (duz 24Feb94)
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "config.h"
#include "forth.h"
#include "term.h"

/* global variables exported by the terminal stuff: */

int rows, cols;                 /* size of text screen */
int xmax, ymax;                 /* size of graphics window in pixels */

/* map function and cursor keys to wordstar key sequences or to EKEY codes: */

static char                     /* replacement strings for the key codes, */
*raw2ws [NO_OF_KEYS] =          /* wordstar-like version for block editor: */
{
  "\013L",                      /* F1 -> ^KL i.e. interpret this line */
  "\013D",                      /* F2 -> ^KD i.e. stamp screen */
  "\013U",                      /* F3 -> ^KU i.e. use other block file */
  NULL,
  "\013X",                      /* F5 -> ^KX i.e. push line */
  "\013E",                      /* F6 -> ^KE i.e. pop line */
  "\013Y",                      /* F7 -> ^KY i.e. push and delete line */
  "\013W",                      /* F8 -> ^KW i.e. pop and insert line */
  "\013B",                      /* F9 -> ^KB i.e. load this block */
  "",                         /* F10 -> ^U i.e. quit editor */

  "\023", "\004", "\005", "\030",
  "\021S","\021D","\003", "\022",
  "\008", "\007", "\026", "\026",
  "\016", "\021Y","\031", "\021Z",
/*NULL
*/},
*raw2ekey [NO_OF_KEYS] =	/* version for EKEY */
{
  "\000", "\001", "\002", "\003", "\004",
  "\005", "\006", "\007", "\010", "\011",

  "\012", "\013", "\014", "\015",
  "\016", "\017", "\020", "\021",
  "\022", "\023", "\024", "\025",
  "\026", "\027", "\030", "\031",
/*"\032"
*/};

static Byte
        keybuf [8],
        *keyptr = NULL;

static int
get_remap_key (Byte *replace [])
{
  int i, n, m;

  if (keyptr)
    if (*keyptr == '\0')
      keyptr = NULL;
    else
      return *keyptr++;
  n = 0;
  for (;;)
    {
      keybuf [n++] = getkey ();
      m = -1;
      for (i = m; ++i < NO_OF_KEYS; )
        {
          if (!rawkey_string [i] || !replace [i])
            continue;
          if (memcmp (keybuf, rawkey_string [i], n) == 0)
            if (rawkey_string [i][n] == '\0')
              {
                keyptr = replace [i];
                return *keyptr++ | 0x100;
              }
            else
              m = i;
        }
      if (m < 0)
        {
          keybuf [n] = '\0';
          keyptr = keybuf;
          return *keyptr++;
        }
    }
}

int
getekey (void)
{
  int key = get_remap_key ((Byte **)raw2ekey);
  if (key == EKEY_kb)
    return '\b';
/*if (key == EKEY_enter)
    return '\r';
*/return key;
}

int
getwskey (void)
{
  return get_remap_key ((Byte **)raw2ws) & ~0x100;
}

int
ekeypressed (void)
{
  return (keyptr && *keyptr) || keypressed ();
}

int
printable (int c)
{
  return
#if EMX
    !isascii (c) || !iscntrl (c) /* almost anything can be displayed */
#else
      isprint (c)
# if defined (ISO_CHARSET)
	|| (0xA0 <= (Byte)c && (Byte)c <= 0xFF)
# endif
#endif
	  ;
}

void				/* write character to screen */
cputc_printable (int c)		/* ensure visible result */
{
  if (printable (c))
    cputc (c);
  else
    {
      standout_on ();
      cputc ((Byte)c < ' ' ? c + '@' : '?');
      standout_off ();
    }
}

int
change_case (int ch)
{
  if (ch >= 0x100)
    return ch;
  if (islower (ch))
    return toupper (ch);
  if (isupper (ch))
    return tolower (ch);
#if defined (ISO_CHARSET)
  if (0xC0 <= (Byte)ch && (Byte)ch <= 0xDE)
    return ch + 0x20;
  if (0xE0 <= (Byte)ch && (Byte)ch <= 0xFE)
    return ch - 0x20;
#elif defined (EMX)
  {
    static char lower [] =
	"\x81\x82\x83\x84\x85\x86\x87\x88" /* IBM 437 lower case characters */
	"\x89\x8A\x8B\x8C\x8D\x91\x93\x94"
	"\x95\x96\x97\x98\xA0\xA1\xA2\xA3"
	"\xA4";
    static char upper [] =
	"\x9A\x90\x41\x8E\x41\x8F\x80\x45" /* IBM 437 upper case characters */
	"\x45\x45\x49\x49\x49\x92\x4F\x99"
	"\x4F\x55\x55\x59\x41\x49\x4F\x55"
	"\xA5";
    char *p;

    if ((p = strchr (lower, (char)ch)))
      return upper [p - lower];
    if ((p = strchr (upper, (char)ch)))
      return lower [p - upper];
  }
#endif
  return ch;
}

int
getckey (void)
{
  char c = getwskey ();
  return c < ' ' ? c + '@'
		 : toupper (c);
}


/*
 * hooks for window size change and job control:
 */

void (*on_stop) (void) =	system_terminal;
void (*on_continue) (void) =	interactive_terminal;
void (*on_winchg) (void) =	query_winsize;
