/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * types.h ---	data types and structures for pfe
 * (duz 09Jul93)
 */

#ifndef __TYPES_H
#define __TYPES_H

#include "config.h"

#ifndef	_SETJMP_H
#include <setjmp.h>
#endif

#ifndef FILE
#include <stdio.h>
#endif

/* the basic integer types ================================================= */

typedef CELL_TYPE		Cell;	/* a stack item */
typedef unsigned CELL_TYPE	uCell;	/* dito unsigned */
typedef struct
	{ uCell hi, lo; }	udCell;	/* dito, double precision unsigned */
typedef struct
	{ Cell hi; uCell lo; }	dCell;	/* dito, double precision signed */

typedef Cell			Flag;
typedef unsigned HALF_CELL_TYPE	hCell;	/* a smaller unsigned number */
typedef unsigned char		Byte;	/* an 8-bit unsigned number */

typedef void (*pCode) (void);		/* pointer to executable code */
typedef pCode *Xt;			/* type of the "execution token" */

typedef struct { Cell	quot, rem; } fdiv_t;
typedef struct { uCell	quot, rem; } udiv_t;

typedef struct			/* "map" of a Cell */
{
#if HIGHBYTE_FIRST
	hCell	hi, lo;
#else
	hCell	lo, hi;
#endif
}	Cell_map;

/* data structures ======================================================== */

typedef struct Head	Head;	/* pointer set comprising a definition */
typedef struct Wordl	Wordl;	/* body of a word list */
typedef struct File	File;	/* neccessary information about open files */
typedef struct Dict	Dict;	/* a dictionary specification */
typedef struct Input	Input;	/* an input source specification */
typedef struct Iframe	Iframe;	/* a frame to save an input specification */
typedef struct Except	Except;	/* an exception frame */

struct Head			/* set of pointers comprising a header */
{				/* when preceeded by a name */
	char *	link;		/* link back to previous word in vocabulary */
	pCode	aux;		/* execution semantics or pointer for DOES> */
	pCode	cfa;		/* compilation/interpretation semantics */
};

struct Wordl			/* body of a word list */
{				/* a word list identifier is a (Wordl *) */
	char *	thread [THREADS];
	Wordl *	prev;		/* word lists are chained */
};

struct File			/* describes a file */
{
	FILE	*f;		/* associated ANSI-C file structure */
	char	mdstr [4];	/* mode string for fopen() */
	Cell	mode;		/* mode code for open_file() */
	uCell	size,		/* if block file: size of file in blocks */
		len,		/* if stream: length of input line */
		n,		/* block in buffer or source line */
		updated;	/* if block file: block updated? */
	char	name [128],	/* file name */
		buffer [1024];	/* buffer for block or input line */
};

struct Input			/* an input source specification */
{
	Cell	source_id;	/* SOURCE-ID */
	File *	block_file;	/* which file is active? */
	uCell	blk;		/* currently loaded block */
	char	*tib;		/* points to TIB or to EVALUATED string */
	uCell	number_tib;	/* #TIB, length of string in TIB */
	uCell	to_in;		/* input parsing position */
};

struct Iframe			/* a frame to save an input specification */
{
	Cell	magic;
	Input	input;
	Iframe *prev;
};

struct Except
{
	Cell	magic;
	Xt *	ip;
	Cell *	sp;
	Iframe *iframe;
	jmp_buf	jmp;
	Except *prev;
};

struct memory			/* contains memory-map, one of lo/hi */
{				/* each start/end of: */
	Byte	*dict;		/*  dictionary */
	Cell	*stack;		/*  data stack */
	double	*fstack;	/*  floating point stack */
	Xt *	*rstack;	/*  return stack */
	char	*pocket;	/*  POCKET for interactive S" */
	char	*tib;		/*  TIB */
	char	*history;	/*  command line history buffer */
	File	*files;		/*  files */
};

struct memsiz			/* size of allocated memory areas */
{				/* in items of type stored there */
	uCell	dict,	stack,
		fstack,	rstack,
		pocket,	tib,
		history,files;
};

struct Dict			/* all system variables needed */
{				/* to describe a dictionary */
	Byte *	dp;		/* actual top of the dictionary */
	Byte *	fence;		/* can't forget below that address */
	char *	last;		/* NFA of most recently CREATEd header */
	Wordl *	voc_link;	/* link to chained word lists */
	Wordl *	context [ORDER_LEN]; /* dictionary search order */
	Wordl *	only;		/* ONLY is always searched */
	Wordl *	current;	/* points to vocabulary in extension */
	Wordl *	forth;		/* points to body of FORTH vocabulary */
	Wordl *	dforder [ORDER_LEN]; /* default dictionary search order */
	Xt	application;	/* word to run initially or 0 */
	char *	hld;		/* auxiliary pointer for number output */
	Cell	dpl;		/* position of input decimal point */
};

struct sysvar			/* all FORTH system variables */
{
	Cell *	s0;		/* base of stack pointer */
	double *f0;		/* base of floating point stack pointer */
#if defined(AIX3) && !defined(__GNUC__)
				/* Workaround for xlc compiler bug */
	void ** r0;		/* base of return stack pointer */
#else
	Xt **	r0;		/* base of return stack pointer */
#endif
	Dict   *dict;		/* the actually used dictionary */
	Input	input;		/* active input source, full specification */
	Input	input_err;	/* input specification when error occurred */
	Iframe *saved_input;	/* links to chain of saved input specs */
	Except *cAtch;		/* links to chain of CATCHed words */
	uCell	span;		/* char count of last EXPECT */
	uCell	scr;		/* latest LISTed block number */
	Cell	out;		/* current output column on screen */
	Cell	state;		/* interpreting (0) or compiling (-1) */
	Cell	*locals;	/* number of locals in current def. */
	char	(*local) [32];	/* names of locals in current def. */
	Cell *	csp;		/* compiler security, saves sp here */
	uCell	base;		/* of number i/o conversion */
	Cell	precision;	/* floating point output precision */

#if defined(AIX3) && !defined(__GNUC__)
	/* avoid "internal compiler error" from AIX 3.2 cc compiler */
	void *	key;		/* executed by KEY */
	void *	emit;		/* executed by EMIT */
	void *	expect;		/* executed by EXPECT */
	void *	type;		/* executed by TYPE */
#else
	Xt	key;		/* executed by KEY */
	Xt	emit;		/* executed by EMIT */
	Xt	expect;		/* executed by EXPECT */
	Xt	type;		/* executed by TYPE */
#endif
	Flag	lower_case;	/* do toupper() before dictionary search */
	Cell	redefined_msg;	/* no `"xxx" is redefined' msg if false */
	Cell	reset_order;	/* if true: reset search order on ABORT */

	File *	stdIn;		/* C-library standard files */
	File *	stdOut;		/* mapped to Forth-files */
	File *	stdErr;
	uCell	more;		/* for a more-like effect */
	uCell	lines;
};

/* file modes */

enum 
{
	FMODE_RO = 1, FMODE_WO,  FMODE_RW,
	FMODE_ROB,    FMODE_WOB, FMODE_RWB
};
#define FMODE_BIN (FMODE_ROB - FMODE_RO)

/* Shortcuts for easy access of some system variables: */

#define DP		(sys.dict->dp)
#define FENCE		(sys.dict->fence)
#define LAST		(sys.dict->last)
#define VOC_LINK	(sys.dict->voc_link)
#define CONTEXT		(sys.dict->context)
#define DEFAULT_ORDER	(sys.dict->dforder)
#define ONLY		(sys.dict->only)
#define CURRENT		(sys.dict->current)
#define FORTH		(sys.dict->forth)
#define APPLICATION	(sys.dict->application)
#define HLD		(sys.dict->hld)
#define DPL		(sys.dict->dpl)

#define SOURCE_ID	(sys.input.source_id)
#define SOURCE_FILE	((File *)SOURCE_ID)
#define BLOCK_FILE	(sys.input.block_file)
#define BLK		(sys.input.blk)
#define TIB		(sys.input.tib)
#define NUMBER_TIB	(sys.input.number_tib)
#define TO_IN		(sys.input.to_in)

#define SPAN		(sys.span)
#define SCR		(sys.scr)
#define OUT		(sys.out)
#define STATE		(sys.state)
#define CSP		(sys.csp)
#define BASE		(sys.base)
#define PRECISION	(sys.precision)
#define LOWER_CASE	(sys.lower_case)
#define REDEFINED_MSG	(sys.redefined_msg)

#endif
