#!/usr/local/bin/ruby -w

#
# install.rb - install the Ruby Standard Library Documentation
#
# 1. Determine the target destination (or accept command-line value).
# 2. See if documentation is already installed and compare versions.  Seek
#    user's permission if installed doco is newer. 
# 3. Copy 'stdlib' directory, and all that's in it, to 'stdlib.tmp' in the
#    target directory.  Include VERSION, which is in the current directory.
# 4. Remove old directory.
# 5. Rename temp directory to 'stdlib'.
#

require 'rbconfig'
require 'fileutils'
require 'optparse'

def log(msg="")
  puts "\n" + msg
end

#
# Cancel the installation.
#
def cancel
  puts "Cancelling install"
  exit 2
end

#
# Ask the user to confrm an action.
#
def confirm(msg)
  return true if @force
  loop do
    STDOUT.print msg
    STDOUT.print " [y/N] "
    STDOUT.sync
    resp = STDIN.gets
    case resp[0,1].upcase
    when 'Y' then return true
    when 'N' then cancel
    end
  end
end

#
# Compare two string versions.  Returns 1 iv v1 > v2, -1 if v1 < v2, and 0 if they're
# equal.
#
def compare_versions(v1, v2)
  v1a = v1.scan(/\d+/).map { |s| s.to_i }   # e.g. "0.8.2" -> [0, 8, 2]
  v2a = v2.scan(/\d+/).map { |s| s.to_i }
  v1a <=> v2a
end

#
# Allow a bit of fancy string formatting down below.
#
class String
  def trim
    self.strip.gsub(/^ */, "")
  end
end


#
# 1. Determine the target destination (or accept command-line value).
#

prefix_dir = Config::CONFIG['prefix']
  # This yields something like /usr/local or C:/Ruby.

doc_base = File.join(prefix_dir, "doc", "ruby")
  # Now we're talkin' /usr/local/doc/ruby or C:/Ruby/doc/ruby.
  # Yes I know the Windows one is a bit repetitive, but look at
  # C:/Ruby/lib/ruby.

@force = false
@doc_base = doc_base

opts = OptionParser.new do |o|
  o.program_name = "ruby install.rb"
  o.separator "Options:"
  o.on('--doc-base DIR', '-d', "Install documentation to DIR/stdlib") { |@doc_base| }
  o.on('--force', '-f', "Don't ask any questions; just install") { @force = true }
  o.on('--interactive', '-i', "Confirm any risky actions") { @force = false }
  o.on_tail('--help', '-h', "Show this message") do
    puts opts
    exit
  end
end

opts.parse!(ARGV)

# Sanity check.
unless File.exists?(@doc_base)
  raise "Can't install to directory #@doc_base; it doesn't exist"
  exit 1
end


#
# 2. See if documentation is already installed and compare versions.  Seek
#    user's permission if installed doco is newer the same or newer.
#

installable_version = File.read("VERSION").strip
Dir.chdir(@doc_base) do
  if File.exists?("stdlib")
    # Either it's got an old version in it, it's got something else in it, or its empty.
    if File.exists?("stdlib/VERSION") and File.directory?("stdlib/libdoc")
      # It's got *a* version in it.
      existing_version = File.read("stdlib/VERSION").strip 
      case compare_versions(existing_version, installable_version)
      when 1
        confirm %{
          #@doc_base/stdlib contains a newer version than the one you're installing.
          Continue?
        }.trim
      when 0
        confirm %{
          #@doc_base/stdlib contains the same version you want to install.
          Shall I bother?
        }.trim
      end
    elsif Dir["stdlib"] > 0
      # It's got *something* in it, but not stdlib doc.
      confirm %{
        #@doc_base/stdlib contains something, but not what I expected.
        Overwrite?
      }.trim
    else
      # It's empty.  That's no concern.
    end
  end
end


#
# 3. Copy 'stdlib' directory, and all that's in it, to 'stdlib.tmp' in the
#    target directory.  Include VERSION, which is in the current directory.
#

@target = File.join(@doc_base, "stdlib")
tmp_target = File.join(@doc_base, "stdlib.tmp")
while File.exists?(tmp_target)
  tmp_target << "x"
end
puts "Installing stdlib directory to #{@target}"
confirm "You happy to proceed?"
puts
puts " * copying files to #{tmp_target}"
FileUtils.cp_r "stdlib", tmp_target
FileUtils.cp "VERSION", tmp_target


#
# 4. Remove old directory.
#

if File.exists?(@target)
  puts " * removing old installation"
  FileUtils.rm_rf @target
end


#
# 5. Rename temp directory to 'stdlib'.
#

puts " * moving temporary files to #@target"
FileUtils.mv tmp_target, @target


#
# Done.
#

puts
puts "Successfully installed to #{@target}."
puts "Please point your browser to #{@target}/index.html."

# vim: tw=90
