(**************************************************************************)
(*                                                                        *)
(*  Program to work out module dependencies in a Modula-2 program.        *)
(*  Copyright (C) 2019   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

MODULE Imports;

        (********************************************************)
        (*                                                      *)
        (*       Discovering all modules used in a program      *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            13 January 2000                 *)
        (*  Last edited:        26 September 2019               *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

IMPORT IV, TextIO, Strings, IOChan, IOConsts, ChanConsts, SeqFile, FileSys;

FROM IOChan IMPORT
    (* type *)  ChanId;

FROM STextIO IMPORT
    (* proc *)  WriteString, WriteLn;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM Files IMPORT
    (* type *)  FilenameString,
    (* proc *)  CreatePathTable, LocateModule;

FROM Scanner IMPORT
    (* proc *)  StripSpaces, SetProjectFilename, StartScan, Scan;

FROM Storage IMPORT
    (* proc *)  ALLOCATE;

(********************************************************************************)

TYPE
    CharSet = SET OF CHAR;

CONST
    Nul = CHR(0);  Tab = CHR(9);
    Letters = CharSet {'A'..'Z', 'a'..'z'};

    Testing = FALSE;
    TestParam = "D:\Dev1\webserve\webserve";      (* used only when testing *)

TYPE
    InfoIndex = [0..2047];
    ExtendedInfoIndex = [0..MAX(InfoIndex)+1];

    (* The fields in a ModuleData record are:                           *)
    (*     name         the name of the module                          *)
    (*     filename     the name in the file system                     *)
    (*     IsDefinition TRUE iff this is a definition module            *)
    (*     IsSource     TRUE iff filename is the name of a source file  *)
    (*                     (To be filled in by LocateModule.)           *)
    (*                                                                  *)
    (* For each module except the main module we actually have two      *)
    (* records, one for the definition module and one for the           *)
    (* implementation module.                                           *)

    ModuleData = RECORD
                     name, filename: FilenameString;
                     IsDefinition, IsSource: BOOLEAN;
                 END (*RECORD*);

    Option = (WriteModuleNames);
    OptionSet = SET OF Option;

    ExclPtr = POINTER TO ExclRecord;
    ExclRecord = RECORD
                    next: ExclPtr;
                    this: FilenameString;
                 END (*RECORD*);

VAR
    ModuleInfo: ARRAY InfoIndex OF ModuleData;

    (* ModuleInfo[NextFree] is the first unused array element. *)

    NextFree: ExtendedInfoIndex;

    Options: OptionSet;

    (* List of modules to be excluded from the results. *)

    ExclList: ExclPtr;

(********************************************************************************)
(*                        PICKING UP PROGRAM ARGUMENTS                          *)
(********************************************************************************)

PROCEDURE RecordExclusions (str: ARRAY OF CHAR);

    (* str should be a space-separated list of module names. *)

    VAR pos: CARDINAL;  found: BOOLEAN;
        p: ExclPtr;
        name: FilenameString;

    BEGIN
        LOOP
            WHILE str[0] = ' ' DO
                Strings.Delete (str, 0, 1);
            END (*WHILE*);
            IF str[0] = Nul THEN
                RETURN;
            END (*IF*);
            Strings.Assign (str, name);
            Strings.FindNext (' ', str, 0, found, pos);
            IF found THEN
                name[pos] := Nul;
                Strings.Delete (str, 0, pos+1);
            ELSE
                str[0] := Nul;
            END (*IF*);
            NEW (p);
            p^.next := ExclList;
            p^.this := name;
            ExclList := p;
        END (*LOOP*);
    END RecordExclusions;

(************************************************************************)

PROCEDURE GetParameters (VAR (*OUT*) AllowBinary: BOOLEAN;
                            VAR (*OUT*) directory, modname: FilenameString);

    (* Picks up an optional program argument from the command line. *)

    VAR args: ChanId;
        ParameterString, excllist: ARRAY [0..255] OF CHAR;
        pos1, pos2: CARDINAL;
        found1, found2: BOOLEAN;

    BEGIN
        AllowBinary := FALSE;
        IF Testing THEN
            ParameterString := TestParam;
        ELSE
            modname := "";
            args := ArgChan();
            IF IsArgPresent() THEN
                TextIO.ReadString (args, ParameterString);
            END (*IF*);
        END (*IF*);

        StripSpaces (ParameterString);
        IF ParameterString[0] = '-' THEN
            IF CAP(ParameterString[1]) = 'B' THEN
                AllowBinary := TRUE;
                Strings.Delete (ParameterString, 0, 2);
            ELSE
                Strings.Delete (ParameterString, 0, 1);
            END (*IF*);
        END (*IF*);
        Strings.Assign (ParameterString, modname);
        StripSpaces (modname);

        (* The -X option, if present, comes after the module name. *)

        Strings.FindNext ('-X', modname, 0, found1, pos1);
        IF NOT found1 THEN
            Strings.FindNext ('-x', modname, 0, found1, pos1);
        END (*IF*);
        IF found1 THEN
            Strings.Assign (modname, excllist);
            modname[pos1] := Nul;
            StripSpaces (modname);
            Strings.Delete (excllist, 0, pos1+2);
            RecordExclusions (excllist);
        END (*IF*);

        directory := "";
        IF modname[0] <> Nul THEN

            (* Separate the result into "directory" and "module name".      *)
            (* We assume the directory separator is either '/' or '\'.      *)

            Strings.FindPrev ('/', modname, LENGTH(modname)-1, found1, pos1);
            Strings.FindPrev ('\', modname, LENGTH(modname)-1, found2, pos2);
            IF NOT found1 OR (found2 AND (pos2 < pos1)) THEN
                pos1 := pos2;
            END (*IF*);
            IF found1 OR found2 THEN
                Strings.Assign (modname, directory);
                directory[pos1] := Nul;
                Strings.Delete (modname, 0, pos1+1);
            END (*IF*);

        END (*IF*);

    END GetParameters;

(************************************************************************)
(*                  WORKING OUT THE MODULE DEPENDENCIES                 *)
(************************************************************************)

PROCEDURE Excluded (name: FilenameString): BOOLEAN;

    (* Returns TRUE iff name is on the "exclude" list. *)

    VAR p: ExclPtr;

    BEGIN
        p := ExclList;
        LOOP
            IF p = NIL THEN
                RETURN FALSE;
            ELSIF Strings.Equal (p^.this, name) THEN
                RETURN TRUE;
            ELSE
                p := p^.next;
            END (*IF*);
        END (*LOOP*);
    END Excluded;

(************************************************************************)

PROCEDURE AddModule (name: FilenameString);

    (* Puts a new module into the ModuleInfo array, unless it's         *)
    (* already there or is on the "exclude" list.                       *)

    VAR pos: ExtendedInfoIndex;

    BEGIN
        IF Excluded(name) THEN
            RETURN;
        END (*IF*);

        pos := 0;
        WHILE (pos < NextFree)
                     AND NOT Strings.Equal(ModuleInfo[pos].name, name) DO
            INC (pos);
        END (*WHILE*);
        IF (pos = NextFree) AND (pos < MAX(InfoIndex)) THEN
            ModuleInfo[pos].name := name;
            ModuleInfo[pos].IsDefinition := TRUE;
            INC (pos);
            ModuleInfo[pos].name := name;
            ModuleInfo[pos].IsDefinition := FALSE;
            INC (pos);
            NextFree := pos;
        END (*IF*);
    END AddModule;

(************************************************************************)

PROCEDURE FindImportsBy (j: InfoIndex);

    (* Reads the IMPORT lines in ModuleInfo[j].filename, adds new       *)
    (* entries to ModuleInfo if appropriate.                            *)

    VAR cid: IOChan.ChanId;  res: ChanConsts.OpenResults;
        token: FilenameString;

    (********************************************************************)

    PROCEDURE SkipToSemicolon;

        (* Scans forward until token is ';' or end of file reached. *)

        VAR ch: CHAR;

        BEGIN
            REPEAT
                Scan (token);
                ch := token[0];
            UNTIL (ch = ';') OR (ch = Nul);
        END SkipToSemicolon;

    (********************************************************************)

    BEGIN
        SeqFile.OpenRead (cid, ModuleInfo[j].filename, SeqFile.text, res);
        IF res = ChanConsts.opened THEN
            StartScan (cid);

            (* Scan past the module header line. *)

            SkipToSemicolon;

            (* Search for IMPORT or FROM lines. *)

            LOOP
                Scan (token);
                IF Strings.Equal (token, "IMPORT") THEN

                    (* Handle IMPORT x, y, z, ... *)

                    LOOP
                        Scan (token);
                        IF token[0] IN Letters THEN
                            AddModule (token);
                        ELSE
                            EXIT (*LOOP*);
                        END (*IF*);
                        Scan (token);
                        (* We're expecting a comma at this point. *)

                        IF token[0] <> ',' THEN
                            EXIT (*LOOP*);
                        END (*IF*);
                    END (*LOOP*);

                ELSIF Strings.Equal (token, "FROM") THEN

                    (* Handle FROM x IMPORT ... *)

                    Scan (token);
                    IF token[0] IN Letters THEN
                        AddModule (token);
                        SkipToSemicolon;
                    ELSE
                        EXIT (*LOOP*);
                    END (*IF*);

                ELSIF token[0] IN Letters THEN

                    EXIT (*LOOP*);

                ELSIF token[0] = Nul THEN

                    EXIT (*LOOP*);

                END (*IF*);

            END (*LOOP*);

            SeqFile.Close (cid);

        END (*IF*);

    END FindImportsBy;

(************************************************************************)

PROCEDURE Expand (j: InfoIndex);

    (* Assuming ModuleInfo[j].name is already set, i.e. the module      *)
    (* name is known: finds the corresponding file name, and updates    *)
    (* ModuleInfo to include modules imported by this one.              *)

    BEGIN
        WITH ModuleInfo[j] DO
            LocateModule (name, IsDefinition, filename, IsSource);
        END (*WITH*);
        IF ModuleInfo[j].IsSource THEN
            FindImportsBy (j);
        END (*IF*);
    END Expand;

(************************************************************************)

PROCEDURE FindAllImports (VAR (*IN*) StartDirectory: FilenameString);

    VAR j: InfoIndex;
        ProjName: FilenameString;

    BEGIN
        NextFree := 0;
        IF ModuleInfo[0].name[0] <> Nul THEN
            ModuleInfo[0].IsDefinition := FALSE;
            NextFree := 1;
            Strings.Assign (StartDirectory, ProjName);
            Strings.Append ('\', ProjName);
            Strings.Append (ModuleInfo[0].name, ProjName);
            Strings.Append (".prj", ProjName);
            IF FileSys.Exists (ProjName) THEN
                SetProjectFilename (ProjName);
            END (*IF*);
        END (*IF*);

        j := 0;
        WHILE j < NextFree DO
            Expand (j);
            INC (j);
        END (*WHILE*);

    END FindAllImports;

(************************************************************************)
(*                       WRITING THE RESULTS                            *)
(************************************************************************)

PROCEDURE WriteResults;

    (* Writes a list of file names to standard output. *)

    VAR j: InfoIndex;

    BEGIN
        IF NextFree = 0 THEN
            WriteString ("No modules");  WriteLn;
        ELSIF NOT ModuleInfo[0].IsSource THEN
            WriteString ("Main module not found.");  WriteLn;
        ELSE
            FOR j := 0 TO NextFree-1 DO
                IF WriteModuleNames IN Options THEN
                    WriteString (ModuleInfo[j].name);
                    WriteString ("=");
                END (*IF*);
                IF (WriteModuleNames IN Options)
                           OR (ModuleInfo[j].filename[0] <> Nul) THEN
                    WriteString (ModuleInfo[j].filename);
                    WriteLn;
                END (*IF*);
            END (*FOR*);
        END (*IF*);
    END WriteResults;

(************************************************************************)
(*                           MAIN PROGRAM                               *)
(************************************************************************)

VAR StartDirectory: FilenameString;  AllowBinary: BOOLEAN;

BEGIN
    ExclList := NIL;
    Options := OptionSet {};
    AllowBinary := FALSE;
    GetParameters (AllowBinary, StartDirectory, ModuleInfo[0].name);
    IF StartDirectory[0] = Nul THEN
        StartDirectory := '.';
    END (*IF*);
    CreatePathTable (AllowBinary, StartDirectory);
    FindAllImports (StartDirectory);
    WriteResults;
END Imports.

