MODULE PLQTest;

        (********************************************************)
        (*                                                      *)
        (*               Test of PLQ factorisation              *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        21 April 2016                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


IMPORT Vec;

FROM LongMath IMPORT
    (* proc *)  sqrt;

FROM LongComplexMath IMPORT
    (* proc *)  abs, conj, scalarMult;

FROM MatExtra IMPORT
    (* proc *)  ReQRFactor, ReLQFactor, CxQRFactor, CxLQFactor,
                ReHouseholder, CxHouseholder,
                ReToBidiag;

FROM Mat IMPORT
    (* type *)  ArrayPtr, CxArrayPtr,
    (* proc *)  Write, CxWrite, NewArray, NewCxArray, DisposeArray,
                DisposeCxArray,
                ReCopy, CxCopy, Unit, CxUnit, Sub, CxSub, Mul, CxMul,
                Random, CxRandom;

FROM MiscM2 IMPORT
    (* proc *)  SelectWindow, WriteChar, WriteString, WriteCard, WriteLn,
                WriteLongReal, WriteLongComplex, PressAnyKey;

(************************************************************************)

TYPE
    Re1x4 = ARRAY [1..1], [1..4] OF LONGREAL;
    Re2x2 = ARRAY [1..2], [1..2] OF LONGREAL;
    Re3x3 = ARRAY [1..3], [1..3] OF LONGREAL;
    Cx1x2 = ARRAY [1..1], [1..2] OF LONGCOMPLEX;

CONST
    small = 1.0E-15;
    CxZero = CMPLX (0.0, 0.0);

(************************************************************************)
(*                          PLQ FACTORISATION                           *)
(************************************************************************)

PROCEDURE RePLQFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                         VAR (*OUT*) P, L, Q: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Factors an rxc matrix A as A = P x L x Q, where:                 *)
    (*      P is a rxq matrix with P*P=I                                *)
    (*      L is a square lower triangular qxq matrix                   *)
    (*      Q is a qxc matrix Q with QQ*=I.                             *)
    (* If A<>0 the integer q, which is returned as the function result, *)
    (* is equal to the rank of A.  If A=0 then we return q=0, and the   *)
    (* result for L is a 1x1 zero matrix.                               *)
    (*                                                                  *)
    (* Assumption: the caller has declared P, L and Q to be large       *)
    (* enough to hold the result. Note that q<=min(r,c), which tells    *)
    (* the caller how much space to reserve even though q is not known  *)
    (* in advance.  Unused rows and columns are left unaltered, except  *)
    (* for some extra 0 entries in L when A = 0.                        *)

    VAR R: ArrayPtr;
        q0, rank, j: CARDINAL;

    BEGIN
        q0 := r;
        IF c < q0 THEN
            q0 := c;
        END (*IF*);
        R := NewArray (q0, c);
        rank := ReQRFactor (A, r, c, P, R^);
        IF rank = 0 THEN
            (* Arbitrary choice for Q. *);
            Q[0,0] := 1.0;
            FOR j := 1 TO c-1 DO
                Q[0,j] := 0.0;
            END (*FOR*);
        ELSE
            rank := ReLQFactor (R^, rank, c, L, Q);
        END (*IF*);
        DisposeArray (R, q0, c);
        RETURN rank;
    END RePLQFactor;

(************************************************************************)

PROCEDURE CxPLQFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                         VAR (*OUT*) P, L, Q: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Factors an rxc matrix A as A = P x L x Q, where:                 *)
    (*      P is a rxq matrix with P*P=I                                *)
    (*      L is a square lower triangular qxq matrix                   *)
    (*      Q is a qxc matrix Q with QQ*=I.                             *)
    (* If A<>0 the integer q, which is returned as the function result, *)
    (* is equal to the rank of A.  If A=0 then we return q=0, and the   *)
    (* result for L is a 1x1 zero matrix.                               *)
    (*                                                                  *)
    (* Assumption: the caller has declared P, L and Q to be large       *)
    (* enough to hold the result. Note that q<=min(r,c), which tells    *)
    (* the caller how much space to reserve even though q is not known  *)
    (* in advance.  Unused rows and columns are left unaltered, except  *)
    (* for some extra 0 entries in L when A = 0.                        *)

    VAR R: CxArrayPtr;
        q0, rank, j: CARDINAL;

    BEGIN
        q0 := r;
        IF c < q0 THEN
            q0 := c;
        END (*IF*);
        R := NewCxArray (q0, c);
        rank := CxQRFactor (A, r, c, P, R^);
        IF rank = 0 THEN
            (* Arbitrary choice for Q. *);
            Q[0,0] := CMPLX (1.0, 0.0);
            FOR j := 1 TO c-1 DO
                Q[0,j] := CxZero;
            END (*FOR*);
        ELSE
            rank := CxLQFactor (R^, rank, c, L, Q);
        END (*IF*);
        DisposeCxArray (R, q0, c);
        RETURN rank;
    END CxPLQFactor;

(************************************************************************)
(*                      Repeated PLQ FACTORISATION                      *)
(************************************************************************)

PROCEDURE MaxElt (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL): LONGREAL;

    (* Maximum absolute value in the array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO rows-1 DO
            FOR j := 0 TO cols-1 DO
                val := ABS(A[i,j]);
                IF val > max THEN
                    max := val;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END MaxElt;

(************************************************************************)

PROCEDURE MaxOffDiag (A: ARRAY OF ARRAY OF LONGREAL;  N: CARDINAL): LONGREAL;

    (* Maximum off-diagonal absolute value in a square array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO N-1 DO
            FOR j := 0 TO N-1 DO
                IF j <> i THEN
                    val := ABS(A[i,j]);
                    IF val > max THEN
                        max := val;
                    END (*IF*);
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END MaxOffDiag;

(************************************************************************)

PROCEDURE CxMaxElt (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL): LONGREAL;

    (* Maximum absolute value in the array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO rows-1 DO
            FOR j := 0 TO cols-1 DO
                val := abs(A[i,j]);
                IF val > max THEN
                    max := val;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END CxMaxElt;

(************************************************************************)

PROCEDURE CxMaxOffDiag (A: ARRAY OF ARRAY OF LONGCOMPLEX;  N: CARDINAL): LONGREAL;

    (* Maximum off-diagonal absolute value in a square array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO N-1 DO
            FOR j := 0 TO N-1 DO
                IF j <> i THEN
                    val := abs(A[i,j]);
                    IF val > max THEN
                        max := val;
                    END (*IF*);
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END CxMaxOffDiag;

(************************************************************************)

PROCEDURE RePLQtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of PLQ factorisation. *)

    CONST
        fieldsize = 10;

    VAR P, L, Q, PL, PLQ, Diff: ArrayPtr;
        q0, rank, count: CARDINAL;

    (********************************************************************)

    PROCEDURE DumpStuff;

        (* Writes diagnostic information to screen. *)

        BEGIN
            (*
            WriteString ("P =");  WriteLn;
            Write (P^, rows, rank, fieldsize);
            *)
            WriteString ("L =");  WriteLn;
            Write (L^, rank, rank, fieldsize);
            (*
            WriteString ("Q =");  WriteLn;
            Write (Q^, rank, cols, fieldsize);
            *)

            (*
            PL := NewArray (rows, rank);
            Mul (P^, L^, rows, rank, rank, PL^);
            PLQ := NewArray (rows, cols);
            Mul (PL^, Q^, rows, rank, cols, PLQ^);
            Diff := NewArray (rows, cols);
            Sub (A, PLQ^, rows, cols, Diff^);
            WriteString ("PL =");  WriteLn;
            Write (PL^, rows, rank, fieldsize);
            WriteString ("PLQ =");  WriteLn;
            Write (PLQ^, rows, cols, fieldsize);
            WriteString ("A - PLQ =");  WriteLn;
            Write (Diff^, rows, cols, fieldsize);
            WriteString ("Maximum element of A-PLQ = ");
            WriteLongReal (MaxElt(Diff^,rows,cols), fieldsize);
            WriteLn;

            DisposeArray (Diff, rows, cols);
            DisposeArray (PLQ, rows, cols);
            DisposeArray (PL, rows, rank);
            *)

            PressAnyKey;

        END DumpStuff;

    (********************************************************************)

    CONST
        MaxIterations = 30;
        small = 1.0E-15;

    VAR oldP, oldL, oldQ, newP, newQ: ArrayPtr;
        dummy: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewArray (rows, q0);
        L := NewArray (q0, q0);
        Q := NewArray (q0, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        Write (A, rows, cols, fieldsize);

        (* Initial factorisation. *)

        count := 0;
        rank := RePLQFactor (A, rows, cols, P^, L^, Q^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;

        DumpStuff;

        oldP := NewArray (rows, rank);
        oldL := NewArray (rank, rank);
        oldQ := NewArray (rank, cols);
        newP := NewArray (rank, rank);
        newQ := NewArray (rank, rank);

        WHILE (count < MaxIterations) AND (MaxOffDiag(L^,rank) > small) DO
            ReCopy (P^, rows, rank, oldP^);
            ReCopy (L^, rank, rank, oldL^);
            ReCopy (Q^, rank, cols, oldQ^);

            dummy := RePLQFactor (oldL^, rank, rank, newP^, L^, newQ^);
            Mul (oldP^, newP^, rows, rank, rank, P^);
            Mul (newQ^, oldQ^, rank, rank, cols, Q^);

            DumpStuff;

            INC (count);

        END (*WHILE*);

        DisposeArray (newQ, rank, cols);
        DisposeArray (newP, rows, rank);
        DisposeArray (oldQ, rank, cols);
        DisposeArray (oldL, rank, rank);
        DisposeArray (oldP, rows, rank);

        DisposeArray (Q, q0, cols);
        DisposeArray (L, q0, q0);
        DisposeArray (P, rows, q0);

    END RePLQtest;

(************************************************************************)

PROCEDURE CxPLQtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of PLQ factorisation. *)

    CONST
        fieldsize = 8;

    VAR P, L, Q, PL, PLQ, Diff: CxArrayPtr;
        q0, rank, count: CARDINAL;

    (********************************************************************)

    PROCEDURE DumpStuff;

        (* Writes diagnostic information to screen. *)

        BEGIN
            (*
            WriteString ("P =");  WriteLn;
            CxWrite (P^, rows, rank, fieldsize);
            *)
            WriteString ("L =");  WriteLn;
            CxWrite (L^, rank, rank, fieldsize);
            (*
            WriteString ("Q =");  WriteLn;
            CxWrite (Q^, rank, cols, fieldsize);
            *)

            (*
            PL := NewCxArray (rows, rank);
            CxMul (P^, L^, rows, rank, rank, PL^);
            PLQ := NewCxArray (rows, cols);
            CxMul (PL^, Q^, rows, rank, cols, PLQ^);
            Diff := NewCxArray (rows, cols);
            CxSub (A, PLQ^, rows, cols, Diff^);
            WriteString ("PL =");  WriteLn;
            CxWrite (PL^, rows, rank, fieldsize);
            WriteString ("PLQ =");  WriteLn;
            CxWrite (PLQ^, rows, cols, fieldsize);
            WriteString ("A - PLQ =");  WriteLn;
            CxWrite (Diff^, rows, cols, fieldsize);
            WriteString ("Maximum element of A-PLQ = ");
            WriteLongReal (CxMaxElt(Diff^,rows,cols), fieldsize);
            WriteLn;

            DisposeCxArray (Diff, rows, cols);
            DisposeCxArray (PLQ, rows, cols);
            DisposeCxArray (PL, rows, rank);
            *)

            PressAnyKey;

        END DumpStuff;

    (********************************************************************)

    CONST
        MaxIterations = 30;
        small = 1.0E-15;

    VAR oldP, oldL, oldQ, newP, newQ: CxArrayPtr;
        dummy: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewCxArray (rows, q0);
        L := NewCxArray (q0, q0);
        Q := NewCxArray (q0, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);

        (* Initial factorisation. *)

        count := 0;
        rank := CxPLQFactor (A, rows, cols, P^, L^, Q^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;

        DumpStuff;

        oldP := NewCxArray (rows, rank);
        oldL := NewCxArray (rank, rank);
        oldQ := NewCxArray (rank, cols);
        newP := NewCxArray (rank, rank);
        newQ := NewCxArray (rank, rank);

        WHILE (count < MaxIterations) AND (CxMaxOffDiag(L^,rank) > small) DO
            CxCopy (P^, rows, rank, oldP^);
            CxCopy (L^, rank, rank, oldL^);
            CxCopy (Q^, rank, cols, oldQ^);

            dummy := CxPLQFactor (oldL^, rank, rank, newP^, L^, newQ^);
            CxMul (oldP^, newP^, rows, rank, rank, P^);
            CxMul (newQ^, oldQ^, rank, rank, cols, Q^);

            DumpStuff;

            INC (count);

        END (*WHILE*);

        DisposeCxArray (newQ, rank, cols);
        DisposeCxArray (newP, rows, rank);
        DisposeCxArray (oldQ, rank, cols);
        DisposeCxArray (oldL, rank, rank);
        DisposeCxArray (oldP, rows, rank);

        DisposeCxArray (Q, q0, cols);
        DisposeCxArray (L, q0, q0);
        DisposeCxArray (P, rows, q0);

    END CxPLQtest;

(************************************************************************)

PROCEDURE DoPLQtests;

    (* Tests of PLQ factorisation. *)

    VAR Rand: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);
        (*
        RePLQtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4);
        RePLQtest (Re2x2 {{1.0, 0.0},
                         {0.0, 1.0}}, 2, 2);
        *)
        RePLQtest (Re2x2 {{1.0, 1.0},
                         {1.0, 1.0}}, 2, 2);
        (*
        RePLQtest (Re3x3 {{0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0}}, 3, 3);
        *)
        r := 6;  c := 6;
        Rand := NewArray (r, c);
        Random (Rand^, r, c);
        RePLQtest (Rand^, r, c);
        DisposeArray (Rand, r, c);

        (* The complex case.  *)

        (*
        r := 2;  c := 2;
        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxPLQtest (B^, r, c);
        DisposeCxArray (B, r, c);
        *)

    END DoPLQtests;

(************************************************************************)

PROCEDURE SVDsteps (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Looking at the convergence rate of a SVD algorithm. *)

    CONST
        fieldsize = 9;


    CONST
        MaxIterations = 30;
        small = 1.0E-15;

    VAR P, B, Q, oldB: ArrayPtr;
        q0, rank, count, dummy: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewArray (rows, q0);
        B := NewArray (q0, q0);
        Q := NewArray (q0, cols);
        WriteString ("------- steps in SVD ---------");  WriteLn;
        WriteString ("Original matrix");  WriteLn;
        Write (A, rows, cols, fieldsize);

        (* Initial factorisation. *)

        rank := RePLQFactor (A, rows, cols, P^, B^, Q^);
        WriteString ("After first PLQ factorisation");  WriteLn;
        Write (B^, rows, cols, fieldsize);
        oldB := NewArray (rank, rank);
        ReCopy (B^, rank, rank, oldB^);

        (* Convert to bidiagonal (upper).  *)

        dummy := ReToBidiag (oldB^, rank, rank, Q^, B^, P^);
        ReCopy (B^, rank, rank, oldB^);
        WriteString ("After bidiagonal conversion");  WriteLn;
        Write (B^, rows, cols, fieldsize);
        PressAnyKey;

        count := 0;
        WHILE (count <= MaxIterations) AND (MaxOffDiag(B^,rank) > small) DO

            dummy := ReLQFactor (oldB^, rank, rank, B^, Q^);
            ReCopy (B^, rank, rank, oldB^);
            WriteString ("After LQ conversion");  WriteLn;
            Write (B^, rows, cols, fieldsize);

            dummy := ReQRFactor (oldB^, rank, rank, Q^, B^);
            ReCopy (B^, rank, rank, oldB^);
            WriteString ("After QR conversion");  WriteLn;
            Write (B^, rows, cols, fieldsize);

            PressAnyKey;

            INC (count);

        END (*WHILE*);

        DisposeArray (Q, q0, cols);
        DisposeArray (oldB, rank, rank);
        DisposeArray (B, q0, q0);
        DisposeArray (P, rows, q0);

    END SVDsteps;

(************************************************************************)

PROCEDURE SVDstepTest;

    (********************************************************************)

    PROCEDURE Rand (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGREAL;
                        rows, cols: CARDINAL);

        (* Like Random, but with bigger numbers. *)

        CONST scale = 50.0;

        VAR i, j: CARDINAL;

        BEGIN
            Random (A, rows, cols);
            FOR i := 0 TO rows-1 DO
                FOR j := 0 TO rows-1 DO
                    A[i,j] := scale*A[i,j];
                END (*FOR*);
            END (*FOR*);
        END Rand;

    (********************************************************************)

    VAR A: ArrayPtr;
        r, c: CARDINAL;

    BEGIN
        r := 7;  c := r;
        A := NewArray (r, c);
        Rand (A^, r, c);
        SVDsteps (A^, r, c);
        DisposeArray (A, r, c);
    END SVDstepTest;

(************************************************************************)
(*                           HOUSEHOLDER TEST                           *)
(************************************************************************)

PROCEDURE DoReHouseholderTest (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGREAL;
                                r, c, r0, c0: CARDINAL);

    CONST fieldsize = 8;

    VAR oldA, H, HA, Diff: ArrayPtr;
        max, newval: LONGREAL;
        i, j: CARDINAL;

    BEGIN
        WriteString ("----------------");  WriteLn;
        WriteString ("Testing real Householder transformation on A =");
        WriteLn;
        Write (A, r, c, fieldsize);
        oldA := NewArray (r, c);
        ReCopy (A, r, c, oldA^);
        H := NewArray (r, r);
        ReHouseholder (A, r, c, r0, c0, H^);
        WriteString ("The updated A is");  WriteLn;
        Write (A, r, c, fieldsize);
        WriteString ("The Householder matrix is");  WriteLn;
        Write (H^, r, r, fieldsize);

        (* Calculate A - H times oldA, as a check. *);

        HA := NewArray (r, c);
        Diff := NewArray (r, c);
        Mul (H^, oldA^, r, r, c, HA^);
        Sub (A, HA^, r, c, Diff^);
        max := 0.0;
        FOR i := 0 TO r-1 DO
            FOR j := 0 TO c-1 DO
                newval := ABS(Diff^[i,j]);
                IF newval > max THEN
                    max := newval;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        WriteString ("Largest element of A minus (H times old A) is ");
        WriteLongReal (max, fieldsize);  WriteLn;
        DisposeArray (Diff, r, c);
        DisposeArray (HA, r, c);

        DisposeArray (oldA, r, c);
        DisposeArray (H, r, r);
        PressAnyKey;

    END DoReHouseholderTest;

(************************************************************************)

PROCEDURE DoCxHouseholderTest (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGCOMPLEX;
                                r, c, r0, c0: CARDINAL);

    CONST fieldsize = 8;

    VAR oldA, H, HA, Diff: CxArrayPtr;
        max, newval: LONGREAL;
        i, j: CARDINAL;

    BEGIN
        WriteString ("----------------");  WriteLn;
        WriteString ("Testing complex Householder transformation on A =");
        WriteLn;
        CxWrite (A, r, c, fieldsize);
        oldA := NewCxArray (r, c);
        CxCopy (A, r, c, oldA^);
        H := NewCxArray (r, r);
        CxHouseholder (A, r, c, r0, c0, H^);
        WriteString ("The updated A is");  WriteLn;
        CxWrite (A, r, c, fieldsize);
        WriteString ("The Householder matrix is");  WriteLn;
        CxWrite (H^, r, r, fieldsize);

        (* Calculate A - H times oldA, as a check. *);

        HA := NewCxArray (r, c);
        Diff := NewCxArray (r, c);
        CxMul (H^, oldA^, r, r, c, HA^);
        CxSub (A, HA^, r, c, Diff^);
        max := 0.0;
        FOR i := 0 TO r-1 DO
            FOR j := 0 TO c-1 DO
                newval := abs(Diff^[i,j]);
                IF newval > max THEN
                    max := newval;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        WriteString ("Largest element of A minus (H times old A) is ");
        WriteLongReal (max, fieldsize);  WriteLn;
        DisposeCxArray (Diff, r, c);
        DisposeCxArray (HA, r, c);

        DisposeCxArray (oldA, r, c);
        DisposeCxArray (H, r, r);
        PressAnyKey;
    END DoCxHouseholderTest;

(************************************************************************)

PROCEDURE DoHouseholderTests;

    VAR A: CxArrayPtr;
        B: ArrayPtr;
        r, c: CARDINAL;

    BEGIN
        r := 6;  c := 3;
        B := NewArray (r, c);
        Random (B^, r, c);
        DoReHouseholderTest (B^, r, c, 0, 0);
        DoReHouseholderTest (B^, r, c, 1, 1);
        DoReHouseholderTest (B^, r, c, 2, 2);
        DisposeArray (B, r, c);

        (* Some rank-deficient cases. *)

        r := 2;  c := 4;
        B := NewArray (r, c);
        B^[0,0] := 5.0;  B^[0,1] := 2.0;  B^[0,2] := 3.0;  B^[0,3] := 4.0;
        B^[1,0] := 5.0;  B^[1,1] := 2.0;  B^[1,2] := 3.0;  B^[1,3] := 4.0;
        DoReHouseholderTest (B^, r, c, 0, 0);
        DoReHouseholderTest (B^, r, c, 1, 1);
        DisposeArray (B, r, c);

        r := 4;  c := 2;
        B := NewArray (r, c);
        B^[0,0] := 5.0;  B^[0,1] := 5.0;
        B^[1,0] := 7.0;  B^[1,1] := 7.0;
        B^[2,0] := 1.0;  B^[2,1] := 1.0;
        B^[3,0] := 1.0;  B^[3,1] := 1.0;
        DoReHouseholderTest (B^, r, c, 0, 0);
        DoReHouseholderTest (B^, r, c, 1, 1);
        DisposeArray (B, r, c);

        (* Now for some complex tests. *)

        (**)
        r := 2;  c := 2;
        A := NewCxArray (r, c);
        A^[0,0] := CMPLX(1.0,0.0);  A^[0,1] := CMPLX(2.0,0.0);
        A^[1,0] := CMPLX(1.0,0.0);  A^[1,1] := CMPLX(1.0,0.0);
        DoCxHouseholderTest (A^, r, c, 0, 0);
        DisposeCxArray (A, r, c);
        (**)

        (**)
        r := 2;  c := 2;
        A := NewCxArray (r, c);
        A^[0,0] := CMPLX(3.0,0.0);  A^[0,1] := CMPLX(2.0,0.0);
        A^[1,0] := CMPLX(4.0,0.0);  A^[1,1] := CMPLX(3.0,0.0);
        DoCxHouseholderTest (A^, r, c, 0, 0);
        DisposeCxArray (A, r, c);
        (**)

        r := 2;  c := 2;
        A := NewCxArray (r, c);
        A^[0,0] := CMPLX(0.9,0.4);  A^[0,1] := CMPLX(0.1,0.9);
        A^[1,0] := CMPLX(0.4,0.1);  A^[1,1] := CMPLX(0.9,0.0);
        DoCxHouseholderTest (A^, r, c, 0, 0);
        DisposeCxArray (A, r, c);
        (**)

        (**)
        r := 6;  c := 3;
        A := NewCxArray (r, c);
        CxRandom (A^, r, c);
        DoCxHouseholderTest (A^, r, c, 0, 0);
        DoCxHouseholderTest (A^, r, c, 1, 0);
        DisposeCxArray (A, r, c);
        (**)

    END DoHouseholderTests;

(************************************************************************)
(*                             MAIN PROGRAM                             *)
(************************************************************************)

BEGIN
    SVDstepTest;
    (**)
    DoPLQtests;
    DoHouseholderTests;
    (**)
END PLQTest.

