MODULE MatExtraTest;

        (********************************************************)
        (*                                                      *)
        (*              Test of MatExtra module                 *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        12 April 2016                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


FROM MatExtra IMPORT
    (* proc *)  ReQRFactor, ReFullQRFactor,
                CxQRFactor, CxFullQRFactor,
                HHReQRFactor, HHCxQRFactor,
                ReLQFactor, CxLQFactor,
                ReLTinverse, CxLTinverse,
                RePseudoInverse, CxPseudoInverse,
                ReCholesky, CxCholesky,
                ReSVD, CxSVD;

FROM MiscM2 IMPORT
    (* proc *)  SelectWindow, WriteString, WriteCard, WriteLJCard,
                WriteLongReal, WriteLn, PressAnyKey;

FROM Mat IMPORT
    (* type *)  ArrayPtr, CxArrayPtr,
    (* proc *)  Write, CxWrite, NewArray, DisposeArray, NewCxArray,
                DisposeCxArray, Transpose, Adjoint,
                Sub, Mul, CxSub, CxMul, Random, CxRandom;

FROM LongComplexMath IMPORT
    (* proc *)  abs, conj;

(************************************************************************)

TYPE
    Re1x4 = ARRAY [1..1], [1..4] OF LONGREAL;
    Re2x2 = ARRAY [1..2], [1..2] OF LONGREAL;
    Re3x3 = ARRAY [1..3], [1..3] OF LONGREAL;
    Cx1x2 = ARRAY [1..1], [1..2] OF LONGCOMPLEX;

CONST
    CxZero = CMPLX (0.0, 0.0);

(************************************************************************)

PROCEDURE MaxElt (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL): LONGREAL;

    (* Maximum absolute value in the array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO rows-1 DO
            FOR j := 0 TO cols-1 DO
                val := ABS(A[i,j]);
                IF val > max THEN
                    max := val;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END MaxElt;

(************************************************************************)

PROCEDURE MaxCxElt (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL): LONGREAL;

    (* Maximum absolute value in the array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO rows-1 DO
            FOR j := 0 TO cols-1 DO
                val := abs(A[i,j]);
                IF val > max THEN
                    max := val;
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END MaxCxElt;

(************************************************************************)
(*                          QR FACTORISATION                            *)
(************************************************************************)

PROCEDURE ReQRtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 10;

    VAR Q, R, QR: ArrayPtr;
        rank: CARDINAL;

    BEGIN
        Q := NewArray (rows, rows);
        R := NewArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        Write (A, rows, cols, fieldsize);
        rank := ReQRFactor (A, rows, cols, Q^, R^);
        WriteString ("rank(A)=");  WriteLJCard (rank);  WriteLn;
        WriteString ("Q =");  WriteLn;
        Write (Q^, rows, rank, fieldsize);
        WriteString ("R =");  WriteLn;
        Write (R^, rank, cols, fieldsize);

        QR := NewArray (rows, cols);
        Mul (Q^, R^, rows, rank, cols, QR^);
        WriteString ("QR =");  WriteLn;
        Write (QR^, rows, cols, fieldsize);

        PressAnyKey;

        IF rank < rows THEN

            (* Do a full QR factorisation as well. *)

            WriteString ("Full QR factorisation:");  WriteLn;
            ReFullQRFactor (A, rows, cols, Q^, R^);
            WriteString ("Q =");  WriteLn;
            Write (Q^, rows, rows, fieldsize);
            WriteString ("R =");  WriteLn;
            Write (R^, rows, cols, fieldsize);

            QR := NewArray (rows, cols);
            Mul (Q^, R^, rows, rank, cols, QR^);
            WriteString ("QR =");  WriteLn;
            Write (QR^, rows, cols, fieldsize);
            PressAnyKey;
        END (*IF*);

        DisposeArray (QR, rows, cols);
        DisposeArray (R, rows, cols);
        DisposeArray (Q, rows, rows);

    END ReQRtest;

(************************************************************************)

PROCEDURE HHReQRtest (A: ARRAY OF ARRAY OF LONGREAL;
                            rows, cols: CARDINAL;  displayA: BOOLEAN);

    (* Test of QR factorisation comparing the Gram-Schmidt approach     *)
    (* with the Householder reflection method.                          *)

    TYPE
        MethodType = (GS, HH);
        LabelType = ARRAY MethodType OF ARRAY [0..3] OF CHAR;

    CONST
        fieldsize = 10;
        label = LabelType {"GS: ", "HH: "};

    VAR Q, R, Product, Diff: ArrayPtr;
        rank, i, j, k: CARDINAL;
        val: LONGREAL;
        method: MethodType;

    BEGIN
        R := NewArray (rows, cols);
        Q := NewArray (rows, rows);
        WriteString ("----------------");  WriteLn;
        IF displayA THEN
            WriteString ("Comparing two approaches to QR factorisation");  WriteLn;
            WriteString ("Factoring matrix");  WriteLn;
            Write (A, rows, cols, fieldsize);
        END (*IF*);
        FOR method := MIN(MethodType) TO MAX(MethodType) DO
            CASE method OF
                GS:  rank := ReQRFactor (A, rows, cols, Q^, R^);
              | HH:  rank := HHReQRFactor (A, rows, cols, Q^, R^);
            END (*CASE*);
            WriteString (label[method]);
            WriteString ("real, ");
            WriteLJCard (rows);  WriteString (" x ");  WriteLJCard (cols);
            WriteString (", rank(A)=");  WriteLJCard (rank);  WriteLn;

            (* If rank = 0, the array dimensions are as if rank = 1.  *)

            IF rank = 0 THEN
                rank := 1;
            END (*IF*);

            (* Calculate A - QR  *)

            Diff := NewArray (rows, cols);
            Product := NewArray (rows, cols);
            Mul (Q^, R^, rows, rank, cols, Product^);
            Sub (A, Product^, rows, cols, Diff^);
            WriteString (label[method]);
            WriteString ("Maximum element in A - QR = ");
            WriteLongReal (MaxElt(Diff^,rows,cols), fieldsize);
            WriteLn;
            DisposeArray (Diff, rows, cols);
            DisposeArray (Product, rows, cols);

            (* Calculate Q*Q - I *)

            Product := NewArray (rank, rank);
            FOR i := 0 TO rank-1 DO
                FOR j := 0 TO rank-1 DO
                    val := 0.0;
                    FOR k := 0 TO rows-1 DO
                        val := val + Q^[k,i] * Q^[k,j]
                    END (*FOR*);
                    IF i = j THEN
                        val := val - 1.0;
                    END (*IF*);
                    Product^[i,j] := val;
                END (*FOR*);
            END (*FOR*);
            WriteString (label[method]);
            WriteString ("Maximum element in Q*Q - I = ");
            WriteLongReal (MaxElt(Product^,rank,rank), fieldsize);
            WriteLn;
            DisposeArray (Product, rank, rank);
        END (*FOR*);

        PressAnyKey;

        DisposeArray (R, rows, cols);
        DisposeArray (Q, rows, rows);

    END HHReQRtest;

(************************************************************************)

PROCEDURE CxQRtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 8;

    VAR Q, R, QR: CxArrayPtr;
        rank: CARDINAL;

    BEGIN
        Q := NewCxArray (rows, rows);
        R := NewCxArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        rank := CxQRFactor (A, rows, cols, Q^, R^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Q =");  WriteLn;
        CxWrite (Q^, rows, rank, fieldsize);
        WriteString ("R =");  WriteLn;
        CxWrite (R^, rank, cols, fieldsize);

        QR := NewCxArray (rows, cols);
        CxMul (Q^, R^, rows, rank, cols, QR^);
        WriteString ("QR =");  WriteLn;
        CxWrite (QR^, rows, cols, fieldsize);

        PressAnyKey;

        IF rank < rows THEN

            (* Do a full QR factorisation as well. *)

            WriteString ("Full QR factorisation:");  WriteLn;
            CxFullQRFactor (A, rows, cols, Q^, R^);
            WriteString ("Q =");  WriteLn;
            CxWrite (Q^, rows, rows, fieldsize);
            WriteString ("R =");  WriteLn;
            CxWrite (R^, rows, cols, fieldsize);

            QR := NewCxArray (rows, cols);
            CxMul (Q^, R^, rows, rank, cols, QR^);
            WriteString ("QR =");  WriteLn;
            CxWrite (QR^, rows, cols, fieldsize);
            PressAnyKey;
        END (*IF*);

        DisposeCxArray (QR, rows, cols);
        DisposeCxArray (R, rows, cols);
        DisposeCxArray (Q, rows, rows);

    END CxQRtest;

(************************************************************************)

PROCEDURE HHCxQRtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;
                            rows, cols: CARDINAL;  displayA: BOOLEAN);

    (* Test of QR factorisation comparing the Gram-Schmidt approach     *)
    (* with the Householder reflection method.                          *)

    TYPE
        MethodType = (GS, HH);
        LabelType = ARRAY MethodType OF ARRAY [0..3] OF CHAR;

    CONST
        fieldsize = 10;
        label = LabelType {"GS: ", "HH: "};

    VAR Q, R, Product, Diff: CxArrayPtr;
        rank, i, j, k: CARDINAL;
        val: LONGCOMPLEX;
        method: MethodType;

    BEGIN
        Q := NewCxArray (rows, rows);
        R := NewCxArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        IF displayA THEN
            WriteString ("Comparing two kinds of QR factorisation");  WriteLn;
            WriteString ("Factoring matrix");  WriteLn;
            CxWrite (A, rows, cols, fieldsize);
        END (*IF*);
        FOR method := MIN(MethodType) TO MAX(MethodType) DO
            CASE method OF
                GS:  rank := CxQRFactor (A, rows, cols, Q^, R^);
              | HH:  rank := HHCxQRFactor (A, rows, cols, Q^, R^);
            END (*CASE*);
            WriteString (label[method]);
            WriteString ("complex, ");
            WriteLJCard (rows);  WriteString (" x ");  WriteLJCard (cols);
            WriteString (", rank(A)=");  WriteLJCard (rank);  WriteLn;

            (* If rank = 0, the array dimensions are as if rank = 1.  *)

            IF rank = 0 THEN
                rank := 1;
            END (*IF*);

            (* Calculate A - QR  *)

            Diff := NewCxArray (rows, cols);
            Product := NewCxArray (rows, cols);
            CxMul (Q^, R^, rows, rank, cols, Product^);
            CxSub (A, Product^, rows, cols, Diff^);
            WriteString (label[method]);
            WriteString ("Maximum element in A - QR = ");
            WriteLongReal (MaxCxElt(Diff^,rows,cols), fieldsize);
            WriteLn;
            DisposeCxArray (Diff, rows, cols);
            DisposeCxArray (Product, rows, cols);

            (* Calculate Q*Q - I *)

            Product := NewCxArray (rank, rank);
            FOR i := 0 TO rank-1 DO
                FOR j := 0 TO rank-1 DO
                    val := CxZero;
                    FOR k := 0 TO rows-1 DO
                        val := val + conj(Q^[k,i]) * Q^[k,j]
                    END (*FOR*);
                    IF i = j THEN
                        val := val - CMPLX (1.0, 0.0);
                    END (*IF*);
                    Product^[i,j] := val;
                END (*FOR*);
            END (*FOR*);
            WriteString (label[method]);
            WriteString ("Maximum element in Q*Q - I = ");
            WriteLongReal (MaxCxElt(Product^,rank,rank), fieldsize);
            WriteLn;
            DisposeCxArray (Product, rank, rank);
        END (*FOR*);

        PressAnyKey;

        DisposeCxArray (R, rows, cols);
        DisposeCxArray (Q, rows, rows);

    END HHCxQRtest;

(************************************************************************)

PROCEDURE DoQRtests;

    (* Tests of QR factorisation. *)

    VAR Rand: ArrayPtr;
        B: CxArrayPtr;

    BEGIN
        SelectWindow (0);
        (**)
        ReQRtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4);
        ReQRtest (Re2x2 {{1.0, 0.0},
                         {0.0, 1.0}}, 2, 2);
        ReQRtest (Re2x2 {{1.0, 1.0},
                         {1.0, 1.0}}, 2, 2);
        ReQRtest (Re3x3 {{0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0}}, 3, 3);
        Rand := NewArray (5, 3);
        Random (Rand^, 5, 3);
        ReQRtest (Rand^, 5, 3);
        DisposeArray (Rand, 5, 3);
        (**)

        (* A complex example.   *)

        B := NewCxArray (2, 2);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxQRtest (B^, 2, 2);
        DisposeCxArray (B, 2, 2);
        (**)

    END DoQRtests;

(************************************************************************)

PROCEDURE DoHHQRtests;

    (* Tests of QR factorisation, including comparison of the   *)
    (* Gram-Schmidt and Householder cases.                      *)

    VAR A: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    (********************************************************************)

    PROCEDURE MiscTests;

        (* Test suite 1: miscellaneous cases. *)

        BEGIN
            (**)
            HHReQRtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4, TRUE);
            HHReQRtest (Re2x2 {{0.0, 0.0},
                             {0.0, 0.0}}, 2, 2, TRUE);
            HHReQRtest (Re2x2 {{1.0, 0.0},
                             {0.0, 1.0}}, 2, 2, TRUE);
            (**)
            HHReQRtest (Re2x2 {{1.0, 1.0},
                             {1.0, 1.0}}, 2, 2, TRUE);
            (**)
            HHReQRtest (Re3x3 {{0.0, 1.0, 1.0},
                             {0.0, 1.0, 1.0},
                             {0.0, 1.0, 1.0}}, 3, 3, TRUE);
            (**)
            r := 6;  c := 3;
            A := NewArray (r, c);
            Random (A^, r, c);
            HHReQRtest (A^, r, c, TRUE);
            DisposeArray (A, r, c);

            r := 3;  c := 6;
            A := NewArray (r, c);
            Random (A^, r, c);
            HHReQRtest (A^, r, c, TRUE);
            DisposeArray (A, r, c);

            (**)

            (* Complex examples.   *)

            (**)
            B := NewCxArray (2, 2);
            B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
            B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
            HHCxQRtest (B^, 2, 2, TRUE);
            DisposeCxArray (B, 2, 2);
            (**)
            (**)
            r := 20;  c := 4;
            B := NewCxArray (r, c);
            CxRandom (B^, r, c);
            HHCxQRtest (B^, r, c, TRUE);
            DisposeCxArray (B, r, c);
            (**)
        END MiscTests;

    (********************************************************************)

    PROCEDURE RandTests;

        (* Test suite 2: putting more pressure on precision checks. *)

        (****************************************************************)

        PROCEDURE Re (r, c: CARDINAL);

            (* Test on real random array. *)

            BEGIN
                A := NewArray (r, c);
                Random (A^, r, c);
                HHReQRtest (A^, r, c, FALSE);
                DisposeArray (A, r, c);
            END Re;

        (****************************************************************)

        PROCEDURE Cx (r, c: CARDINAL);

            (* Test on complex random array. *)

            BEGIN
                B := NewCxArray (r, c);
                CxRandom (B^, r, c);
                HHCxQRtest (B^, r, c, FALSE);
                DisposeCxArray (B, r, c);
            END Cx;

        (****************************************************************)

        BEGIN
            (**)
            Re (20, 10);
            Re (10, 20);
            Re (50, 50);
            Re (4, 200);
            Re (100, 4);
            (**)
            Cx (20, 10);
            Cx (10, 20);
            Cx (40, 50);
            Cx (4, 200);
            Cx (50, 4);
        END RandTests;

    (********************************************************************)

    BEGIN
        SelectWindow (0);
        MiscTests;
        RandTests;
    END DoHHQRtests;

(************************************************************************)
(*                          LQ FACTORISATION                            *)
(************************************************************************)

PROCEDURE ReLQtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of LQ factorisation. *)

    CONST
        fieldsize = 10;

    VAR L, Q, LQ: ArrayPtr;
        rank: CARDINAL;

    BEGIN
        L := NewArray (rows, rows);
        Q := NewArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        Write (A, rows, cols, fieldsize);
        rank := ReLQFactor (A, rows, cols, L^, Q^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("L =");  WriteLn;
        Write (L^, rows, rank, fieldsize);
        WriteString ("Q =");  WriteLn;
        Write (Q^, rank, cols, fieldsize);

        LQ := NewArray (rows, cols);
        Mul (L^, Q^, rows, rank, cols, LQ^);
        WriteString ("LQ =");  WriteLn;
        Write (LQ^, rows, cols, fieldsize);

        PressAnyKey;

        DisposeArray (LQ, rows, cols);
        DisposeArray (Q, rows, cols);
        DisposeArray (L, rows, rows);

    END ReLQtest;

(************************************************************************)

PROCEDURE CxLQtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 8;

    VAR L, Q, LQ: CxArrayPtr;
        rank: CARDINAL;

    BEGIN
        L := NewCxArray (rows, rows);
        Q := NewCxArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        rank := CxLQFactor (A, rows, cols, L^, Q^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("L =");  WriteLn;
        CxWrite (L^, rows, rank, fieldsize);
        WriteString ("Q =");  WriteLn;
        CxWrite (Q^, rank, cols, fieldsize);

        LQ := NewCxArray (rows, cols);
        CxMul (L^, Q^, rows, rank, cols, LQ^);
        WriteString ("LQ =");  WriteLn;
        CxWrite (LQ^, rows, cols, fieldsize);

        PressAnyKey;

        DisposeCxArray (LQ, rows, cols);
        DisposeCxArray (Q, rows, cols);
        DisposeCxArray (L, rows, rows);

    END CxLQtest;

(************************************************************************)

PROCEDURE DoLQtests;

    (* Tests of LQ factorisation. *)

    VAR Rand: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);
        (**)
        ReLQtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4);
        ReLQtest (Re2x2 {{1.0, 0.0},
                         {0.0, 1.0}}, 2, 2);
        ReLQtest (Re2x2 {{1.0, 1.0},
                         {1.0, 1.0}}, 2, 2);
        ReLQtest (Re3x3 {{0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0}}, 3, 3);
        r := 4;  c := 5;
        Rand := NewArray (r, c);
        Random (Rand^, r, c);
        ReLQtest (Rand^, r, c);
        DisposeArray (Rand, r, c);
        (**)

        (* Complex test.        *)

        (**)
        B := NewCxArray (2, 2);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxLQtest (B^, 2, 2);
        DisposeCxArray (B, 2, 2);
        (**)

    END DoLQtests;

(************************************************************************)
(*                              PSEUDOINVERSE                           *)
(************************************************************************)

PROCEDURE RePStest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of real pseudoinverse. *)

    CONST
        fieldsize = 10;

    VAR B, Product: ArrayPtr;
        rank: CARDINAL;

    BEGIN
        B := NewArray (cols, rows);
        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix A =");  WriteLn;
        Write (A, rows, cols, fieldsize);
        rank := RePseudoInverse (A, rows, cols, B^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Its pseudoinverse B =");  WriteLn;
        Write (B^, cols, rows, fieldsize);

        Product := NewArray (rows, rows);
        Mul (A, B^, rows, cols, rows, Product^);
        WriteString ("AB =");  WriteLn;
        Write (Product^, rows, rows, fieldsize);
        DisposeArray (Product, rows, rows);

        Product := NewArray (cols, cols);
        Mul (B^, A, cols, rows, cols, Product^);
        WriteString ("BA =");  WriteLn;
        Write (Product^, cols, cols, fieldsize);
        DisposeArray (Product, cols, cols);

        DisposeArray (B, rows, cols);

        PressAnyKey;

    END RePStest;

(************************************************************************)

PROCEDURE CxPStest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of complex pseudoinverse. *)

    CONST
        fieldsize = 8;

    VAR B, Product: CxArrayPtr;
        rank: CARDINAL;

    BEGIN
        B := NewCxArray (cols, rows);
        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix A =");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        rank := CxPseudoInverse (A, rows, cols, B^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Its pseudoinverse B =");  WriteLn;
        CxWrite (B^, cols, rows, fieldsize);

        Product := NewCxArray (rows, rows);
        CxMul (A, B^, rows, cols, rows, Product^);
        WriteString ("AB =");  WriteLn;
        CxWrite (Product^, rows, rows, fieldsize);
        DisposeCxArray (Product, rows, rows);

        Product := NewCxArray (cols, cols);
        CxMul (B^, A, cols, rows, cols, Product^);
        WriteString ("BA =");  WriteLn;
        CxWrite (Product^, cols, cols, fieldsize);
        DisposeCxArray (Product, cols, cols);

        DisposeCxArray (B, rows, cols);

        PressAnyKey;

    END CxPStest;

(************************************************************************)

PROCEDURE psinvtest;

    (* Test of pseudo-inverse. *)

    VAR A: ArrayPtr;  B: CxArrayPtr;
    CONST r = 2;  c = 3;

    BEGIN
        SelectWindow (0);
        (**)
        A := NewArray (1, 4);
        A^[0,0] := 1.0;  A^[0,1] := 2.0;
        A^[0,2] := 3.0;  A^[0,3] := 4.0;
        RePStest (A^, 1, 4);
        DisposeArray (A, 1, 4);
        (**)
        A := NewArray (5, 3);
        Random (A^, 5, 3);
        RePStest (A^, 5, 3);
        DisposeArray (A, 5, 3);
        (**)
        A := NewArray (r, c);
        Random (A^, r, c);
        RePStest (A^, r, c);
        DisposeArray (A, r, c);

        (* Complex tests.  *)

        (**)
        B := NewCxArray (r, c);
        CxRandom (B^, r, c);
        CxPStest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)

        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);  B^[0,2] := CMPLX(0.2,0.5);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);  B^[1,2] := CMPLX(0.9,0.0);
        CxPStest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)
    END psinvtest;

(************************************************************************)
(*                           TRIANGULAR MATRICES                        *)
(************************************************************************)

PROCEDURE ReLTInvTest (A: ARRAY OF ARRAY OF LONGREAL;  rows: CARDINAL);

    (* Inverse of a nonsingular lower triangular matrix. *)

    CONST
        fieldsize = 8;

    VAR B, Product: ArrayPtr;

    BEGIN
        B := NewArray (rows, rows);
        ReLTinverse (A, rows, B^);

        WriteString ("The inverse of");  WriteLn;
        Write (A, rows, rows, fieldsize);
        WriteString ("is");  WriteLn;
        Write (B^, rows, rows, fieldsize);
        Product := NewArray (rows, rows);
        Mul (A, B^, rows, rows, rows, Product^);
        WriteString ("Their product is");  WriteLn;
        Write (Product^, rows, rows, fieldsize);

        DisposeArray (Product, rows, rows);
        DisposeArray (B, rows, rows);
        PressAnyKey;

    END ReLTInvTest;

(************************************************************************)

PROCEDURE CxLTInvTest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows: CARDINAL);

    (* Inverse of a nonsingular lower triangular matrix. *)

    CONST
        fieldsize = 8;

    VAR B, Product: CxArrayPtr;

    BEGIN
        B := NewCxArray (rows, rows);
        CxLTinverse (A, rows, B^);

        WriteString ("The inverse of");  WriteLn;
        CxWrite (A, rows, rows, fieldsize);
        WriteString ("is");  WriteLn;
        CxWrite (B^, rows, rows, fieldsize);
        Product := NewCxArray (rows, rows);
        CxMul (A, B^, rows, rows, rows, Product^);
        WriteString ("Their product is");  WriteLn;
        CxWrite (Product^, rows, rows, fieldsize);

        DisposeCxArray (Product, rows, rows);
        DisposeCxArray (B, rows, rows);
        PressAnyKey;

    END CxLTInvTest;

(************************************************************************)

PROCEDURE ReRanLT (VAR (*OUT*) A: ARRAY OF ARRAY OF LONGREAL;  N: CARDINAL);

    (* Generates a random lower triangular matrix. *)

    VAR i, j: CARDINAL;

    BEGIN
        Random (A, N, N);
        FOR i := 0 TO N-2 DO
            FOR j := i+1 TO N-1 DO
                A[i,j] := 0.0;
            END (*FOR*);
        END (*FOR*);
    END ReRanLT;

(************************************************************************)

PROCEDURE CxRanLT (VAR (*OUT*) A: ARRAY OF ARRAY OF LONGCOMPLEX;  N: CARDINAL);

    (* Generates a random lower triangular matrix. *)

    VAR i, j: CARDINAL;

    BEGIN
        CxRandom (A, N, N);
        FOR i := 0 TO N-2 DO
            FOR j := i+1 TO N-1 DO
                A[i,j] := CxZero;
            END (*FOR*);
        END (*FOR*);
    END CxRanLT;

(************************************************************************)

PROCEDURE TriTest;

    VAR rows: CARDINAL;
        A: ArrayPtr;  B: CxArrayPtr;

    BEGIN
        (* For now, the only tests are for inverting lower triangular matrices. *)

        rows := 2;
        A := NewArray (rows, rows);
        A^[0,0] := 1.0;  A^[0,1] := 0.0;
        A^[1,0] := 3.0;  A^[1,1] := 4.0;
        ReLTInvTest (A^, rows);
        DisposeArray (A, rows, rows);

        rows := 6;
        A := NewArray (rows, rows);
        ReRanLT (A^, rows);
        ReLTInvTest (A^, rows);
        DisposeArray (A, rows, rows);

        rows := 4;
        B := NewCxArray (rows, rows);
        CxRanLT (B^, rows);
        CxLTInvTest (B^, rows);
        DisposeCxArray (B, rows, rows);

    END TriTest;

(************************************************************************)
(*                          CHOLESKY FACTORISATION                      *)
(************************************************************************)

PROCEDURE CholeskyTest;

    (* Test of Cholesky factorisation of a positive definite matrix. *)

    CONST
        fieldsize = 8;
        rows = 3;  cols = 3;

    VAR T, Ttr, R, S, Str, Product, Diff: ArrayPtr;
        cT, cTtr, cR, cS, cStr, cProduct, cDiff: CxArrayPtr;
        k: CARDINAL;  success: BOOLEAN;

    BEGIN
        (* One easy way to generate a positive definite matrix is to    *)
        (* compute S*S for an arbitrary S, and then add some diagonal   *)
        (* entries to ensure that the result is not singular.           *)

        T := NewArray (rows, cols);
        Ttr := NewArray (cols, rows);
        R := NewArray (rows, rows);
        S := NewArray (rows, rows);
        Str := NewArray (rows, rows);
        Product := NewArray (rows, rows);
        Diff := NewArray (rows, rows);

        Random (T^, rows, cols);
        Transpose (T^, rows, cols, Ttr^);
        Mul (T^, Ttr^, rows, cols, rows, R^);
        FOR k := 0 TO rows-1 DO
            R^[k,k] := R^[k,k] + 0.1;
        END (*FOR*);

        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix R =");  WriteLn;
        Write (R^, rows, rows, fieldsize);
        success := ReCholesky (R^, rows, S^);
        WriteString ("Its Cholesky factor S =");  WriteLn;
        Write (S^, cols, rows, fieldsize);
        Transpose (S^, rows, rows, Str^);
        Mul (Str^, S^, rows, rows, rows, Product^);
        Sub (Product^, R^, rows, cols, Diff^);
        WriteString ("S*S - R =");  WriteLn;
        Write (Diff^, rows, rows, fieldsize);

        DisposeArray (Diff, rows, rows);
        DisposeArray (Product, rows, rows);
        DisposeArray (Str, rows, rows);
        DisposeArray (S, rows, rows);
        DisposeArray (R, rows, rows);
        DisposeArray (Ttr, cols, rows);
        DisposeArray (T, rows, cols);
        PressAnyKey;

        (* Now repeat the same test for the complex case. *)

        cT := NewCxArray (rows, cols);
        cTtr := NewCxArray (cols, rows);
        cR := NewCxArray (rows, rows);
        cS := NewCxArray (rows, rows);
        cStr := NewCxArray (rows, rows);
        cProduct := NewCxArray (rows, rows);
        cDiff := NewCxArray (rows, rows);

        CxRandom (cT^, rows, cols);
        Adjoint (cT^, rows, cols, cTtr^);
        CxMul (cT^, cTtr^, rows, cols, rows, cR^);
        FOR k := 0 TO rows-1 DO
            cR^[k,k] := cR^[k,k] + CMPLX(0.1,0.0);
        END (*FOR*);

        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix R =");  WriteLn;
        CxWrite (cR^, rows, rows, fieldsize);
        success := CxCholesky (cR^, rows, cS^);
        WriteString ("Its Cholesky factor S =");  WriteLn;
        CxWrite (cS^, cols, rows, fieldsize);
        Adjoint (cS^, rows, rows, cStr^);
        CxMul (cStr^, cS^, rows, rows, rows, cProduct^);
        WriteString ("S*S =");  WriteLn;
        CxWrite (cProduct^, rows, rows, fieldsize);
        CxSub (cProduct^, cR^, rows, cols, cDiff^);
        WriteString ("S*S - R =");  WriteLn;
        CxWrite (cDiff^, rows, rows, fieldsize);

        DisposeCxArray (cDiff, rows, rows);
        DisposeCxArray (cProduct, rows, rows);
        DisposeCxArray (cStr, rows, rows);
        DisposeCxArray (cS, rows, rows);
        DisposeCxArray (cR, rows, rows);
        DisposeCxArray (cTtr, cols, rows);
        DisposeCxArray (cT, rows, cols);
        PressAnyKey;

    END CholeskyTest;

(************************************************************************)
(*                     SINGULAR VALUE DECOMPOSITION                     *)
(************************************************************************)

PROCEDURE ReSVDtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of singular value decomposition. *)

    CONST
        fieldsize = 10;

    VAR P, L, Q: ArrayPtr;
        q0, rank: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewArray (rows, q0);
        L := NewArray (q0, q0);
        Q := NewArray (q0, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix A =");  WriteLn;
        Write (A, rows, cols, fieldsize);

        rank := ReSVD (A, rows, cols, P^, L^, Q^);

        WriteString ("A = PLQ, where P =");  WriteLn;
        Write (P^, rows, rank, fieldsize);
        WriteString ("L =");  WriteLn;
        Write (L^, rank, rank, fieldsize);
        WriteString ("Q =");  WriteLn;
        Write (Q^, rank, cols, fieldsize);
        PressAnyKey;

        DisposeArray (Q, q0, cols);
        DisposeArray (L, q0, q0);
        DisposeArray (P, rows, q0);

    END ReSVDtest;

(************************************************************************)

PROCEDURE CxSVDtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of singular value decomposition. *)

    CONST
        fieldsize = 10;

    VAR P, L, Q: CxArrayPtr;
        q0, rank: CARDINAL;

    BEGIN
        q0 := rows;
        IF cols < q0 THEN
            q0 := cols;
        END (*IF*);
        P := NewCxArray (rows, q0);
        L := NewCxArray (q0, q0);
        Q := NewCxArray (q0, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix A =");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);

        rank := CxSVD (A, rows, cols, P^, L^, Q^);

        WriteString ("A = PLQ, where P =");  WriteLn;
        CxWrite (P^, rows, rank, fieldsize);
        WriteString ("L =");  WriteLn;
        CxWrite (L^, rank, rank, fieldsize);
        WriteString ("Q =");  WriteLn;
        CxWrite (Q^, rank, cols, fieldsize);
        PressAnyKey;

        DisposeCxArray (Q, q0, cols);
        DisposeCxArray (L, q0, q0);
        DisposeCxArray (P, rows, q0);

    END CxSVDtest;

(************************************************************************)

PROCEDURE DoSVDtests;

    (* Tests of singular value decomposition. *)

    VAR Rand: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);
        (**)
        ReSVDtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4);
        ReSVDtest (Re2x2 {{1.0, 0.0},
                         {0.0, 1.0}}, 2, 2);
        (**)
        ReSVDtest (Re2x2 {{1.0, 1.0},
                         {1.0, 1.0}}, 2, 2);
        (**)
        ReSVDtest (Re3x3 {{0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0}}, 3, 3);
        (**)
        r := 5;  c := 3;
        Rand := NewArray (r, c);
        Random (Rand^, r, c);
        ReSVDtest (Rand^, r, c);
        DisposeArray (Rand, r, c);

        (* The complex case.  *)

        (**)
        r := 2;  c := 2;
        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxSVDtest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)

    END DoSVDtests;

(************************************************************************)
(*                              MAIN PROGRAM                            *)
(************************************************************************)

BEGIN
    (**)
    DoQRtests;
    (**)
    DoHHQRtests;
    (**)
    DoLQtests;
    (**)
    TriTest;
    (**)
    psinvtest;
    (**)
    CholeskyTest;
    (**)
    DoSVDtests;
    (**)
END MatExtraTest.

