IMPLEMENTATION MODULE MatExtra;

        (********************************************************)
        (*                                                      *)
        (*          Some additional matrix operations           *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        21 April 2016                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (* Note that many of the functions in this module occur *)
        (*   in pairs, one for real and one for complex data.   *)
        (*                                                      *)
        (********************************************************)


FROM LongMath IMPORT
    (* proc *)  sqrt;

FROM LongComplexMath IMPORT
    (* proc *)  abs, conj, scalarMult;

FROM MiscM2 IMPORT
    (* proc *)  WriteString, WriteLn;

IMPORT Vec;

FROM Vec IMPORT
    (* type *)  VectorPtr, CxVectorPtr,
    (* proc *)  NewVector, DisposeVector, NewCxVector, DisposeCxVector;

FROM Mat IMPORT
    (* type *)  ArrayPtr, CxArrayPtr,
    (* proc *)  NewArray, DisposeArray, NewCxArray, DisposeCxArray,
                Unit, CxUnit,
                ReCopy, CxCopy, Transpose, Adjoint, Mul, CxMul;

(************************************************************************)

CONST
    small = 1.0E-15;
    CxZero = CMPLX (0.0, 0.0);

(************************************************************************)
(*                          QR FACTORISATION                            *)
(************************************************************************)
(*                                                                      *)
(* Background: the QR factorisation of a real or complex matrix A is    *)
(* a factorisation A = QR, where Q has the property that Q*Q = I and    *)
(* R is upper triangular.  Here, Q* means the adjoint of Q: the complex *)
(* conjugate transpose in the complex case, or just the transpose in    *)
(* the real case. Thus, the columns of Q have unit norm and are         *)
(* orthogonal to one another. The triangular shape of R can simplify a  *)
(* lot of calculations, especially when A has many more rows than       *)
(* columns.                                                             *)
(*                                                                      *)
(* Let A be an rxc matrix. In the traditional "thick" factorisation     *)
(* Q is a square matrix, i.e. Q is rxr and R is rxc. In this case Q     *)
(* is an orthonormal matrix, a slightly stronger property than Q*Q=I.   *)
(* Whether this matters depends on what you are going to do with Q and  *)
(* R once you have calculated them. In many situations it does not      *)
(* matter. If r>c, then the last (r-c) rows of R are zero, because of   *)
(* the uppper triangular property. In this case it is obvious that we   *)
(* can delete those zero rows, and delete the last (r-c) columns of Q,  *)
(* to obtain the so-called "thin" QR factorisation, where Q is rxc and  *)
(* R is cxc.                                                            *)
(*                                                                      *)
(* Remark: if A has more columns than rows the above argument does not  *)
(* apply, and there is no thin factorisation. In the practical          *)
(* applications I have worked on, however, the case r>>c is precisely   *)
(* the case where a QR factorisation is useful.                         *)
(*                                                                      *)
(* The code below gives an "even thinner" QR factorisation, where Q is  *)
(* rxq and R is qxc, where q=rank(A). It works in both cases r>c and    *)
(* r<=c. I have not found this case described in the literature, but    *)
(* I have worked out the theory (details on request) to justify the     *)
(* algorithm. In my opinion this is the most useful QR factorisation    *)
(* of all, because it guarantees that the rows of R are linearly        *)
(* independent. One consequence of this is a new and (I believe)        *)
(* computationally efficient way to calculate a pseudo-inverse -- see   *)
(* the code later in this module.                                       *)
(*                                                                      *)
(************************************************************************)

PROCEDURE ReQRFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                      VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* QR factorisation of an rxc matrix A, not necessarily square.     *)
    (* The result is an rxq matrix Q such that Q*Q=I, and an upper      *)
    (* triangular qxc matrix R.  If A<>0 the integer q, which is        *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the rows of R are linearly independent.                          *)
    (* If A=0 then we return q=0, and R is a single row of zeros.       *)
    (* Assumption: the caller has declared Q and R to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* This is the "even thinner" version of QR factorisation. If the   *)
    (* "thick" version is needed, see ReFullQRFactor below.             *)

    (* This version uses the modified Gram-Schmidt algorithm, with      *)
    (* provision for the possibility that A is rank-deficient.          *)

    VAR
        j, col, row, rank: CARDINAL;
        temp, norm, val: LONGREAL;
        W: ArrayPtr;
        allzero: BOOLEAN;

    BEGIN
        (* The case A = 0 needs separate handling. *)

        allzero := TRUE;  row := 0;
        WHILE allzero AND (row < r) DO
            col := 0;
            WHILE allzero AND (col < c) DO
                IF ABS(A[row, col]) >= small THEN
                     allzero := FALSE;
                END (*IF*);
                INC (col);
            END (*WHILE*);
            INC (row);
        END (*WHILE*);
        IF allzero THEN
            Q[0,0] := 1.0;
            FOR row := 1 TO r-1 DO
                Q[row, 0] := 0.0;
            END (*FOR*);
            FOR col := 0 TO c-1 DO
                R[0, col] := 0.0;
            END (*FOR*);
            RETURN 0;
        END (*IF*);

        (* From now on we can assume that A <> 0. *)

        (* Set subdiagonal elements of R to zero. Here rank is being    *)
        (* used as a temporary variable.                                *)

        IF r >= c THEN rank := c ELSE rank := r END(*IF*);
        FOR row := 1 TO rank-1 DO
            FOR col := 0 TO row-1 DO
                R[row, col] := 0.0;
            END (*FOR*);
        END (*FOR*);

        (* To avoid destroying the original A, copy A to a work matrix  *)
        (* W^.  As the calculation works through column 'col', the      *)
        (* final Q will be stored in columns 0 to rank-1 of the work    *)
        (* matrix, while columns col+1 upwards will hold a modified A.  *)
        (* In the case r >= c we could have used Q as the work matrix,  *)
        (* but Q might not have enough columns in the general case.     *)

        (* Note that col is the current column of A we are working      *)
        (* with, rank is both the current column of Q and the current   *)
        (* row of R.  In the full-rank case, rank=col, but if A is      *)
        (* rank-deficient we will sometimes step to the next column     *)
        (* of W^ without incrementing rank.                             *)

        rank := 0;
        W := NewArray (r, c);
        ReCopy (A, r, c, W^);
        col := 0;
        WHILE col < c DO
            norm := 0.0;

            (* Set norm equal to the norm of the current column. *)

            FOR row := 0 TO r-1 DO
                val := W^[row, col];
                norm := norm + val*val;
            END (*FOR*);
            norm := sqrt(norm);

            IF norm < small THEN

                (* Special case: first column is zero. *)

                R[rank, col] := 0.0;

            ELSE

                (* DEBUGGING CODE *)
                (*
                WriteChar ('[');  WriteLJCard(rank);  WriteChar (',');
                WriteLJCard (col);  WriteChar (']');
                *)
                (* /DEBUGGING CODE *)

                (* Now current column of W^ must be set equal to a copy,    *)
                (* normalised, of the current column, and the diagonal      *)
                (* element of R is the normalisation factor.  Subdiagonal   *)
                (* elements of R have already been set to zero.             *)

                R[rank, col] := norm;
                temp := 1.0/norm;
                allzero := (col+1 < c);     (* FALSE if in last column *)
                FOR row := 0 TO r-1 DO
                    W^[row, rank] := temp * W^[row, col];
                END (*FOR*);

                (* Set the rest of the current row of R. *)

                IF allzero THEN            (* i.e. if not in last column *)
                    FOR j := col+1 TO c-1 DO
                        val := 0.0;
                        FOR row := 0 TO r-1 DO
                            val := val + W^[row, rank] * W^[row, j];
                        END (*FOR*);
                        R[rank, j] := val;

                        (* Adjust the remaining columns of W^ in preparation *)
                        (* for the next step in the calculation.             *)

                        FOR row := 0 TO r-1 DO
                            W^[row, j] := W^[row, j] - val * W^[row, rank];
                            IF ABS(W^[row, j]) >= small THEN
                                 allzero := FALSE;
                            END (*IF*);
                        END (*FOR*);
                    END (*FOR "adjust remaining columns" loop*);
                END (*IF*);

                INC (rank);

                (* There are two more special cases to handle. One is    *)
                (* where there are more columns than rows, and we have   *)
                (* run out of rows.  The other is where the adjusted     *)
                (* remaining part of W^ turns out to be zero. In either  *)
                (* case we have finished the calculations and don't need *)
                (* to look at the rest of W^.                            *)

                IF allzero OR (rank >= r) THEN col := c END (*IF*);

            END (* IF norm *);
            INC (col);

        END; (* loop over columns *)

        ReCopy (W^, r, rank, Q);
        DisposeArray (W, r, c);
        RETURN rank;

    END ReQRFactor;

(************************************************************************)

PROCEDURE ReFullQRFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                                VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGREAL);

    (* QR factorisation of an rxc matrix A, with r>=c.  This is the     *)
    (* same as ReQRFactor, except that we guarantee to return a square  *)
    (* matrix Q. That is, Q is rxr and R is rxc, with extra zero rows   *)
    (* added to R as needed.  We still have the property Qadj*Q=I.      *)

    VAR col, onepos, i, k: CARDINAL;
        scale: LONGREAL;
        u: VectorPtr;

    BEGIN
        u := NewVector (r);
        col := ReQRFactor (A, r, c, Q, R);
        onepos := 0;
        WHILE col < r DO

            (* The first col columns of Q have been filled in.  We will now *)
            (* use the Gram-Schmidt procedure to generate the next column.  *)

            scale := 0.0;
            WHILE scale < small DO

                (* Create the vector u.  *)

                FOR i := 0 TO r-1 DO
                    u^[i] := 0.0;
                END (*FOR*);
                u^[onepos] := 1.0;
                IF col > 0 THEN
                    FOR k := 0 TO col-1 DO
                        scale := Q[onepos,k];
                        FOR i := 0 TO r-1 DO
                            u^[i] := u^[i] - scale*Q[i,k];
                        END (*FOR*);
                    END (*FOR*);
                END (*IF*);
                scale := 0.0;
                FOR i := 0 TO r-1 DO
                    scale := scale + u^[i]*u^[i];
                END (*FOR*);
                scale := sqrt (scale);
                INC (onepos);

                (* If scale is too small, we just try again. *)

            END (*WHILE*);

            (* Fill in the new column of Q, and zero a new row of R.  *)

            scale := 1.0/scale;
            FOR i := 0 TO r-1 DO
                Q[i,col] := scale * u^[i];
            END (*FOR*);
            FOR k := 0 TO c-1 DO
                R[col,k] := 0.0;
            END (*FOR*);
            INC (col);
        END (*WHILE*);

        DisposeVector (u, r);

    END ReFullQRFactor;

(************************************************************************)

PROCEDURE CxQRFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                         VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* QR factorisation of an rxc matrix A, not necessarily square.     *)
    (* The result is an rxq matrix Q such that Q*Q=I, and an upper      *)
    (* triangular qxc matrix R.  If A<>0 the integer q, which is        *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the rows of R are linearly independent.                          *)
    (* If A=0 then we return q=0, and R is a single row of zeros.       *)
    (* Assumption: the caller has declared Q and R to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* This is the "even thinner" version of QR factorisation. If the   *)
    (* "thick" version is needed, see CxFullQRFactor below.             *)

    (* This version uses the modified Gram-Schmidt algorithm, with      *)
    (* provision for the possibility that A is rank-deficient.          *)

    VAR
        j, col, row, rank: CARDINAL;
        norm: LONGREAL;
        val: LONGCOMPLEX;
        W: CxArrayPtr;
        allzero: BOOLEAN;

    BEGIN

        (* The case A = 0 needs separate handling. *)

        allzero := TRUE;  row := 0;
        WHILE allzero AND (row < r) DO
            col := 0;
            WHILE allzero AND (col < c) DO
                IF abs(A[row, col]) >= small THEN
                     allzero := FALSE;
                END (*IF*);
                INC (col);
            END (*WHILE*);
            INC (row);
        END (*WHILE*);
        IF allzero THEN
            Q[0,0] := CMPLX (1.0, 0.0);
            FOR row := 1 TO r-1 DO
                Q[row, 0] := CxZero;
            END (*FOR*);
            FOR col := 0 TO c-1 DO
                R[0, col] := CxZero;
            END (*FOR*);
            RETURN 0;
        END (*IF*);

        (* From now on we can assume that A <> 0.  *)

        (* Set subdiagonal elements of R to zero. Here rank is being    *)
        (* used as a temporary variable.                                *)

        IF r >= c THEN rank := c ELSE rank := r END(*IF*);
        FOR row := 1 TO rank-1 DO
            FOR col := 0 TO row-1 DO
                R[row, col] := CxZero;
            END (*FOR*);
        END (*FOR*);

        (* To avoid destroying the original A, copy A to a work matrix  *)
        (* W^.  As the calculation works through column 'col', the      *)
        (* final Q will be stored in columns 0 to rank-1 of the work    *)
        (* matrix, while columns col+1 upwards will hold a modified A.  *)
        (* In the case r >= c we could have used Q as the work matrix,  *)
        (* but Q might not have enough columns in the general case.     *)

        (* Note that col is the current column of W^ we are working     *)
        (* with, rank is both the current column of Q and the current   *)
        (* row of R.  In the full-rank case, rank=col, but if A is      *)
        (* rank-deficient we will sometimes step to the next column     *)
        (* of A without incrementing rank.                              *)

        rank := 0;
        W := NewCxArray (r, c);
        CxCopy (A, r, c, W^);
        col := 0;
        WHILE col < c DO
            norm := 0.0;

            (* Set norm equal to the norm of the current column. *)

            FOR row := 0 TO r-1 DO
                val := W^[row, col];
                norm := norm + RE(val)*RE(val) + IM(val)*IM(val);
            END (*FOR*);
            norm := sqrt(norm);

            IF norm < small THEN

                (* Special case: first column is zero. *)

                R[rank, col] := CxZero;

            ELSE

                (* Now current column of W^ must be normalised, and the     *)
                (* diagonal element of R is the normalisation factor.       *)
                (* Subdiagonal elements of R have already been set to zero. *)

                R[rank, col] := CMPLX (norm, 0.0);
                norm := 1.0/norm;
                allzero := (col+1 < c);     (* FALSE if in last column *)
                FOR row := 0 TO r-1 DO
                    W^[row, rank] := scalarMult (norm, W^[row, col]);
                END (*FOR*);

                (* Set the rest of the current row of R. *)

                IF allzero THEN            (* i.e. if not in last column *)
                    FOR j := col+1 TO c-1 DO
                        val := CxZero;
                        FOR row := 0 TO r-1 DO
                            val := val + conj(W^[row, rank]) * W^[row, j];
                        END (*FOR*);
                        R[rank, j] := val;

                        (* Adjust the remaining columns of our copy of W^ in    *)
                        (* preparation for the next step in the calculation.    *)

                        FOR row := 0 TO r-1 DO
                            W^[row, j] := W^[row, j] - val * W^[row, rank];
                            IF abs(W^[row, j]) >= small THEN
                                 allzero := FALSE;
                            END (*IF*);
                        END (*FOR*);
                    END (*FOR "adjust remaining columns" loop*);
                END (*IF*);

                INC (rank);

                (* There are two more special cases to handle. One is    *)
                (* where there are more columns than rows, and we have   *)
                (* run out of rows.  The other is where the adjusted     *)
                (* remaining part of W^ turns out to be zero. In either  *)
                (* case we have finished the calculations and don't need *)
                (* to look at the rest of W^.                            *)

                IF allzero OR (rank >= r) THEN col := c END (*IF*);

            END (* IF norm *);
            INC (col);

        END; (* loop over columns *)

        CxCopy (W^, r, rank, Q);
        DisposeCxArray (W, r, c);
        RETURN rank;

    END CxQRFactor;

(************************************************************************)

PROCEDURE CxFullQRFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                                VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* QR factorisation of an rxc matrix A, with r>=c.  This is the     *)
    (* same as CxQRFactor, except that we guarantee to return a square  *)
    (* matrix Q. That is, Q is rxr and R is rxc, with extra zero rows   *)
    (* added to R as needed.  We still have the property Qadj*Q=I.      *)

    VAR col, onepos, i, k: CARDINAL;
        norm: LONGREAL;
        cxscale: LONGCOMPLEX;
        u: CxVectorPtr;

    BEGIN
        u := NewCxVector (r);
        col := CxQRFactor (A, r, c, Q, R);
        onepos := 0;
        WHILE col < r DO

            (* The first col columns of Q have been filled in.  We will now *)
            (* use the Gram-Schmidt procedure to generate the next column.  *)

            norm := 0.0;
            WHILE norm < small DO

                (* Create the vector u.  *)

                FOR i := 0 TO r-1 DO
                    u^[i] := CxZero;
                END (*FOR*);
                u^[onepos] := CMPLX (1.0, 0.0);
                IF col > 0 THEN
                    FOR k := 0 TO col-1 DO
                        cxscale := conj (Q[onepos,k]);
                        FOR i := 0 TO r-1 DO
                            u^[i] := u^[i] - cxscale*Q[i,k];
                        END (*FOR*);
                    END (*FOR*);
                END (*IF*);
                norm := 0.0;
                FOR i := 0 TO r-1 DO
                    norm := norm + RE(u^[i])*RE(u^[i]) + IM(u^[i])*IM(u^[i]);
                END (*FOR*);
                norm := sqrt (norm);
                INC (onepos);

                (* If norm is too small, we just try again. *)

            END (*WHILE*);

            (* Fill in the new column of Q, and zero a new row of R.  *)

            norm := 1.0/norm;
            FOR i := 0 TO r-1 DO
                Q[i,col] := scalarMult (norm, u^[i]);
            END (*FOR*);
            FOR k := 0 TO c-1 DO
                R[col,k] := CxZero;
            END (*FOR*);
            INC (col);
        END (*WHILE*);

        DisposeCxVector (u, r);

    END CxFullQRFactor;

(************************************************************************)

PROCEDURE HHReQRFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                         VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* QR factorisation of an rxc matrix A, not necessarily square,     *)
    (* using Householder reflections.                                   *)
    (* The result is an rxr matrix Q such that Q*Q=I, and an upper      *)
    (* triangular rxc matrix R.  If A<>0 the integer q, which is        *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the rows of R are linearly independent.                          *)
    (* If A=0 then we return q=0, and R is a single row of zeros.       *)
    (* Assumption: the caller has declared Q and R to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* This is the fully reduced version of QR factorisation. I haven't *)
    (* bothered to do the thick version.                                *)

    CONST fieldsize = 7;

    VAR newQ, oldR, H: ArrayPtr;
        r0, c0, i, j, k: CARDINAL;
        val: LONGREAL;

    BEGIN
        newQ := NewArray (r, r);
        oldR := NewArray (r, c);
        ReCopy (A, r, c, R);
        Unit (Q, r);
        H := NewArray (r, r);
        r0 := 0;  c0 := 0;
        REPEAT
            ReCopy (R, r, c, oldR^);
            ReHouseholder (R, r, c, r0, c0, H^);

            (* Set new Q = Q times H*  *)

            FOR i := 0 TO r-1 DO
                FOR j := 0 TO r-1 DO
                    val := 0.0;
                    FOR k := 0 TO r-1 DO
                        val := val + Q[i,k] * H^[j,k];
                    END (*FOR*);
                    newQ^[i,j] := val;
                END (*FOR*);
            END (*FOR*);
            ReCopy (newQ^, r, r, Q);

        UNTIL (r0 >= r) OR (c0 >= c);

        DisposeArray (oldR, r, c);
        DisposeArray (newQ, r, r);
        DisposeArray (H, r, r);

        (* Return min(r0, c0) as the rank of the matrix. *)

        IF r0 < c0 THEN
            RETURN r0;
        ELSE
            RETURN c0;
        END (*IF*);

    END HHReQRFactor;

(************************************************************************)

PROCEDURE HHCxQRFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                         VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* QR factorisation of an rxc matrix A, not necessarily square,     *)
    (* using Householder reflections.                                   *)
    (* The result is an rxr matrix Q such that Q*Q=I, and an upper      *)
    (* triangular rxc matrix R.  If A<>0 the integer q, which is        *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the rows of R are linearly independent.                          *)
    (* If A=0 then we return q=0, and R is a single row of zeros.       *)
    (* Assumption: the caller has declared Q and R to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* This is the fully reduced version of QR factorisation. I haven't *)
    (* bothered to do the thick version.                                *)

    CONST fieldsize = 7;

    VAR newQ, oldR, H: CxArrayPtr;
        r0, c0, i, j, k: CARDINAL;
        val: LONGCOMPLEX;

    BEGIN
        newQ := NewCxArray (r, r);
        oldR := NewCxArray (r, c);
        CxCopy (A, r, c, R);
        CxUnit (Q, r);
        H := NewCxArray (r, r);
        r0 := 0;  c0 := 0;
        REPEAT
            CxCopy (R, r, c, oldR^);
            CxHouseholder (R, r, c, r0, c0, H^);

            (* Set new Q = Q times H*  *)

            FOR i := 0 TO r-1 DO
                FOR j := 0 TO r-1 DO
                    val := CxZero;
                    FOR k := 0 TO r-1 DO
                        val := val + Q[i,k] * conj(H^[j,k]);
                    END (*FOR*);
                    newQ^[i,j] := val;
                END (*FOR*);
            END (*FOR*);
            CxCopy (newQ^, r, r, Q);

        UNTIL (r0 >= r) OR (c0 >= c);

        DisposeCxArray (oldR, r, c);
        DisposeCxArray (newQ, r, r);
        DisposeCxArray (H, r, r);

        (* Return min(r0, c0) as the rank of the matrix. *)

        IF r0 < c0 THEN
            RETURN r0;
        ELSE
            RETURN c0;
        END (*IF*);

    END HHCxQRFactor;

(************************************************************************)
(*                          LQ FACTORISATION                            *)
(************************************************************************)
(*                                                                      *)
(* Background: the LQ factorisation of a real or complex matrix A is    *)
(* a factorisation A = LR, where Q has the property that QQ* = I and    *)
(* L is lower triangular.  The details are the same as for the QR       *)
(* factorisation, but with rows and columns swapped.                    *)
(*                                                                      *)
(************************************************************************)

PROCEDURE ReLQFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                         VAR (*OUT*) L, Q: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Minimal LQ factorisation of an rxc matrix A, not necessarily     *)
    (* square.The result is a lower triangular rxq matrix L and a qxc   *)
    (* matrix Q such that QQ*=I.  If A<>0 the integer q, which is       *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the columns of L are linearly independent.                       *)
    (* If A=0 then we return q=0, and L is a single column of zeros.    *)
    (* Assumption: the caller has declared L and Q to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* Not recommended for the "long thin" case m>>n, because it that   *)
    (* case the resulting L is also long and thin. On the other hand,   *)
    (* we get a good size reduction when A is short and wide.           *)

    VAR
        i, col, row, rank: CARDINAL;
        norm, val: LONGREAL;
        W: ArrayPtr;
        allzero: BOOLEAN;

    BEGIN

        (* The case A = 0 needs separate handling. *)

        allzero := TRUE;  row := 0;
        WHILE allzero AND (row < r) DO
            col := 0;
            WHILE allzero AND (col < c) DO
                IF ABS(A[row, col]) >= small THEN
                     allzero := FALSE;
                END (*IF*);
                INC (col);
            END (*WHILE*);
            INC (row);
        END (*WHILE*);
        IF allzero THEN
            Q[0,0] := 1.0;
            FOR col := 1 TO c-1 DO
                Q[0, col] := 0.0;
            END (*FOR*);
            FOR row := 0 TO r-1 DO
                L[row, 0] := 0.0;
            END (*FOR*);
            RETURN 0;
        END (*IF*);

        (* From now on we can assume that A <> 0.  *)

        (* Set superdiagonal elements of L to zero. Here rank is being  *)
        (* used as a temporary variable.                                *)

        IF r >= c THEN rank := c ELSE rank := r END(*IF*);
        FOR col := 1 TO rank-1 DO
            FOR row := 0 TO col-1 DO
                L[row, col] := 0.0;
            END (*FOR*);
        END (*FOR*);

        (* To avoid destroying the original A, copy A to a work matrix  *)
        (* W^.  As the calculation works through row 'row', the final   *)
        (* Q will be stored in rows 0 to rank-1 of the work matrix, and *)
        (* rows row+1 upwards will hold a modified A.                   *)
        (* In the case c >= r we could have used Q as the work matrix,  *)
        (* but Q might not have enough rows in the general case.        *)

        (* Note that row is the current row of W^ we are working with,  *)
        (* rank is both the current row of Q and the current column of  *)
        (* L.  In the full-rank case, rank=row, but if A is             *)
        (* rank-deficient we will sometimes step to the next row of A   *)
        (* A without incrementing rank.                                 *)

        rank := 0;
        W := NewArray (r, c);
        ReCopy (A, r, c, W^);
        row := 0;
        WHILE row < r DO
            norm := 0.0;

            (* Set norm equal to the norm of the current row. *)

            FOR col := 0 TO c-1 DO
                val := W^[row, col];
                norm := norm + val*val;
            END (*FOR*);
            norm := sqrt(norm);

            IF norm < small THEN

                (* Special case: first row is zero. *)

                L[row, rank] := 0.0;

            ELSE

                (* Now current row of W^ must be normalised, and the    *)
                (* diagonal element of L is the normalisation factor.   *)
                (* Superdiagonal elements of L have already been set    *)
                (* to zero.                                             *)

                L[row, rank] := norm;
                norm := 1.0/norm;
                allzero := (row+1 < r);     (* FALSE if in last row *)
                FOR col := 0 TO c-1 DO
                    W^[row, col] := norm * W^[row, col];
                END (*FOR*);

                (* Set the rest of the current column of L. *)

                IF allzero THEN            (* i.e. if not in last column *)
                    FOR i := row+1 TO r-1 DO
                        val := 0.0;
                        FOR col := 0 TO c-1 DO
                            val := val + W^[i, col] * W^[rank, col];
                        END (*FOR*);
                        L[i, rank] := val;

                        (* Adjust the remaining rows of our copy of W^ in    *)
                        (* preparation for the next step in the calculation. *)

                        FOR col := 0 TO c-1 DO
                            W^[i, col] := W^[i, col] - val * W^[rank, col];
                            IF ABS(W^[i, col]) >= small THEN
                                 allzero := FALSE;
                            END (*IF*);
                        END (*FOR*);
                    END (*FOR "adjust remaining rows" loop*);
                END (*IF*);

                INC (rank);

                (* There is still one special case to handle, and that  *)
                (* is where the adjusted remaining part of W^ turns out *)
                (* to be zero. In that case we have finished the        *)
                (* calculations and don't need to process more columns. *)

                IF allzero THEN row := r END (*IF*);

            END (* IF norm *);
            INC (row);

        END; (* loop over rows *)

        ReCopy (W^, rank, c, Q);
        DisposeArray (W, r, c);
        RETURN rank;

    END ReLQFactor;

(************************************************************************)

PROCEDURE CxLQFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                         VAR (*OUT*) L, Q: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Minimal LQ factorisation of an rxc matrix A, not necessarily     *)
    (* square.The result is a lower triangular rxq matrix L and a qxc   *)
    (* matrix Q such that QQ*=I.  If A<>0 the integer q, which is       *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the columns of L are linearly independent.                       *)
    (* If A=0 then we return q=0, and L is a single column of zeros.    *)
    (* Assumption: the caller has declared L and Q to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* Not recommended for the "long thin" case m>>n, because it that   *)
    (* case the resulting L is also long and thin. On the other hand,   *)
    (* we get a good size reduction when A is short and wide.           *)

    VAR
        i, col, row, rank: CARDINAL;
        norm: LONGREAL;
        val: LONGCOMPLEX;
        W: CxArrayPtr;
        allzero: BOOLEAN;

    BEGIN

        (* The case A = 0 needs separate handling. *)

        allzero := TRUE;  row := 0;
        WHILE allzero AND (row < r) DO
            col := 0;
            WHILE allzero AND (col < c) DO
                IF abs(A[row, col]) >= small THEN
                     allzero := FALSE;
                END (*IF*);
                INC (col);
            END (*WHILE*);
            INC (row);
        END (*WHILE*);
        IF allzero THEN
            Q[0,0] := CMPLX (1.0, 0.0);
            FOR col := 1 TO c-1 DO
                Q[0, col] := CxZero;
            END (*FOR*);
            FOR row := 0 TO r-1 DO
                L[row, 0] := CxZero;
            END (*FOR*);
            RETURN 0;
        END (*IF*);

        (* From now on we can assume that A <> 0.  *)

        (* Set superdiagonal elements of L to zero. Here rank is being  *)
        (* used as a temporary variable.                                *)

        IF r >= c THEN rank := c ELSE rank := r END(*IF*);
        FOR col := 1 TO rank-1 DO
            FOR row := 0 TO col-1 DO
                L[row, col] := CxZero;
            END (*FOR*);
        END (*FOR*);

        (* To avoid destroying the original A, copy A to a work matrix  *)
        (* W^.  As the calculation works through row 'row', the final   *)
        (* Q will be stored in rows 0 to rank-1 of the work matrix, and *)
        (* rows row+1 upwards will hold a modified A.                   *)
        (* In the case c >= r we could have used Q as the work matrix,  *)
        (* but Q might not have enough rows in the general case.        *)

        (* Note that row is the current row of W^ we are working with,  *)
        (* rank is both the current row of Q and the current column of  *)
        (* L.  In the full-rank case, rank=row, but if A is             *)
        (* rank-deficient we will sometimes step to the next row of A   *)
        (* A without incrementing rank.                                 *)

        rank := 0;
        W := NewCxArray (r, c);
        CxCopy (A, r, c, W^);
        row := 0;
        WHILE row < r DO
            norm := 0.0;

            (* Set norm equal to the norm of the current row. *)

            FOR col := 0 TO c-1 DO
                val := W^[row, col];
                norm := norm + RE(val)*RE(val) + IM(val)*IM(val);
            END (*FOR*);
            norm := sqrt(norm);

            IF norm < small THEN

                (* Special case: first row is zero. *)

                L[row, rank] := CxZero;

            ELSE

                (* Now current row of W^ must be normalised, and the    *)
                (* diagonal element of L is the normalisation factor.   *)
                (* Superdiagonal elements of L have already been set    *)
                (* to zero.                                             *)

                L[row, rank] := CMPLX (norm, 0.0);
                norm := 1.0/norm;
                allzero := (row+1 < r);     (* FALSE if in last row *)
                FOR col := 0 TO c-1 DO
                    W^[row, col] := scalarMult (norm, W^[row, col]);
                END (*FOR*);

                (* Set the rest of the current column of L. *)

                IF allzero THEN            (* i.e. if not in last column *)
                    FOR i := row+1 TO r-1 DO
                        val := CxZero;
                        FOR col := 0 TO c-1 DO
                            val := val + W^[i, col] * conj(W^[rank, col]);
                        END (*FOR*);
                        L[i, rank] := val;

                        (* Adjust the remaining rows of our copy of W^ in    *)
                        (* preparation for the next step in the calculation. *)

                        FOR col := 0 TO c-1 DO
                            W^[i, col] := W^[i, col] - val * W^[rank, col];
                            IF abs(W^[i, col]) >= small THEN
                                 allzero := FALSE;
                            END (*IF*);
                        END (*FOR*);
                    END (*FOR "adjust remaining rows" loop*);
                END (*IF*);

                INC (rank);

                (* There is still one special case to handle, and that  *)
                (* is where the adjusted remaining part of W^ turns out *)
                (* to be zero. In that case we have finished the        *)
                (* calculations and don't need to process more columns. *)

                IF allzero THEN row := r END (*IF*);

            END (* IF norm *);
            INC (row);

        END; (* loop over rows *)

        CxCopy (W^, rank, c, Q);
        DisposeCxArray (W, r, c);
        RETURN rank;

    END CxLQFactor;

(************************************************************************)
(*                          PLQ FACTORISATION                           *)
(************************************************************************)

PROCEDURE RePLQFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                         VAR (*OUT*) P, L, Q: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Factors an rxc matrix A as A = P x L x Q, where:                 *)
    (*      P is a rxq matrix with P*P=I                                *)
    (*      L is a square lower triangular qxq matrix                   *)
    (*      Q is a qxc matrix Q with QQ*=I.                             *)
    (* If A<>0 the integer q, which is returned as the function result, *)
    (* is equal to the rank of A.  If A=0 then we return q=0, and the   *)
    (* result for L is a 1x1 zero matrix.                               *)
    (*                                                                  *)
    (* Assumption: the caller has declared P, L and Q to be large       *)
    (* enough to hold the result. Note that q<=min(r,c), which tells    *)
    (* the caller how much space to reserve even though q is not known  *)
    (* in advance.  Unused rows and columns are left unaltered, except  *)
    (* for some extra 0 entries in L when A = 0.                        *)

    VAR R: ArrayPtr;
        q0, rank, j: CARDINAL;

    BEGIN
        q0 := r;
        IF c < q0 THEN
            q0 := c;
        END (*IF*);
        R := NewArray (q0, c);
        rank := ReQRFactor (A, r, c, P, R^);
        IF rank = 0 THEN
            (* Arbitrary choice for Q. *);
            Q[0,0] := 1.0;
            FOR j := 1 TO c-1 DO
                Q[0,j] := 0.0;
            END (*FOR*);
        ELSE
            rank := ReLQFactor (R^, rank, c, L, Q);
        END (*IF*);
        DisposeArray (R, q0, c);
        RETURN rank;
    END RePLQFactor;

(************************************************************************)

PROCEDURE CxPLQFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                         VAR (*OUT*) P, L, Q: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Factors an rxc matrix A as A = P x L x Q, where:                 *)
    (*      P is a rxq matrix with P*P=I                                *)
    (*      L is a square lower triangular qxq matrix                   *)
    (*      Q is a qxc matrix Q with QQ*=I.                             *)
    (* If A<>0 the integer q, which is returned as the function result, *)
    (* is equal to the rank of A.  If A=0 then we return q=0, and the   *)
    (* result for L is a 1x1 zero matrix.                               *)
    (*                                                                  *)
    (* Assumption: the caller has declared P, L and Q to be large       *)
    (* enough to hold the result. Note that q<=min(r,c), which tells    *)
    (* the caller how much space to reserve even though q is not known  *)
    (* in advance.  Unused rows and columns are left unaltered, except  *)
    (* for some extra 0 entries in L when A = 0.                        *)

    VAR R: CxArrayPtr;
        q0, rank, j: CARDINAL;

    BEGIN
        q0 := r;
        IF c < q0 THEN
            q0 := c;
        END (*IF*);
        R := NewCxArray (q0, c);
        rank := CxQRFactor (A, r, c, P, R^);
        IF rank = 0 THEN
            (* Arbitrary choice for Q. *);
            Q[0,0] := CMPLX (1.0, 0.0);
            FOR j := 1 TO c-1 DO
                Q[0,j] := CxZero;
            END (*FOR*);
        ELSE
            rank := CxLQFactor (R^, rank, c, L, Q);
        END (*IF*);
        DisposeCxArray (R, q0, c);
        RETURN rank;
    END CxPLQFactor;

(************************************************************************)
(*                      HOUSEHOLDER TRANSFORMATION                      *)
(************************************************************************)

PROCEDURE ReHouseholder (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGREAL;
                         r, c: CARDINAL;
                         VAR (*INOUT*) r0, c0: CARDINAL;
                         VAR (*OUT*) H: ARRAY OF ARRAY OF LONGREAL);

    (* Householder transformation H on an rxc matrix A, to set the      *)
    (* elements below diagonal A[r0,c0] to zero.  We update r0 and c0   *)
    (* so that the caller is ready for the next call.                   *)

    CONST
        fieldsize = 8;
        gamma = 2.0;

    VAR alpha, val: LONGREAL;
        norm: LONGREAL;
        v, w: Vec.VectorPtr;
        i, j, k: CARDINAL;

    BEGIN
        Unit(H, r);

        (* Find the norm of the vector to be transformed. We only want  *)
        (* to transform A from row r0 downwards, so this vector is the  *)
        (* lower part of column c0.                                     *)

        norm := 0.0;
        FOR i := r0 TO r-1 DO
            val := A[i, c0];
            norm := norm + val*val;
        END (*FOR*);
        norm := sqrt(norm);
        alpha := norm;

        (* norm = 0 at this point means an all-zero column.  This means *)
        (* that the caller must move on to the next column, so we       *)
        (* update c0 without updating r0.                               *)

        IF norm < small THEN
            INC (c0);
            RETURN;
        END (*IF*);

        (* Let u = x - alpha y, where y is a unit vector, and let       *)
        (*   v = u/norm(u).  At the same time we can fill in the first  *)
        (* column of the result, to save ourselves some work further on.*)

        v := Vec.NewVector(r-r0);

        norm := 0.0;
        IF r0 < r-1 THEN
            FOR i := r0+1 TO r-1 DO
                val := A[i, c0];
                A[i,c0] := 0.0;
                v^[i-r0] := val;
                norm := norm + val*val;
            END (*FOR*);
        END (*IF*);

        (* Complete the u and v calculation. *)

        val := A[r0, c0] - alpha;
        A[r0, c0] := alpha;
        v^[0] := val;
        norm := norm + val*val;

        (* If norm = 0 at this point then we have a "do nothing"    *)
        (* transformation, so return with H = I.    Note that this  *)
        (* does not necessarily mean an all-zero column, a case     *)
        (* that was detected above.                                 *)

        IF norm < small THEN
            Vec.DisposeVector (v, r-r0);
            INC (r0);  INC (c0);
            RETURN;
        END (*IF*);

        norm := 1.0/sqrt(norm);
        FOR i := 0 TO r-r0-1 DO
            v^[i] := norm * v^[i];
        END (*FOR*);

        (* Define w = v*A -- where here we interpret A to mean only *)
        (* the lower right block of the original A.  We cannot find *)
        (* the first element of w because we have already altered   *)
        (* the first column of A, but that does not matter because  *)
        (* that first element won't be used below.                  *)

        w := Vec.NewVector(c-c0);
        FOR j := 1 TO c-c0-1 DO
            val := 0.0;
            FOR k := 0 TO r-r0-1 DO
                val := val + v^[k] * A[r0+k,c0+j];
            END (*FOR*);
            w^[j] := val;
        END (*FOR*);

        (* Subtract gamma v w from A.  We can skip the first column,    *)
        (* which was dealt with while calculating u.                    *)

        FOR i := r0 TO r-1 DO
            FOR j := c0+1 TO c-1 DO
                A[i,j] := A[i,j] - gamma * v^[i-r0] * w^[j-c0];
            END (*FOR*);
        END (*FOR*);

        (* Subtract gamma v v* from H.  *)

        FOR i := r0 TO r-1 DO
            FOR j := r0 TO r-1 DO
                H[i,j] := H[i,j] - gamma * v^[i-r0] * v^[j-r0];
            END (*FOR*);
        END (*FOR*);

        Vec.DisposeVector (w, c-c0);
        Vec.DisposeVector (v, r-r0);
        INC (r0);  INC (c0);

    END ReHouseholder;

(************************************************************************)

PROCEDURE CxHouseholder (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGCOMPLEX;
                         r, c: CARDINAL;
                         VAR (*INOUT*) r0, c0: CARDINAL;
                         VAR (*OUT*) H: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Householder transformation H on an rxc matrix A, to set the      *)
    (* elements below diagonal A[r0,c0] to zero.  We update r0 and c0   *)
    (* so that the caller is ready for the next call.                   *)

    CONST fieldsize = 8;

    VAR alpha, gamma, val: LONGCOMPLEX;
        norm: LONGREAL;
        v, w: Vec.CxVectorPtr;
        i, j, k: CARDINAL;

    BEGIN
        CxUnit(H, r);

        (* Find the norm of the vector to be transformed. We only want  *)
        (* to transform A from row r0 downwards, so this vector is the  *)
        (* lower part of column c0.                                     *)

        norm := 0.0;
        FOR i := r0 TO r-1 DO
            val := A[i, c0];
            norm := norm + RE(val)*RE(val) + IM(val)*IM(val);
        END (*FOR*);
        norm := sqrt(norm);

        (* norm = 0 at this point means an all-zero column.  This means *)
        (* that the caller must move on to the next column, so we       *)
        (* update c0 without updating r0.                               *)

        IF norm < small THEN
            INC (c0);
            RETURN;
        END (*IF*);

        (* Choose a suitable alpha. *)

        alpha := CMPLX(norm, 0.0);

        (* Let u = x - alpha y, where y is a unit vector,   *)
        (* and let v = u/norm(u)                            *)

        v := Vec.NewCxVector(r-r0);

        norm := 0.0;
        IF r0 < r-1 THEN
            FOR i := r0+1 TO r-1 DO
                val := A[i, c0];
                v^[i-r0] := val;
                norm := norm + RE(val)*RE(val) + IM(val)*IM(val);
            END (*FOR*);
        END (*IF*);

        (* Complete the u and v calculation. *)

        val := A[r0, c0] - alpha;
        v^[0] := val;
        norm := norm + RE(val)*RE(val) + IM(val)*IM(val);

        (* If norm = 0 at this point then we have a "do nothing"    *)
        (* transformation, so return with H = I.    Note that this  *)
        (* does not necessarily mean an all-zero column, a case     *)
        (* that was detected above.                                 *)

        IF norm < small THEN
            Vec.DisposeCxVector (v, r-r0);
            INC (r0);  INC (c0);
            RETURN;
        END (*IF*);

        norm := 1.0/sqrt(norm);
        FOR i := 0 TO r-r0-1 DO
            v^[i] := scalarMult (norm, v^[i]);
        END (*FOR*);

        (* Define w = v*A -- where here we interpret A to mean only *)
        (* the lower right block of the original A.                 *)

        w := Vec.NewCxVector(c-c0);
        FOR j := 0 TO c-c0-1 DO
            val := CxZero;
            FOR k := 0 TO r-r0-1 DO
                val := val + conj(v^[k]) * A[r0+k,c0+j];
            END (*FOR*);
            w^[j] := val;
        END (*FOR*);

        (* Now beta = x*v/v*x, and gamma = 1 + beta.    *)
        (* We have already calculated beta* = w[0].     *)

        gamma := w^[0];        (* which is beta*  *)
        gamma := CMPLX(1.0,0.0) + conj(gamma)/gamma;

        (* Subtract gamma x v x w from A.  *)

        FOR i := r0 TO r-1 DO
            FOR j := c0 TO c-1 DO
                A[i,j] := A[i,j] - gamma * v^[i-r0] * w^[j-c0];
            END (*FOR*);
        END (*FOR*);

        (* Subtract gamma v v* from H.  *)

        FOR i := r0 TO r-1 DO
            FOR j := r0 TO r-1 DO
                H[i,j] := H[i,j] - gamma * v^[i-r0] * conj(v^[j-r0]);
            END (*FOR*);
        END (*FOR*);

        Vec.DisposeCxVector (w, c-c0);
        Vec.DisposeCxVector (v, r-r0);
        INC (r0);  INC (c0);

    END CxHouseholder;

(************************************************************************)

PROCEDURE ReColHouseholder (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGREAL;
                         r, c: CARDINAL;
                         VAR (*INOUT*) r0, c0: CARDINAL;
                         VAR (*OUT*) Hstar: ARRAY OF ARRAY OF LONGREAL);

    (* Like the Householder transformation, but we are finding an H     *)
    (* such that AH has zero entries in row r0 from columns c0+1        *)
    (* onwards.  The returned Hstar is the adjoint of this H.           *)

    VAR Astar: ArrayPtr;

    BEGIN
        (* In this version we take the shortcut of simply reducing it   *)
        (* to the usual Householder transformation.  The efficiency is  *)
        (* hurt only by a small amount.                                 *)

        Astar := NewArray (c, r);
        Transpose (A, r, c, Astar^);
        ReHouseholder (Astar^, c, r, c0, r0, Hstar);
        Transpose (Astar^, c, r, A);
        DisposeArray (Astar, c, r);
    END ReColHouseholder;

(************************************************************************)

PROCEDURE CxColHouseholder (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGCOMPLEX;
                         r, c: CARDINAL;
                         VAR (*INOUT*) r0, c0: CARDINAL;
                         VAR (*OUT*) Hstar: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Like the Householder transformation, but we are finding an H     *)
    (* such that AH has zero entries in row r0 from columns c0+1        *)
    (* onwards.  The returned Hstar is the adjoint of this H.           *)

    VAR Astar: CxArrayPtr;

    BEGIN
        (* In this version we take the shortcut of simply reducing it   *)
        (* to the usual Householder transformation.  The efficiency is  *)
        (* hurt only by a small amount.                                 *)

        Astar := NewCxArray (c, r);
        Adjoint (A, r, c, Astar^);
        CxHouseholder (Astar^, c, r, c0, r0, Hstar);
        Adjoint (Astar^, c, r, A);
        DisposeCxArray (Astar, c, r);
    END CxColHouseholder;

(************************************************************************)
(*                 REDUCING A MATRIX TO BIDIAGONAL FORM                 *)
(************************************************************************)

PROCEDURE ReSqToBidiag (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGREAL;
                        r: CARDINAL;
                        VAR (*OUT*) Q, P: ARRAY OF ARRAY OF LONGREAL);

    (* Assumption: A is a square nonsingular rxr matrix.  This is the   *)
    (* internal version where the caller has already reduced the        *)
    (* original matrix to this case.                                    *)

    (* Does a factorisation A = Q.B.P where all matrices are  rxr and   *)
    (* B is bidiagonal.  On return B has replaced the original A.       *)

    CONST fieldsize = 8;

    VAR
        j, r0, c0: CARDINAL;
        H, Qcopy: ArrayPtr;

    BEGIN
        Unit (Q, r);
        Unit (P, r);
        IF r < 2 THEN
            RETURN;
        END (*IF*);
        H := NewArray (r, r);
        Qcopy := NewArray (r, r);
        j := 0;

        (* Remark: in the multiplications in the loop below we rely on  *)
        (* the fact that a real Householder matrix is symmetric, so we  *)
        (* can avoid two "transpose" operations.                        *)

        LOOP
            r0 := j;  c0 := j;
            ReHouseholder (A, r, r, r0, c0, H^);
            ReCopy (Q, r, r, Qcopy^);
            Mul (Qcopy^, H^, r, r, r, Q);

            IF j > r-2 THEN
                EXIT (*LOOP*);
            END (*IF*);

            (* Now a column operation. *)

            r0 := j;  c0 := j+1;
            ReColHouseholder (A, r, r, r0, c0, H^);
            ReCopy (P, r, r, Qcopy^);
            Mul (H^, Qcopy^, r, r, r, P);
            INC (j);
        END (*LOOP*);

        DisposeArray (Qcopy, r, r);
        DisposeArray (H, r, r);

    END ReSqToBidiag;

(************************************************************************)

PROCEDURE ReToBidiag (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                   VAR (*OUT*) Q, B, P: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Does a factorisation A = Q.B.P where Q*Q = I, PP* = I, and B is  *)
    (* a qxq bidiagonal matrix.  The return value q is the rank of A.   *)
    (* Exception: For an all-zero A we return 1 as the "rank", to be    *)
    (* consistent with the returned array sizes.                        *)

    CONST fieldsize = 8;

    VAR
        q0, rank: CARDINAL;
        L, Qcopy, Q1, P1, Pcopy: ArrayPtr;

    BEGIN
        (* Start with a PLQ factorisation to ensure that from then on   *)
        (* we will only have to deal with the square nonsingular case.  *)

        q0 := r;
        IF c < q0 THEN
            q0 := c;
        END (*IF*);
        L := NewArray (q0, q0);

        (* Initial factorisation. *)

        rank := RePLQFactor (A, r, c, Q, L^, P);
        IF rank = 0 THEN rank := 1 END (*IF*);

        (* Dimensions are now: Q is r by rank, L is rank by rank,   *)
        (* P is rank by c.                                          *)

        IF rank > 1 THEN
            Q1 := NewArray (r, rank);
            P1 := NewArray (rank, c);
            ReSqToBidiag (L^, rank, Q1^, P1^);

            Qcopy := NewArray (r, rank);
            Pcopy := NewArray (rank, c);

            ReCopy (Q, r, rank, Qcopy^);
            Mul (Qcopy^, Q1^, r, rank, rank, Q);
            ReCopy (P, rank, c, Pcopy^);
            Mul (P1^, Pcopy^, rank, rank, c, P);

            DisposeArray (Qcopy, r, rank);
            DisposeArray (Pcopy, rank, c);
            DisposeArray (P1, rank, c);
            DisposeArray (Q1, r, rank);
        END (*IF*);

        ReCopy (L^, q0, q0, B);
        DisposeArray (L, q0, q0);
        RETURN rank;

    END ReToBidiag;

(************************************************************************)

PROCEDURE CxSqToBidiag (VAR (*INOUT*) A: ARRAY OF ARRAY OF LONGCOMPLEX;
                        r: CARDINAL;
                        VAR (*OUT*) Q, P: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Assumption: A is a square nonsingular rxr matrix.  This is the   *)
    (* internal version where the caller has already reduced the        *)
    (* original matrix to this case.                                    *)

    (* Does a factorisation A = Q.B.P where all matrices are  rxr and   *)
    (* B is bidiagonal.  On return B has replaced the original A.       *)

    CONST fieldsize = 8;

    VAR
        j, r0, c0: CARDINAL;
        H, Hadj, Qcopy: CxArrayPtr;

    BEGIN
        CxUnit (Q, r);
        CxUnit (P, r);
        IF r < 2 THEN
            RETURN;
        END (*IF*);
        H := NewCxArray (r, r);
        Hadj := NewCxArray (r,r);
        Qcopy := NewCxArray (r, r);
        j := 0;

        (* Remark: in the multiplications in the loop below we rely on  *)
        (* the fact that a real Householder matrix is symmetric, so we  *)
        (* can avoid two "transpose" operations.                        *)

        LOOP
            r0 := j;  c0 := j;
            CxHouseholder (A, r, r, r0, c0, H^);
            Adjoint (H^, r, r, Hadj^);
            CxCopy (Q, r, r, Qcopy^);
            CxMul (Qcopy^, Hadj^, r, r, r, Q);

            IF j > r-2 THEN
                EXIT (*LOOP*);
            END (*IF*);

            (* Now a column operation. *)

            r0 := j;  c0 := j+1;
            CxColHouseholder (A, r, r, r0, c0, Hadj^);
            CxCopy (P, r, r, Qcopy^);
            CxMul (Hadj^, Qcopy^, r, r, r, P);
            INC (j);
        END (*LOOP*);

        DisposeCxArray (Qcopy, r, r);
        DisposeCxArray (Hadj, r, r);
        DisposeCxArray (H, r, r);

    END CxSqToBidiag;

(************************************************************************)

PROCEDURE CxToBidiag (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                   VAR (*OUT*) Q, B, P: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Does a factorisation A = Q.B.P where Q*Q = I, PP* = I, and B is  *)
    (* a qxq bidiagonal matrix.  The return value q is the rank of A.   *)
    (* Exception: For an all-zero A we return 1 as the "rank", to be    *)
    (* consistent with the returned array sizes.                        *)

    CONST fieldsize = 8;

    VAR
        q0, rank: CARDINAL;
        L, Qcopy, Q1, P1, Pcopy: CxArrayPtr;

    BEGIN
        (* Start with a PLQ factorisation to ensure that from then on   *)
        (* we will only have to deal with the square nonsingular case.  *)

        q0 := r;
        IF c < q0 THEN
            q0 := c;
        END (*IF*);
        L := NewCxArray (q0, q0);

        (* Initial factorisation. *)

        rank := CxPLQFactor (A, r, c, Q, L^, P);
        IF rank = 0 THEN rank := 1 END (*IF*);

        (* Dimensions are now: Q is r by rank, L is rank by rank,   *)
        (* P is rank by c.                                          *)

        IF rank > 1 THEN
            Q1 := NewCxArray (r, rank);
            P1 := NewCxArray (rank, c);
            CxSqToBidiag (L^, rank, Q1^, P1^);

            Qcopy := NewCxArray (r, rank);
            Pcopy := NewCxArray (rank, c);

            CxCopy (Q, r, rank, Qcopy^);
            CxMul (Qcopy^, Q1^, r, rank, rank, Q);
            CxCopy (P, rank, c, Pcopy^);
            CxMul (P1^, Pcopy^, rank, rank, c, P);

            DisposeCxArray (Qcopy, r, rank);
            DisposeCxArray (Pcopy, rank, c);
            DisposeCxArray (P1, rank, c);
            DisposeCxArray (Q1, r, rank);
        END (*IF*);

        CxCopy (L^, q0, q0, B);
        DisposeCxArray (L, q0, q0);
        RETURN rank;

    END CxToBidiag;

(************************************************************************)
(*                       UPPER TRIANGULAR MATRICES                      *)
(************************************************************************)

PROCEDURE ReUTinverse (S: ARRAY OF ARRAY OF LONGREAL;  r: CARDINAL;
                                VAR (*OUT*) T: ARRAY OF ARRAY OF LONGREAL);

    (* Returns T = inverse(S), for the special case where S is an       *)
    (* upper triangular rxr matrix.  The caller must guarantee that     *)
    (* the matrix is nonsingular.                                       *)

    VAR i, j, k: CARDINAL;
        temp, Tii: LONGREAL;

    BEGIN
        FOR i := r-1 TO 0 BY -1 DO
            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    T[i,j] := 0.0;
                END (*FOR*);
            END (*IF*);
            Tii := 1.0/S[i,i];
            T[i,i] := Tii;
            IF i < r-1 THEN
                FOR j := i+1 TO r-1 DO
                    temp := 0.0;
                    FOR k := i+1 TO j DO
                        temp := temp + S[i,k] * T[k,j];
                    END (*FOR*);
                    T[i,j] := -temp*Tii;
                END (*FOR*);
             END (*IF*);
        END (*FOR*);
    END ReUTinverse;

(************************************************************************)

PROCEDURE CxUTinverse (S: ARRAY OF ARRAY OF LONGCOMPLEX;  r: CARDINAL;
                                VAR (*OUT*) T: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Returns T = inverse(S), for the special case where S is an       *)
    (* upper triangular rxr matrix.  The caller must guarantee that     *)
    (* the matrix is nonsingular.                                       *)

    VAR i, j, k: CARDINAL;
        temp, Tii: LONGCOMPLEX;

    BEGIN
        FOR i := r-1 TO 0 BY -1 DO
            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    T[i,j] := CxZero;
                END (*FOR*);
            END (*IF*);
            Tii := CMPLX (1.0, 0.0) / S[i,i];
            T[i,i] := Tii;
            IF i < r-1 THEN
                FOR j := i+1 TO r-1 DO
                    temp := CxZero;
                    FOR k := i+1 TO j DO
                        temp := temp + S[i,k] * T[k,j];
                    END (*FOR*);
                    T[i,j] := -temp*Tii;
                END (*FOR*);
             END (*IF*);
        END (*FOR*);
    END CxUTinverse;

(************************************************************************)
(*                       LOWER TRIANGULAR MATRICES                      *)
(************************************************************************)

PROCEDURE ReLTinverse (S: ARRAY OF ARRAY OF LONGREAL;  r: CARDINAL;
                                VAR (*OUT*) T: ARRAY OF ARRAY OF LONGREAL);

    (* Returns T = inverse(S), for the special case where S is an       *)
    (* lower triangular rxr matrix.  The caller must guarantee that     *)
    (* the matrix is nonsingular.                                       *)

    VAR i, j, k: CARDINAL;
        temp, Tii: LONGREAL;

    BEGIN
        FOR i := 0 TO r-1 DO
            IF i < r-1 THEN
                FOR j := i+1 TO r-1 DO
                    T[i,j] := 0.0;
                END (*FOR*);
            END (*IF*);
            Tii := 1.0/S[i,i];
            T[i,i] := Tii;
            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    temp := 0.0;
                    FOR k := j TO i-1 DO
                        temp := temp + S[i,k] * T[k,j];
                    END (*FOR*);
                    T[i,j] := -temp*Tii;
                END (*FOR*);
             END (*IF*);
        END (*FOR*);
    END ReLTinverse;

(************************************************************************)

PROCEDURE CxLTinverse (S: ARRAY OF ARRAY OF LONGCOMPLEX;  r: CARDINAL;
                                VAR (*OUT*) T: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Returns T = inverse(S), for the special case where S is an       *)
    (* lower triangular rxr matrix.  The caller must guarantee that     *)
    (* the matrix is nonsingular.                                       *)

    VAR i, j, k: CARDINAL;
        temp, Tii: LONGCOMPLEX;

    BEGIN
        FOR i := 0 TO r-1 DO
            IF i < r-1 THEN
                FOR j := i+1 TO r-1 DO
                    T[i,j] := CxZero;
                END (*FOR*);
            END (*IF*);
            Tii := CMPLX (1.0, 0.0) / S[i,i];
            T[i,i] := Tii;
            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    temp := CxZero;
                    FOR k := j TO i-1 DO
                        temp := temp + S[i,k] * T[k,j];
                    END (*FOR*);
                    T[i,j] := -temp*Tii;
                END (*FOR*);
             END (*IF*);
        END (*FOR*);
    END CxLTinverse;

(************************************************************************)
(*                            PSEUDOINVERSE                             *)
(************************************************************************)

PROCEDURE ReUTpsinverse (S: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL;
                           VAR (*OUT*) T: ARRAY OF ARRAY OF LONGREAL);

    (* Returns the Moore-Penrose pseudoinverse T = psinv(S), where S is *)
    (* an upper triangular rows x cols matrix, with full row rank.      *)
    (* (The caller is responsible for ensuring that S has these         *)
    (* properties.) The result is a cols x rows matrix.                 *)

    VAR Q, L, Linv: ArrayPtr;
        temp: LONGREAL;
        i, j, k, rank: CARDINAL;

    BEGIN
        IF rows = cols THEN
            ReUTinverse (S, rows, T);
        ELSE
            (* rows < cols, so S has an LQ decomposition S = L x Q for  *)
            (* which L is a square (rows x rows) nonsingular upper      *)
            (* triangular matrix.                                       *)

            Linv := NewArray (rows, rows);
            Q := NewArray (rows, cols);
            L := NewArray (rows, rows);

            rank := ReLQFactor (S, rows, cols, L^, Q^);
            IF rank <> rows THEN
                WriteString ("ERROR: rank condition failed in ReUTpsinverse");
                WriteLn;
            END (*IF*);

            ReLTinverse (L^, rows, Linv^);
            DisposeArray (L, rows, rows);

            (* The result is adjoint(Q) times Linv. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    temp := 0.0;
                    FOR k := j TO rows-1 DO
                        temp := temp + Q^[k,i] * Linv^[k,j];
                    END (*FOR*);
                    T[i,j] := temp;
                END (*FOR*);
            END (*FOR*);
            DisposeArray (Q, rows, cols);
            DisposeArray (Linv, rows, rows);
        END (*IF*);
    END ReUTpsinverse;

(************************************************************************)

PROCEDURE CxUTpsinverse (S: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL;
                           VAR (*OUT*) T: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Returns the Moore-Penrose pseudoinverse T = psinv(S), where S is *)
    (* an upper triangular rows x cols matrix, with full row rank.      *)
    (* (The caller is responsible for ensuring that S has these         *)
    (* properties.) The result is a cols x rows matrix.                 *)

    VAR Q, L, Linv: CxArrayPtr;
        temp: LONGCOMPLEX;
        i, j, k, rank: CARDINAL;

    BEGIN
        IF rows = cols THEN
            CxUTinverse (S, rows, T);
        ELSE
            (* rows < cols, so S has an LQ decomposition S = L x Q for  *)
            (* which L is a square (rows x rows) nonsingular upper      *)
            (* triangular matrix.                                       *)

            Linv := NewCxArray (rows, rows);
            Q := NewCxArray (rows, cols);
            L := NewCxArray (rows, rows);

            rank := CxLQFactor (S, rows, cols, L^, Q^);
            IF rank <> rows THEN
                WriteString ("ERROR: rank condition failed in CxUTpsinverse");
                WriteLn;
            END (*IF*);

            CxLTinverse (L^, rows, Linv^);
            DisposeCxArray (L, rows, rows);

            (* The result is adjoint(Q) times Linv. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    temp := CxZero;
                    FOR k := j TO rows-1 DO
                        temp := temp + conj(Q^[k,i]) * Linv^[k,j];
                    END (*FOR*);
                    T[i,j] := temp;
                END (*FOR*);
            END (*FOR*);
            DisposeCxArray (Q, rows, cols);
            DisposeCxArray (Linv, rows, rows);
        END (*IF*);
    END CxUTpsinverse;

(************************************************************************)

PROCEDURE RePseudoInverse (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL;
                             VAR (*OUT*) PSI: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Returns the Moore-Penrose pseudoinverse PSI of an arbitrary      *)
    (* rowsxcols matrix A, and also returns the rank of A.              *)
    (* The result is a colsxrows matrix.                                *)

    VAR Q, R, Rpsinv: ArrayPtr;
        val: LONGREAL;
        rank, i, j, k: CARDINAL;

    BEGIN
        (* Perform a QR factorisation of A. *)

        Q := NewArray (rows, rows);
        R := NewArray (rows, cols);

        rank := ReQRFactor (A, rows, cols, Q^, R^);

        IF rank = 0 THEN

            (* Special case: a zero matrix. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    PSI[i,j] := 0.0;
                END (*FOR*);
            END (*FOR*);

        ELSE
            (* Calculate the pseudoinverse of R. *)

            Rpsinv := NewArray (cols, rank);
            ReUTpsinverse (R^, rank, cols, Rpsinv^);

            (* Postmultiply by Q adjoint. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    val := 0.0;
                    FOR k := 0 TO rank-1 DO
                        val := val + Rpsinv^[i,k] * Q^[j,k];
                    END (*FOR*);
                    PSI[i,j] := val;
                END (*FOR*);
            END (*FOR*);

            DisposeArray (Rpsinv, cols, rank);

        END (*IF*);

        DisposeArray (R, rows, cols);
        DisposeArray (Q, rows, rows);

        RETURN rank;

    END RePseudoInverse;

(************************************************************************)

PROCEDURE CxPseudoInverse (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL;
                             VAR (*OUT*) PSI: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Returns the Moore-Penrose pseudoinverse PSI of an arbitrary      *)
    (* rowsxcols matrix A, and also returns the rank of A.              *)
    (* The result is a colsxrows matrix.                                *)

    VAR Q, R, Rpsinv: CxArrayPtr;
        val: LONGCOMPLEX;
        rank, i, j, k: CARDINAL;

    BEGIN
        (* Perform a QR factorisation of A. *)

        Q := NewCxArray (rows, rows);
        R := NewCxArray (rows, cols);

        rank := CxQRFactor (A, rows, cols, Q^, R^);

        IF rank = 0 THEN

            (* Special case: a zero matrix. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    PSI[i,j] := CxZero;
                END (*FOR*);
            END (*FOR*);

        ELSE
            (* Calculate the pseudoinverse of R. *)

            Rpsinv := NewCxArray (cols, rank);
            CxUTpsinverse (R^, rank, cols, Rpsinv^);

            (* Postmultiply by Q adjoint. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    val := CxZero;
                    FOR k := 0 TO rank-1 DO
                        val := val + Rpsinv^[i,k] * conj(Q^[j,k]);
                    END (*FOR*);
                    PSI[i,j] := val;
                END (*FOR*);
            END (*FOR*);

            DisposeCxArray (Rpsinv, cols, rank);

        END (*IF*);

        DisposeCxArray (R, rows, cols);
        DisposeCxArray (Q, rows, rows);

        RETURN rank;

    END CxPseudoInverse;

(************************************************************************)
(*                       CHOLESKY FACTORISATION                         *)
(************************************************************************)

PROCEDURE ReCholesky (R: ARRAY OF ARRAY OF LONGREAL;  N: CARDINAL;
                   VAR (*OUT*) S: ARRAY OF ARRAY OF LONGREAL): BOOLEAN;

    (* Cholesky decomposition of a positive definite symmetric NxN      *)
    (* matrix R as S*S, where S is upper triangular, and its diagonal   *)
    (* entries are real and positive. If R is not positive definite and *)
    (* symmetric the calculation will fail.                             *)

    (* We could have done an in-place calculation, but the way we call  *)
    (* this function it turns out to be more convenient to have         *)
    (* separate input and output arrays.                                *)

    (* Returns FALSE if matrix was not positive definite.               *)

    VAR
        i, j, k: CARDINAL;
        sum, Sii, Ski: LONGREAL;

    BEGIN
        FOR i := 0 TO N-1 DO

            (* Put zero in the below-diagonal entries of the result. *)

            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    S[i,j] := 0.0;
                END (*FOR*);
            END (*IF*);

            (* Calculate the diagonal entry. *)

            sum := R[i,i];
            IF i > 0 THEN
                FOR k := 0 TO i-1 DO
                    Ski := S[k,i];
                    sum := sum - Ski * Ski;
                END (*FOR*);
            END (*IF*);

            IF sum <= 0.0 THEN RETURN FALSE END(*IF*);
            Sii := sqrt(sum);

            S[i,i] := Sii;

            (* Now the rest of row i. *)

            FOR j := i+1 TO N-1 DO
                sum := R[i,j];
                IF i > 0 THEN
                    FOR k := 0 TO i-1 DO
                        sum := sum - S[k,i] * S[k,j];
                    END (*FOR*);
                END (*IF*);
                S[i,j] := sum / Sii;
            END (*FOR*);
        END (*FOR*);
        RETURN TRUE;
    END ReCholesky;

(************************************************************************)

PROCEDURE CxCholesky (R: ARRAY OF ARRAY OF LONGCOMPLEX;  N: CARDINAL;
                   VAR (*OUT*) S: ARRAY OF ARRAY OF LONGCOMPLEX): BOOLEAN;

    (* Cholesky decomposition of a positive definite Hermitian NxN      *)
    (* matrix R as S*S, where S is upper triangular, and its diagonal   *)
    (* entries are real and positive. If R is not positive definite and *)
    (* Hermitian the calculation will fail.                             *)

    (* We could have done an in-place calculation, but the way we call  *)
    (* this function it turns out to be more convenient to have         *)
    (* separate input and output arrays.                                *)

    (* Returns FALSE if matrix was not positive definite.               *)

    VAR
        i, j, k: CARDINAL;
        Sii, resum: LONGREAL;
        Ski, cxsum: LONGCOMPLEX;

    BEGIN
        FOR i := 0 TO N-1 DO

            (* Put zero in the below-diagonal entries of the result. *)

            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    S[i,j] := CxZero;
                END (*FOR*);
            END (*IF*);

            (* Calculate the diagonal entry. *)

            resum := RE(R[i,i]);
            IF i > 0 THEN
                FOR k := 0 TO i-1 DO
                    Ski := S[k,i];
                    resum := resum - RE(Ski) * RE(Ski) - IM(Ski) * IM(Ski);
                END (*FOR*);
            END (*IF*);

            IF resum <= 0.0 THEN RETURN FALSE END(*IF*);
            Sii := sqrt(resum);

            S[i,i] := CMPLX (Sii, 0.0);

            (* Now the rest of row i. *)

            FOR j := i+1 TO N-1 DO
                cxsum := R[i,j];
                IF i > 0 THEN
                    FOR k := 0 TO i-1 DO
                        cxsum := cxsum - conj(S[k,i]) * S[k,j];
                    END (*FOR*);
                END (*IF*);
                S[i,j] := scalarMult (1.0/Sii, cxsum);
            END (*FOR*);
        END (*FOR*);
        RETURN TRUE;
    END CxCholesky;

(************************************************************************)
(*                                                                      *)
(*                    SINGULAR VALUE DECOMPOSITION                      *)
(*                                                                      *)
(* A singular value decomposition is a factorisation A = U S V*, where  *)
(* where U is an mxm real or complex unitary matrix, S is a mxn         *)
(* rectangular diagonal matrix with non-negative real numbers on the    *)
(* diagonal, and V is an nxn real or complex unitary matrix. The        *)
(* diagonal entries S are known as the singular values of A.            *)
(*                                                                      *)
(* One way to compute the decomposition is by a succession of QR and LQ *)
(* decompositions, and this is what we do in this module.               *)
(*                                                                      *)
(* We do, in any case, depart from the conventional approaches to SVD.  *)
(* Because we are doing minimal QR and LQ factorisations, we end up     *)
(* with a square S matrix that gives only the nonzero singular values.  *)
(* No information is lost, because we know that the remaining singular  *)
(* values that we haven't computed are all zero.                        *)
(*                                                                      *)
(************************************************************************)

PROCEDURE MaxOffDiag (A: ARRAY OF ARRAY OF LONGREAL;  N: CARDINAL): LONGREAL;

    (* Maximum off-diagonal absolute value in a square array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO N-1 DO
            FOR j := 0 TO N-1 DO
                IF j <> i THEN
                    val := ABS(A[i,j]);
                    IF val > max THEN
                        max := val;
                    END (*IF*);
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END MaxOffDiag;

(************************************************************************)

PROCEDURE CxMaxOffDiag (A: ARRAY OF ARRAY OF LONGCOMPLEX;  N: CARDINAL): LONGREAL;

    (* Maximum off-diagonal absolute value in a square array. *)

    VAR i, j: CARDINAL;
        val, max: LONGREAL;

    BEGIN
        max := 0.0;
        FOR i := 0 TO N-1 DO
            FOR j := 0 TO N-1 DO
                IF j <> i THEN
                    val := abs(A[i,j]);
                    IF val > max THEN
                        max := val;
                    END (*IF*);
                END (*IF*);
            END (*FOR*);
        END (*FOR*);
        RETURN max;
    END CxMaxOffDiag;

(************************************************************************)

PROCEDURE ReSVD (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL;
                    VAR (*OUT*) P, L, Q: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Factors A as A = PxLxQ, where P*P=I, QQ*=I, and L is a diagonal  *)
    (* matrix whose diagonal entries are the nonzero singular values    *)
    (* of A.  The dimensions of P, L, and Q are rowsxq, qxq, and qxcols *)
    (* respectively, where q, which is returned as the function result, *)
    (* is the rank of A.  Exception: if A = 0 we return q=0, but the    *)
    (* matrix sizes are as if q had been 1.                             *)

    CONST
        MaxIterations = 500;
        UseBidiag = TRUE;

    VAR oldP, oldL, oldQ, newP, newQ: ArrayPtr;
        rank, count, dummy: CARDINAL;

    BEGIN

        (* Initial factorisation. *)

        count := 0;
        rank := RePLQFactor (A, rows, cols, P, L, Q);

        oldP := NewArray (rows, rank);
        oldL := NewArray (rank, rank);
        oldQ := NewArray (rank, cols);
        newP := NewArray (rank, rank);
        newQ := NewArray (rank, rank);

        IF UseBidiag AND (rank > 1) THEN

            (* Reduce to bidiagonal form.  We do this only because      *)
            (* everyone else is doing it.  My tests seem to show that   *)
            (* this step reduces accuracy slightly, and has almost no   *)
            (* effect on the number of iterations, because the latter   *)
            (* is dominated by the long stretch at the end where we are *)
            (* trying to reduce one or two very small numbers.          *)

            ReCopy (P, rows, rank, oldP^);
            ReCopy (L, rank, rank, oldL^);
            ReCopy (Q, rank, cols, oldQ^);
            rank := ReToBidiag (oldL^, rank, rank, newP^, L, newQ^);
            Mul (oldP^, newP^, rows, rank, rank, P);
            Mul (newQ^, oldQ^, rank, rank, cols, Q);
        END (*IF*);

        (* Iterative reduction of off-diagonal elements. *)

        WHILE (count < MaxIterations) AND (MaxOffDiag(L,rank) > small) DO

            ReCopy (P, rows, rank, oldP^);
            ReCopy (L, rank, rank, oldL^);
            ReCopy (Q, rank, cols, oldQ^);

            dummy := RePLQFactor (oldL^, rank, rank, newP^, L, newQ^);
            Mul (oldP^, newP^, rows, rank, rank, P);
            Mul (newQ^, oldQ^, rank, rank, cols, Q);

            INC (count);

        END (*WHILE*);

        DisposeArray (newQ, rank, cols);
        DisposeArray (newP, rows, rank);
        DisposeArray (oldQ, rank, cols);
        DisposeArray (oldL, rank, rank);
        DisposeArray (oldP, rows, rank);

        RETURN rank;

    END ReSVD;

(************************************************************************)

PROCEDURE CxSVD (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL;
                    VAR (*OUT*) P, L, Q: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Factors A as A = PxLxQ, where P*P=I, QQ*=I, and L is a diagonal  *)
    (* matrix whose diagonal entries are the nonzero singular values    *)
    (* of A.  The dimensions of P, L, and Q are rowsxq, qxq, and qxcols *)
    (* respectively, where q, which is returned as the function result, *)
    (* is the rank of A.  Exception: if A = 0 we return q=0, but the    *)
    (* matrix sizes are as if q had been 1.                             *)

    CONST
        MaxIterations = 100;
        UseBidiag = TRUE;

    VAR oldP, oldL, oldQ, newP, newQ: CxArrayPtr;
        rank, count, dummy: CARDINAL;

    BEGIN

        (* Initial factorisation. *)

        count := 0;
        rank := CxPLQFactor (A, rows, cols, P, L, Q);

        oldP := NewCxArray (rows, rank);
        oldL := NewCxArray (rank, rank);
        oldQ := NewCxArray (rank, cols);
        newP := NewCxArray (rank, rank);
        newQ := NewCxArray (rank, rank);

        IF UseBidiag AND (rank > 1) THEN

            (* Reduce to bidiagonal form.  We do this only because      *)
            (* everyone else is doing it.  My tests seem to show that   *)
            (* this step reduces accuracy slightly, and has almost no   *)
            (* effect on the number of iterations, because the latter   *)
            (* is dominated by the long stretch at the end where we are *)
            (* trying to reduce one or two very small numbers.          *)

            CxCopy (P, rows, rank, oldP^);
            CxCopy (L, rank, rank, oldL^);
            CxCopy (Q, rank, cols, oldQ^);
            rank := CxToBidiag (oldL^, rank, rank, newP^, L, newQ^);
            CxMul (oldP^, newP^, rows, rank, rank, P);
            CxMul (newQ^, oldQ^, rank, rank, cols, Q);
        END (*IF*);

        (* Iterative reduction of off-diagonal elements. *)

        WHILE (count < MaxIterations) AND (CxMaxOffDiag(L,rank) > small) DO

            CxCopy (P, rows, rank, oldP^);
            CxCopy (L, rank, rank, oldL^);
            CxCopy (Q, rank, cols, oldQ^);

            dummy := CxPLQFactor (oldL^, rank, rank, newP^, L, newQ^);
            CxMul (oldP^, newP^, rows, rank, rank, P);
            CxMul (newQ^, oldQ^, rank, rank, cols, Q);

            INC (count);

        END (*WHILE*);

        DisposeCxArray (newQ, rank, cols);
        DisposeCxArray (newP, rows, rank);
        DisposeCxArray (oldQ, rank, cols);
        DisposeCxArray (oldL, rank, rank);
        DisposeCxArray (oldP, rows, rank);

        RETURN rank;

    END CxSVD;

(************************************************************************)

END MatExtra.

