#!/usr/bin/env python
#############################################################################
# Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1997, 1998, 1999
# All Rights Reserved.
#
# The software contained on this media is the property of the DSTC Pty
# Ltd.  Use of this software is strictly in accordance with the
# license agreement in the accompanying LICENSE.HTML file.  If your
# distribution of this software does not contain a LICENSE.HTML file
# then you have no rights to use this software in any manner and
# should contact DSTC at the address below to determine an appropriate
# licensing arrangement.
# 
#      DSTC Pty Ltd
#      Level 7, GP South
#      Staff House Road
#      University of Queensland
#      St Lucia, 4072
#      Australia
#      Tel: +61 7 3365 4310
#      Fax: +61 7 3365 4311
#      Email: enquiries@dstc.edu.au
# 
# This software is being provided "AS IS" without warranty of any
# kind.  In no event shall DSTC Pty Ltd be liable for damage of any
# kind arising out of or in connection with the use or performance of
# this software.
#
# Project:      Fnorb
# File:         $Source: /units/arch/src/Fnorb/compiler/RCS/CodeGenerator.py,v $
# Version:      @(#)$RCSfile: CodeGenerator.py,v $ $Revision: 1.8 $
#
#############################################################################
""" Base class for Python code generators for OMG IDL. """


# Standard/built-in modules.
import errno, os

# Fnorb modules.
from Fnorb.orb import CORBA, Util


class CodeGenerator:
    """ Base class for Python code generators for OMG IDL. """

    def generate(self, context, file, ifr_object, indent):
	""" Generate Python code for an Interface Repository object. """

	# Find out the kind of definition that the IR object represents.
	def_kind = ifr_object._get_def_kind()

	# Attribute.
	if def_kind == CORBA.dk_Attribute:
	    self._generate_attribute(context, file, ifr_object, indent)

	# Constant.
	elif def_kind == CORBA.dk_Constant:
	    self._generate_constant(context, file, ifr_object, indent)

	# Exception.
	elif def_kind == CORBA.dk_Exception:
	    self._generate_exception(context, file, ifr_object, indent)

	# Interface.
	elif def_kind == CORBA.dk_Interface:
	    self._generate_interface(context, file, ifr_object, indent)

	# Module.
	elif def_kind == CORBA.dk_Module:
	    self._generate_module(context, file, ifr_object, indent)

	# Operation.
	elif def_kind == CORBA.dk_Operation:
	    self._generate_operation(context, file, ifr_object, indent)

	# Structure.
	elif def_kind == CORBA.dk_Struct:
	    self._generate_struct(context, file, ifr_object, indent)

	# Union.
	elif def_kind == CORBA.dk_Union:
	    self._generate_union(context, file, ifr_object, indent)

	# Enumeration.
	elif def_kind == CORBA.dk_Enum:
	    self._generate_enum(context, file, ifr_object, indent)

	# Alias.
	elif def_kind == CORBA.dk_Alias:
	    self._generate_alias(context, file, ifr_object, indent)

	# Anything else, we just ignore ;^)
	else:
	    pass

	return

    #########################################################################
    # Protected interface.
    #########################################################################

    def _indent(self, indent):
	""" Produce the indentation for the specified indent level! """

	#       1234 ;^)
	return '    ' * indent

    def _create_package(self, context, package, module_id=''):
	""" Create a new Python package. """

	# Create the full package name relative to the context's package.
	package = context.package() + Util.PackageName(package)

	# Make sure that all of the intermediate packages in the path actually
	# exist (and create them if they don't!).
	for i in range(1, len(package)):
	    # Create the pathname of the intermediate package relative to the
	    # context's target directory.
	    package_path = context.create_package_path(package[:i])

	    # Create the package directory.
	    try:
		os.mkdir(package_path)

	    # If the package already exists then carry on!
            except os.error, (code, message):
		if code != errno.EEXIST:
		    raise os.error, (code, message)

            # See if the package already has an '__init__.py' module.
            try:
                file = open('%s/__init__.py' % package_path, 'r')
                file.close()
 
	    # If it doesn't, then create one.
	    except IOError:
                file = open('%s/__init__.py' % package_path, 'w')
                file.close()

	# Create the pathname of the package relative to the context's target
	# directory.
	package_path = context.create_package_path(package)

	# Create the package directory.
	try:
	    os.mkdir(package_path)

	# If the package already exists then carry on!
        except os.error, (code, message):
	    if code != errno.EEXIST:
		raise os.error, (code, message)

	# Create/open the package's '__init__' module.
	file = open('%s/__init__.py' % package_path, 'w')

	# Documentation string.
	file.write('""" Module: %s\n\n' % module_id)
	file.write("Automagically generated by:-\n")
	file.write("\n")
	file.write("%s\n" % CORBA.ORB_ID)
	file.write("\n")
	file.write('"""\n\n')

	# Repository id attribute.
	file.write('_FNORB_ID = "%s"\n\n' % module_id)

	# Python 'import' statements.
	file.write("# Fnorb modules.\n")
	file.write("import Fnorb.orb.CORBA\n")
	file.write("import Fnorb.orb.TypeManager\n")
	file.write("import Fnorb.orb.Util\n\n")
##	file.write("from Fnorb.orb import CORBA, TypeManager, Util\n\n")

	return file

    def _end_package(self, file, indent):
	""" Add any stuff required at the end of the package!

	Currently this is just an EOF marker!

	"""
	# EOF marker.
	file.write(self._indent(indent))
        file.write('#' * 77)
	file.write('\n')

	return

    def _write_typecode_constant(self, file, indent, ifr_id, tc, name):
	""" Generate code to add a typecode constant. """

	# Create the typecode constant.
	typecode_constant = tc._fnorb_to_constant()

	# Generate the code to add the type to the Type Manager.
	file.write(self._indent(indent))
	file.write('Fnorb.orb.TypeManager.TypeManager_init().add_type')
	file.write('("%s", "%s", %s)\n\n' % (ifr_id, typecode_constant, name))

	return

    def _get_typecode(self, typecode):
	""" Get the typecode 'string' for the given typecode. """

	# Primitive typecodes have a '_fnorb_code' method that generates the
	# appropriate string.
	if hasattr(typecode, '_fnorb_code'):
	    result = typecode._fnorb_code()

	# Otherwise, generate a call to get the typecode from the typecode
	# manager.
	else:
	    result = 'Fnorb.orb.CORBA.typecode("%s")' % typecode.id()

	return result

    # Override these methods to generate the code for each IFR object that
    # requires mapping to a Python data structure.
    def _generate_attribute(self, context, file, ifr_object, indent):
	pass

    def _generate_constant(self, context, file, ifr_object, indent):
	pass

    def _generate_exception(self, context, file, ifr_object, indent):
	pass

    def _generate_interface(self, context, file, ifr_object, indent):
	pass

    def _generate_module(self, context, file, ifr_object, indent):
	pass

    def _generate_operation(self, context, file, ifr_object, indent):
	pass

    def _generate_struct(self, context, file, ifr_object, indent):
	pass

    def _generate_union(self, context, file, ifr_object, indent):
	pass

    def _generate_enum(self, context, file, ifr_object, indent):
	pass

    def _generate_alias(self, context, file, ifr_object, indent):
	pass

#############################################################################
