/* tif_os2.c */

/*
 * TIFF Library OS/2-specific Routines.
 */
#include "tif_config.h"

#include <stdlib.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#if HAVE_FCNTL_H
# include <fcntl.h>
#endif

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#if HAVE_IO_H
# include <io.h>
#endif

#ifdef GBMMEM
# include "gbmmem.h"
#endif


#include <sys/stat.h>

#include <string.h>

#include "tiffiop.h"

static tsize_t
_tiffReadProc(thandle_t fd, tdata_t buf, tsize_t size)
{
	return ((tsize_t) read((int) fd, buf, (size_t) size));
}

static tsize_t
_tiffWriteProc(thandle_t fd, tdata_t buf, tsize_t size)
{
	return ((tsize_t) write((int) fd, buf, (size_t) size));
}

static toff_t
_tiffSeekProc(thandle_t fd, toff_t off, int whence)
{
	return ((toff_t) lseek((int) fd, (off_t) off, whence));
}

static int
_tiffCloseProc(thandle_t fd)
{
	return (close((int) fd));
}

static toff_t
_tiffSizeProc(thandle_t fd)
{
   const off_t fpos  = tell((int) fd);
   const off_t fsize = lseek((int) fd, 0, SEEK_END);
   if (lseek((int) fd, fpos, SEEK_SET) < 0)
   {
      return (toff_t) 0;
   }
   return (toff_t) (fsize < 0 ? 0 : fsize);
}

#ifdef HAVE_MMAP
 "Memory Mapped I/O is not available"
#endif /* !HAVE_MMAP */

/*
 * Open a TIFF file descriptor for read/writing.
 */
TIFF*
TIFFFdOpen(int fd, const char* name, const char* mode)
{
	TIFF* tif;

	tif = TIFFClientOpen(name, mode,
	    (thandle_t) fd,
	    _tiffReadProc, _tiffWriteProc,
	    _tiffSeekProc, _tiffCloseProc, _tiffSizeProc,
	    NULL, NULL);
	if (tif)
		tif->tif_fd = fd;
	return (tif);
}

/*
 * Open a TIFF file for read/writing.
 */
TIFF*
TIFFOpen(const char* name, const char* mode)
{
	static const char module[] = "TIFFOpen";
	int m, fd;
        TIFF* tif;

	m = _TIFFgetMode(mode, module);
	if (m == -1)
		return ((TIFF*)0);

        m |= O_BINARY;

	fd = open(name, m, S_IREAD | S_IWRITE);

	if (fd < 0) {
		TIFFError(module, "%s: Cannot open", name);
		return ((TIFF *)0);
	}

	tif = TIFFFdOpen((int)fd, name, mode);
	if(!tif)
		close(fd);
	return tif;
}

void*
_TIFFmalloc(tsize_t s)
{
#ifdef GBMMEM
	return (gbmmem_malloc((size_t) s));
#else
	return (malloc((size_t) s));
#endif
}

void
_TIFFfree(tdata_t p)
{
#ifdef GBMMEM
	gbmmem_free(p);
#else
	free(p);
#endif
}

void*
_TIFFrealloc(tdata_t p, tsize_t s)
{
#ifdef GBMMEM
	return (gbmmem_realloc(p, (size_t) s));
#else
	return (realloc(p, (size_t) s));
#endif
}

void
_TIFFmemset(tdata_t p, int v, tsize_t c)
{
	memset(p, v, (size_t) c);
}

void
_TIFFmemcpy(tdata_t d, const tdata_t s, tsize_t c)
{
	memcpy(d, s, (size_t) c);
}

int
_TIFFmemcmp(const tdata_t p1, const tdata_t p2, tsize_t c)
{
	return (memcmp(p1, p2, (size_t) c));
}

static void
os2WarningHandler(const char* module, const char* fmt, va_list ap)
{
	if (module != NULL)
		fprintf(stderr, "%s: ", module);
	fprintf(stderr, "Warning, ");
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, ".\n");
}
TIFFErrorHandler _TIFFwarningHandler = os2WarningHandler;

static void
os2ErrorHandler(const char* module, const char* fmt, va_list ap)
{
	if (module != NULL)
		fprintf(stderr, "%s: ", module);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, ".\n");
}
TIFFErrorHandler _TIFFerrorHandler = os2ErrorHandler;

