/*
 * func.c --
 */

#define INCL_REXXSAA

#include <os2.h>
#include <rexxsaa.h>

#include <stdio.h>
#include <ctype.h>

#include "func.h"
#include "varpool.h"

/*
 * SetVar -- A REXX external function that sets a variable...
 *
 *       SETVAR( varname, value )
 */

RexxFunctionHandler SetVar;

ULONG SetVar( PUCHAR funcname, ULONG numargs, PRXSTRING args,
              PSZ queuename, PRXSTRING result )
  {
    LONG     ret;

    /* Get rid of compiler warnings... */

    funcname  = funcname;
    queuename = queuename;

    /* Need exactly two arguments... */

    if( numargs != 2 ) return( 40 );

    ret = VarSetValue( args[0].strptr, &args[1] );

    result->strptr[0] = ( ( ret == RXSHV_OK || ret == RXSHV_NEWV ) ? '1' : '0' );
    result->strlength = 1;

    return( 0 );
  }

/*
 * DumpVars -- A REXX external function that dumps all the currently
 *             defined rexx variables.
 */

RexxFunctionHandler DumpVars;

ULONG DumpVars( PUCHAR funcname, ULONG numargs, PRXSTRING args,
                PSZ queuename, PRXSTRING result )
  {
    RXSTRING name;
    RXSTRING value;
    ULONG    count = 0;

    /* Get rid of compiler warnings... */

    funcname  = funcname;
    queuename = queuename;
    args      = args;

    /* Need exactly no arguments... */

    if( numargs != 0 ) return( 40 );

    printf( "---- Dumping variable pool:\n\n" );

    /* Scan through the list and print each one... */

    while( 1 ){
        name.strlength = value.strlength = 0;
        name.strptr    = value.strptr    = NULL;

        VarGetNextValue( &name, &value );
        if( !name.strptr ) break;

        if( value.strptr ){
            printf( "%s: %s\n", name.strptr, value.strptr );

            DosFreeMem( value.strptr );
        } else {
            printf( "%s: (NULL)\n", name.strptr );
        }

        DosFreeMem( name.strptr );

        ++count;
    }

    printf( "\n---- End of variable pool\n" );

    /* Return # of variables that were dumped... */

    result->strlength = sprintf( result->strptr, "%d", count );

    return( 0 );
  }

/*
 * Function registration... before starting any REXX macros, we have
 * to register our external functions with the interpreter.  We just
 * keep a table of functions and register each one in sequence.
 */

typedef struct {
    PSZ name;
    PFN function;
} rxfunc_entry, *rxfunc_entry_p;

static rxfunc_entry REXXFuncs[] =
  {
    { "SETVAR",   (PFN) SetVar },
    { "DUMPVARS", (PFN) DumpVars },
    { NULL,             NULL  }
  };

/*
 * RegisterREXXFuncs -- Register each external REXX function in the
 *                      table.  We use RexxRegisterFunctionExe to do
 *                      the registration.
 *
 *  NOTE: RexxRegisterFunctionExe takes two parms, one is the name of
 *        the function to register and the second is the address of the
 *        function.  The function does not need to be exported.  After
 *        registration the function is only available to the process
 *        that registered it.
 *
 *        In contrast, RexxRegisterFunctionDll requires that the function
 *        be exported from a DLL and is then made available to all REXX
 *        programs in any process.
 *
 *        When the REXX interpreter looks for a function, it first looks
 *        in the list of functions registered using RexxRegisterFunctionExe
 *        before looking at those registered using RexxRegisterFunctionDll.
 */

BOOL RegisterREXXFuncs( void )
  {
    rxfunc_entry_p ptr;

    for( ptr = REXXFuncs; ptr->name != NULL; ++ptr ){
        RexxRegisterFunctionExe( ptr->name, ptr->function );
    }

    return( TRUE );
  }

/*
 * DeregisterREXXFuncs -- Deregister each external REXX function.  Strictly
 *                        speaking, this is not necessary since the
 *                        registrations are freed once the process exits,
 *                        but it's always good practice to clean up nicely.
 *
 *  NOTE: Never deregister external REXX functions that were registered
 *        using RexxRegisterFunctionDll, as this swipes them out from
 *        under the feet of ANY and ALL REXX programs, even those
 *        currently running!  Deregistering functions that were
 *        registered with RexxRegisterFunctionEXE is OK to do since they
 *        were only registered as part of your process space.
 */

void DeregisterREXXFuncs( void )
  {
    rxfunc_entry_p ptr;

    for( ptr = REXXFuncs; ptr->name != NULL; ++ptr ){
        RexxDeregisterFunction( ptr->name );
    }
  }
