/* string.c -- changed for emx by Eberhard Mattes -- Nov 1998 */
/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/string.c
 * Summary:
 *	Specially checked string and memory functions, replacing those
 *	found in your normal C library.
 * Other notes:
 *	Some of these functions are built into GCC, so you may need to
 *	use the '-fno-builtin' flag to get these checked versions. The
 *	rest of this library uses the '__bounds_*' versions which are
 *	unchecked.
 * Author      	Date		Notes
 * RWMJ		27/2/95		Initial implementation.
 * RWMJ		4/4/95		After profiling, some optimizations.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "bounds-lib.h"
#include "check.h"

/* Undefine the definitions made in "bounds-lib.h" above. See that header file
 * for comments.
 */

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/* On `memchr' and `memccpy', it is probably valid to give a `n' (max. len.)
 * argument longer than the object, provided you know the `c' (character)
 * will match always before the end of the object. If you want strict
 * checking of `n', set these to `1'. If you do this, certain valid (?)
 * software fails, in particular GNU's `sprintf'.
 */
#define STRICT_MEMCHR		0
#define STRICT_MEMCCPY		0

/* Since `bcopy' varies from machine to machine in the way it handles
 * overlapping arguments, you may set its behaviour here. Set
 * `BCOPY_OVERLAPPING_ARGUMENTS' to 1 to get memmove-type behaviour, and
 * to 0 to get memcpy behaviour. Set `BCOPY_WARN_OVERLAPPING_ARGUMENTS'
 * if you want a warning when `bcopy' is called with overlapping arguments
 * (since relying on this is not necessarily portable).
 *
 * - RWMJ 16/1/96
 * - Thanks to Mark W. Snitily (mark@sgcs.com) for clarification here.
 */
#define BCOPY_OVERLAPPING_ARGUMENTS		1
#define BCOPY_WARN_OVERLAPPING_ARGUMENTS	0


static inline void *
check_ptr(char *filename, int line, void *pointer,char *function,char *ptr_name)
{
  if (pointer == NULL || pointer == ILLEGAL)
    {
      __bounds_errorf (filename, line, pointer, NULL,
		       "NULL or ILLEGAL %s used in %s", ptr_name, function);
      return NULL;
    }
  return pointer;
}

/*----------------------------------------------------------------------
 *	Inline function that does the donkey work checking a single
 *	pointer is valid over a range of n bytes. If the pointer is
 *	invalid, the function exits with an appropriate message.
 *	  If size == 0, the pointer points to a string. We work out the
 *	size ourselves here, and check the extent will be OK.
 *----------------------------------------------------------------------*/

static inline size_t
check (char *filename, int line, void *pointer, size_t size, char *function, char *ptr_name)
{
  object *obj;

  if (pointer == NULL || pointer == ILLEGAL)
    {
      __bounds_errorf (filename, line, pointer, NULL,
		       "NULL or ILLEGAL %s used in %s", ptr_name, function);
      return 0;
    }
  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, NULL, 0, function)
	|| maybe_is_unchecked_stack (pointer, NULL, 0, function)) {
      /* Unchecked pointer is OK. We have already delivered a warning at
       * this point. If this is a string, return its length, else return
       * anything.
       */
      if (size)		/* not a string */
	return 0;
      else {		/* unchecked operation => may fail */
	char *p = (char *) pointer;

        while (*p++);
        return p - (char *)pointer;
      }
    }
    __bounds_errorf (filename, line, pointer, NULL,
		     "invalid %s used in %s", ptr_name, function);
    return 0;
  }

  if (size) {
    /* The pointer itself is valid, and points to a checked object. Now make
     * sure that we won't overrun the object by using this pointer and size.
     */
    if ((char *)pointer + size > (char *)obj->extent)
      __bounds_errorf (filename, line, pointer, obj,
		       "%s with this %s and size %u would overrun the end of the object's allocated memory",
		       function, ptr_name, size);
    return 0;
  } else {
    /* This is a string.
     * Work out the size ourselves, and whether the pointer will be valid.
     * Return the length of the string + 1.
     */
    char *p = (char *) pointer;

    while (*p++);
    if (p > (char *)obj->extent)
      __bounds_errorf (filename, line, pointer, obj,
		       "in %s, %s is a string with size %u overrunning the end of the object's allocated memory",
		       function, ptr_name, p - (char *)pointer);
    return p - (char *)pointer;
  }
}

/*----------------------------------------------------------------------
 *	Check for overlapping objects in functions (such as memcpy, strcpy)
 *	where this is not permitted.
 *----------------------------------------------------------------------*/

static inline void
check_overlap (char *filename, int line,
	       void *p1, size_t n1,
	       void *p2, size_t n2, char *function_name)
{
  void *p2e = (void *) ((char *) p2 + n2 - 1);

  if (p1 > p2 && p1 <= p2e)
    __bounds_errorf (filename, line, p1, NULL,
		     "in %s, source and destination objects overlap",
		     function_name);
}

/*----------------------------------------------------------------------
 *	These are the checked functions themselves. The '__bounds_*'
 *	versions are internal unchecked functions used by this library
 *	itself (to avoid reentrancy problems).
 *   1.	The memory functions.
 *----------------------------------------------------------------------*/

#ifdef Lbounds_memcpy

inline void *
__bounds_memcpy (void *dest, const void *src, size_t n)
{
  char *d = (char *) dest;
  const char *s = (char *) src;

  for (;n;n--)
    *d++ = *s++;
  return dest;
}

void *
__bounds_check_memcpy (char *filename, int line, void *dest, const void *src, size_t n)
{
  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "memcpy",
			  "NULL source or destination for a zero-sized memcpy");
      return dest;
    }
  if (n == 0) return dest;

  check (filename, line, dest, n, "memcpy", "destination pointer");
  check (filename, line, (void *)src,  n, "memcpy", "source pointer");
  check_overlap (filename, line, dest, n, (void *)src, n, "memcpy");
  return __bounds_memcpy (dest, src, n);
}

#endif

#ifdef Lmemcpy

extern void *__bounds_check_memcpy (char *filename, int line, void *dest, const void *src, size_t n);

void *
memcpy(void *dest, const void *src, size_t n)
{
  return __bounds_check_memcpy(NULL, 0, dest, src, n);
}

#endif

#ifdef Lbounds_memmove

inline void *
__bounds_memmove (void *dest, const void *src, size_t n)
{
  char *d = (char *) dest;
  const char *s = (char *) src;

  if (dest < src)
    for (;n;n--)
      *d++ = *s++;
  else if (dest > src) {
    d += n;
    s += n;
    for (;n;n--)
      *--d = *--s;
  }
  return dest;
}

void *
__bounds_check_memmove (char *filename, int line, void *dest, const void *src, size_t n)
{
  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "memmove",
			  "NULL source or destination for a zero-sized memmove");
      return dest;
    }
  if (n == 0) return dest;

  check (filename, line, dest, n, "memmove", "destination pointer");
  check (filename, line, (void *)src,  n, "memmove", "source pointer");
  return __bounds_memmove (dest, src, n);
}

#endif

#ifdef Lmemmove

extern void *__bounds_check_memmove (char *filename, int line, void *dest, const void *src, size_t n);

void *
memmove (void *dest, const void *src, size_t n)
{
  return __bounds_check_memmove(NULL, 0, dest, src, n);
}

#endif

#ifdef Lbounds_bcopy

/* See comment on `bcopy' at the top of this file.
 */
#ifdef bcopy
#undef bcopy
#endif
extern void *__bounds_check_memmove (char *filename, int line, void *dest, const void *src, size_t n);

void
__bounds_check_bcopy ( char *filename, int line, const void *src, void *dest, int n)
{
#if BCOPY_OVERLAPPING_ARGUMENTS
#if BCOPY_WARN_OVERLAPPING_ARGUMENTS
  check_overlap (filename, line, dest, n, (void *)src, n, "bcopy");
#endif
  __bounds_check_memmove (filename, line, dest, src, n);
#else /* !BCOPY_OVERLAPPING_ARGUMENTS */
  __bounds_check_memcpy (filename, line, dest, src, n);
#endif /* !BCOPY_OVERLAPPING_ARGUMENTS */
}

#endif

#ifdef Lbcopy

extern void __bounds_check_bcopy ( char *filename, int line, const void *src, void *dest, int n);

void
bcopy (const void *src, void *dest, int n)
{
  __bounds_check_bcopy(NULL, 0, src, dest, n);
}

#endif

#ifdef Lbounds_memset

inline void *
__bounds_memset (void *dest, int c, size_t n)
{
  /* `memset' is called very frequently, and therefore has to be fast. In a
   * future release of GCC, we will call '__builtin_memset' in this library,
   * but that isn't implemented yet. In the mean time, we spot the common
   * case when the memory is aligned, and write integers.
   */
  if (PTR_TO_UNSIGNED (dest) & (sizeof (int) - 1)) /* not aligned */
    {
      char *d = (char *) dest;

      for (;n;n--)
	*d++ = c;
    }
  else						/* aligned */
    {
      int *di = (int *) dest;
      size_t n_longs = n / sizeof (int);
      int wr = (unsigned char) c << 24 |
	       (unsigned char) c << 16 |
	       (unsigned char) c << 8  |
	       (unsigned char) c;
      char *dc;

      n &= sizeof (int) - 1;
      for (;n_longs;n_longs--)
	*di++ = wr;
      dc = (char *) di;
      for (;n;n--)
	*dc++ = c;
    }

  return dest;
}

void *
__bounds_check_memset (char *filename, int line, void *dest, int c, size_t n)
{
  if (n == 0 && dest == NULL)
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "memset",
			  "NULL destination for a zero-sized memset");
      return dest;
    }
  if (n == 0) return dest;

  check (filename, line, dest, n, "memset", "destination pointer");
  return __bounds_memset (dest, c, n);
}

#endif

#ifdef Lmemset

extern void *__bounds_check_memset (char *filename, int line, void *dest, int c, size_t n);

void *
memset (void *dest, int c, size_t n)
{
  return __bounds_check_memset(NULL, 0, dest, c, n);
}

#endif

#ifdef Lbounds_bzero

#ifdef bzero
#undef bzero
#endif

extern void *__bounds_check_memset (char *filename, int line, void *dest, int c, size_t n);

void
__bounds_check_bzero (char *filename, int line, char *dest, int n)
{
  __bounds_check_memset (filename, line, (void *)dest, 0, n);
}

#endif

#ifdef Lbzero

extern void __bounds_check_bzero (char *filename, int line, char *dest, int n);

void
bzero (char *dest, int n)
{
  __bounds_check_bzero(NULL, 0, dest, n);
}

#endif

#ifdef Lbounds_memcmp

inline int
__bounds_memcmp (const void *s1, const void *s2, size_t n)
{
  const char *cs1 = (char *) s1;
  const char *cs2 = (char *) s2;
  for (;n;n--) {
    if (*cs1 - *cs2 != 0)
      return *cs1 - *cs2;
    cs1++, cs2++;
  }
  return 0;
}

int
__bounds_check_memcmp (char *filename, int line, const void *s1, const void *s2, size_t n)
{
  if (n == 0 && (s1 == NULL || s2 == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "memcmp",
			  "NULL source or destination for a zero-sized memcmp");
      return 1;
    }
  if (n == 0) return 0;

  check (filename, line, (void *)s1, n, "memcmp", "first pointer argument");
  check (filename, line, (void *)s2, n, "memcmp", "second pointer argument");
  return __bounds_memcmp (s1, s2, n);
}

#endif

#ifdef Lmemcmp

extern int __bounds_check_memcmp (char *filename, int line, const void *s1, const void *s2, size_t n);

int
memcmp (const void *s1, const void *s2, size_t n)
{
  return __bounds_check_memcmp (NULL, 0, s1, s2, n);
}

#endif

#ifdef Lbounds_bcmp

/* I'm assuming here that bcmp is the pre-ANSI memcmp function. Since I've
 * never actually used the b* functions, I'm not sure if this is totally
 * right.
 */
#ifdef bcmp
#undef bcmp
#endif

extern int __bounds_check_memcmp (char *filename, int line, const void *s1, const void *s2, size_t n);

int
__bounds_check_bcmp (char *filename, int line, void *s1, void *s2, int n)
{
  return __bounds_check_memcmp (filename, line, s1, s2, n);
}

#endif

#ifdef Lbcmp

extern int __bounds_check_bcmp (char *filename, int line, void *s1, void *s2, int n);

int
bcmp (void *s1, void *s2, int n)
{
  return __bounds_check_bcmp(NULL, 0, s1, s2, n);
}

#endif

/*----------------------------------------------------------------------
 *   2. The string functions.
 *----------------------------------------------------------------------*/

#ifdef Lbounds_strcpy

inline char *
__bounds_strcpy (char *dest, const char *src)
{
  char *d = dest;

  while ((*dest++ = *src++) != 0);
  return d;
}

char *
__bounds_check_strcpy (char *filename, int line, char *dest, const char *src)
{
  size_t n = check (filename, line, (void *)src, 0, "strcpy", "source string");
  check (filename, line, dest, n, "strcpy", "destination string");
  check_overlap (filename, line, dest, n, (void *)src, n, "strcpy");
  return __bounds_strcpy (dest, src);
}

#endif

#ifdef Lstrcpy

extern char *__bounds_check_strcpy (char *filename, int line, char *dest, const char *src);

char *
strcpy (char *dest, const char *src)
{
  return __bounds_check_strcpy(NULL, 0, dest, src);
}

#endif

#ifdef Lbounds_strncpy

static inline char *
__bounds_strncpy (char *dest, const char *src, size_t n)
{
  char *d = dest;

  if (n) {
    for (;;) {
      if (--n == -1) return d;
      else if ((*dest++ = *src++) == 0) break;
    }
    while (n--) *dest++ = 0;
  }
  return d;
}

char *
__bounds_check_strncpy (char *filename, int line, char *dest, const char *src, size_t n)
{
  char *t1;
  size_t n1;

  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "strncpy",
			  "NULL source or destination for a zero-sized strncpy");
      return dest;
    }
  if (n == 0) return dest;
  t1 = check_ptr(filename, line, (void *)src, "strncpy", "source argument");
  if (t1 == NULL) return NULL;
  n1 = n;
  while (n1 && *t1) { n1--; t1++; }
  n1 = (((t1 - src) + 1) < n) ? (t1 - src) + 1 : n;
  check (filename, line, (void *)src, n1, "strncpy", "source string");
  check (filename, line, dest, n, "strncpy", "destination string");
  check_overlap (filename, line, dest, n, (void *)src, n1, "strncpy");
  return __bounds_strncpy (dest, src, n);
}

#endif

#ifdef Lstrncpy

extern char *__bounds_check_strncpy (char *filename, int line, char *dest, const char *src, size_t n);

char *
strncpy (char *dest, const char *src, size_t n)
{
  return __bounds_check_strncpy(NULL, 0, dest, src, n);
}

#endif

#ifdef Lbounds_strlen

inline size_t
__bounds_strlen (const char *s)
{
  const char *r = s;

  while (*s++);
  return (s - r) - 1;
}

size_t
__bounds_check_strlen (char *filename, int line, const char *s)
{
  return check (filename, line, (void *)s, 0, "strlen", "string argument") - 1;
}

#endif

#ifdef Lstrlen

extern size_t __bounds_check_strlen (char *filename, int line, const char *s);

size_t
strlen (const char *s)
{
  return __bounds_check_strlen(NULL, 0, s);
} 

#endif

#ifdef Lbounds_strcmp

inline int
__bounds_strcmp (const char *s1, const char *s2)
{ 
  while (*s1 && *s1 == *s2) { s1++; s2++; }
  return(*s1 - *s2);
}

int
__bounds_check_strcmp(char *filename, int line, const char *s1, const char *s2)
{
  char *t1 = check_ptr(filename, line, (void *)s1, "strcmp", "string1 argument");
  char *t2 = check_ptr(filename, line, (void *)s2, "strcmp", "string2 argument");
  size_t n;

  if (t1 == NULL || t2 == NULL) return -1;

  while (*t1 && *t2) { t1++; t2++; }
  n = (t1 - s1) + 1;
  check (filename, line, (void *)s1, n, "strcmp", "string1 argument");
  check (filename, line, (void *)s2, n, "strcmp", "string2 argument");
  return(__bounds_strcmp (s1, s2));
}

#endif

#ifdef Lstrcmp

extern int __bounds_check_strcmp(char *filename, int line, const char *s1, const char *s2);

int
strcmp(const char *s1, const char *s2)
{
  return __bounds_check_strcmp(NULL, 0, s1, s2);
}

#endif

#ifdef Lbounds_strncmp

inline int
__bounds_strncmp (const char *s1, const char *s2, size_t n)
{ 
  do {
    if (--n == -1) break;
    else if (*s1 != *s2) return *s1 - *s2;
    s2++;
  } while (*s1++);
  return 0;
}

int
__bounds_check_strncmp(char *filename, int line, const char *s1, const char *s2, size_t n)
{
  char *t1;
  char *t2;
  size_t n1;

  if (n == 0 && (s1 == NULL || s2 == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "strncmp",
			  "NULL source or destination for a zero-sized strncmp");
      return 0;
    }
  if (n == 0) return 0;
  t1 = check_ptr(filename, line, (void *)s1, "strncmp", "string1 argument");
  t2 = check_ptr(filename, line, (void *)s2, "strncmp", "string2 argument");
  if (t1 == NULL || t2 == NULL) return -1;
  n1 = n;
  while (n1 && *t1 && *t2) { n1--; t1++; t2++; }
  if (((t1 - s1) + 1) < n) n = (t1 - s1) + 1;
  check (filename, line, (void *)s1, n, "strncmp", "string1 argument");
  check (filename, line, (void *)s2, n, "strncmp", "string2 argument");
  return(__bounds_strncmp (s1, s2,n));
}

#endif

#ifdef Lstrncmp

extern int __bounds_check_strncmp(char *filename, int line, const char *s1, const char *s2, size_t n);

int
strncmp(const char *s1, const char *s2, size_t n)
{
  return __bounds_check_strncmp(NULL, 0, s1, s2, n);
}

#endif

#ifdef Lbounds_strcat

static inline char *
__bounds_strcat(char *dest, const char *src)
{
  char *r = dest;

  while (*dest++);
  dest--;
  while ((*dest++ = *src++) != 0);
  return r;
}

char *
__bounds_check_strcat(char *filename, int line, char *dest, const char *src)
{
  char *r = check_ptr(filename, line, dest, "strcat", "destination argument");
  size_t n = check(filename, line, (void *)src, 0, "strcat", "source argument");

  if (r == NULL) return NULL;

  while (*r++);
  check(filename, line, dest, n + (r - dest) - 1, "strcat", "destination argument");
  return(__bounds_strcat(dest,src));
}

#endif

#ifdef Lstrcat

extern char *__bounds_check_strcat(char *filename, int line, char *dest, const char *src);

char *
strcat(char *dest, const char *src)
{
  return __bounds_check_strcat(NULL, 0, dest, src);
}

#endif

#ifdef Lbounds_strncat

static inline char *
__bounds_strncat(char *dest, const char *src, size_t n)
{
  char *r = dest;

  while (*dest++);
  dest--;
  for (;;) {
	if (n-- == 0) break;
	else if ((*dest++ = *src++) == 0) return(r);
  }
  *dest = 0;
  return(r);
}


char *
__bounds_check_strncat(char *filename, int line, char *dest, const char *src, size_t n)
{
  char *r;
  char *p;
  size_t n1;

  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "strncat",
			  "NULL source or destination for a zero-sized strncat");
      return dest;
    }
  if (n == 0) return dest;
  r = check_ptr(filename, line, dest, "strncat", "destination argument");
  p = check_ptr(filename, line, (void *)src, "strncat", "source argument");
  if (r == NULL || p == NULL) return NULL;
  n1 = n;
  while (n1 && *p) { n1--; p++; }
  if (((p - src) + 1) < n) n = (p - src) + 1;
  check(filename, line, (void *)src, n, "strncat", "source argument");
  while (*r++);
  check(filename, line, dest, n + (r - dest) - 1, "strncat", "destination argument");
  return(__bounds_strncat(dest,src,n));
}

#endif

#ifdef Lstrncat

extern char *__bounds_check_strncat(char *filename, int line, char *dest, const char *src, size_t n);

char *
strncat(char *dest, const char *src, size_t n)
{
  return __bounds_check_strncat(NULL, 0, dest, src, n);
}

#endif

#ifdef Lbounds_strpbrk

static inline char *
__bounds_strpbrk(const char *string, const char *chs)
{
  char ch;

  while ((ch = *string++) != 0) {
    const char *chk = chs;
    char tch;

    while ((tch = *chk++) != 0)
      if (ch == tch) return (char *)string - 1;
  }
  return(NULL);
}

char *
__bounds_check_strpbrk(char *filename, int line, const char *string, const char *chs)
{
  check(filename, line, (void *)string, 0, "strpbrk", "string argument");
  check(filename, line, (void *)chs, 0, "strpbrk", "check argument");
  return __bounds_strpbrk(string, chs);
}

#endif

#ifdef Lstrpbrk

extern char *__bounds_check_strpbrk(char *filename, int line, const char *string, const char *chs);

char *
strpbrk(const char *string, const char *chs)
{
  return __bounds_check_strpbrk(NULL, 0, string, chs);
}

#endif

#ifdef Lbounds_strrchr

char *
__bounds_strrchr(const char *string, int c)
{
  unsigned char ch = c;
  unsigned char tch;
  unsigned char *result = NULL;

  for (;;) {
    if (ch == (tch = (unsigned char) (*string++))) result = (char *)string - 1;
    if (tch == 0) break;
  }
  return result;
}

char *
__bounds_check_strrchr(char *filename, int line, const char *string, int c)
{
  check(filename, line, (void *)string, 0, "strrchr", "string argument");
  return __bounds_strrchr(string, c);
}

#endif

#ifdef Lstrrchr

extern char *__bounds_check_strrchr(char *filename, int line, const char *string, int c);

char *
strrchr(const char *string, int c)
{
  return __bounds_check_strrchr(NULL, 0, string, c);
}

#endif

#ifdef Lbounds_rindex

#ifdef rindex
#undef rindex
#endif

extern char * __bounds_check_strrchr(char *filename, int line, const char *string, int c);

char *
__bounds_check_rindex (char *filename, int line, const char *string, int c)
{
  return __bounds_check_strrchr (filename, line, string, c);
}

#endif

#ifdef Lrindex

extern char *__bounds_check_rindex (char *filename, int line, const char *string, int c);

char *
rindex (const char *string, int c)
{
  return __bounds_check_rindex(NULL, 0, string, c);
}

#endif

#ifdef Lbounds_strspn

size_t
__bounds_strspn(const char *string, const char *chs)
{
  const char *r = string;
  char ch;

  while ((ch = *r++) != 0) {
    const char *chk = chs;
    char tch;

    do {
      if ((tch = *chk++) == 0) return (r - string) - 1;
    } while (tch != ch);
  }
  return (r - string) - 1;
}

size_t
__bounds_check_strspn(char *filename, int line, const char *string, const char *chs)
{
  check(filename, line, (void *)string, 0, "strspn", "string argument");
  check(filename, line, (void *)chs, 0, "strspn", "check argument");
  return __bounds_strspn(string, chs);
}

#endif

#ifdef Lstrspn

extern size_t __bounds_check_strspn(char *filename, int line, const char *string, const char *chs);

size_t
strspn(const char *string, const char *chs)
{
  return __bounds_check_strspn(NULL, 0, string, chs);
}

#endif

#ifdef Lbounds_strcspn

size_t
__bounds_strcspn(const char *string, const char *chs)
{
  const char *r = string;
  char ch;

  while ((ch = *r++) != 0) {
    const char *chk = chs;
    char tch;

    while ((tch = *chk++) != 0) {
      if (ch == tch) return (r - string) - 1;
    }
  }
  return (r - string) - 1;
}

size_t
__bounds_check_strcspn(char *filename, int line, const char *string, const char *chs)
{
  check(filename, line, (void *)string, 0, "strcspn", "string argument");
  check(filename, line, (void *)chs, 0, "strcspn", "check argument");
  return __bounds_strcspn(string, chs);
}

#endif

#ifdef Lstrcspn

extern size_t __bounds_check_strcspn(char *filename, int line, const char *string, const char *chs);

size_t
strcspn(const char *string, const char *chs)
{
  return __bounds_check_strcspn(NULL, 0, string, chs);
}

#endif

#ifdef Lbounds_strstr

static inline char *
__bounds_strstr(const char *s1, const char *s2)
{
  char cp1, cp2;
  const char *pos = s1;

  while (*s1) {
    const char *cmp = s2;

    while (((cp1 = *s1++) == (cp2 = *cmp++)) && cp1);
    if (cp2 == 0) return (char *)pos;
    s1 = ++pos;
  }
  return NULL;
}

char *
__bounds_check_strstr(char *filename, int line, const char *s1, const char *s2)
{
  check(filename, line, (void *)s1, 0, "strstr", "string1 argument");
  check(filename, line, (void *)s2, 0, "strstr", "string2 argument");
  return __bounds_strstr(s1, s2);
}

#endif

#ifdef Lstrstr

extern char *__bounds_check_strstr(char *filename, int line, const char *s1, const char *s2);

char *
strstr(const char *s1, const char *s2)
{
  return __bounds_check_strstr(NULL, 0, s1, s2);
}

#endif

#ifdef Lbounds_strtok

extern size_t __bounds_strspn(const char *string, const char *chs);
extern size_t __bounds_strcspn(const char *string, const char *chs);

static inline char *
__bounds_strtok(char *str1, const char *str2)
{
  /* NOTE: For multi-threaded libraries, make last_end a thread-local
   * variable.
   */
  static char *last_end = NULL;
  char        *start;

  if (str1) last_end = str1;

  if (!last_end) return NULL;

  last_end += __bounds_strspn(last_end, str2);
  if (*last_end == '\0') return NULL;

  start = last_end; 
  last_end += __bounds_strcspn(last_end, str2);

  if (*last_end != '\0') *last_end++ = '\0';

  return start;
}

char *
__bounds_check_strtok(char *filename, int line, char *str1, const char *str2)
{
  if (str1 != NULL)
    check(filename, line, str1, 0, "strtok", "first argument");
  check(filename, line, (void *)str2, 0, "strtok", "second argument");
  return __bounds_strtok(str1, str2);
}

#endif

#ifdef Lstrtok

extern char *__bounds_check_strtok(char *filename, int line, char *str1, const char *str2);

char *
strtok(char *str1, const char *str2)
{
  return __bounds_check_strtok(NULL, 0, str1, str2);
}

#endif

#ifdef Lbounds_strchr

static inline char *
__bounds_strchr(const char *string, int ch)
{
  int   tch;

  for (;;)
    if      ( (tch = (unsigned char) (*string++))
	      == (unsigned char) ch ) return (char *)string - 1;
    else if ( tch == 0 ) return NULL;
}

char *
__bounds_check_strchr(char *filename, int line, const char *string, int ch)
{
  check(filename, line, (void *)string, 0, "strchr", "string argument");
  return __bounds_strchr(string, ch);
}

#endif

#ifdef Lstrchr

extern char *__bounds_check_strchr(char *filename, int line, const char *string, int ch);

char *
strchr(const char *string, int ch)
{
  return __bounds_check_strchr(NULL, 0, string, ch);
}

#endif

#ifdef Lbounds_strdup

char *
__bounds_check_strdup(char *filename, int line, const char *string)
{
  char *new;
  size_t n = check(filename, line, (void *)string, 0, "strdup", "string argument");
  new = __bounds_malloc(n + 1);
  if (new == NULL) return NULL;
  __bounds_add_heap_object (new, n, 1, "strdup", filename, line);
  return __bounds_strcpy(new, string);
}

#endif

#ifdef Lstrdup

extern char *__bounds_check_strdup(char *filename, int line, const char *string);

char *
strdup(const char *string)
{
  return __bounds_check_strdup(NULL, 0, string);
}

#endif

#ifdef Lbounds_index

#ifdef index
#undef index
#endif

extern char * __bounds_check_strchr(char *filename, int line, const char *string, int ch);

char *
__bounds_check_index (char *filename, int line, const char *string, int ch)
{
  return __bounds_check_strchr (filename, line, string, ch);
}

#endif

#ifdef Lindex

extern char *__bounds_check_index (char *filename, int line, const char *string, int ch);

char *
index (const char *string, int ch)
{
  return __bounds_check_index(NULL, 0, string, ch);
}

#endif

#ifdef Lbounds_strcoll

static inline int
__bounds_strcoll(const char *string1, const char *string2)
{
   while ( *string1 && (*string1 == *string2)) {
		string1++; string2++;
   }
   return *string1 - *string2;
}

int
__bounds_check_strcoll(char *filename, int line, const char *string1, const char *string2)
{
  check(filename, line, (void *)string1, 0, "strcoll", "first argument");
  check(filename, line, (void *)string2, 0, "strcoll", "second argument");
  return __bounds_strcoll(string1, string2);
}

#endif

#ifdef Lstrcoll

extern int __bounds_check_strcoll(char *filename, int line, const char *string1, const char *string2);

int
strcoll(const char *string1, const char *string2)
{
  return __bounds_check_strcoll(NULL, 0, string1, string2);
}

#endif

#ifdef Lbounds_strxfrm

static inline size_t
__bounds_strxfrm(char *to, const char *from, size_t n)
{
  int count = 0; 
  while (*from++) count++;
  from -= count;
  from--;

  if (n != 0) while((*to++ = *from++) && (--n != 0));

  return ((size_t) count);
}

size_t
__bounds_check_strxfrm(char *filename, int line, char *to, const char *from, size_t n)
{
  check(filename, line, (void *)from, 0, "strxfrm", "src argument");
  if (n) check(filename, line, to, n, "strxfrm", "dest argument");
  return __bounds_strxfrm(to, from, n);
}

#endif

#ifdef Lstrxfrm

extern size_t __bounds_check_strxfrm(char *filename, int line, char *to, const char *from, size_t n);

size_t
strxfrm(char *to, const char *from, size_t n)
{
  return __bounds_check_strxfrm(NULL, 0, to, from, n);
}

#endif

#define	LOW(c)	(islower((int)(c)) ? c : tolower((int)(c)))

#ifdef Lbounds_strcasecmp

static inline int
__bounds_strcasecmp(const char *s1, const char *s2)
{
  while (*s1 && LOW(*s1) == LOW(*s2)) { s1++; s2++; }
  return(LOW(*s1) - LOW(*s2));
}

int
__bounds_check_strcasecmp(char *filename, int line, const char *s1, const char *s2)
{
  char *t1 = check_ptr(filename, line, (void *)s1, "strcasecmp", "first argument");
  char *t2 = check_ptr(filename, line, (void *)s2, "strcasecmp", "second argument");
  size_t n;

  if (t1 == NULL || t2 == NULL) return 0;

  while (*t1 && *t2) { t1++; t2++; }
  n = (t1 - s1) + 1;
  check (filename, line, (void *)s1, n, "strcasecmp", "string1 argument");
  check (filename, line, (void *)s2, n, "strcasecmp", "string2 argument");
  return __bounds_strcasecmp(s1,s2);
}

#endif

#ifdef Lstrcasecmp

extern int __bounds_check_strcasecmp(char *filename, int line, const char *s1, const char *s2);

int
strcasecmp(const char *s1, const char *s2)
{
  return __bounds_check_strcasecmp(NULL, 0, s1, s2);
}

#endif

#ifdef Lbounds_strncasecmp

static inline int
__bounds_strncasecmp(const char *s1, const char *s2, size_t n)
{
  do {
    if (--n == -1) break;
    else if (LOW(*s1) != LOW(*s2)) return LOW(*s1) - LOW(*s2);
    s2++;
  } while (*s1++);
  return 0;
}

int
__bounds_check_strncasecmp(char *filename, int line, const char *s1, const char *s2, size_t n)
{
  char *t1;
  char *t2;
  size_t n1;

  if (n == 0 && (s1 == NULL || s2 == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (filename, line, "strncasecmp",
			  "NULL source or destination for a zero-sized strncasecmp");
      return 0;
    }
  if (n == 0) return 0;
  t1 = check_ptr(filename, line, (void *)s1, "strncasecmp", "string1 argument");
  t2 = check_ptr(filename, line, (void *)s2, "strncasecmp", "string2 argument");
  if (t1 == NULL || t2 == NULL) return 0;
  n1 = n;
  while (n1 && *t1 && *t2) { n1--; t1++; t2++; }
  if (((t1 - s1) + 1) < n) n = (t1 - s1) + 1;
  check (filename, line, (void *)s1, n, "strncasecmp", "string1 argument");
  check (filename, line, (void *)s2, n, "strncasecmp", "string2 argument");
  return __bounds_strncasecmp(s1,s2,n);
}

#endif

#ifdef Lstrncasecmp

extern int __bounds_check_strncasecmp(char *filename, int line, const char *s1, const char *s2, size_t n);

int
strncasecmp(const char *s1, const char *s2, size_t n)
{
  return __bounds_check_strncasecmp(NULL, 0, s1, s2, n);
}

#endif

/*----------------------------------------------------------------------
 *	The odd functions `memccpy' and `memchr' are treated separately
 *	here. With these functions, it is permissible to have a `size'
 *	argument that is too large, so long as we will always meet
 *	the `c' terminating character before the end of the object.
 *----------------------------------------------------------------------
 */

static inline void
check2 (char *filename, int line, void *pointer, size_t size, int c, char *function, char *ptr_name)
{
  object *obj;

#if 0 /* This check is redundant. All callers should cast the argument to
       * unsigned char before calling.
       * - Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
       * & RWMJ.
       */
  if (c < 0 || c > 255)
    __bounds_errorf (filename, line, pointer, NULL,
		     "%s called with terminating character < 0 or > 255",
		     function);
#endif

  if (pointer == NULL || pointer == ILLEGAL)
    {
      __bounds_errorf (filename, line, pointer, NULL,
		       "NULL or ILLEGAL %s used in %s", ptr_name, function);
      return;
    }
  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, NULL, 0, function)
	|| maybe_is_unchecked_stack (pointer, NULL, 0, function)) {
      /* Unchecked pointer is OK. We have already delivered a warning at
       * this point.
       */
      return;
    }
    __bounds_errorf (filename, line, pointer, NULL,
		     "invalid %s used in %s", ptr_name, function);
    return;
  }

  /* The pointer itself is valid, and points to a checked object. Now make
   * sure that we won't overrun the object by using this pointer and size.
   * However, if size does overrun, give it a second chance by looking for
   * the terminating character.
   */
  if (pointer + size > obj->extent)
    {
      char *p = (char *) pointer;
      size_t n = obj->extent - pointer;

      for (; n && *p != c; n--, p++)
	;
      if (n == 0)
	{
	  __bounds_errorf (filename, line, pointer, obj,
			   "%s with this %s would overrun the end of the object's allocated memory",
			   function, ptr_name);
	  return;
	}
      /* If n > 0, then OK: The area is still acceptable. */
      }
  return;
}

#ifdef Lbounds_memchr

inline void *
__bounds_memchr (const void *s, int c, size_t n)
{
  const unsigned char *cs = (const unsigned char *) s;
  for (;n;n--) {
    if (*cs == (unsigned char) c)
      return (void *)cs;
    cs++;
  }
  return NULL;
}

void *
__bounds_check_memchr (char *filename, int line, const void *s, int c, size_t n)
{
#if STRICT_MEMCHR
  check (filename, line, (void *)s, n, "memchr", "pointer argument");
#else
  check2 (filename, line, (void *)s, n, (unsigned char) c, "memchr", "pointer argument");
#endif
  return __bounds_memchr (s, c, n);
}

#endif

#ifdef Lmemchr

extern void *__bounds_check_memchr (char *filename, int line, const void *s, int c, size_t n);

void *
memchr (const void *s, int c, size_t n)
{
  return __bounds_check_memchr(NULL, 0, s, c, n);
}

#endif

#ifdef Lbounds_memccpy

inline void *
__bounds_memccpy (void *dest, const void *src, int c, size_t n)
{
  int c2;

  for (;n;n--) {
    c2 = *(char *)dest++ = *(char *)src++;
    if (c == c2)
      return dest;
  }
  return NULL;
}

void *
__bounds_check_memccpy (char *filename, int line, void *dest, const void *src, int c, size_t n)
{
#if STRICT_MEMCHR
  check (filename, line, (void *)src, n, "memccpy", "source pointer");
#else
  check2 (filename, line, (void *)src, n, (unsigned char) c, "memccpy", "source pointer");
#endif
  check (filename, line, dest, n, "memccpy", "destination pointer");
  check_overlap (filename, line, dest, n, (void *)src, n, "memccpy");
  return __bounds_memccpy (dest, src, c, n);
}

#endif

#ifdef Lmemccpy

extern void *__bounds_check_memccpy (char *filename, int line, void *dest, const void *src, int c, size_t n);

void *
memccpy (void *dest, const void *src, int c, size_t n)
{
  return __bounds_check_memccpy(NULL, 0, dest, src, c, n);
}

#endif
