/*
 * Legalesy-free Unicode API interface for OS/2
 * Interface definitions for basic Unicode API functions
 *
 * Written by Andrew Zabolotny <bit@eltech.ru>
 *
 * This file is put into public domain. You are free to do
 * literally anything you wish with it: modify, print, sell,
 * rent, eat, throw out of window: in all (esp. in later)
 * cases I am not responsible for any damage it causes.
 */

#ifndef __UNIDEF_H__
#define __UNIDEF_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>
#include <time.h>

#include <ulserrno.h>
#include <ulsitem.h>

typedef wchar_t UniChar;
typedef void *LocaleObject;
typedef unsigned int LocaleToken;
typedef void *AttrObject;
typedef void *XformObject;
typedef int ulsBool;

#ifndef TRUE
# define TRUE	1
#endif
#ifndef FALSE
# define FALSE	0
#endif

#define UNI_TOKEN_POINTER	1
#define UNI_MBS_STRING_POINTER	2
#define UNI_UCS_STRING_POINTER	3

/* Locale categories */
#undef LANG
#undef LC_ALL
#undef LC_COLLATE
#undef LC_CTYPE
#undef LC_NUMERIC
#undef LC_MONETARY
#undef LC_TIME
#undef LC_MESSAGES

#define LANG			(-2)
#define LC_ALL			(-1)
#define LC_COLLATE		0
#define LC_CTYPE		1
#define LC_NUMERIC		2
#define LC_MONETARY		3
#define LC_TIME			4
#define LC_MESSAGES		5

/* Locale conventions structure */
struct UniLconv
{
  UniChar *decimal_point;	/* non-monetary decimal point */
  UniChar *thousands_sep;	/* non-monetary thousands separator */
  short   *grouping;		/* non-monetary size of grouping */
  UniChar *int_curr_symbol;	/* int'l currency symbol and separator */
  UniChar *currency_symbol;	/* local currency symbol */
  UniChar *mon_decimal_point;	/* monetary decimal point */
  UniChar *mon_thousands_sep;	/* monetary thousands separator */
  short   *mon_grouping;	/* monetary size of grouping */
  UniChar *positive_sign;	/* non-negative values sign */
  UniChar *negative_sign;	/* negative values sign */
  short   int_frac_digits;	/* no of fractional digits int currency */
  short   frac_digits;		/* no of fractional digits loc currency */
  short   p_cs_precedes;	/* nonneg curr sym 1-precedes,0-succeeds */
  short   p_sep_by_space;	/* nonneg curr sym 1-space,0-no space */
  short   n_cs_precedes;	/* neg curr sym 1-precedes,0-succeeds */
  short   n_sep_by_space;	/* neg curr sym 1-space 0-no space */
  short   p_sign_posn;		/* positioning of nonneg monetary sign */
  short   n_sign_posn;		/* positioning of negative monetary sign */
  short   os2_mondecpt;		/* os2 curr sym positioning */
  UniChar *debit_sign;		/* non-neg-valued monetary sym - "DB" */
  UniChar *credit_sign;		/* negative-valued monetary sym - "CR" */
  UniChar *left_parenthesis;	/* negative-valued monetary sym - "(" */
  UniChar *right_parenthesis;	/* negative-valued monetary sym - ")" */
};

/* Locale Management Function Prototypes */

int UniCreateLocaleObject (int locale_spec_type, const void *locale_spec,
  LocaleObject *locale_object_ptr);
int UniQueryLocaleObject (const LocaleObject locale_object, int category,
  int locale_spec_type, void **locale_spec_ptr);
int UniFreeLocaleObject (LocaleObject locale_object);
int UniFreeMem (void *memory_ptr);
int UniLocaleStrToToken (int locale_string_type, const void *locale_string,
  LocaleToken *locale_token_ptr);
int UniLocaleTokenToStr (const LocaleToken locale_token,
  int locale_string_type, void **locale_string_ptr);

/* Locale Information Function Prototypes */
int UniQueryLocaleInfo (const LocaleObject locale_object,
  struct UniLconv **unilconv_addr_ptr);
int UniFreeLocaleInfo (struct UniLconv *unilconv_addr);
int UniQueryLocaleItem (const LocaleObject locale_object, LocaleItem item,
  UniChar **info_item_addr_ptr);

/* Date and Time Function Prototypes */
size_t UniStrftime (const LocaleObject locale_object, UniChar *ucs,
  size_t maxsize, const UniChar *format, const struct tm *time_ptr);
UniChar *UniStrptime (const LocaleObject locale_object, const UniChar *buf,
  const UniChar *format, struct tm *time_ptr);

/* Monetary Formatting Function Prototype */
int UniStrfmon (const LocaleObject locale_object, UniChar *ucs, size_t maxsize,
  const UniChar *format, ...);

/* String/Character Function Prototypes */
UniChar *UniStrcat (UniChar *ucs1, const UniChar *ucs2);
UniChar *UniStrchr (const UniChar *ucs, UniChar uc);
int UniStrcmp (const UniChar *ucs1, const UniChar *ucs2);
UniChar *UniStrcpy (UniChar *ucs1, const UniChar *ucs2);
size_t UniStrcspn (const UniChar *ucs1, const UniChar *ucs2);
size_t UniStrlen (const UniChar *ucs1);
UniChar *UniStrncat (UniChar *ucs1, const UniChar *ucs2, size_t n);
int UniStrncmp (const UniChar *ucs1, const UniChar *ucs2, size_t n);
UniChar *UniStrncpy (UniChar *ucs1, const UniChar *ucs2, size_t n);
UniChar *UniStrpbrk (const UniChar *ucs1, const UniChar *ucs2);
UniChar *UniStrrchr (const UniChar *ucs, UniChar uc);
size_t UniStrspn (const UniChar *ucs1, const UniChar *ucs2);
UniChar *UniStrstr (const UniChar *ucs1, const UniChar *ucs2);
UniChar *UniStrtok (UniChar *ucs1, const UniChar *ucs2);

/* Character Attribute Function Prototypes */
int UniCreateAttrObject (const LocaleObject locale_object,
  const UniChar *attr_name, AttrObject *attr_object_ptr);
int UniQueryCharAttr (AttrObject attr_object, UniChar uc);
int UniScanForAttr (AttrObject attr_object, const UniChar *ucs,
  size_t num_elems, ulsBool inverse_op, size_t *offset_ptr);
int UniFreeAttrObject (AttrObject attr_object);
int UniQueryAlnum (const LocaleObject locale_object, UniChar uc);
int UniQueryAlpha (const LocaleObject locale_object, UniChar uc);
int UniQueryBlank (const LocaleObject locale_object, UniChar uc);
int UniQueryCntrl (const LocaleObject locale_object, UniChar uc);
int UniQueryDigit (const LocaleObject locale_object, UniChar uc);
int UniQueryGraph (const LocaleObject locale_object, UniChar uc);
int UniQueryLower (const LocaleObject locale_object, UniChar uc);
int UniQueryPrint (const LocaleObject locale_object, UniChar uc);
int UniQueryPunct (const LocaleObject locale_object, UniChar uc);
int UniQuerySpace (const LocaleObject locale_object, UniChar uc);
int UniQueryUpper (const LocaleObject locale_object, UniChar uc);
int UniQueryXdigit (const LocaleObject locale_object, UniChar uc);

/* String Transformation Function Prototypes */
int UniCreateTransformObject (const LocaleObject locale_object,
  const UniChar *xtype, XformObject *xform_object_ptr);
int UniTransformStr (XformObject xform_object, const UniChar *inp_buf,
  int *inp_size, UniChar *out_buf, int *out_size);
int UniFreeTransformObject (XformObject xform_object);
UniChar UniTransLower (const LocaleObject locale_object, UniChar uc);
UniChar UniTransUpper (const LocaleObject locale_object, UniChar uc);

/* String Conversion Function Prototypes */
int UniStrtod (const LocaleObject locale_object, const UniChar *ucs,
  UniChar **end_ptr, double *result_ptr);
int UniStrtol (const LocaleObject locale_object, const UniChar *ucs,
  UniChar **end_ptr, int base, long int *result_ptr);
int UniStrtoul (const LocaleObject locale_object, const UniChar *ucs,
  UniChar **end_ptr, int base, unsigned long int *result_ptr);

/* String Comparison Function Prototypes */
int UniStrcoll (const LocaleObject locale_object,
  const UniChar *ucs1, const UniChar *ucs2);
size_t UniStrxfrm (const LocaleObject locale_object, UniChar *ucs1,
  const UniChar *ucs2, size_t n);
int UniStrcmpi (const LocaleObject locale_object,
  const UniChar *ucs1, const UniChar *ucs2);
int UniStrncmpi (const LocaleObject locale_object,
  const UniChar *ucs1, const UniChar *ucs2, const size_t n);

/* Unicode Case Mapping Function Prototypes */
UniChar UniToupper (UniChar uc);
UniChar UniTolower (UniChar uc);
UniChar *UniStrupr (UniChar *ucs);
UniChar *UniStrlwr (UniChar *ucs);

int UniMapCtryToLocale (unsigned long Country, UniChar *LocaleName, size_t n);

#ifdef __cplusplus
}
#endif

#endif /* __UNIDEF_H__ */
