/****************************** Module Header ******************************\
*
* Module Name: _mmf.c
*
* memory mapped files manager sample
*
* Copyright (c) WPS Toolkit Project - Christian Langanke 2000
*
* $Id: _mmf.c,v 1.3 2008/10/15 12:48:53 cla Exp $
*
* ===========================================================================*
*
* This file is part of the WPS Toolkit package and is free software.  You can
* redistribute it and/or modify it under the terms of the GNU Library General
* Public License as published by the Free Software Foundation, in version 2
* as it comes in the "COPYING.LIB" file of the WPS Toolkit main distribution.
* This library is distributed in the hope that it will be useful, but WITHOUT
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
* License for more details.
*
\***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define INCL_ERRORS
#define INCL_DOSMISC
#include <os2.h>

#define INCL_WTKMMF
#define INCL_WTKUTLFILE
#include <wtk.h>

// -----------------------------------------------------------------------------

#define PRINTSEPARATOR printf("\n------------------------------------------\n")

int main ( int argc, char *argv[])
{

         APIRET         rc = NO_ERROR;
         ULONG          i;
         PSZ            p;

         HMMF           hmmf = NULLHANDLE;
         CHAR           szFile[ _MAX_PATH];
         PSZ            pszMemory = NULL;
         PSZ            pszFileContents = NULL;
         ULONG          ulFilesize = 32 * MMF_MAXSIZE_KB;
         ULONG          ulCurrentSize;
         ULONG          ulBytesWritten;

         HFILE          hfile = NULLHANDLE;
         ULONG          ulAction;

static   PSZ            pszFile = "\?:\\CONFIG.SYS";

#define FILEHEAD_MAXLINES 10

do
   {
   // --------------------------------------------------------------------------------

   PRINTSEPARATOR;

   // ----------  create in-memory only file --------------

   printf( "initialize\n");
   printf( "- WtkInitializeMmf: initialize support for memory mapped files\n");
   rc = WtkInitializeMmf( &hmmf, 4);
   if (rc != NO_ERROR)
      {
      printf( " WtkInitializeMmf: error: cannot initialize MMF support, rc=%u\n", rc);
      break;
      }

   // --------------------------------------------------------------------------------

   PRINTSEPARATOR;

   printf( "Testing in-memory pseudo file\n");

   // ----------  create in-memory only file --------------

   printf( "- WtkAllocMmf: allocate in-memory only buffer\n");
   rc = WtkAllocMmf( hmmf, (PVOID*) &pszMemory, MMF_FILE_INMEMORY, MMF_ACCESS_READWRITE, ulFilesize);
   if (rc != NO_ERROR)
      {
      printf( " WtkAllocMmf:  error: cannot allocate in-memory mapped file, rc=%u\n", rc);
      break;
      }

   // set memory to all hashmarks
   printf( "- access dynamic memory buffer, writing %u bytes\n", ulFilesize);
   memset( pszMemory, '#', ulFilesize);
   printf( "- no trap so far !\n");

   // --------------------------------------------------------------------------------

   PRINTSEPARATOR;

   printf( "Testing readwrite file access\n");

   // ----------  write dummy file --------------

   // allocate test file
   sprintf( szFile, "%s\\mmftest.txt", getenv( "TMP"));
   printf( "- WtkAllocMmf: allocate readwrite file buffer: %s\n", szFile);
   rc = WtkAllocMmf( hmmf, (PVOID*) &pszFileContents, szFile,
                  MMF_ACCESS_READWRITE | MMF_OPENMODE_RESETFILE,
                  ulFilesize);
   if (rc != NO_ERROR)
      {
      printf( " WtkAllocMmf: error: cannot allocate memory mapped file %s, rc=%u\n", szFile, rc);
      break;
      }

   // access memory by copying the contents of the in-memory file
   // but leave out first 64 bytes
   printf( "- access dynamic memory buffer of file, writing %u bytes\n", ulFilesize - 64);
   strcpy( pszFileContents, "This is the WPSTK testcase for memory mapped files !");
   memcpy( pszFileContents + 64, pszMemory, ulFilesize - 64);
   printf( "- no trap so far !\n");

   // cut size to half of size
   ulFilesize /= 2;
   printf( "- WtkSetMmfSize: cut size to %u\n", ulFilesize);
   rc = WtkSetMmfSize( hmmf, pszFileContents, ulFilesize);
   if (rc != NO_ERROR)
      {
      printf( " WtkSetMmfSize: error: cannot set size of memory buffer for file %s, rc=%u\n", szFile, rc);
      break;
      }

   // query size of memory area
   rc = WtkQueryMmfSize( hmmf, pszFileContents, &ulCurrentSize);
   if (rc != NO_ERROR)
      {
      printf( " WtkQueryMmfSize: error: cannot query size of allocated memory for memory mapped file %s, rc=%u\n", szFile, rc);
      break;
      }
   printf( "- WtkQueryMmfSize: file area size is %u\n", ulCurrentSize);

   // update the file
   printf( "- WtkUpdateMmf: update file\n");
   rc = WtkUpdateMmf( hmmf, pszFileContents);
   if (rc != NO_ERROR)
      {
      printf( " WtkUpdateMmf: error: cannot update: %s\n", szFile);
      break;
      }

   // query file size with file API
   printf( "- WtkQueryFileSize: checked size of updated file is %u\n", WtkQueryFileSize( szFile));

   // --------------------------------------------------------------------------------

   PRINTSEPARATOR;

   printf( "Release dynamic memory buffers used until here\n");


   // free memory and file again
   printf( "- WtkFreeMmf: free in-memory buffer\n");
   rc = WtkFreeMmf( hmmf, pszFileContents);
   if (rc != NO_ERROR)
      {
      printf( " WtkFreeMmf: error: cannot free memory for: %s\n", szFile);
      break;
      }

   // free memory or in-memory file
   printf( "- WtkFreeMmf: free file buffer\n");
   rc = WtkFreeMmf( hmmf, pszMemory);
   if (rc != NO_ERROR)
      {
      printf( " WtkFreeMmf: error: cannot free memory of in-memory file\n", szFile);
      break;
      }

   // --------------------------------------------------------------------------------

   PRINTSEPARATOR;

   printf( "Reading file\n");

   // ----------  read config sys file --------------

   // open up config.sys
   printf( "- WtkAllocMmf: allocate readonly file: %s\n", pszFile);
   rc = WtkAllocMmf( hmmf, (PVOID*) &pszFileContents, pszFile, MMF_ACCESS_READONLY, 1024*1024);
   if (rc != NO_ERROR)
      {
      printf( " WtkAllocMmf: error: cannot allocate memory mapped file %s, rc=%u\n", pszFile, rc);
      break;
      }

   // display length of config.sys
   printf( "- strlen: contents of file is %u bytes long\n", strlen( pszFileContents));

   // free memory and file again
   printf( "- WtkFreeMmf: free file area\n");
   rc = WtkFreeMmf( hmmf, pszFileContents);
   if (rc != NO_ERROR)
      {
      printf( " WtkFreeMmf: error: cannot free memory for: %s\n", pszFile);
      break;
      }

   // --------------------------------------------------------------------------------

   PRINTSEPARATOR;

   printf( "mapping opened file\n");

   // ----------  open config sys file --------------

   WtkQueryFullname( pszFile, szFile, sizeof( szFile));
   printf( "- DosOpen: open file readonly: %s\n", szFile);
   // open the file and read it
   rc = DosOpen( szFile,
                 &hfile,
                 &ulAction,
                 0,
                 FILE_NORMAL,
                 OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                 OPEN_ACCESS_READONLY | OPEN_SHARE_DENYWRITE,
                 (PEAOP2)NULL);
   if (rc != NO_ERROR)
      {
      printf( " DosOpen: error: cannot open file, rc=%u\n", rc);
      break;
      }
   // mapping open file
   printf( "- WtkAllocMmfFile: map open file\n");
   rc = WtkAllocMmfFile( hmmf, (PVOID*) &pszFileContents, hfile, MMF_ACCESS_READONLY, 1024*1024);
   if (rc != NO_ERROR)
      {
      printf( " WtkAllocMmfFile: error: cannot map file %s, rc=%u\n", szFile, rc);
      break;
      }

   // display length of config.sys
   printf( "- strlen: contents of file is %u bytes long\n", strlen( pszFileContents));

   // display head of config.sys
   for (p = pszFileContents, i = 0;
        ((p) && (i < FILEHEAD_MAXLINES));
        i++)
       {
       // search end of line
       p = strchr( p, 13);
       if (p) p++;
       }
   // cut off after ten lines
   if (p)
      *(p - 1) = 0;

   printf( "- %u first lines of file\n"
           ">>>>>>>>>>>\n"
           "%s\n"
           "<<<<<<<<<<<\n",
           FILEHEAD_MAXLINES,
           pszFileContents);

   // free memory again - this does not close the file !!!
   printf( "- WtkFreeMmfFile: free file area\n");
   rc = WtkFreeMmfFile( hmmf, pszFileContents);
   if (rc != NO_ERROR)
      {
      printf( " WtkFreeMmfFile: error: cannot free memory for: %s\n", szFile);
      break;
      }


   // close handle here
   printf( "- DosClose: close file again\n");
   rc = DosClose( hfile); 
   if (rc != NO_ERROR)
      {
      printf( " DosClose: error: cannot close file, rc=%u\n", rc);
      break;
      }
   hfile = NULLHANDLE;

   } while (FALSE);

// cleanup
PRINTSEPARATOR;
printf( "cleanup\n");
if (hfile) DosClose( hfile);
if (hmmf)
   {
   printf( "- WtkTerminateMmf: terminate support for memory mapped files\n");
   WtkTerminateMmf( hmmf);
   }
printf( "\n");

return rc;
}

