////////////////////////////////////////////////////////////////
// File - GT64_LIB.C
//
// Library for 'WinDriver for Galileo GT64' API. 
// The basic idea is to get a handle for the board
// with GT64_Open() and use it in the rest of the program
// when calling WD functions.  Call GT64_Close() when done.
// 
////////////////////////////////////////////////////////////////

#include "../../../include/windrvr.h"
#include "../../../include/windrvr_int_thread.h"
#include "gt64_lib.h"
#include <stdio.h>

// this string is set to an error message, if one occurs
CHAR GT64_ErrorString[1024];

// internal function used by GT64_Open()
BOOL GT64_DetectCardElements(GT64_HANDLE hGT);

DWORD GT64_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    GT64_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( GT64_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( GT64_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( GT64_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}


BOOL GT64_Open (GT64_HANDLE *phGT, GT64_VERSION gt64Ver, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD dwOptions)
{
    GT64_HANDLE hGT = (GT64_HANDLE) malloc (sizeof (GT64_STRUCT));

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;

    *phGT = NULL;
    GT64_ErrorString[0] = '\0';
    BZERO(*hGT);

    hGT->cardReg.hCard = 0;
    hGT->hWD = WD_Open();

    // check if handle valid & version OK
    if (hGT->hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( GT64_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hGT->hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( GT64_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }
    hGT->gt64Ver = gt64Ver;
    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hGT->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( GT64_ErrorString, "Could not find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( GT64_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    WD_PciGetCardInfo (hGT->hWD, &pciCardInfo);
    hGT->pciSlot = pciCardInfo.pciSlot;
    hGT->cardReg.Card = pciCardInfo.Card;

    hGT->fUseInt = (dwOptions & GT64_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hGT->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hGT->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hGT->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hGT->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hGT->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hGT->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hGT->hWD, &hGT->cardReg);
    if (hGT->cardReg.hCard==0)
    {
        sprintf ( GT64_ErrorString, "Failed locking device\n");
        goto Exit;
    }

    if (!GT64_DetectCardElements(hGT))
    {
        sprintf ( GT64_ErrorString, "Card does not have all items expected for Galileo GT64\n");
        goto Exit;
    }

    // Open finished OK
    *phGT = hGT;
    return TRUE;

Exit:
    // Error durin Open
    if (hGT->cardReg.hCard) 
        WD_CardUnregister(hGT->hWD, &hGT->cardReg);
    if (hGT->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hGT->hWD);
    free (hGT);
    return FALSE;
}

DWORD GT64_ReadPCIReg(GT64_HANDLE hGT, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO (pciCnf);
    pciCnf.pciSlot = hGT->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hGT->hWD,&pciCnf);
    return dwVal;
}

void GT64_WritePCIReg(GT64_HANDLE hGT, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hGT->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hGT->hWD,&pciCnf);
}

BOOL GT64_DetectCardElements(GT64_HANDLE hGT)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hGT->Int);
    BZERO(hGT->addrDesc);

    for (i=0; i<hGT->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hGT->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwAddr;
                DWORD dwPhysAddr;
                DWORD dwAddrDirect = 0;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwAddr = pItem->I.Mem.dwTransAddr;
                    dwAddrDirect = pItem->I.Mem.dwUserDirectAddr;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwAddr = pItem->I.IO.dwAddr;
                    dwPhysAddr = dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=GT64_ADDR_BAR0; ad_sp<=GT64_ADDR_BAR5; ad_sp++)
                {
                    DWORD dwPCIAddr;
                    DWORD dwPCIReg;

                    if (hGT->addrDesc[ad_sp].dwAddr) continue;
                    else if (ad_sp<GT64_ADDR_BAR5) 
                        dwPCIReg = PCI_BAR0 + 4*(ad_sp-GT64_ADDR_BAR0);
                    else dwPCIReg = PCI_ERBAR;
                    dwPCIAddr = GT64_ReadPCIReg(hGT, dwPCIReg);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<=GT64_ADDR_BAR5)
                {
                    DWORD j;
                    hGT->addrDesc[ad_sp].dwBytes = dwBytes;
                    hGT->addrDesc[ad_sp].dwAddr = dwAddr;
                    hGT->addrDesc[ad_sp].dwAddrDirect = dwAddrDirect;
                    hGT->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hGT->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<hGT->addrDesc[ad_sp].dwBytes && j!=0x80000000; j *= 2)
                    {
                        hGT->addrDesc[ad_sp].dwMask = 
                            (hGT->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hGT->Int.Int.hInterrupt) return FALSE;
            hGT->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hGT->fUseInt && !hGT->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that the registers space was found
    if (!GT64_IsAddrSpaceActive(hGT, GT64_ADDR_REG)) 
            //|| hGT->addrDesc[GT64_ADDR_REG].dwBytes!=GT64_RANGE_REG)
        return FALSE;

    // check that address space 0 was found
    if (!GT64_IsAddrSpaceActive(hGT, GT64_ADDR_BAR0)) return FALSE;

    return TRUE;
}

void GT64_Close(GT64_HANDLE hGT)
{
    // disable interrupts
    if (GT64_IntIsEnabled(hGT))
        GT64_IntDisable(hGT);

    // unregister card
    if (hGT->cardReg.hCard) 
        WD_CardUnregister(hGT->hWD, &hGT->cardReg);

    // close WinDriver
    WD_Close(hGT->hWD);

    free (hGT);
}

BOOL GT64_IsAddrSpaceActive(GT64_HANDLE hGT, GT64_ADDR addrSpace)
{
    return hGT->addrDesc[addrSpace].dwAddr!=0;
}

DWORD GT64_ReadReg (GT64_HANDLE hGT, DWORD dwReg)
{
    return GT64_ReadDWord(hGT, GT64_ADDR_REG, dwReg);
}

void GT64_WriteReg (GT64_HANDLE hGT, DWORD dwReg, DWORD dwData)
{
    GT64_WriteDWord(hGT, GT64_ADDR_REG, dwReg, dwData);
}

DWORD GT64_ReadDWord (GT64_HANDLE hGT, GT64_ADDR addrSpace, DWORD dwOffset)
{
    if (hGT->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hGT->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        return *pDword;
    }
    else
    {
        DWORD dwAddr = hGT->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_DWORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hGT->hWD, &trans);
        return trans.Data.Dword;
    }
}

void GT64_WriteDWord (GT64_HANDLE hGT, GT64_ADDR addrSpace, DWORD dwOffset, DWORD data)
{
    if (hGT->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hGT->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        *pDword = data;
    }
    else
    {
        DWORD dwAddr = hGT->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_DWORD;
        trans.dwPort = dwAddr;
        trans.Data.Dword = data;
        WD_Transfer (hGT->hWD, &trans);
    }
}

void GT64_ReadWriteBlock (GT64_HANDLE hGT, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, GT64_ADDR addrSpace)
{
    DWORD dwAddr = hGT->addrDesc[addrSpace].dwAddr + dwOffset;
    WD_TRANSFER trans;

    BZERO(trans);

    if (hGT->addrDesc[addrSpace].fIsMemory) 
    {
        if (fIsRead) 
            trans.cmdTrans = RM_SDWORD;
        else 
            trans.cmdTrans = WM_SDWORD;
    }
    else 
    {
        if (fIsRead) 
            trans.cmdTrans = RP_SDWORD;
        else 
            trans.cmdTrans = WP_SDWORD;
    }
    trans.dwPort = dwAddr;
    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hGT->hWD, &trans);
}

void GT64_ReadBlock (GT64_HANDLE hGT, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, GT64_ADDR addrSpace)
{
    GT64_ReadWriteBlock (hGT, dwOffset, buf, dwBytes, TRUE, addrSpace);
}

void GT64_WriteBlock (GT64_HANDLE hGT, DWORD dwOffset, PVOID buf, 
                     DWORD dwBytes, GT64_ADDR addrSpace)
{
    GT64_ReadWriteBlock (hGT, dwOffset, buf, dwBytes, FALSE, addrSpace);
}

BOOL GT64_IntIsEnabled (GT64_HANDLE hGT)
{
    if (!hGT->fUseInt) return FALSE;
    if (!hGT->Int.hThread) return FALSE;
    return TRUE;
}

VOID GT64_IntHandler (PVOID pData)
{
    GT64_HANDLE hGT = (GT64_HANDLE) pData;
    GT64_INT_RESULT intResult;
    intResult.dwCounter = hGT->Int.Int.dwCounter;
    intResult.dwLost = hGT->Int.Int.dwLost;
    intResult.fStopped = hGT->Int.Int.fStopped;
    intResult.dwStatusReg = hGT->Int.Trans[0].Data.Dword;
    hGT->Int.funcIntHandler(hGT, &intResult);  
}

BOOL GT64_IntEnable (GT64_HANDLE hGT, GT64_INT_HANDLER funcIntHandler)
{
    DWORD dwAddr;

    if (!hGT->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hGT->Int.hThread) return FALSE;

    BZERO(hGT->Int.Trans);
    // One transfer commands is issued to CANCEL the source of the interrupt,
    // otherwise, the PC will hang when an interrupt occurs!
    dwAddr = hGT->addrDesc[GT64_ADDR_REG].dwAddr + GT64_INTERRUPT_ACKNOWLEDGE;
    hGT->Int.Trans[0].cmdTrans = hGT->addrDesc[GT64_ADDR_REG].fIsMemory ? RM_DWORD : RP_DWORD;
    hGT->Int.Trans[0].dwPort = dwAddr;
    hGT->Int.Int.dwCmds = 1; 
    hGT->Int.Int.Cmd = hGT->Int.Trans;
    hGT->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;
 
    // this calls WD_IntEnable() and creates an interrupt handler thread
    hGT->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hGT->Int.hThread, hGT->hWD, &hGT->Int.Int, GT64_IntHandler, (PVOID) hGT))
        return FALSE;

    return TRUE;
}

void GT64_IntDisable (GT64_HANDLE hGT)
{
    if (!hGT->fUseInt) return;
    if (!hGT->Int.hThread) return;

    // this calls WD_IntDisable()
    InterruptThreadDisable(hGT->Int.hThread);
    hGT->Int.hThread = NULL;
}

GT64_DMA_HANDLE GT64_DMAOpen (GT64_HANDLE hGT, DWORD dwLocalAddr, PVOID buf, 
    DWORD dwBytes, BOOL fIsRead, GT64_DAT_TRANS_LIM lim_mode, GT64_DMA_CHANNEL dmaChannel)
{
    DWORD dwChannelShift = dmaChannel*GT64_DMA_CHANNEL_SHIFT;
    BOOL fAutoinc = TRUE;
    GT64_DMA_HANDLE hDma;
    
    if (hGT->gt64Ver == GT64_VERSION_64111) 
    {
        sprintf( GT64_ErrorString, "GT-64111 DMA transfer are not supported\n");
        goto Exit;
    }
    hDma = malloc (sizeof(GT64_DMA_STRUCT));
    if (hDma==NULL)
    {
        sprintf( GT64_ErrorString, "Failed allocating memory for dma handle!\n");
        goto Exit;
    }
    BZERO (*hDma);
    hDma->dmaChannel = dmaChannel;
    hDma->dma.dwBytes = dwBytes;
    hDma->dma.pUserAddr = buf; 
    hDma->dma.dwOptions = 0; 
    WD_DMALock (hGT->hWD, &hDma->dma);
    if (!hDma->dma.hDma) 
    {
        sprintf( GT64_ErrorString, "Failed locking the buffer!\n");
        goto Exit;
    }
    
    hDma->dwDmaControl =  
          ((!fAutoinc && fIsRead)? 0 : BIT3)    // hold source address 
        | ((!fAutoinc && !fIsRead)? 0 : BIT5)   // hold destension address
        | lim_mode       // burst size
        | BIT11          // block transfer mode 
        | BIT12          // to initiate the DMA
        | BIT21 | BIT23 | BIT25; // source, destension, and next record is in PCI_0 Memory space
    if (hDma->dma.dwPages==1)
    {
        hDma->dwDmaControl |= BIT9;  // do not use chain mode
        //dma of one page ==> direct dma

        GT64_WriteReg (hGT, GT64_CHANNEL_0_DMA_BYTE_COUNT + dwChannelShift, hDma->dma.Page[0].dwBytes);
        if (fIsRead)
        {
            GT64_WriteReg (hGT, GT64_CHANNEL_0_DMA_SOURCE_ADDRESS + dwChannelShift, dwLocalAddr);
            GT64_WriteReg (hGT, GT64_CHANNEL_0_DMA_DESTINATION_ADDRESS + dwChannelShift
                , (DWORD) hDma->dma.Page[0].pPhysicalAddr);
        }
        else
        {
            GT64_WriteReg (hGT, GT64_CHANNEL_0_DMA_DESTINATION_ADDRESS + dwChannelShift, dwLocalAddr);
            GT64_WriteReg (hGT, GT64_CHANNEL_0_DMA_SOURCE_ADDRESS + dwChannelShift
                , (DWORD) hDma->dma.Page[0].pPhysicalAddr);
        }
        GT64_WriteReg (hGT, GT64_CHANNEL_0_NEXT_RECORD_POINTER + dwChannelShift, 0);  //One block
    }
    else
    {
        DWORD dwPageNumber, dwMemoryCopied;
        typedef struct {
            DWORD dwByteCt;
            DWORD dwSrcAdd;
            DWORD dwDestAdd;
            DWORD dwNextRecPtr;
        } DMA_LIST;
        DMA_LIST *pList;

        //dma of more then one page ==> chain dma
        hDma->dmaList.dwBytes = hDma->dma.dwPages * sizeof(DMA_LIST);
        hDma->dmaList.pUserAddr = NULL;
        hDma->dmaList.dwOptions = DMA_KERNEL_BUFFER_ALLOC;
        WD_DMALock (hGT->hWD, &hDma->dmaList);
        if (!hDma->dmaList.hDma)
        {
            sprintf (GT64_ErrorString, "Failed locking the chain buffer!\n");
            goto Exit;
        }

        //setting chain of dma pages in the memory
        dwMemoryCopied = 0;
        pList = (DMA_LIST *) ((DWORD) hDma->dmaList.pUserAddr);
        for (dwPageNumber=0; dwPageNumber<hDma->dma.dwPages; dwPageNumber++)
        {
            pList[dwPageNumber].dwByteCt = hDma->dma.Page[dwPageNumber].dwBytes;
            if (fIsRead)
            {
                pList[dwPageNumber].dwSrcAdd = dwLocalAddr + (fAutoinc ? dwMemoryCopied : 0);
                pList[dwPageNumber].dwDestAdd = (DWORD) hDma->dma.Page[dwPageNumber].pPhysicalAddr;
            }
            else
            {
                pList[dwPageNumber].dwDestAdd = dwLocalAddr + (fAutoinc ? dwMemoryCopied : 0);
                pList[dwPageNumber].dwSrcAdd = (DWORD) hDma->dma.Page[dwPageNumber].pPhysicalAddr;
            }
            pList[dwPageNumber].dwNextRecPtr =  
                ((DWORD) hDma->dmaList.Page[0].pPhysicalAddr + sizeof(DMA_LIST)*(dwPageNumber+1));
            dwMemoryCopied += hDma->dma.Page[dwPageNumber].dwBytes;
        }
        pList[dwPageNumber - 1].dwNextRecPtr = 0 ; // mark end of chain
    
        hDma->dwDmaControl |= BIT10    // interrupt only after finishing all pages
            | BIT13;   // fetching the first record
        // adjusting the first record
        GT64_WriteReg (hGT, GT64_CHANNEL_0_NEXT_RECORD_POINTER + dwChannelShift,
            (DWORD)hDma->dmaList.Page[0].pPhysicalAddr);
    }
    return hDma;

Exit:
    if (hDma!=NULL)
        GT64_DMAClose(hGT,hDma);
    return NULL;
}

void GT64_DMAClose (GT64_HANDLE hGT, GT64_DMA_HANDLE hDma)
{
    if (hDma->dma.hDma)
        WD_DMAUnlock(hGT->hWD, &hDma->dma);
    if (hDma->dmaList.hDma)
        WD_DMAUnlock(hGT->hWD, &hDma->dmaList);
    free (hDma);
}

BOOL GT64_DMAIsDone (GT64_HANDLE hGT, GT64_DMA_HANDLE hDma)
{
    return (GT64_ReadReg (hGT, GT64_CHANNEL_0_CONTROL + hDma->dmaChannel * GT64_DMA_CHANNEL_SHIFT) & BIT14)==BIT14;
}

void GT64_DMAStart (GT64_HANDLE hGT, GT64_DMA_HANDLE hDma, BOOL fBlocking)
{
    GT64_WriteReg (hGT, GT64_CHANNEL_0_CONTROL + hDma->dmaChannel * GT64_DMA_CHANNEL_SHIFT, hDma->dwDmaControl);

    //Busy wait for GT to finish transfer
    if (fBlocking) 
        while (!GT64_DMAIsDone(hGT, hDma));
}

BOOL GT64_DMAReadWriteBlock (GT64_HANDLE hGT, GT64_ADDR addrSpace, DWORD dwLocalAddrOffset, PVOID buf, 
    DWORD dwBytes, BOOL fIsRead, GT64_DAT_TRANS_LIM lim_mode, GT64_DMA_CHANNEL dmaChannel)
{
    DWORD dwLocalAddr = hGT->addrDesc[addrSpace].dwAddrDirect + dwLocalAddrOffset;
    GT64_DMA_HANDLE hDma;
    if (dwBytes==0) 
        return TRUE;

    hDma = GT64_DMAOpen(hGT, dwLocalAddr, buf, dwBytes, fIsRead, lim_mode, dmaChannel);
    if (!hDma) 
        return FALSE;
    GT64_DMAStart (hGT, hDma, TRUE);

    GT64_DMAClose(hGT, hDma);
    return TRUE;
}

