////////////////////////////////////////////////////////////////
// File - P9050_LIB.C
//
// Library for 'WinDriver for PLX 9050' API. 
// The basic idea is to get a handle for the board
// with P9050_Open() and use it in the rest of the program
// when calling WD functions.  Call P9050_Close() when done.
// 
////////////////////////////////////////////////////////////////

#include "p9050_lib.h"
#include "../../../include/windrvr_int_thread.h"
#include <stdio.h>

// this string is set to an error message, if one occurs
CHAR P9050_ErrorString[1024];

// internal data structures
typedef struct
{
    WD_INTERRUPT Int;
    HANDLE hThread;
    WD_TRANSFER Trans[2];
    P9050_INT_HANDLER funcIntHandler;
} P9050_INTERRUPT;

typedef struct
{
    DWORD dwLocalBase;
    DWORD dwMask;
    DWORD dwBytes;
    DWORD dwAddr;
    DWORD dwAddrDirect;
    BOOL  fIsMemory;
} P9050_ADDR_DESC;

typedef struct P9050_STRUCT
{
    HANDLE hWD;
    WD_CARD cardLock;
    WD_PCI_SLOT pciSlot;
    WD_CARD_REGISTER cardReg;
    P9050_ADDR_DESC addrDesc[AD_PCI_BARS];
    BOOL   fUseInt;
    P9050_INTERRUPT Int;
} P9050_STRUCT;

// internal function used by P9050_Open()
BOOL P9050_DetectCardElements(P9050_HANDLE hPlx);
// internal function used by P9050_Read... and P9050_Write... functions
void P9050_SetMode (P9050_HANDLE hPlx, P9050_ADDR addrSpace, P9050_MODE mode, DWORD dwLocalAddr);

DWORD P9050_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    P9050_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( P9050_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( P9050_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( P9050_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}


BOOL P9050_Open (P9050_HANDLE *phPlx, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD options)
{
    P9050_HANDLE hPlx = (P9050_HANDLE) malloc (sizeof (P9050_STRUCT));

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;
    DWORD dwData;

    *phPlx = NULL;
    P9050_ErrorString[0] = '\0';
    BZERO(*hPlx);

    hPlx->cardReg.hCard = 0;
    hPlx->hWD = WD_Open();

    // check if handle valid & version OK
    if (hPlx->hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( P9050_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hPlx->hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( P9050_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hPlx->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( P9050_ErrorString, "Could not find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( P9050_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    hPlx->pciSlot = pciCardInfo.pciSlot;

    if (options & P9050_OPEN_FIX_BIT7)
    {
        // this section fixes the BIT7 errata on PLX 9050 chip
        dwData = P9050_ReadPCIReg(hPlx, PCI_BAR0) & ~BIT7;
        P9050_WritePCIReg(hPlx, PCI_BAR0, dwData);
        // There is usually no need to fix also BAR1
        //dwData = P9050_ReadPCIReg(hPlx, PCI_BAR1) & ~BIT7;
        //P9050_WritePCIReg(hPlx, PCI_BAR1, dwData);
    }

    WD_PciGetCardInfo (hPlx->hWD, &pciCardInfo);
    hPlx->cardReg.Card = pciCardInfo.Card;

    hPlx->fUseInt = (options & P9050_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hPlx->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hPlx->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hPlx->hWD, &hPlx->cardReg);
    if (hPlx->cardReg.hCard==0)
    {
        sprintf ( P9050_ErrorString, "Failed locking device\n");
        goto Exit;
    }

    if (!P9050_DetectCardElements(hPlx))
    {
        sprintf ( P9050_ErrorString, "Card does not have all items expected for PLX 9050\n");
        goto Exit;
    }

    // Open finished OK
    *phPlx = hPlx;
    return TRUE;

Exit:
    // Error durin Open
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);
    if (hPlx->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hPlx->hWD);
    free (hPlx);
    return FALSE;
}

void P9050_GetPciSlot(P9050_HANDLE hPlx, WD_PCI_SLOT *pPciSlot)
{
    *pPciSlot = hPlx->pciSlot;
}

DWORD P9050_ReadPCIReg(P9050_HANDLE hPlx, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
    return dwVal;
}

void P9050_WritePCIReg(P9050_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
}

BOOL P9050_DetectCardElements(P9050_HANDLE hPlx)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hPlx->Int);
    BZERO(hPlx->addrDesc);

    for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwAddr;
                DWORD dwAddrDirect = 0;
                DWORD dwPhysAddr;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwAddr = pItem->I.Mem.dwTransAddr;
                    dwAddrDirect = pItem->I.Mem.dwUserDirectAddr;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwAddr = pItem->I.IO.dwAddr;
                    dwPhysAddr = dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=P9050_ADDR_REG; ad_sp<=P9050_ADDR_EPROM; ad_sp++)
                {
                    DWORD dwPCIAddr;
                    DWORD dwPCIReg;

                    if (hPlx->addrDesc[ad_sp].dwAddr) continue;
                    if (ad_sp==P9050_ADDR_REG) dwPCIReg = PCI_BAR0;
                    else if (ad_sp<P9050_ADDR_EPROM) 
                        dwPCIReg = PCI_BAR2 + 4*(ad_sp-P9050_ADDR_SPACE0);
                    else dwPCIReg = PCI_ERBAR;
                    dwPCIAddr = P9050_ReadPCIReg(hPlx, dwPCIReg);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<=P9050_ADDR_EPROM)
                {
                    DWORD j;
                    hPlx->addrDesc[ad_sp].dwBytes = dwBytes;
                    hPlx->addrDesc[ad_sp].dwAddr = dwAddr;
                    hPlx->addrDesc[ad_sp].dwAddrDirect = dwAddrDirect;
                    hPlx->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hPlx->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<hPlx->addrDesc[ad_sp].dwBytes && j!=0x80000000; j *= 2)
                    {
                        hPlx->addrDesc[ad_sp].dwMask = 
                            (hPlx->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hPlx->Int.Int.hInterrupt) return FALSE;
            hPlx->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hPlx->fUseInt && !hPlx->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that the registers space was found
    if (!P9050_IsAddrSpaceActive(hPlx, P9050_ADDR_REG))
            //|| hPlx->addrDesc[P9050_ADDR_REG].dwBytes!=P9050_RANGE_REG)
        return FALSE;

    // check that at least one memory space was found
    // for (i = P9050_ADDR_SPACE0; i<=P9050_ADDR_EPROM; i++)
    //     if (P9050_IsAddrSpaceActive(hPlx, i)) break;
    // if (i>P9050_ADDR_EPROM) return FALSE;

    return TRUE;
}

void P9050_Close(P9050_HANDLE hPlx)
{
    // disable interrupts
    if (P9050_IntIsEnabled(hPlx))
        P9050_IntDisable(hPlx);

    // unregister card
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);

    // close WinDriver
    WD_Close(hPlx->hWD);

    free (hPlx);
}

BOOL P9050_IsAddrSpaceActive(P9050_HANDLE hPlx, P9050_ADDR addrSpace)
{
    return hPlx->addrDesc[addrSpace].dwAddr!=0;
}

DWORD P9050_ReadReg (P9050_HANDLE hPlx, DWORD dwReg)
{
    return P9050_ReadSpaceDWord(hPlx, P9050_ADDR_REG, dwReg);
}

void P9050_WriteReg (P9050_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    P9050_WriteSpaceDWord(hPlx, P9050_ADDR_REG, dwReg, dwData);
}

void P9050_SetMode (P9050_HANDLE hPlx, P9050_ADDR addrSpace, P9050_MODE mode, DWORD dwLocalAddr)
{
    DWORD dwRegOffset = 4*(addrSpace-P9050_ADDR_SPACE0);
    P9050_ADDR_DESC *addrDesc = &hPlx->addrDesc[addrSpace];
    addrDesc->dwLocalBase = dwLocalAddr & ~addrDesc->dwMask;
    addrDesc->dwLocalBase |= BIT0;
    P9050_WriteReg (hPlx, P9050_LAS0BA + dwRegOffset, addrDesc->dwLocalBase);
}

BYTE P9050_ReadSpaceByte (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        return *pByte;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_BYTE;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Byte;
    }
}

void P9050_WriteSpaceByte (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwOffset, BYTE data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        *pByte = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_BYTE;
        trans.dwPort = dwAddr;
        trans.Data.Byte = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

WORD P9050_ReadSpaceWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        return *pWord;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_WORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Word;
    }
}

void P9050_WriteSpaceWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwOffset, WORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        *pWord = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_WORD;
        trans.dwPort = dwAddr;
        trans.Data.Word = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

DWORD P9050_ReadSpaceDWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        return *pDword;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_DWORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Dword;
    }
}

void P9050_WriteSpaceDWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwOffset, DWORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        *pDword = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_DWORD;
        trans.dwPort = dwAddr;
        trans.Data.Dword = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

void P9050_ReadWriteSpaceBlock (P9050_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, P9050_ADDR addrSpace, P9050_MODE mode)
{
    WD_TRANSFER trans;
    DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;

    BZERO(trans);

    if (hPlx->addrDesc[addrSpace].fIsMemory) 
    {
        if (fIsRead) 
        {
            if (mode==P9050_MODE_BYTE) trans.cmdTrans = RM_SBYTE;
            else if (mode==P9050_MODE_WORD) trans.cmdTrans = RM_SWORD;
            else trans.cmdTrans = RM_SDWORD;
        }
        else 
        {
            if (mode==P9050_MODE_BYTE) trans.cmdTrans = WM_SBYTE;
            else if (mode==P9050_MODE_WORD) trans.cmdTrans = WM_SWORD;
            else trans.cmdTrans = WM_SDWORD;
        }
    }
    else 
    {
        if (fIsRead) 
        {
            if (mode==P9050_MODE_BYTE) trans.cmdTrans = RP_SBYTE;
            else if (mode==P9050_MODE_WORD) trans.cmdTrans = RP_SWORD;
            else trans.cmdTrans = RP_SDWORD;
        }
        else 
        {
            if (mode==P9050_MODE_BYTE) trans.cmdTrans = WP_SBYTE;
            else if (mode==P9050_MODE_WORD) trans.cmdTrans = WP_SWORD;
            else trans.cmdTrans = WP_SDWORD;
        }
    }
    trans.dwPort = dwAddr;
    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hPlx->hWD, &trans);
}

void P9050_ReadSpaceBlock (P9050_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, P9050_ADDR addrSpace, P9050_MODE mode)
{
    P9050_ReadWriteSpaceBlock (hPlx, dwOffset, buf, dwBytes, TRUE, addrSpace, mode);
}

void P9050_WriteSpaceBlock (P9050_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                     DWORD dwBytes, P9050_ADDR addrSpace, P9050_MODE mode)
{
    P9050_ReadWriteSpaceBlock (hPlx, dwOffset, buf, dwBytes, FALSE, addrSpace, mode);
}

BYTE P9050_ReadByte (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, P9050_MODE_BYTE, dwLocalAddr);
    return P9050_ReadSpaceByte(hPlx, addrSpace, dwOffset);
}

void P9050_WriteByte (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwLocalAddr, BYTE data)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, P9050_MODE_BYTE, dwLocalAddr);
    P9050_WriteSpaceByte(hPlx, addrSpace, dwOffset, data);
}

WORD P9050_ReadWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, P9050_MODE_WORD, dwLocalAddr);
    return P9050_ReadSpaceWord(hPlx, addrSpace, dwOffset);
}

void P9050_WriteWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwLocalAddr, WORD data)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, P9050_MODE_WORD, dwLocalAddr);
    P9050_WriteSpaceWord(hPlx, addrSpace, dwOffset, data);
}

DWORD P9050_ReadDWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, P9050_MODE_DWORD, dwLocalAddr);
    return P9050_ReadSpaceDWord(hPlx, addrSpace, dwOffset);
}

void P9050_WriteDWord (P9050_HANDLE hPlx, P9050_ADDR addrSpace, DWORD dwLocalAddr, DWORD data)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, P9050_MODE_DWORD, dwLocalAddr);
    P9050_WriteSpaceDWord(hPlx, addrSpace, dwOffset, data);
}

void P9050_ReadWriteBlock (P9050_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, P9050_ADDR addrSpace, P9050_MODE mode)
{
    DWORD dwOffset = hPlx->addrDesc[addrSpace].dwMask & dwLocalAddr;
    P9050_SetMode (hPlx, addrSpace, mode, dwLocalAddr);
    P9050_ReadWriteSpaceBlock(hPlx, dwOffset, buf, dwBytes, fIsRead, addrSpace, mode);
}

void P9050_ReadBlock (P9050_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                    DWORD dwBytes, P9050_ADDR addrSpace, P9050_MODE mode)
{
    P9050_ReadWriteBlock (hPlx, dwLocalAddr, buf, dwBytes, TRUE, addrSpace, mode);
}

void P9050_WriteBlock (P9050_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                     DWORD dwBytes, P9050_ADDR addrSpace, P9050_MODE mode)
{
    P9050_ReadWriteBlock (hPlx, dwLocalAddr, buf, dwBytes, FALSE, addrSpace, mode);
}

BOOL P9050_IntIsEnabled (P9050_HANDLE hPlx)
{
    if (!hPlx->fUseInt) return FALSE;
    if (!hPlx->Int.hThread) return FALSE;
    return TRUE;
}

void P9050_IntHandler (PVOID pData)
{
    P9050_HANDLE hPlx = (P9050_HANDLE) pData;
    P9050_INT_RESULT intResult;
    
    intResult.dwCounter = hPlx->Int.Int.dwCounter;
    intResult.dwLost = hPlx->Int.Int.dwLost;
    intResult.fStopped = hPlx->Int.Int.fStopped;
    intResult.dwStatusReg = hPlx->Int.Trans[0].Data.Dword;
    hPlx->Int.funcIntHandler(hPlx, &intResult);
}

BOOL P9050_IntEnable (P9050_HANDLE hPlx, P9050_INT_HANDLER funcIntHandler)
{
    DWORD dwIntStatus;
    DWORD dwAddr;

    if (!hPlx->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hPlx->Int.hThread) return FALSE;

    dwIntStatus = P9050_ReadReg (hPlx, P9050_INTCSR);

    BZERO(hPlx->Int.Trans);
    // This is a samlpe of handling interrupts:
    // Two transfer commands are issued. First the value of the interrrupt control/status
    // register is read. Then, a value of ZERO is written.
    // This will cancel interrupts after the first interrupt occurs.
    // When using interrupts, this section will have to change:
    // you must put transfer commands to CANCEL the source of the interrupt, otherwise, the 
    // PC will hang when an interrupt occurs!
    dwAddr = hPlx->addrDesc[P9050_ADDR_REG].dwAddr + P9050_INTCSR;
    hPlx->Int.Trans[0].cmdTrans = hPlx->addrDesc[P9050_ADDR_REG].fIsMemory ? RM_DWORD : RP_DWORD;
    hPlx->Int.Trans[0].dwPort = dwAddr;
    hPlx->Int.Trans[1].cmdTrans = hPlx->addrDesc[P9050_ADDR_REG].fIsMemory ? WM_DWORD : WP_DWORD;
    hPlx->Int.Trans[1].dwPort = dwAddr;
    hPlx->Int.Trans[1].Data.Dword = dwIntStatus & ~BIT6; // put here the data to write to the control register
    hPlx->Int.Int.dwCmds = 2; 
    hPlx->Int.Int.Cmd = hPlx->Int.Trans;
    hPlx->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;

    // this calls WD_IntEnable() and creates an interrupt handler thread
    hPlx->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hPlx->Int.hThread, hPlx->hWD, &hPlx->Int.Int, P9050_IntHandler, (PVOID) hPlx))
        return FALSE;

    // this physically enables interrupts
    P9050_WriteReg (hPlx, P9050_INTCSR, dwIntStatus | BIT6);

    return TRUE;
}

void P9050_IntDisable (P9050_HANDLE hPlx)
{
    DWORD dwIntStatus;

    if (!hPlx->fUseInt) return;
    if (!hPlx->Int.hThread) return;

    // this disables interrupts
    dwIntStatus = P9050_ReadReg (hPlx, P9050_INTCSR);
    P9050_WriteReg (hPlx, P9050_INTCSR, dwIntStatus & ~BIT6);

    // this calls WD_IntDisable()
    InterruptThreadDisable(hPlx->Int.hThread);

    hPlx->Int.hThread = NULL;
}

void P9050_EEPROMDelay(P9050_HANDLE hPlx)
{
    WD_SLEEP sleep;
    BZERO (sleep);
    sleep.dwMicroSeconds = 500;
    WD_Sleep( hPlx->hWD, &sleep);
}

void P9050_EEPROMChipSelect(P9050_HANDLE hPlx, BOOL fSelect)
{
    DWORD dwCNTRL = P9050_ReadReg(hPlx, P9050_CNTRL);
    if (fSelect)
        dwCNTRL |= BIT25;
    else
        dwCNTRL &= ~BIT25;
    P9050_WriteReg(hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);
}

BOOL P9050_EEPROMValid(P9050_HANDLE hPlx)
{
    return (P9050_ReadReg(hPlx, P9050_CNTRL) & BIT28)==BIT28;
}

void P9050_EEPROMWriteBit(P9050_HANDLE hPlx, BOOL fBit)
{
    DWORD dwCNTRL = P9050_ReadReg(hPlx, P9050_CNTRL);

    dwCNTRL &= ~BIT24;
    if (fBit) // data
        dwCNTRL |= BIT26;
    else
        dwCNTRL &= ~BIT26;
    P9050_WriteReg( hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);

    dwCNTRL |= BIT24; // clock
    P9050_WriteReg( hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);

    dwCNTRL &= ~BIT24;
    P9050_WriteReg( hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);
}

BOOL P9050_EEPROMReadBit(P9050_HANDLE hPlx)
{
    BOOL fRet;
    DWORD dwCNTRL = P9050_ReadReg(hPlx, P9050_CNTRL);

    dwCNTRL &= ~BIT24;
    P9050_WriteReg( hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);

    dwCNTRL |= BIT24; // clock
    P9050_WriteReg( hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);

    dwCNTRL &= ~BIT24;
    P9050_WriteReg( hPlx, P9050_CNTRL, dwCNTRL);
    P9050_EEPROMDelay(hPlx);

    fRet = (P9050_ReadReg( hPlx, P9050_CNTRL) & BIT27)==BIT27;

    return fRet;
}

void P9050_EEPROMWriteEnableDisable(P9050_HANDLE hPlx, BOOL fEnable)
{
    P9050_EEPROMChipSelect(hPlx, TRUE);

    // send a WEN instruction
    P9050_EEPROMWriteBit(hPlx, 1);
    P9050_EEPROMWriteBit(hPlx, 0);
    P9050_EEPROMWriteBit(hPlx, 0);
    P9050_EEPROMWriteBit(hPlx, fEnable ? 1 : 0);
    P9050_EEPROMWriteBit(hPlx, fEnable ? 1 : 0);

    P9050_EEPROMWriteBit(hPlx, 0);
    P9050_EEPROMWriteBit(hPlx, 0);
    P9050_EEPROMWriteBit(hPlx, 0);
    P9050_EEPROMWriteBit(hPlx, 0);

    P9050_EEPROMChipSelect(hPlx, FALSE);
}

BOOL P9050_EEPROMReadWord(P9050_HANDLE hPlx, DWORD dwOffset, PWORD pwData)
{
    DWORD dwAddr = dwOffset >> 1;
    DWORD i;

    *pwData = 0;

    P9050_EEPROMChipSelect(hPlx, TRUE);
    P9050_EEPROMWriteBit(hPlx, 1);
    P9050_EEPROMWriteBit(hPlx, 1);
    P9050_EEPROMWriteBit(hPlx, 0);
    // if it's a CS46 EEPROM send only 5 bit address
    for (i=BIT5; i; i = i>>1)
    {
        P9050_EEPROMWriteBit(hPlx, (dwAddr & i) == i);
    }
    for (i=BIT15; i; i = i>>1)
    {
        *pwData |= P9050_EEPROMReadBit(hPlx) ? i : 0;
    }

    P9050_EEPROMWriteEnableDisable(hPlx, FALSE);
    
    return TRUE;
}

BOOL P9050_EEPROMWriteWord(P9050_HANDLE hPlx, DWORD dwOffset, WORD wData)
{
    DWORD dwAddr = dwOffset >> 1;
    DWORD i;
    WORD readback;

    P9050_EEPROMWriteEnableDisable(hPlx, TRUE);

    P9050_EEPROMChipSelect(hPlx, TRUE);

    // send a PRWRITE instruction
    P9050_EEPROMWriteBit(hPlx, 1);
    P9050_EEPROMWriteBit(hPlx, 0);
    P9050_EEPROMWriteBit(hPlx, 1);
    // if it's a CS46 EEPROM send only a 6 bit address
    for (i=BIT5; i; i = i>>1)
    {
        P9050_EEPROMWriteBit(hPlx, (dwAddr & i) == i);
    }
    for (i=BIT15; i; i = i>>1)
    {
        P9050_EEPROMWriteBit(hPlx, (wData & i) == i);
    }

    P9050_EEPROMChipSelect(hPlx, FALSE);
    
    P9050_EEPROMWriteEnableDisable(hPlx, FALSE);

    if (P9050_EEPROMReadWord(hPlx, dwOffset, &readback))
    {
        if (wData != readback)
        {
            sprintf( P9050_ErrorString, "Write 0x%04x, Read 0x%04x\n", wData, readback);
            return FALSE;
        }
    }
    else
        return FALSE;

    return TRUE;
}

BOOL P9050_EEPROMReadDWord(P9050_HANDLE hPlx, DWORD dwOffset, PDWORD pdwData)
{
    WORD wData1, wData2;
    if (dwOffset % 4)
    {
        sprintf( P9050_ErrorString, "The offset is not a multiple of 4\n");
        return FALSE;
    }

    if (!P9050_EEPROMReadWord(hPlx, dwOffset, &wData1))
        return FALSE;
    if (!P9050_EEPROMReadWord(hPlx, dwOffset+2, &wData2))
        return FALSE;

    *pdwData = (DWORD) ((wData1 << 16) + wData2);
    return TRUE;
}

BOOL P9050_EEPROMWriteDWord(P9050_HANDLE hPlx, DWORD dwOffset, DWORD dwData)
{
    WORD wData1, wData2;

    if (dwOffset % 4)
    {
        sprintf( P9050_ErrorString, "The offset is not a multiple of 4\n");
        return FALSE;
    }

    wData1 = (WORD) (dwData >> 16);
    wData2 = (WORD) (dwData & 0xffff);

    if (!P9050_EEPROMWriteWord(hPlx, dwOffset, wData1))
        return FALSE;
    if (!P9050_EEPROMWriteWord(hPlx, dwOffset+2, wData2))
        return FALSE;

    return TRUE;
}
