/*****************************************************************************
 * game.cpp - General game loop class
 *****************************************************************************/
#include "SDL.h"
#include "actioneventqueue.h"
#include "aiplugman.h"
#include "ccmap.h"
#include "common.h"
#include "config.h"
#include "cursor.h"
#include "dispatcher.h"
#include "game.h"
#include "graphicsengine.h"
#include "input.h"
#include "loadingscreen.h"
#include "logger.h"
#include "playerpool.h"
#include "sidebar.h"
#include "soundengine.h"
#include "vqa.h"

using std::string;

/** Constructor, loads the map, sidebar and such. plays briefing and actionmovie
 */
Game::Game() {
    ConfigType config;
    VQAMovie *mov;
    string tmp;
    INIFile* fileini;
    LoadingScreen *loadscreen;
    config = getConfig();
    /// @TODO We've already loaded files.ini in the vfs.
    fileini = GetConfig("files.ini");
    INIKey key = fileini->readIndexedKeyValue("general", config.gamenum, "PLAY");
    if (!pc::sfxeng->CreatePlaylist(key->second.c_str())) {
        logger->error("Could not create playlist!\n");
        throw GameError();
    }
    loadscreen = new LoadingScreen();
    gamemode = config.gamemode;

    /* reset the tickcounter, should be a function in the class conatining the
     * counter */
    loadscreen->setCurrentTask("Creating the ActionEventQueue");
    p::aequeue = new ActionEventQueue();
    /* load the map */
    loadscreen->setCurrentTask("Loading the map.");
    try {
        p::ccmap = new CnCMap();
        p::ccmap->loadMap(config.mapname.c_str(), loadscreen);
    } catch (CnCMap::LoadMapError&) {
        delete loadscreen;
        // loadmap will have printed the error
        throw GameError();
    }
    p::dispatcher = new Dispatcher::Dispatcher();
    switch (config.dispatch_mode) {
        case 0:
            break;
        case 1:
            // Record
            break;
        case 2:
            // Playback
            break;
        default:
            logger->error("Invalid dispatch mode: %i\n",config.dispatch_mode);
            throw GameError();
            break;
    }

    ps::aiplugman = new AI::AIPlugMan(getBinaryLocation().c_str());

    delete loadscreen;
    switch (gamemode) {
    case 0:
        /* play briefing */
        try {
            mov = new VQAMovie(p::ccmap->getMissionData().brief);
            mov->play();
            delete mov;
        } catch (VQAError&) {
        }
        try {
            mov = new VQAMovie(p::ccmap->getMissionData().action);
            mov->play();
            delete mov;
        } catch (VQAError&) {
        }
        break;
    case 1:
        p::ppool->setupAIs();
        break;
    case 2:
        break;
    default:
        break;
    }
    /* init sidebar */
    try {
        pc::sidebar = new Sidebar(p::ppool->getLPlayer(), pc::gfxeng->getHeight(),
                p::ccmap->getMissionData().theater);
    } catch (Sidebar::SidebarError&) {
        throw GameError();
    }
    /* init cursor */
    pc::cursor = new Cursor();
    /* init the input functions */
    pc::input = new Input(pc::gfxeng->getWidth(), pc::gfxeng->getHeight(),
                          pc::gfxeng->getMapArea());
}

/** Destructor, frees up some memory */
Game::~Game()
{
    delete p::dispatcher;
    delete p::aequeue;
    delete p::ccmap;
    delete pc::input;
    delete pc::sidebar;
    delete pc::cursor;
    delete ps::aiplugman;
}

/** Play the mission. */
void Game::play()
{
    VQAMovie *mov;
    // Start the music
    pc::sfxeng->PlayTrack(p::ccmap->getMissionData().theme);
    pc::gfxeng->setupCurrentGame();
    // Jump to start location
    // @TODO: Jump to correct start location in multiplayer games.
    p::ccmap->restoreLocation(0);
    /* main gameloop */
    while (!pc::input->shouldQuit()) {
        // Draw the scene
        pc::gfxeng->renderScene();
        // Run scheduled events
        p::aequeue->runEvents();
        // Handle the input
        pc::input->handle();
        // Run the AI scripts
        p::ppool->runAIs();
        if (gamemode == 2) {
            // Synchronise events with server
        }
    }
    // Stop the music
    pc::sfxeng->StopMusic();

    if (gamemode == 0) {
        if (p::ppool->hasWon() ) {
            try {
                mov = new VQAMovie(p::ccmap->getMissionData().winmov);
                mov->play();
                delete mov;
            } catch (VQAError&) {
            }
        } else if (p::ppool->hasLost() ) {
            try {
                mov = new VQAMovie(p::ccmap->getMissionData().losemov);
                mov->play();
                delete mov;
            } catch (VQAError&) {
            }
        }
    }
    dumpstats();
}

void Game::dumpstats()
{
    Player* pl;
    Uint8 h,m,s,i;
    Uint32 uptime = p::aequeue->getElapsedTime();
    uptime /= 1000;
    h = uptime/3600;
    uptime %= 3600;
    m = uptime/60;
    s = uptime%60;
    logger->renderGameMsg(false);
    logger->gameMsg("Time wasted: %i hour%s%i minute%s%i second%s",
                    h,(h!=1?"s ":" "),m,(m!=1?"s ":" "),s,(s!=1?"s ":" "));
    for (i = 0; i < p::ppool->getNumPlayers(); i++) {
        pl = p::ppool->getPlayer(i);
        logger->gameMsg("%s\nUnit kills:  %i\n     losses: %i\n"
                        "Structure kills:  %i\n          losses: %i\n",
                        pl->getName(),pl->getUnitKills(),pl->getUnitLosses(),
                        pl->getStructureKills(),pl->getStructureLosses());
    }
}
