/* ls.c - directory utility */
/* Written by Les Bell, 12/18/85 onwards */
/* Version 3.0 for IBM C Set/2 under OS/2 2.1  */

#define INCL_DOSPROCESS
#define INCL_DOSFILEMGR
#define INCL_BASE

#include <os2.h>

#include "stdio.h"
#include "stdlib.h"
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <sys\types.h>
#include <sys\stat.h>

/* #include "subcalls.h" */

extern void main(int argc,char * *argv);
extern void print_attributes(USHORT a_byte);
extern void print_date16(FDATE date);
extern void print_time16(FTIME time);
extern void init(void );
extern void process_options(int argc,char * *argv, char *dirstr);
extern  long round_up(long val,int mult);
extern  int fcomp(const void *entry1, const void *entry2);
extern  int tcomp(const void *entry1, const void *entry2);
extern  int scomp(const void *entry1, const void *entry2);
extern void help(void );
extern void pause(void );
extern void wprint(int nentries);
extern void lprint(int nentries);
extern void crt_cls(void );
extern int isdir(char * pathname);
void eprint(char *filename);

/* File attribute byte bit values */
#define    NORMAL    0x00
#define    READONLY    0x01
#define    HIDDEN    0x02
#define    SYSTEM    0x04
#define    VOLUME    0x08
#define    SUBDIR    0x10
#define    ARCHIVE    0x20

#define TRUE    1
#define FALSE    0

#define MAXFILES 1024
#define FNAMELEN 15
#define SCRDEPTH 25
#define SCRWIDTH 80

/* Structure returned by DOS Find First/Next File call */
FILEFINDBUF4 *finfo[MAXFILES];    /* Array of pointers to file data */

/* Control flags */
struct {
    unsigned lflag:1, dflag:1, helpflag:1, sortflag:1, tsflag:1,
    pauseflag:1, ssort:1, hflag:1, wideflag:1, eaflag:1;
    } flags;

void main(int argc, char **argv)
{

    /* Filename/size comparison functions */
    int fcomp(),tcomp(), scomp();

    USHORT result;

    char dirstr[CCHMAXPATH] = "*.*";
    FILEFINDBUF4 dta;
    int count = 0;
    ULONG lcount = 1;
    USHORT matchatt;
    HDIR MyHandle = HDIR_CREATE;

    /* Initialise flags, etc */
    init();
    process_options(argc,argv, dirstr);

    matchatt = NORMAL;        /* Find normal files only - unless: */
    if (flags.hflag)
        matchatt |= HIDDEN | SYSTEM;
    if (flags.dflag)
        matchatt |= SUBDIR;

    if ((result=DosFindFirst(dirstr,
        &MyHandle,
        matchatt,
        &dta,
        sizeof(FILEFINDBUF4),
        &lcount,
        FIL_QUERYEASIZE))==0) {        /* Find First File function */
                    /* dta contains dir entry */
        while (result == 0) {
            /* Make somewhere to store it and set pointer to it */
            finfo[count] = (PFILEFINDBUF4) malloc(sizeof(FILEFINDBUF4));
            /* Then copy it there */
            memcpy(finfo[count++],&dta,sizeof(FILEFINDBUF4));
            /* Then do it again! */
            result = DosFindNext(MyHandle,&dta,sizeof(FILEFINDBUF4),&lcount);
        }
       DosFindClose(MyHandle);
    }
    else {
        if(result == 0x0012)
            printf("File not found\n");
        else
            printf ("Unknown error: result = %x",result);
    }

    if(flags.sortflag)
        qsort(finfo,count,sizeof(FILEFINDBUF4 *),fcomp);
    else if (flags.ssort)
        qsort(finfo,count,sizeof(FILEFINDBUF4 *),scomp);
    else if (flags.tsflag)
        qsort(finfo,count,sizeof(FILEFINDBUF4 *),tcomp);
    if (flags.wideflag)
        wprint(count);
    else lprint(count);
    eprint("test.txt");
}

void print_attributes(USHORT a_byte)
{
    int i;
    char *c;
    unsigned char mask;

    static char attribute[] = {"12advshr"};

    c = attribute;
    mask = 0x80;
    for (i = 0; i < 8; i++, mask >>= 1, c++)
    fputc(a_byte & mask ? *c : '-',stdout);
}

void print_date16(date)
FDATE date;
{
    static char *mname[] = {
    "illegal month",
    "Jan","Feb","Mar","Apr","May","Jun",
    "Jul","Aug","Sep","Oct","Nov","Dec"};

    printf("%2u %3s %4u  ",date.day,mname[date.month],date.year + 1980);
}

void print_time16(time)
FTIME time;
{
    printf("%02u:%02u:%02u  ",time.hours,time.minutes,time.twosecs);
}

void init()
{
    flags.dflag = FALSE;
    flags.hflag = FALSE;
    flags.lflag = FALSE;
    flags.pauseflag = FALSE;
    flags.ssort = FALSE;
    flags.sortflag = TRUE;
    flags.tsflag = FALSE;
    flags.wideflag = FALSE;
    flags.eaflag = FALSE;
}

void process_options(int argc, char **argv, char *dirstr)
{
    /* Initial default drivespec */
    char defdrv[CCHMAXPATH] = "",
         defpath[CCHMAXPATH]= "",
         deffile[CCHMAXPATH] = "*",
         deftype[CCHMAXPATH] = ".*";

    int i,j;

    for(i = 1; i < argc; i++) {
        if (argv[i][0] == '-' || argv[i][0] == '/') {
            j = 1;
            while (argv[i][j]) {
                switch(toupper(argv[i][j])) {
                    case '?':
                        help();
                        pause();
                        break;
                    case 'C':
                        crt_cls();
                        break;
                    case 'L':
                        flags.lflag = TRUE;
                        flags.wideflag = FALSE;
                        break;
                    case 'D':
                        flags.dflag = TRUE;
                        break;
                    case 'N':
                        flags.sortflag = FALSE;
                        break;
                    case 'P':
                        flags.pauseflag = TRUE;
                        break;
                    case 'S':
                        flags.sortflag = FALSE;
                        flags.ssort = TRUE;
                        break;
                    case 'T':
                        flags.sortflag = FALSE;
                        flags.tsflag = TRUE;
                        break;
                    case 'H':
                        flags.hflag = TRUE;
                        break;
                    case 'W':
                        if (!flags.lflag)
                        flags.wideflag = TRUE;
                        break;
                    case 'E':
                        flags.lflag = TRUE;
                        flags.wideflag = FALSE;
                        flags.eaflag = TRUE;
                }
                j++;
            }
        }
        else {
            _splitpath(argv[i],defdrv,defpath,deffile,deftype);
            if (isdir(argv[i])) { /* argv[i] is a directory, so deffile is tail of path */
                strcat(defpath, deffile);
                strcpy(deffile, "*");
            }
            if(strlen(deftype) == 0)
                strcpy(deftype, ".*");
            if(strlen(deffile) == 0)
                strcpy(deffile, "*");
            _makepath(dirstr,defdrv,defpath,deffile,deftype);
        }
    }
}

long round_up(long val,int mult)
{
    long result = 0;

    while (result < val) result += mult;
    return result;
}

int fcomp(const void *entry1, const void *entry2)
{
    return(strcmp(((FILEFINDBUF4 *)entry1)->achName,((FILEFINDBUF4 *)entry2)->achName));
}

int tcomp(const void *entry1, const void *entry2)
{
    if (((FILEFINDBUF4 *)entry1)->fdateLastWrite.year > ((FILEFINDBUF4 *)entry2)->fdateLastWrite.year)
    return 1;
    if (((FILEFINDBUF4 *)entry1)->fdateLastWrite.year < ((FILEFINDBUF4 *)entry2)->fdateLastWrite.year)
    return -1;
    /* Same year - what about the months? */
    if (((FILEFINDBUF4 *)entry1)->fdateLastWrite.month > ((FILEFINDBUF4 *)entry2)->fdateLastWrite.month)
    return 1;
    if (((FILEFINDBUF4 *)entry1)->fdateLastWrite.month < ((FILEFINDBUF4 *)entry2)->fdateLastWrite.month)
    return -1;
    if (((FILEFINDBUF4 *)entry1)->fdateLastWrite.day > ((FILEFINDBUF4 *)entry2)->fdateLastWrite.day)
    return 1;
    if (((FILEFINDBUF4 *)entry1)->fdateLastWrite.day < ((FILEFINDBUF4 *)entry2)->fdateLastWrite.day)
    return -1;

    /* Same date, now compare times */
    if (((FILEFINDBUF4 *)entry1)->ftimeLastWrite.hours > ((FILEFINDBUF4 *)entry2)->ftimeLastWrite.hours)
    return 1;
    if (((FILEFINDBUF4 *)entry1)->ftimeLastWrite.hours < ((FILEFINDBUF4 *)entry2)->ftimeLastWrite.hours)
    return -1;
    if (((FILEFINDBUF4 *)entry1)->ftimeLastWrite.minutes > ((FILEFINDBUF4 *)entry2)->ftimeLastWrite.minutes)
    return 1;
    if (((FILEFINDBUF4 *)entry1)->ftimeLastWrite.minutes < ((FILEFINDBUF4 *)entry2)->ftimeLastWrite.minutes)
    return -1;
    if (((FILEFINDBUF4 *)entry1)->ftimeLastWrite.twosecs > ((FILEFINDBUF4 *)entry2)->ftimeLastWrite.twosecs)
    return 1;
    if (((FILEFINDBUF4 *)entry1)->ftimeLastWrite.twosecs < ((FILEFINDBUF4 *)entry2)->ftimeLastWrite.twosecs)
    return -1;
    /* Same all through */
    return 0;
}

int scomp(const void *entry1, const void *entry2)
{
    if (((FILEFINDBUF4 *)entry1)->cbFile == ((FILEFINDBUF4 *)entry2)->cbFile) return 0;
    else if (((FILEFINDBUF4 *)entry1)->cbFile > ((FILEFINDBUF4 *)entry2)->cbFile) return 1;
    else return -1;
}

void help()
{
    printf("\nls directory utility version 3.0, May 1994 by Les Bell");
    printf("\nUsage:-");
    printf("\n      ls [d:][\\path\\][filename.typ] [-flags]");
    printf("\nFlags are:-");
    printf("\n      c       -       clear screen");
    printf("\n      d       -       include directories in search");
    printf("\n      h       -       include system and hidden files");
    printf("\n      l       -       produce long format listing");
    printf("\n      n       -       do not sort output");
    printf("\n      p       -       pause output every screenful");
    printf("\n      s       -       sort by size");
    printf("\n      t       -       sort by date and time of creation/update");
    printf("\n      w       -       print wide listing");
    printf("\n");
}

void pause()
{
    char *prompt;
    USHORT i;
    KBDINFO kbdstuff;
    KBDKEYINFO keystuff;
    int IOWait = 0;

    prompt = "Press any key to continue";
    fputs(prompt,stderr);
    kbdstuff.cb = sizeof(kbdstuff);
    KbdGetStatus(&kbdstuff,0);
    kbdstuff.fsMask |= 4;
    KbdSetStatus(&kbdstuff,0);
    KbdCharIn(&keystuff,IOWait,0);
    for (i = 0 ; i < strlen(prompt) ; i++)
    fputs("\b \b",stderr);
}

void wprint(int nentries)
{
    short i,j;
    int nacross, nlines;
    long total_size = 0, total_k = 0;
    char format[10];

    nacross = SCRWIDTH / FNAMELEN;
    strcpy(format,"%-");
    _itoa(FNAMELEN,format+2,10);
    strcat(format,"s");

    if ((nentries % nacross) == 0)
    nlines = nentries / nacross;
    else
    nlines = nentries / nacross + 1;
    for (i = 0 ; i < nlines ; i++) {
    for (j = i ; j < nentries; j += nlines) {
        printf(format,strlwr(strdup(finfo[j]->achName)));
        total_size += finfo[j]->cbFile;
        total_k += round_up(finfo[j]->cbFile,1024);
    }
    fputs("\n",stdout);
    if (flags.pauseflag)
        if((i+1)%(SCRDEPTH-1) == 0) pause();
    }
    printf("Total Size %6lu - %lu Kbytes on floppy",total_size,total_k/1024);
}

void lprint(nentries)
int nentries;
{
    int i;
    long total_size = 0, total_k = 0;

    for (i = 0; i < nentries; i++) {
        if (flags.lflag) {
            print_attributes(finfo[i]->attrFile);
            printf("    %6lu ",finfo[i]->cbFile);
            printf("    %6lu ",finfo[i]->cbList);
            print_date16(finfo[i]->fdateLastWrite);
            print_time16(finfo[i]->ftimeLastWrite);
            total_size += finfo[i]->cbFile;
            total_k += round_up(finfo[i]->cbFile,1024);
        }
        printf("%-13s\n",strlwr(strdup(finfo[i]->achName)));
        if (flags.pauseflag)
            if((i+1)%(SCRDEPTH-1) == 0)
                pause();
    }
    if(flags.lflag)
    printf("Total Size %6lu - %lu Kbytes on floppy",total_size,total_k/1024);
}

void crt_cls()
{
    char *space = " ";

    VioScrollUp(0,0,-1,-1,25,space,0);
    VioSetCurPos(0,0,0);
}

int isdir(pathname)
char * pathname;
{
    FILESTATUS3 buffer;

    if (!DosQueryPathInfo(pathname, FIL_STANDARD, (PVOID)&buffer, sizeof(FILESTATUS3))) {
        if (buffer.attrFile & FILE_DIRECTORY)
            return TRUE;
        else
            return FALSE;
    }
    else
        return FALSE;
}

void eprint(char *filename)
{
    APIRET rc;
    BYTE abEAList[1024];
    ULONG ulEACount = -1;
    PBYTE pv, pe;
    PFEA2 pfea2;

    rc = DosEnumAttribute(ENUMEA_REFTYPE_PATH,
                  filename,
                  1,
                  abEAList,
                  sizeof(abEAList),
                  &ulEACount,
                  ENUMEA_LEVEL_NO_VALUE);

    if(rc==0) {
        pv = abEAList;
        do{
            pfea2 = (PFEA2)pv;
            /* Print the EA name */
            printf("%20s",pfea2->szName);
            pe = pv+sizeof(FEA2)+pfea2->cbName;
            printf("%s\n", (PSZ)pe);
            pv = pv + pfea2->oNextEntryOffset;
        } while (pfea2->oNextEntryOffset);
    }
}


