/* Shows a sector based on Drive and either LSN or C/H/S */

PARSE UPPER ARG drive lsn . 1 '0X' hex . 1 '0Y' y1 y2 y3 y4 .,
      1 . cyl '/' hd '/' sec .   /* Parse parm line 4 times */

/* There must be at least two parms supplied */
IF drive = '' | lsn = '' THEN
   CALL HELP

/* Only parm supplied was for help */
IF WordPos(drive,"? /? /H HELP") \= 0 THEN
   CALL Help

/* Register external functions */
CALL RxFuncAdd 'QDrive','sector','QDrive'
CALL RxFuncAdd 'ReadSect','sector','ReadSect'
CALL DriveInfo            /* Determine drive geometry */
secPerCyl = (totalHd + 1) * secPerTrk

/* Adjust for presence of other secs before LSN 0 */
lsnStartPos = GetHiddenSecs()

SELECT
WHEN hex='' & y1='' & hd='' & sec='' THEN   /*Dec LSN format*/
    CALL WithLSN
WHEN hex \= '' THEN       /* 0x hex format */
    DO
    lsn = X2D(hex)
    CALL WithLSN
    END
WHEN Y1 \= '' THEN        /* 0y hex format */
    DO
    lsn = X2D(Right('00'y4,2)||Right('00'y3,2)||,
              Right('00'y2,2)||Right('00'y1,2))
    CALL WithLSN
    END
WHEN sec \= '' THEN       /* C/H/S format */
    DO
    CALL WithCHS
    CALL CheckForNegativeLSN
    END
OTHERWISE CALL Help
END

'@cls'
SAY "LSN" lsn "(0x"D2X(lsn,8)") on" drive " Cyl:"cyl", Hd: "hd", Sec:"sec " (Max CHS:" totalCyl"/"totalHd"/"secPerTrk")"

SAY
sec = ReadSect(drive,lsn)        /* Read in required sector */

/* Display the sector */
DO paraBoundary = 0 TO 511 BY 16
   /* Display hex char & a space */
   CALL Charout, D2X(paraBoundary,4)" "
   DO offset = 1 TO 16
      IF offset \= 9 THEN
         CALL Charout, " "C2X(Substr(sec,paraBoundary+offset,1))
      ELSE    /* Put a dash between hex bytes 7 & 8 */
         CALL Charout, "-"C2X(Substr(sec,paraBoundary+offset,1))
   END offset

   CALL Charout, "   " /* 3 spaces between hex & ASCII */
   DO offset=1 TO 16   /* Display the printable ASCII chars */
      char = Substr(sec,paraBoundary+offset,1)
      IF char < " " | char > "~" THEN
         CALL Charout, "." /* Swap non-print char for a dot */
      ELSE
         CALL Charout, char
   END offset

   SAY                       /* Start a new line */
END paraBoundary
EXIT  /****************EXECUTION ENDS HERE****************/


DriveInfo:          /* Determine drive geometry */
PARSE VALUE QDrive(drive) WITH totalSec totalCyl totalHd secPerTrk .
totalCyl = totalCyl - 1
totalHd  = totalHd - 1
RETURN


GetHiddenSecs: /* Given Drive, return dec value of Dword */
/* Num of hidden secs is specified in the logical boot sec
   (LSN 0) in the dword stored at offset 29 decimal.
   This region is part of the BPB (BIOS Parameter Block) */
logicalBootSec = ReadSect(drive, 0)
hiddenSecs = Reverse(Substr(logicalBootSec,29,4))
RETURN C2D(hiddenSecs)


WithCHS:                    /* Given CHS, determine LSN */
IF dec < 1 THEN
   CALL ShowErrMsg "Physical sector numbering starts from 1"

IF cyl < 0 THEN
   CALL ShowErrMsg "Cylinder numbering starts from 0"

IF cyl > totalCyl | hd > totalHd | sec > secPerTrk THEN
   DO
   SAY
   SAY "Requested C/H/S is greater than geometry of" drive "("TotalCyl"/"TotalHd"/"secPerTrk")"
   EXIT
   END

IF hd<0 THEN
   CALL ShowErrMsg "Head numbering starts from 0"

/* Work out LSN, given Cyl, Hds, Sec */
lsn = cyl * secPerCyl
lsn = lsn + (hd * secPerTrk)
lsn = lsn + sec
lsn = lsn - lsnStartPos - 1         /* LSN is zero-based */
RETURN


ShowErrMsg:
ARG errMsg
SAY
SAY errMsg
EXIT


CheckForNegativeLSN:
/* Neg LSN will occur if specified CHS is in hidden area */
IF lsn < 0 THEN
   DO
   lsn = 0
   CALL WithLSN
   displayString = cyl"/"hd"/"sec
   SAY
   SAY "Requested C/H/S is below LSN 0 which starts at:" displayString
   EXIT
   END
RETURN


WithLSN:          /* Given LSN, determine CHS */
sec = lsn + lsnStartPos

IF lsn + 1 > totalSec THEN  /* LSN is zero-based */
   DO
   SAY
   SAY "Requested sector is greater than the maximum LSN"
   SAY "of this volume ("totalSec-1")"
   SAY
   EXIT
   END

cyl = sec % secPerCyl           /* Determine cyl value */
sec = sec - (Cyl*secPerCyl)  /* Determine remainder secs */
hd = (sec % secPerTrk) // (totalHd+1)  /* Determine hd */
sec = sec - (hd*secPerTrk)+1    /* Determine sec value */
RETURN


Help:
SAY
SAY "Purpose:"
SAY "  SEC dumps disk sectors to STDOUT"
SAY
SAY "Usage:"
SAY "  SEC drive logical_sector_number in decimal, hex (0x) or dump style (0y)"
SAY "       or"
SAY "  SEC drive Cyl/Hd/Sec"
SAY
SAY "Examples:"
SAY "  SEC C: 1015"
SAY "  SEC C: 0x3F7"
SAY "  SEC C: 0yF7 03 01  or  0yF7 3 1  or  0y F7 3 1 0                                                      = 0x000103F7"
SAY "  SEC C: 2/14/5"
SAY
SAY "Notes:"
SAY "  LSN, Cyl & Hd are zero-based while Sec is one-based."
SAY "  LSN, being a logical numbering scheme, does not include"
SAY "  the hidden secs before the logical boot sec (LSN 0)."
EXIT
