/*****************************************************************************
 * client.exe
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This module contain the source for the RMX test
 *           program client.exe.
 *
 * $Log$
 *
 ***************************************************************************** 
 *
 * Copyright (c) 1995 Johan Wikman
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

/*****************************************************************************
 * Include Files
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSDATETIME
#define INCL_DOSMEMMGR
#define INCL_DOSPROCESS
#include <common.h>
#include <rmxcomms.h>

#include <iostream.h>
#include <process.h>
#include <stdio.h>
#include <string.h>
#include <os2.h>


/*****************************************************************************
 * EXTERNAL VARIABLES
 *****************************************************************************/

PSZ pszAppName = "Client"; // Used by common


/*****************************************************************************
 * MODULE TYPES
 *****************************************************************************/

struct QualifiedName
{
  PCSZ pcszHost;
  PCSZ pcszPort;
};


/*****************************************************************************
 * MODULE VARIABLES
 *****************************************************************************/

const CHAR usage[] = "usage: client [-t #] ([-h host]|-e engine) [-v]\n\n"
                     "    -t #      : Number of threads\n"
                     "    -h host   : The name of the host\n"
		     "    -e engine : Use this as engine connection\n"
		     "    -v verbose: More chit chat\n\n" 
                     "Note the blank between the flag and the value\n\n"
                     "example: client -t 5 -h \\\\ODIN -v\n"
                     "         client -t 5 -h 192.26.110.21 -v\n"
		     "         client -v";  

const ULONG RETRY_COUNT        = 10;
const ULONG THREAD_COUNT       = 5;

const ULONG SLEEP_MILLISECONDS = 200;

const int   EXIT_OK            = 0;
const int   EXIT_FLAGS         = 1;
const int   EXIT_SERVER        = 2;
const int   EXIT_ENGINE        = 3;

static BOOL bVerbose           = FALSE;


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************/

static VOID        CloseConnection(HCONNECTION hConn);
static PSZ         GetEngineName  (PCSZ pszHost);
static VOID        EnterClientLoop(HCONNECTION hConn);
static HCONNECTION OpenConnection (PCSZ pcszHost, PCSZ pcszPort);
static PSZ         QueryEngineName(HCONNECTION hConn);
static VOID        StartClientLoop(PCSZ pcszHost, PCSZ pszEngine);
static VOID        Thread         (VOID*); 


/*****************************************************************************
 * MAIN
 *****************************************************************************/

int main(int argc, char* argv[])
{
  if (argc > 6)
    {
      cerr << usage << endl;
      return EXIT_FLAGS;
    }

  BOOL
    bError = FALSE;
  PSZ
    pszEngine = 0,
    pszHost = 0;
  LONG
    lThreads = THREAD_COUNT;
  
  for (int i = 1; i < argc; i++)
    {
      if (argv[i][0] != '-')
	{
	  pszHost = 0;
	  break;
	}
      
      CHAR
	chFlag = argv[i][1];

      switch (chFlag)
	{
	case 'e':
	  {
	    if ((i == argc - 1) || pszHost)
	      {
		// We've reached the end as there is no argument
		// following the flag, or the host has been specified.
		
		bError = TRUE;
		i = argc;
	      }
	    else
	      pszEngine = argv[++i];
	  }
	  break;
	  
	case 'h':
	  {
	    if ((i == argc - 1) || pszEngine)
	      {
		// We've reached the end as there is no argument
		// following the flag, or the engine has been specified.
		
		bError = TRUE;
		i = argc;
	      }
	    else
	      pszHost = argv[++i];
	  }
	  break;
	  
	case 't':
	  {
	    if (i == argc - 1)
	      {
		// We've reached the end. There is no argument
		// following the flag. 
		
		bError = TRUE;
		i = argc;    // To break the for-loop.
	      }
	    else
	      {
		lThreads = atol(argv[++i]);
		
		if (lThreads <= 0)
		  {
		    bError = TRUE;
		    i = argc;
		  }
	      }
	  }
	  break;
	  
	case 'v':
	  bVerbose = TRUE;
	  break;

	default:
	  bError = TRUE;
	  i = argc;
	}
    }

  if (bError)
    {
      cerr << usage << endl;
      return EXIT_FLAGS;
    }

  if (!pszEngine)
    pszEngine = GetEngineName(pszHost);

  if (!pszEngine)
    return EXIT_SERVER;

  // Now it is time to start the threads.

  QualifiedName
    name;
  
  name.pcszHost = pszHost;
  name.pcszPort = pszEngine;

  for (i = 0; i < lThreads - 1; i++)
    {
      if (_beginthread(Thread, SIZE_STACK, &name) == -1)
	{
	  CHAR
	    acMessage[128]; // More than enough.
	  
	  sprintf(acMessage, 
		  "client: Failed to create thread %d\n", i);
	  Print(acMessage);
	}
    }
  
  // So that thread 1 also has something to do we'll put it into a
  // message loop as well.
  
  StartClientLoop(pszHost, pszEngine);

  // We'll only end up here if something fails.
  
  Print("Main thread exited, terminating application");

  // Yes, pszEngine is never freed.
  
  return EXIT_ENGINE;
}


/*****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 *****************************************************************************
 * 
 * FUNCTION: PSZ GetEngineName(PCSZ psczHost);
 *
 * INPUT:
 *    pszHost: The name of the host computer. What it actually should
 *             contain is up to the communications DLL to define.
 *
 * RETURN:
 *    The full name of the engine.
 *
 * OVERVIEW:
 *    This function opens a connection to the server and instructs it
 *    to starts an engine dedicated for providing services to this
 *    client. If that succeeds the function returns the fully
 *    qualified name of the engine. Failure to do this is reported by
 *    returning NULL. It is the callers responsibility to free the
 *    returned string.
 *
 *****************************************************************************/

PSZ GetEngineName(PCSZ pcszHost)
{
  PSZ
    pszPort = GetPortName();
  
  if (!pszPort)
    return 0;
  
  Print("The name of the server is: ", pszPort);

  HCONNECTION
    hConn = OpenConnection(pcszHost, pszPort);

  delete [] pszPort;
  
  if (!hConn)
    return 0;

  Print("Connection to the server successfully opened");

  PSZ
    pszEngine = QueryEngineName(hConn);

  CloseConnection(hConn);

  if (!pszEngine)
    return 0;
  
  Print("The name of the engine is: ", pszEngine);

  return pszEngine;
}


/*****************************************************************************
 * 
 * FUNCTION: HCONNECTION OpenConnection(PCSZ pcszHost,
 *                                      PCSZ pcszPort);
 *
 * INPUT:
 *    pcszHost: The host name.
 *    pcszPort: The port name.
 *
 * RETURN:
 *    The connection handle.
 *
 * OVERVIEW:
 *    This function returns a connection handle using which the caller
 *    can send and receive data from the remote process. Failure is
 *    reported by returning 0.
 *
 *****************************************************************************/

HCONNECTION OpenConnection(PCSZ pcszHost, PCSZ pcszPort)
{
  ULONG
    rc;
  HCONNECTION
    hConn = 0;
  ULONG
    iAttempt = 1,
    cMaxAttempts = RETRY_COUNT;
  
  do
    {
      rc = RmxOpen(pcszHost, pcszPort, &hConn);
      
      if (rc == RMXERR_CONNECTION_BUSY)
	{
	  Print("Engine busy, sleeping and retrying");
	  
	  DosSleep(iAttempt * SLEEP_MILLISECONDS);
	}
    }
  while ((rc == RMXERR_CONNECTION_BUSY) && (iAttempt++ <= cMaxAttempts));
  
  if (rc)
    {
      Print("RmxOpen:", ErrorAsString(rc));
      return 0;
    }
  else
    return hConn;
}

  
/*****************************************************************************
 * 
 * FUNCTION: VOID CloseConnection(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The connection to be closed.
 *
 * OVERVIEW:
 *    This function closes a connection.
 *
 *****************************************************************************/

VOID CloseConnection(HCONNECTION hConn)
{
  ULONG
    rc = RmxClose(hConn);

  if (rc)
    Print("RmxClose:", ErrorAsString(rc));
}


/*****************************************************************************
 * 
 * FUNCTION: PSZ QueryEngineName(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn:    The handle of the server connection.
 *
 * RETURN:
 *    The full name of the engine.
 *
 * OVERVIEW:
 *    This function returns the fully qualified name of the
 *    engine. Failure is reported by returning 0. It is the callers
 *    responsibility to free the string.
 *
 *****************************************************************************/

static PSZ QueryEngineName(HCONNECTION hConn)
{
  ULONG
    aulRequest[2];

  aulRequest[0] = REQUEST_CONNECT;
  aulRequest[1] = bVerbose;
  
  if (!WriteMessage(hConn, (PBYTE) aulRequest, sizeof(aulRequest)))
    return 0;

  Print("Connect request successfully sent to the server");

  // Now, let's wait for the reply.
  
  ULONG
    ulBufferSize = 0;
  PBYTE
    pbBuffer = 0;
  ULONG
    ulBytesRead = ReadMessage(hConn, &pbBuffer, &ulBufferSize);

  if (ulBytesRead == 0)
    {
      delete [] pbBuffer;
      return 0;
    }
  
  if (ulBytesRead < sizeof(ULONG))
    {
      Print("Too short a reply received");
      delete [] pbBuffer;
      return 0;
    }

  Print("Reply successfully read");

  PSZ
    pszEngine = 0;
  
  // The first four bytes treated as an unsigned long indicate the
  // result. 
  
  ULONG
    ulReply = *(ULONG*)pbBuffer;

  switch (ulReply)
    {
    case REPLY_OK:
      {
	// A correct reply is followed by the name of the engine.
	
	PSZ
	  pszName = pbBuffer + sizeof(ULONG);
	
	pszEngine = new CHAR [strlen(pszName)];

	strcpy(pszEngine, pszName);
      }
      break;

    case REPLY_INVALID_REQUEST:
      Print("Server failed to recognize the request");
      break;

    case REPLY_SPAWNING_FAILED:
      Print("Server failed to spawn engine");
      break;

    case REPLY_ENGINE_FAILED:
      Print("The engine initialization failed");
      break;

    default:
      Print("Server returned unknown error code");
    };
  
  delete [] pbBuffer;
  
  return pszEngine;
}


/*****************************************************************************
 * 
 * FUNCTION: VOID Thread(VOID* pvArgument);
 *
 * INPUT:
 *    pvArgument: The thread argument.
 *
 * OVERVIEW:
 *    The thread function.
 *
 *****************************************************************************/

static VOID Thread(VOID* pvArgument)
{
  QualifiedName
    *name = (QualifiedName*) pvArgument;
  
  StartClientLoop(name->pcszHost, name->pcszPort);
}


/*****************************************************************************
 * 
 * FUNCTION: VOID StartClientLoop(PCSZ pcszHost, PCSZ pcszPort);
 *
 * INPUT:
 *    pcszHost: The hostname of the computer where the engine is running.
 *    pcszPort: The port the engine is lsitening to.
 *
 * OVERVIEW:
 *    This function opens a connection to the engine and enters
 *    the loop of sending and receiving messages.
 *
 *****************************************************************************/

VOID StartClientLoop(PCSZ pcszHost, PCSZ pcszPort)
{
  HCONNECTION
    hConn = OpenConnection(pcszHost, pcszPort);
  
  if (!hConn)
    return;

  Print("Connection to the engine successfully opened");

  EnterClientLoop(hConn);

  CloseConnection(hConn);
}


/*****************************************************************************
 * 
 * FUNCTION: VOID EnterClientLoop(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The connection handle of the engine.
 *
 * OVERVIEW:
 *    The alghoritm for this function:
 *
 *    while no fatal error do
 *    begin
 *        generate message with size between 32 and 65536 bytes
 *
 *        initialize message data
 *
 *        for MESSAGE_COUNT times do
 *        begin
 *           send message
 *
 *           receive message
 *
 *           compare sent and received messages
 *        end
 *    end
 *
 *****************************************************************************/

static VOID EnterClientLoop(HCONNECTION hConn)
{
  BOOL
    bError = 0;
  ULONG
    ulInSize  = 0,
    ulOutSize = 0;
  PBYTE
    pbInMessage  = 0,
    pbOutMessage = 0;

  while (!bError)
    {
      DATETIME
	dateTime;

      DosGetDateTime(&dateTime);
      
      // We'll generate the message length from the current time.
	  
      ULONG
	ulPower = dateTime.hundredths; // 0 - 99
	  
      ulPower /= 11; // 0 - 9
      
      ULONG
	ulMessageSize = 16 * (2 << ulPower);

      if (ulMessageSize > ulOutSize)
	{
	  delete [] pbOutMessage;
	  
	  ulOutSize = ulMessageSize;
	  
	  pbOutMessage = new BYTE [ulOutSize];
	}
      
      // Initialize the message.

      UCHAR
	seed = dateTime.hundredths;
      
      for (int i = 0; i < ulMessageSize; i++)
	pbOutMessage[i] = seed++;

      CHAR
	acReport[256]; // More than enough
      
      if (bVerbose)
	{
	  sprintf(acReport, "Sending message of %lu bytes",
		  ulMessageSize);
	  
	  Print(acReport);
	}

      if (!WriteMessage(hConn, pbOutMessage, ulMessageSize))
	{
	  Print("Failed to write message");
	  
	  bError = TRUE;
	}
      else
	{
	  // The allocation and updating of pbInMessage and
	  // ulInSize is handled by ReadMessage.
	  
	  ULONG
	    ulBytesRead = ReadMessage(hConn, &pbInMessage, &ulInSize);
	  
	  if (ulBytesRead == 0)
	    {
	      // This indicates an error
	      bError = TRUE;
	    }
	  else if (ulBytesRead != ulMessageSize)
	    Print("Mismatch in written and read message sizes");
	  else
	    {
	      for (int k = 0; k < ulMessageSize; k++)
		if (pbOutMessage[k] != pbInMessage[k])
		  {
		    Print("Mismatch in written and read data");
		    k = ulMessageSize;
		  }

	      // To simulate that we are doing something in between.
	      //DosSleep(200);
	    }
	}
    }
}
