/*****************************************************************************
 * RMXPATCH.EXE
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This file contains the main function of rmxpatch.
 *
 * $Log$
 *
 *****************************************************************************
 * 
 * Copyright (c) 1994 Johan Wikman
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 ****************************************************************************/

#define INCL_DOSFILEMGR
#define INCL_DOSERRORS
#define INCL_ORDINALS
#include "rmxpatch.h"
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <os2.h>


/*****************************************************************************
 * MODULE VARIABLES
 *****************************************************************************/

static char usage[] = 
"usage: rmxpatch (-p|-u) [-d] file [file ...]\n\n"
"       -p: Patch the files for RMX.\n"
"       -u: Unpatch the files, i.e., remove references to RMX.\n"
"       -d: Do it. Without this flag rmxpatch only shows what it would do\n"
"           had the flag been given.";

const int EXIT_OK     = 0;
const int EXIT_ARGS   = 1;
const int EXIT_SYSTEM = 2;
const int EXIT_FILE   = 3;


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************/

static int  rmxPatch       (PCSZ name,   BOOL doIt, ULONG patchMode);

static void rmxApplyPatches(HFILE hFile, ULONG patchMode, 
			    ULONG dllCount, const RMXDLL* dlls);
static BOOL rmxApplyPatch  (HFILE hFile, ULONG patchMode, const RMXDLL* dll);

static void rmxApplyQueries(ULONG patchMode, 
			    ULONG dllCount, const RMXDLL* dlls);
static BOOL rmxApplyQuery  (ULONG patchMode, const RMXDLL* dll);

static void rmxPrintIntro  (const RMXDLL* dll);
static BOOL rmxPrintNotes  (ULONG patchMode, ULONG state);


/*****************************************************************************
 * MAIN
 *****************************************************************************/

int main(int argc, char* argv[])
{
  ios::sync_with_stdio();
  
  if (argc < 2)
    {
      cerr << usage << endl;
      return EXIT_ARGS;
    }

  ULONG
    patchMode = 0;
  ULONG
    doIt = FALSE;
  
  for (int i = 1; i < argc; i++)
    {
      if (argv[i][0] == '-')
	{
	  switch (argv[i][1])
	    {
	    case 'd':
	      if (doIt == FALSE)
		doIt = TRUE;
	      else
		i = argc;
	      break;
		  
	    case 'p':
	      if (patchMode == 0)
		patchMode = RMX_PATCH;
	      else
		i = argc;
	      break;

	    case 'u':
	      if (patchMode == 0)
		patchMode = RMX_UNPATCH;
	      else
		i = argc;
	      break;

	    default:
	      cerr << usage << endl;
	      return EXIT_ARGS;
	    }
	}
      else
	break;
    }

  if ((i == argc) || (patchMode == 0))
    {
      cerr << usage << endl;
      return EXIT_ARGS;
    }

  int
    xc = EXIT_OK;

  // Yes, we'll return the code for the last patched application.
  
  for (int j = i; j < argc; j++)
    xc = rmxPatch((PCSZ) argv[j], doIt, patchMode);

  return xc;
}



/*****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 *****************************************************************************/

static int rmxPatch(PCSZ name, BOOL doIt, ULONG patchMode)
{
  ULONG
    openMode;

  if (doIt == TRUE)
    openMode = RMX_OPEN_PATCH;
  else
    openMode = RMX_OPEN_QUERY;
  
  HFILE
    hFile;
  ULONG
    rc;

  rc = RmxPatchOpen((PSZ)name, openMode, &hFile);

  switch (rc)
    {
    case NO_ERROR:
      break;

    case ERROR_ACCESS_DENIED:
    case ERROR_FILE_NOT_FOUND:
    case ERROR_OPEN_FAILED:
    case ERROR_PATH_NOT_FOUND:
      cerr << "rmxpatch: unable to open " << name << "." << endl;
      return EXIT_SYSTEM;

    case ERROR_SHARING_VIOLATION:
      cerr << "rmxpatch: unable to open " << name 
	   << " with required sharing level." << endl;
      return EXIT_SYSTEM;

    default:
      cerr << "rmxpatch: unspecified error (" 
	   << rc 
	   << ") occurred when " << name 
	   << " was opened." << endl;
      return EXIT_SYSTEM;
    }
  
  ULONG
    dllCount;
  RMXDLL
    *dlls;
  
  rc = RmxPatchReadDlls(hFile, patchMode, &dllCount, &dlls);

  if ((rc != NO_ERROR) && (rc != ERROR_BAD_FORMAT))
    {
      DosClose(hFile);

      switch (rc)
	{
	case ERROR_BAD_EXE_FORMAT:
	  cerr << "rmxpatch: " << name << " is not a 32-bit executable." 
	       << endl;
	  return EXIT_FILE;
	  
	case ERROR_NOT_ENOUGH_MEMORY:
	  cerr << "rmxpatch: unable to allocate memory." << endl;
	  return EXIT_SYSTEM;
	  
	default:
	  cerr << "rmxpatch: unspecified fatal error occurred when " << name 
	       << " was queried for dll info." << endl;
	  return EXIT_SYSTEM;
	}
    }

  cout << name << ":\n";
  cout << "    Offset    From      To        Comment\n" 
          "    ---------------------------------------\n";

  if ((rc == NO_ERROR) && (doIt == TRUE))
    rmxApplyPatches(hFile, patchMode, dllCount, dlls);
  else
    rmxApplyQueries(patchMode, dllCount, dlls);

  cout << endl;
  
  free(dlls);
  
  DosClose(hFile);
  
  return EXIT_OK;
}


/*****************************************************************************/

static void rmxApplyPatches(HFILE hFile, ULONG patchMode,
			    ULONG dllCount, const RMXDLL* dlls)
{
  BOOL
    success = TRUE;
  
  for (int i = 0; i < dllCount; i++)
    success = rmxApplyPatch(hFile, patchMode, &dlls[i]) && success;
  
  cout << "\n";
  
  if (!success)
    cout << "    Error: the executable could not be patched.\n";
  else
    cout << "    Success: the executable was successfully patched.\n";
}


/*****************************************************************************/

static void rmxApplyQueries(ULONG patchMode, 
			    ULONG dllCount, const RMXDLL* dlls)
{
  BOOL
    success = TRUE;
  
  for (int i = 0; i < dllCount; i++)
    success = rmxApplyQuery(patchMode, &dlls[i]) && success;
  
  cout << "\n";
  
  if (!success)
    cout << "    Error: the executable cannot be patched.\n";
  else
    cout << "    Success: the executable can be patched.\n";
}


/*****************************************************************************/

static BOOL rmxApplyPatch(HFILE hFile, ULONG patchMode, const RMXDLL* dll)
{
  rmxPrintIntro(dll);

  BOOL
    success = TRUE;

  switch (dll->flState)
    {
    case RMXDLL_REPLACABLE:
      if (!RmxPatch(hFile, dll))
	cout << "successfully replaced\n";
      else
	{
	  cout << "replacing failed, executable may be unusable\n";
	  success = FALSE;
	}
      break;

    default:
      success = rmxPrintNotes(patchMode, dll->flState);
    }
  
  return success;
}


/*****************************************************************************/

static BOOL rmxApplyQuery(ULONG patchMode, const RMXDLL* dll)
{
  rmxPrintIntro(dll);
  return rmxPrintNotes(patchMode, dll->flState);
}


/*****************************************************************************/

static void rmxPrintIntro(const RMXDLL* dll)
{
  PCSZ
    replacement = dll->pcszReplacement;
  
  if (!replacement)
    replacement = "";
  
  printf("    %-8d  %-8s  %-8s  ", 
	 dll->ulOffset, dll->achOriginal, replacement);
}


/*****************************************************************************/

static BOOL rmxPrintNotes(ULONG patchMode, ULONG state)
{
  BOOL
    success = TRUE;
  
  switch (state)
    {
    case RMXDLL_REPLACABLE:
      cout << "replacable\n";
      break;
      
    case RMXDLL_TRANSPARENT:
      cout << "ignored\n";
      break;
      
    case RMXDLL_SUSPICIOUS:
      if (patchMode == RMX_PATCH)
	cout << "warning, not replaced\n";
      else
	cout << "ignored\n";
      break;
      
    case RMXDLL_USEDNONPURELY:
      cout << "16-bit interface of DLL is used\n";
      success = FALSE;
      break;
      
    case RMXDLL_NOTSUPPORTED:
      cout << "DLL not supported\n";
      success = FALSE;
    }
  
  return success;
}
