/***************************************************************************
 *
 * MODULE:	RMX Communications API
 * SOURCE:	$Source$
 * OVERVIEW:	This file contains a TCP/IP based implementation of
 *              the RMX Communications API.
 *
 * Copyright (c) 1995 Johan Wikman
 *
 * $Log$
 *
 ***************************************************************************** 
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

/*****************************************************************************
 * Include Files
 *****************************************************************************/

// This define is needed by the socket headers.
#define OS2

#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSERRORS
#define INCL_DOSMEMMGR
#define INCL_DOSSEMAPHORES
#include <rmxcomms.h>
#include <os2.h>

#ifdef __BCPLUSPLUS__
#define _System APIENTRY
#endif
extern "C"
{
#include <netdb.h>
#include <types.h>
#include <netinet\in.h>
#include <sys\socket.h>
}
#ifdef __BCPLUSPLUS__
#undef _System
#endif


/****************************************************************************
 * CONSTANTS & DEFINES
 ****************************************************************************/

const ULONG SIZE_PAGE        = 4096;
const ULONG SIZE_CONNECTION  = SIZE_PAGE;
const ULONG SIZE_NAME        = 12;             // Enough for 2 ^ 31
const ULONG SIZE_POOL        = 4 * SIZE_PAGE;  // /sizeof(SOCKET) = 910
const ULONG SIZE_MAXTRANSFER = 32767;


/****************************************************************************
 * MODULE TYPES
 ****************************************************************************/

struct SOCKET
{
  unsigned short port;   // The port number in network format.
  int            socket; // The socket handle.
  unsigned       users;  // Number of users of this instance.
  SOCKET*        next;   // Pointer to next instance (or NULL).
  SOCKET*        prev;   // Pointer to previous instance (or NULL).
};

const ULONG SIZE_BUFFER = SIZE_CONNECTION - sizeof(SOCKET*) - sizeof(int);

struct CONNECTION
{
  SOCKET* sharedSocket;
  int     privateSocket;
  BYTE    bytes[SIZE_BUFFER];
};

typedef CONNECTION* PCONNECTION;


/****************************************************************************
 * MODULE VARIABLES
 ****************************************************************************/

static PVOID   pPool; // Memory pool to allocate small object from.
static HMTX    hmtx;  // Mutex semaphore to synchronize creation of sockets.
static SOCKET* head;  // Pointer to first instance.


/****************************************************************************
 * MODULE FUNCTIONS
 ****************************************************************************/

static SOCKET*     AllocSOCKET     ();
static VOID        FreeSOCKET      (SOCKET*);
static SOCKET*     PickSOCKET      (unsigned short port);
static VOID        UnpickSOCKET    (SOCKET*);

static int         CreateSocket    ();

static ULONG       CreateConnection(unsigned short port, HCONNECTION* phConn);
static VOID        ReleaseSOCKET   (SOCKET* pSocket);
static ULONG       InitializeSOCKET(SOCKET* pSocket);
static PCONNECTION AllocConnection ();


/****************************************************************************
 * DLL INITIALIZATION ROUTINE
 ****************************************************************************
 * 
 * FUNCTION: ULONG DLLMAIN(ULONG terminating, HMODULE hmod);
 *
 * INPUT:
 *    terminating: 0, the DLL is being initialized.
 *                 1, the DLL is being terminated.
 *    hmod:        The module handle of this DLL. 
 *
 * RETURN:
 *    0: initialization/termination failed.
 *    1: initialization/termination succeeded.
 *
 * OVERVIEW:
 *    The initialization routine initializes the socket library.
 *
 *****************************************************************************/

DLL_PREAMBLE;

ULONG DLLMAIN(ULONG terminating, HMODULE)
{
  if (!terminating)
    {
      if (sock_init())
	return 0;       // Failed

      // The semaphore is needed for synchronizing the creation and
      // deletion of SOCKET instances.
      
      if (DosCreateMutexSem(0, &hmtx, 0, FALSE))
	return 0;

      // We need a memory pool for storing information about each
      // socket. As we are allocating 4 pages it means that an
      // application can simultanously have roughly 910 distinct sockets.
      // Should be enough for most circumstances. As we allow the
      // DosSub... function to deal with the committing/decommitting
      // of pages the memory consumption (due to this arrangement) for
      // an ordinary server application will seldom exceed 1 page.
      
      ULONG
	rc;

      rc = DosAllocMem((PVOID*) &pPool, SIZE_POOL, PAG_READ | PAG_WRITE);
      
      if (rc != NO_ERROR)
	{
	  DosCloseMutexSem(hmtx);

	  return 0;
	}
      
      rc = DosSubSetMem(pPool, DOSSUB_INIT | DOSSUB_SPARSE_OBJ, SIZE_POOL);
      
      if (rc != NO_ERROR)
	{
	  DosFreeMem(pPool);
	  DosCloseMutexSem(hmtx);

	  return 0;
	}
    }
  else
    {
      DosSubUnsetMem(pPool);
      DosFreeMem(pPool);
      DosCloseMutexSem(hmtx);
    }
  
  return 1;
}


/****************************************************************************
 * EXPORTED FUNCTIONS
 ****************************************************************************
 * 
 * FUNCTION: ULONG RmxClose(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The handle of the connection to be closed.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_INVALID_HANDLE
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    This function closes the socket and releases the memory
 *    associated with the handle. 
 *
 *****************************************************************************/

ULONG RMXENTRY RmxClose(HCONNECTION hConn)
{
  PCONNECTION
    pconn = (PCONNECTION) hConn;

  // If this API is used the way it is supposed to, either
  // sharedSocket, or privateSocket, but not both, is NULL. 
  
  if (pconn->sharedSocket)
    ReleaseSOCKET(pconn->sharedSocket);

  int
    socket = pconn->privateSocket;

  DosFreeMem(pconn);

  ULONG
    rc = NO_ERROR;
  
  if (socket)
    {
      int
	sc = ::soclose(socket);
  
      if (sc < 0)
	{
	  switch (sock_errno())
	    {
	    case SOCENOTSOCK:
	      rc = ERROR_INVALID_HANDLE;
	      break;
	      
	    default:
	      rc = RMXERR_GENERAL;
	    }
	}
    }
  
  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxConnect(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The handle of the connection to be connected.
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    This function blocks on the socket until some client opens it.
 *    Accept returns a new handle that is used for further
 *    communication. 
 *
 *****************************************************************************/

ULONG RMXENTRY RmxConnect(HCONNECTION hConn)
{
  PCONNECTION
    pconn = (PCONNECTION) hConn;

  int
    socket = ::accept(pconn->sharedSocket->socket, 0, 0);

  if (socket < 0)
    return RMXERR_GENERAL;

  pconn->privateSocket = socket;
  
  return NO_ERROR;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxCreate(PCSZ pcszPort, HCONNECTION* phConn);
 *
 * INPUT:
 *    pcszPort: The name of the socket to be created.
 *
 * OUTPUT:
 *    phConn:   Points to a variable where the connection handle will
 *              be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_FILE_NOT_FOUND
 *
 *    CreateConnection
 *
 * OVERVIEW:
 *    This function creates a socket with the given name and returns
 *    the handle of the connection. Everything is handled by CreateConnection.
 * 
 *****************************************************************************/

ULONG RMXENTRY RmxCreate(PCSZ pcszPort, HCONNECTION* phConn)
{
  int
    port = atoi(pcszPort);
  
  if (port <= -1)
    ERROR_FILE_NOT_FOUND;
  
  return CreateConnection(htons((unsigned short) port), phConn);
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxCreateUnique(ULONG*       pulSize, 
 *                                 PSZ          pszPort, 
 *                                 HCONNECTION* phConn);
 *
 * INPUT/OUTPUT:
 *    pulSize: If NULL on input, the used unique name will not be
 *             returned in pszName. Otherwise should point to a
 *             variable that specifies the size of the buffer pointed
 *             to by pszName. If the function fails because the buffer
 *             is too small, the variable pointed to will contain the
 *             needed size.
 *
 * OUTPUT:
 *    pszName: If pulSize is non-NULL the buffer pointed to by pszName
 *             will on return contain the name used when the
 *             connection was created. 
 *    phConn:  Points to a variable where the connection handle will
 *             be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_ENLARGE_BUFFER
 *
 *    CreateConnection
 *
 * OVERVIEW:
 *    This function creates a socket with a unique name and returns
 *    the name and handle of the connection.  
 *
 *****************************************************************************/

ULONG RMXENTRY RmxCreateUnique(ULONG*       pulSize, 
			       PSZ          pszPort, 
			       HCONNECTION* phConn)
{
  if (pulSize && (*pulSize < SIZE_NAME))
    {
      *pulSize = SIZE_NAME;
      return RMXERR_ENLARGE_BUFFER;
    }
  
  ULONG
    rc = CreateConnection(0, phConn);

  if ((rc == NO_ERROR) && pulSize)
    {
      PCONNECTION
	pconn = (PCONNECTION) *phConn;

      // The port name is returned in host not network format.
      
      sprintf(pszPort, "%d", ntohs(pconn->sharedSocket->port));
    }
  
  return rc;
}
  
  
/****************************************************************************
 * 
 * FUNCTION: ULONG RmxDisConnect(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The handle of the connection to be disconnected.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_BROKEN_PIPE
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    The socket used for communication is closed.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxDisConnect(HCONNECTION hConn)
{
  ULONG
    rc = NO_ERROR;
  
  CONNECTION
    *pconn = (CONNECTION*) hConn;
  int
    sc;
  
  sc = ::soclose(pconn->privateSocket);
  
  if (sc < 0)
    {
      switch (sock_errno())
	{
	case SOCENOTCONN:
	  rc = ERROR_BROKEN_PIPE;
	  break;
	  
	default:
	  rc = RMXERR_GENERAL;
	}
    }

  pconn->privateSocket = 0;

  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxGetServiceName(PCSZ pcszService, ULONG* pulSize,
 *				     PSZ pszName);
 *
 * INPUT:
 *    pcszService: The service whose name is queried.
 *
 * INPUT/OUTPUT:
 *    pulSize:     Points to a variable that upon call should contain
 *                 the size of pszName. On return will contain the
 *                 length of pszName. If the function fails because
 *                 the buffer is too small, the variable will contain
 *                 the required size of the buffer.
 *
 * OUTPUT:
 *    pszName:     Points to a buffer where the name of the service
 *                 will be written.  
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_UNKNOWN_SERVICE
 *    RMXERR_ENLARGE_BUFFER
 *
 * OVERVIEW:
 *    The function is directly mapped on getservbyname.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxGetServiceName(PCSZ pcszService, ULONG* pulSize, PSZ pszName)
{
  if (*pulSize < SIZE_NAME)
    {
      *pulSize = SIZE_NAME;
      return RMXERR_ENLARGE_BUFFER;
    }
  
  servent
    *s = getservbyname((PSZ) pcszService, "tcp");
  
  if (!s)
    return RMXERR_UNKNOWN_SERVICE;

  // The port is returned in host format.
  
  sprintf(pszName, "%d", ntohs((unsigned short) s->s_port));
  
  return NO_ERROR;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxOpen(PCSZ         pcszHost
 *                         PCSZ         pcszPort, 
 *                         HCONNECTION* phConn);
 *
 * INPUT:
 *    pcszHost: The name of the host computer. If it is NULL then the
 *              local host will be used. 
 *    pcszPort: The port name to be opened.
 *
 * OUTPUT:
 *    phConn:   Points to a variable where the connection handle will
 *              be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_FILE_NOT_FOUND
 *    ERROR_PATH_NOT_FOUND
 *    ERROR_NOT_ENOUGH_MEMORY
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    This function opens an existing connection and returns its handle.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxOpen(PCSZ         pcszHost,
		       PCSZ         pcszPort, 
		       HCONNECTION* phConn)
{
  // A NULL host is ok, but an empty host is not.
  
  if (pcszHost && (*pcszHost == 0))
    return ERROR_PATH_NOT_FOUND;

  // No negative or 0 port names.
  
  int
    port = atoi(pcszPort);

  if (port <= 0)
    return ERROR_FILE_NOT_FOUND;

  unsigned long
    address = 0;

  if (pcszHost)
    {
      address = inet_addr((PSZ) pcszHost);
  
      if (address == (unsigned long) -1)
	{
	  // Ok, so it wasn't an address in standard dotted-decimal
	  // format. Let's see if it is a textual hostname.
	  
	  hostent
	    *host = gethostbyname((PSZ) pcszHost);
	  
	  if (!host)
	    return ERROR_PATH_NOT_FOUND;

	  address = *(unsigned long*) host->h_addr;
	}
    }

  // All set, let's create the socket.
  
  int
    socket = CreateSocket();
  
  if (!socket)
    return RMXERR_GENERAL;

  sockaddr_in
    sin;
  
  sin.sin_family      = AF_INET;
  sin.sin_port        = htons((unsigned short) port);
  sin.sin_addr.s_addr = address;

  int
    sc;

  sc = ::connect(socket, (sockaddr*) &sin, sizeof(sockaddr_in));

  if (sc < 0)
    {
      switch (sock_errno())
	{
	case SOCEADDRNOTAVAIL:
	  ::soclose(socket);
	  return ERROR_PATH_NOT_FOUND;

	default:
	  ::soclose(socket);
	  return RMXERR_GENERAL;
	}
    }

  // All set. Now we allocate the connection handle.
  
  ULONG
    rc = NO_ERROR;
  PCONNECTION
    pconn = AllocConnection();
  
  if (pconn)
    {
      pconn->privateSocket = socket;
      
      *phConn = pconn;
    }
  else
    {
      ::soclose(socket);
      rc = ERROR_NOT_ENOUGH_MEMORY;
    }
  
  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxRead(HCONNECTION hConn, PBYTE pbBuffer, 
 *                         ULONG ulSize, ULONG* pulBytesRead);
 *
 * INPUT:
 *    hConn:        The handle of the connection.
 *    pbBuffer:     Pointer to buffer where the data will be returned.
 *    ulSize:       The size of the buffer.
 *
 * OUTPUT:
 *    pulBytesRead: Points to a variable where the number of read
 *                  bytes (or the required buffer size) will be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_ENLARGE_BUFFER
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    This function returns the next message from the connection. The
 *    caller need not know in advance how many bytes are arriving. If
 *    the buffer is too small, the caller is instructed to enlarge it.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxRead(HCONNECTION hConn, 
		       PBYTE       pbBuffer,
		       ULONG       ulSize,
		       ULONG*      pulBytesRead)
{
  CONNECTION
    *pconn = (PCONNECTION) hConn;
  int
    socket = pconn->privateSocket;

  // As we are mixing unsigned and signed 32 bit values here, things
  // will start to bomb if somebody attempts to transfer more than 2
  // giga at a time, but in that case he deserves what he gets.
  
  // The length of the message is encoded in the first four bytes.
  
  ULONG
    *pulLength = (ULONG*) pconn->bytes;

  // If the lenght is zero we havn't read anything yet. If it is
  // non-zero we have read the length of the message and concluded
  // that the user's buffer is too small and have instructed him to
  // enlarge it.

  if (*pulLength == 0)
    {
      // Here we read the size of the message.
      
      int
	bytesLeft = sizeof(ULONG),
        bytesRead = 0;
      
      do
	{
	  PBYTE
	    pbData = ((PBYTE) pulLength) + bytesRead;
	  
	  bytesRead = recv(socket, (char*) pbData, bytesLeft, 0);
      
	  if (bytesRead <= 0)
	    {
	      *pulLength = 0;
	      *pulBytesRead = 0;

	      // A read size of 0 indictaes EOF.

	      return (bytesRead == 0) ? NO_ERROR : RMXERR_GENERAL;
	    }
	  
	  bytesLeft -= bytesRead;
	}
      while (bytesLeft != 0);

      *pulLength = (ULONG) ntohl(*pulLength);

      if (*pulLength > ulSize)
	{
	  // If the buffer is too small the caller must enlarge it.
	  
	  *pulBytesRead = *pulLength;
	  return RMXERR_ENLARGE_BUFFER;
	}
    }
  else
    {
      // Ok, the caller has previously been told to enlarge his
      // buffer. Let's see if it now is big enough.
      
      if (*pulLength > ulSize)
	{
	  // Hmm, the caller didn't enlarge it enough.
      
	  *pulBytesRead = *pulLength;
	  return RMXERR_ENLARGE_BUFFER;
	}
    }

  // Now that we know how many bytes we are expecting we can read the
  // actual message.

  int
    bytesLeft = *pulLength,
    bytesRead = 0,
    totalBytesRead = 0;

  do
    {
      PBYTE
	pbData = pbBuffer + totalBytesRead;
      int
	bytesToRead = (SIZE_MAXTRANSFER < bytesLeft) ?
	               SIZE_MAXTRANSFER : bytesLeft; 
      
      bytesRead = recv(socket, (char*) pbData, bytesToRead, 0);
      
      if (bytesRead <= 0)
	{
	  *pulLength = 0;
	  *pulBytesRead = 0;

	  // A read size of 0 indictaes EOF.

	  return (bytesRead == 0) ? NO_ERROR : RMXERR_GENERAL;
	}
      
      bytesLeft -= bytesRead;
      totalBytesRead += bytesRead;
    }
  while (bytesLeft != 0);
  
  *pulBytesRead = *pulLength;

  // We set the count to zero as an indication that all data
  // belonging to the current message has been read.
      
  *pulLength = 0;
  
  return NO_ERROR;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxWrite(HCONNECTION hConn, PCBYTE pcbBuffer, 
 *                          ULONG ulBytesToWrite);
 *
 * INPUT:
 *    hConn:          The handle of the connection.
 *    pcbBuffer:      Pointer to buffer containing data.
 *    ulBytesToWrite: Number of bytes to write.
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    This function writes the specified amount of bytes to the
 *    connection. If the buffer is too large for one packet, it is
 *    transparently split into separate packets.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxWrite(HCONNECTION hConn, 
			PCBYTE      pbBuffer,	
			ULONG       ulBytesToWrite)
{
  if (ulBytesToWrite == 0)
    return NO_ERROR;
  
  PCONNECTION
    pconn = (PCONNECTION) hConn;
  int
    socket = pconn->privateSocket;

  // The size of the message is encoded in the first four bytes of the
  // message.
  
  BYTE
    *pbData = pconn->bytes;
  
  *((ULONG*) pbData) = htonl(ulBytesToWrite);
  
  ULONG
    ulBytesInBuffer = SIZE_BUFFER - sizeof(ULONG);
  ULONG
    ulBytesToCopy   = ulBytesInBuffer < ulBytesToWrite ?
                      ulBytesInBuffer : ulBytesToWrite;

  // The first packet of a message (along with the size info) is
  // always copied to the buffer associated with the connection
  // handle, and only then is the buffer sent. This arrangement
  // guarantees that when the message is less than SIZE_BUFFER,
  // everything will be sent in one single packet. If the size was
  // sent separately, a message - no matter how small - would always
  // require two system calls. Copying bytes around is a lot cheaper
  // than system calls.

  memcpy(pbData + sizeof(ULONG), pbBuffer, ulBytesToCopy);
  
  // The adding of sizeof(ULONG) in the lines below is required for
  // compensating for the first 4 bytes that actually are not part of
  // the message. I don't quite like this mixing of signed and
  // unsigned variables.
  
  ULONG
    ulBytesLeft   = ulBytesToWrite + sizeof(ULONG),
    ulBytesToSend = ulBytesToCopy  + sizeof(ULONG); 
  LONG
    lBytesSent = -sizeof(ULONG);

  ULONG
    rc = NO_ERROR;
  int
    sc;
  
  do
    {
      sc = ::send(socket, (char*) pbData, ulBytesToSend, 0);
      
      if (sc < 0)
	rc = RMXERR_GENERAL;
      else
	{
	  lBytesSent  += sc;
	  ulBytesLeft -= sc;
	  
	  if (ulBytesLeft != 0)
	    {
	      pbData = (PSZ)pbBuffer + lBytesSent;
	   
	      // We attempt to send as much as possible at a time.
	      
	      ulBytesToSend = SIZE_MAXTRANSFER < ulBytesLeft ?
		              SIZE_MAXTRANSFER : ulBytesLeft;
	    }
	}
    }
  while ((ulBytesLeft != 0) && (sc != -1));

  *((ULONG*) pconn->bytes) = 0;

  return rc;
}


/****************************************************************************
 * MODULE FUNCTIONS
 ****************************************************************************
 * 
 * FUNCTION: SOCKET* AllocSOCKET();
 *
 * RETURN:
 *    An instance of SOCKET or NULL if the allocation fails.
 *
 * OVERVIEW:
 *    Initialized to 0.
 *
 *****************************************************************************/

static SOCKET* AllocSOCKET()
{
  SOCKET
    *pSocket;
  ULONG
    rc = DosSubAllocMem(pPool, (PVOID*) &pSocket, sizeof(SOCKET));
      
  if (rc == NO_ERROR)
    memset(pSocket, 0, sizeof(SOCKET));
  else
    pSocket = 0;
  
  return pSocket;
}


/****************************************************************************
 * 
 * FUNCTION: VOID FreeSOCKET(SOCKET* pSocket);
 *
 * INPUT:
 *    pSocket: Pointer to the SOCKET instance to be freed.
 *
 * OVERVIEW:
 *    As the pool is only used for allocating SOCKETs, the size of the
 *    allocated chunk is implicitely known.
 *
 *****************************************************************************/

static VOID FreeSOCKET(SOCKET* pSocket)
{
  DosSubFreeMem(pPool, pSocket, sizeof(SOCKET));
}


/****************************************************************************
 * 
 * FUNCTION: SOCKET* PickSOCKET(unsigned short port);
 *
 * INPUT:
 *    port: The required port number of the socket.
 *
 * RETURN:
 *    A pointer to a suitable SOCKET instance.
 *
 * OVERVIEW:
 *    If the application already has a socket with that port number,
 *    it is returned. Otherwise a new instance is allocated.
 * 
 *    NOTE: This function must be called only when the semaphore
 *          'hmtx' has been successfully requested.
 *
 *****************************************************************************/

static SOCKET* PickSOCKET(unsigned short port)
{
  SOCKET
    *pSocket = 0;
  
  if (head == 0)
    {
      // All new situation, a new instance must be allocated.
      
      head = AllocSOCKET();
      
      if (head != 0)
	{
	  pSocket = head;
	  pSocket->port = port;
	}
    }
  else if (port == 0)
    {
      // A unique socket is required. A new instance must be allocated.

      pSocket = AllocSOCKET();
      
      if (pSocket)
	{
	  pSocket->next = head;
	  head = pSocket;
	}
    }
  else
    {
      // Let's see if a suitable socket can be found.
      
      SOCKET
	*pPrev = 0;
      
      pSocket = head;
      
      while (pSocket)
	{
	  if (pSocket->port == port)
	    break;
	  else
	    {
	      pPrev = pSocket;
	      pSocket = pSocket->next;
	    }
	}
      
      if (!pSocket)
	{
	  // Nope, so we have to create a new one.
	  
	  pSocket = AllocSOCKET();
	  
	  if (pSocket)
	    {
	      pSocket->port = port;
	      
	      if (pPrev)
		pPrev = pSocket;
	      
	      pSocket = pPrev;
	    }
	}
    }

  if (pSocket)
    // Increase usecount, i.e., it is 1 for new a instance.
    pSocket->users++;
  
  return pSocket;
}


/****************************************************************************
 * 
 * FUNCTION: VOID UnpickSOCKET(SOCKET* pSocket);
 *
 * INPUT:
 *    pSocket: The SOCKET instance to be removed from the list and deleted.
 *
 * OVERVIEW:
 *    NOTE: This function must be called only when the semaphore
 *          'hmtx' has been successfully requested.
 *
 *****************************************************************************/

static VOID UnpickSOCKET(SOCKET* pSocket)
{
  if (pSocket->prev)
    pSocket->prev = pSocket->next;

  if (pSocket->next)
    pSocket->next = pSocket->prev;

  if (pSocket == head)
    head = 0;

  FreeSOCKET(pSocket);
}


/****************************************************************************
 * 
 * FUNCTION: int CreateSocket();
 *
 * RETURN:
 *    A socket handle.
 *
 * OVERVIEW:
 *    This function makes sure the creation does not fail because the
 *    create function is interrupted.
 *
 *****************************************************************************/

static int CreateSocket()
{
  int
    serrno = 0,
    socket = 0;
  
  do
    {
      socket = ::socket(AF_INET, SOCK_STREAM, 0);
      
      if (socket < 0)
	serrno = sock_errno();
    }
  while (serrno == SOCEINTR);
  
  if (serrno != 0)
    return 0;
  
  return socket;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG InitializeSOCKET(SOCKET* pSocket);
 *
 * INPUT:
 *    pSocket: The SOCKET instance to be initialized.
 * RETURN:
 *    NO_ERROR
 *    ERROR_PIPE_BUSY;
 *    ERROR_OUT_OF_STRUCTURES;
 *    RMXERR_GENERAL
 *
 * OVERVIEW:
 *    This function creates and prepares a server socket.
 *
 *****************************************************************************/

ULONG InitializeSOCKET(SOCKET* pSocket)
{
  // First we create a socket.
  
  int
    socket = CreateSocket();

  if (!socket)
    return RMXERR_GENERAL;
  
  // Then we bind the socket. That causes a local address to be
  // assigned to the socket. If the port is 0, the system allocates an
  // available port. 
  
  sockaddr_in
    in;
  
  in.sin_family      = AF_INET;
  in.sin_port        = pSocket->port;
  in.sin_addr.s_addr = INADDR_ANY;
  
  int
    sc;
  
  sc = ::bind(socket, (sockaddr*) &in, sizeof(sockaddr_in));
  
  if (sc < 0)
    {
      ::soclose(socket);

      switch (sock_errno())
	{
	case SOCEADDRINUSE:
	  return ERROR_PIPE_BUSY;
	  
	case SOCENOBUFS:
	  return ERROR_OUT_OF_STRUCTURES;
	  
	default:
	  return RMXERR_GENERAL;
	}
    }
  
  // The list calls completes the binding of the socket and creates a
  // connection request queue for incoming requests.

  sc = ::listen(socket, SOMAXCONN);
  
  if (sc < 0)
    {
      ::soclose(socket);
      return RMXERR_GENERAL;
    }

  // If the port was 0, we ask the system what the allocated port
  // number is.

  if (pSocket->port == 0)
    {
      int
	size = sizeof(sockaddr_in);
      
      getsockname(socket, (sockaddr*) &in, &size);
      
      pSocket->port = in.sin_port; // Note, network order
    }
      
  pSocket->socket = socket;
  
  return NO_ERROR;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG CreateConnection(unsigned short port, HCONNECTION* phConn);
 *
 * INPUT:
 *    port:   The port to be used when creating the connection.
 *
 * OUTPUT:
 *    phConn: Points to variable where the handle will be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_NOT_ENOUGH_MEMORY
 *    RMXERR_GENERAL
 *
 *    InitializeSOCKET
 *
 * OVERVIEW:
 *    This function creates and prepares a connection handle. The
 *    entire operation is done protected by a mutex semaphore.
 *
 *****************************************************************************/

static ULONG CreateConnection(unsigned short port, HCONNECTION* phConn)
{
  ULONG
    rc = NO_ERROR;

  DosRequestMutexSem(hmtx, SEM_INDEFINITE_WAIT);
  
  // First we pick a SOCKET instance. If the application already has
  // an instance for the speicifed port the very same instance is
  // returned. The call can only fail due to memory exhaustion.

  SOCKET
    *pSocket = PickSOCKET(port);
  
  if (!pSocket)
    rc = ERROR_NOT_ENOUGH_MEMORY;
  else
    {
      // If a new instance was created there is no socket yet. In that
      // case we create it.

      if (pSocket->socket == 0)
	rc = InitializeSOCKET(pSocket);

      // If it succeeds we create the actual connection handle. 
      // Otherwise we have no option but to abort the whole thing.  
      
      if (rc == NO_ERROR)
	{
	  PCONNECTION
	    pconn = AllocConnection();
	 
	  if (pconn)
	    {
	      pconn->sharedSocket = pSocket;
	      
	      *phConn = pconn;
	    }
	  else
	    rc = ERROR_NOT_ENOUGH_MEMORY;
	}

      if (rc != NO_ERROR)
	ReleaseSOCKET(pSocket);
    }
  
  DosReleaseMutexSem(hmtx);
  
  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: VOID ReleaseSOCKET(SOCKET* pSocket);
 *
 * INPUT:
 *    pSocket: The socket instance to be released.
 *
 * OVERVIEW:
 *    The usecount of the instance is reduced and if it reaches 0 the
 *    instance itself is deleted.
 *
 *****************************************************************************/

static VOID ReleaseSOCKET(SOCKET* pSocket)
{
  DosRequestMutexSem(hmtx, SEM_INDEFINITE_WAIT);

  pSocket->users--;

  if (pSocket->users == 0)
    {
      if (pSocket->socket)
	::soclose(pSocket->socket);

      UnpickSOCKET(pSocket);
    }

  DosReleaseMutexSem(hmtx);
}


/****************************************************************************
 * 
 * FUNCTION: PCONNECTION AllocConnection();
 *
 * RETURN:
 *    Pointer to a connection data structure.
 *
 * OVERVIEW:
 *    No system call interruptions are allowed to interfere.
 *
 *****************************************************************************/

static PCONNECTION AllocConnection()
{
  PVOID
    pvMemory;
      
  // Make sure we get the memory or a proper error code.
  
  ULONG
    rc;
  
  do
    {
      rc = DosAllocMem(&pvMemory, sizeof(CONNECTION), 
		       PAG_READ | PAG_WRITE | PAG_COMMIT);
    }
  while (rc == ERROR_INTERRUPT);

  PCONNECTION
    pconn = 0;
  
  if (rc == NO_ERROR)
    {
      pconn = (PCONNECTION) pvMemory;
      
      memset(pconn, sizeof(CONNECTION), 0);
    }
  
  return pconn;
}
