/***************************************************************************
 *
 * MODULE:	RMX Starter Daemon
 * SOURCE:	$Source$
 * OVERVIEW:	This file contains the implementation of rmxstrtr.exe,
 *              a daemon application that starts applications upon
 *              requests from rmxstart.exe 
 *
 * Copyright (c) 1995 Johan Wikman (johan.wikman@ntc.nokia.com)
 *
 * $Log$
 *
 ****************************************************************************/

#define INCL_NOCOMMON
#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#define INCL_DOSMISC
#define INCL_DOSMODULEMGR
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES

#include "rmxstart.h"
//#include <rmxapi.h>
#include <rmxcomms.h>

#include <io.h>
#include <iostream.h>
#include <new.h>
#include <process.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <os2.h>


/****************************************************************************
 * MODULE TYPES
 ****************************************************************************/

struct Request
{
  PBYTE pbRequest;
  ULONG ulSize;
};


/****************************************************************************
 * MODULE VARIABLES
 ****************************************************************************/

const  CHAR       RMXDISPLAY[]    = "RMXDISPLAY=";

const  ULONG      SIZE_RMXDISPLAY = sizeof(RMXDISPLAY) - 1;

const  ULONG      SIZE_BUFFER     = 4096;
const  ULONG      SIZE_STACK      = 64*1024;

const  ULONG      TWO_SECONDS     = 2000;

static HEV        hev             = 0;
static HMTX       hmtx            = 0;
static PID        pid             = 0;


/****************************************************************************
 * MODULE FUNCTIONS
 ****************************************************************************/

static VOID APIENTRY CleanUp             (ULONG);
static PSZ           CloneEnvironment    (PCSZ);
static HEV           CreateEventSemaphore();
static HMTX          CreateMutexSemaphore();
static PID           CurrentPid          ();
static TID           CurrentTid          ();
static PSZ           GetStarterName      ();
static VOID          HandlerThread       (VOID*);
static ULONG         LaunchApp           (PCSZ pcszDisplay, 
					  PCSZ psczApplication,
					  PCSZ pcszArguments);
static BOOL          ParseRequest        (const Request* pRequest, 
					  PSZ*           ppszDisplay, 
					  PSZ*           ppszApplication,
					  PSZ*           ppszArguments);
static VOID          Print               (PCSZ pcszFormat, ...);
static BOOL          ReadRequest         (HCONNECTION, Request* pRequest);
static VOID          SendResponse        (HCONNECTION hConn, ULONG response);

static VOID          StarterLoop         (PCSZ pcszName);
static ULONG         StartApp            (PCSZ pcszDisplay, 
				          PCSZ pcszApplication,
				          PCSZ pcszArguments);


/****************************************************************************
 * MAIN
 ****************************************************************************/

int main(int, char* [])
{
  set_new_handler(OutOfMem);
  ios::sync_with_stdio();

  // We begin by registering a clean-up function that will be called by
  // the OS when the application is terminated.
  
  ULONG
    ulFunctionOrder = MAKEUSHORT(EXLST_ADD, 0xFF);
	    
  DosExitList(ulFunctionOrder, CleanUp);

  // If this call fails, it is an indication that an instance of this
  // program is running already.

  hev = CreateEventSemaphore();

  if (!hev)
    {
      cerr << "rmxstrtr: RMX starter daemon (using " 
	   << getenv(RMXCOMMS)
	   << ") is running already." << endl;
      
      return EXIT_INIT;
    }

  // If we fail to create the mutex semaphore it is an indication that
  // the system is messed up. We can just aswell exit.
  
  hmtx = CreateMutexSemaphore();
  
  if (!hmtx)
    {
      cerr << "rmxstrtr: Failed to create serialization semaphore." 
	   << endl;
      
      return EXIT_INIT;
    }

  // If the starter name is not found, it means that the initialization
  // file for whichever communications DLL we are using is not up-to-date.

  PSZ
    pszName = GetStarterName();
  
  if (!pszName)
    {
      cerr << "rmxstrtr: Failed to get the starter port. Is the "
	   << "initialization file of the communications DLL you "
	   << "are using up to date?"
	   << endl;
      
      return EXIT_INIT;
    }

  // The pid is always so we put it in a global variable.

  pid = CurrentPid();

  // All set, let's start the server-client loop.

  cout << "RMX Starter Daemon (pid = " << pid << "): using "
       << getenv(RMXCOMMS) 
       << ", listening on " << pszName << "." << endl;

  // A little bit of cheating ...
  
  void (*firstThread)(void*) = (void (*)(void*)) StarterLoop;
  
  int
    tid = _beginthread(firstThread, SIZE_STACK, pszName);
  
  if (tid == -1)
    {
      cerr << "rmxstrtr: Failed to start first handler thread. Exiting." 
	   << endl;
      
      return EXIT_INIT;
    }
  else
    {
      Print("First handler thread started. Waiting on stop semaphore.\n");
      
      DosWaitEventSem(hev, SEM_INDEFINITE_WAIT);
      
      Print("Semaphore posted. Exiting.");

      return EXIT_OK;
    }
}


/****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 ****************************************************************************
 *
 * FUNCTION: VOID APIENTRY CleanUp(ULONG);
 *
 * INPUT:
 *    Not used.
 *
 * OVERVIEW:
 *    This function is called by the OS when the application is terminated.
 *    We make sure the application exits in a clean way.
 *
 ****************************************************************************/

static VOID APIENTRY CleanUp(ULONG)
{
  if (hev)
    DosCloseEventSem(hev);
  
  if (hmtx)
    DosCloseMutexSem(hev);
  
  DosExitList(EXLST_EXIT, CleanUp);
}


/****************************************************************************
 *
 * FUNCTION: PSZ CloneEnvironment(PCSZ pcszDisplay);
 *
 * INPUT:
 *    pcszDisplay: The desired value of RMXDISPLAY.
 *
 * OUTPUT:
 *    return: Points to a copy of the environment.
 *
 * OVERVIEW:
 *    This function clones the current environment. If RMXDISPLAY has
 *    a value, it is removed. A proper RMXDISPLAY value is provided
 *    instead. 
 *
 ****************************************************************************/

static PSZ CloneEnvironment(PCSZ pcszDisplay)
{
  static ULONG
    ulEnvSize = 0;
  static PSZ
    pszEnvironment = 0;
  
  if (!pszEnvironment)
    {
      // The first time the function is called we make a copy of the
      // environment.
      
      DosEnterCritSec();
      
      // We have to make this check twice. Under pathological
      // conditions two threads might pass the first if-test
      // simultaneously. 

      if (!pszEnvironment)
	{
	  PPIB
	    ppib;
	  PTIB
	    ptib;
	  
	  DosGetInfoBlocks(&ptib, &ppib);

	  // Then we determine the total size.
	  
	  PSZ
	    psz = ppib->pib_pchenv;
	  
	  // The environment ends with a double NULL.
	  
	  while (*psz)
	    {
	      while (*psz)
		psz++;
	      
	      psz++;
	    }
	  
	  psz++;

	  pszEnvironment = new CHAR [psz - ppib->pib_pchenv];
	  
	  // Then we can copy all variables.
	  
	  psz = ppib->pib_pchenv;
	  
	  while (*psz)
	    {
	      int
		length = strlen(psz);
	      
	      // But we skip RMXDISPLAY
	      
	      if (strncmp(psz, RMXDISPLAY, SIZE_RMXDISPLAY) != 0)
		{
		  strcpy(pszEnvironment + ulEnvSize, psz);
		  ulEnvSize += length + 1;
		}
	      
	      psz += length + 1;
	    }
	}
      
      DosExitCritSec();
    }

  // First we calculate the total size of the environment. 
  
  ULONG
    ulLength = strlen(pcszDisplay);
  ULONG
    ulSize = ulEnvSize + sizeof(RMXDISPLAY) + ulLength + 1;
  PSZ
    pszEnv = new CHAR [ulSize],
    psz    = pszEnv;
  
  memcpy(psz, pszEnvironment, ulEnvSize);
  psz += ulEnvSize;
  strcpy(psz, RMXDISPLAY);
  psz += sizeof(RMXDISPLAY) - 1;
  strcpy(psz, pcszDisplay);
  psz += ulLength + 1;
  *psz = 0;                     // Double NULL

  return pszEnv;
}


/****************************************************************************
 *
 * FUNCTION: HEV CreateEventSemaphore();
 *
 * RETURN:
 *    A semaphore handle or NULL if creation fails.
 *
 * OVERVIEW:
 *    This function creates a public event semaphore. The name is
 *    built partly built from the name of the current RMXCOMMS DLL
 *    being used. This way the semaphore name can also be built by
 *    another process. This works in this context because it does not
 *    make sense to have several starters using the same DLL.
 *
 ****************************************************************************/

static HEV CreateEventSemaphore()
{
  // As RMXCOMMS.DLL is used we won't get here unless the RMXCOMMS
  // environment variables has properly been set. Hence the following
  // call will never return NULL.

  PCSZ
    pcszRmxComms = getenv(RMXCOMMS);
  
  // The semaphore name is built from a common prefix and the name of
  // the communications DLL.

  CHAR
    achSemName[CCHMAXPATH];
  
  strcpy(achSemName, RMXSEMPREFIX);
  strcat(achSemName, pcszRmxComms);

  HEV
    hev = 0;
  
  if (DosCreateEventSem(achSemName, &hev, DC_SEM_SHARED, FALSE) != NO_ERROR)
    hev = 0;

  return hev;
}


/****************************************************************************
 *
 * FUNCTION: HMTX CreateMutexSemaphore();
 *
 * RETURN:
 *    A semaphore handle or NULL if creation fails.
 *
 * OVERVIEW:
 *    This function creates an anonymous mutex semaphore. 
 *
 ****************************************************************************/

static HMTX CreateMutexSemaphore()
{
  HMTX
    hmtx = 0;

  if (DosCreateMutexSem(0, &hmtx, 0, FALSE) != NO_ERROR)
    hmtx = 0;

  return hmtx;
}


/****************************************************************************
 *
 * FUNCTION: PID CurrentPid();
 *           TID CurrentTid();
 *
 * RETURN:
 *    The specific identifier.
 *
 * OVERVIEW:
 *    These functions return the process and thread identifier respectively.
 *
 ****************************************************************************/

static PID CurrentPid()
{
  PPIB
    ppib;
  PTIB
    ptib;
  
  DosGetInfoBlocks(&ptib, &ppib);
  
  return ppib->pib_ulpid;
}

static TID CurrentTid()
{
  PPIB
    ppib;
  PTIB
    ptib;

  DosGetInfoBlocks(&ptib, &ppib);
  
  return ptib->tib_ptib2->tib2_ultid;
}


/****************************************************************************
 *
 * FUNCTION: VOID HandlerThread(VOID* pvArg);
 *
 * INPUT:
 *    pvArg: the thread argument. In this context it will be a connection
 *           handle. 
 *
 * OVERVIEW:
 *    This is the thread function that reads the request, parses it and if
 *    it is ok starts the application requested.
 *
 ****************************************************************************/

static VOID HandlerThread(VOID* pvArg)
{
  HCONNECTION
    hConn = (HCONNECTION) pvArg;
  
  Request
    request;

  if (ReadRequest(hConn, &request))
    {
      PSZ
	pszDisplay,
        pszApplication,
        pszArguments;
      ULONG
	rc;
      
      if (!ParseRequest(&request, &pszDisplay, &pszApplication, &pszArguments))
	{
	  Print("Received garbled request.\n");
	  rc = RMXAPP_INVALID_REQUEST;
	}
      else
	rc = StartApp(pszDisplay, pszApplication, pszArguments);

      delete [] request.pbRequest;

      SendResponse(hConn, rc);
    }
  else
    Print("Failed to read request.\n");

  RmxDisConnect(hConn);
  RmxClose(hConn);
}



/****************************************************************************
 *
 * FUNCTION: ULONG LaunchApp(PCSZ pcszDisplay,
 *			     PCSZ pcszApplication, 
 *			     PCSZ pcszArguments);
 *
 * INPUT:
 *    pcszDisplay:     The display that should be used.
 *    pcszApplication: The application to be started.
 *    pcszArguments:   The arguments of the application (can be NULL).
 *
 * RETURN:
 *    The result code of the operation.
 *    FALSE: Failure.
 *
 * OVERVIEW:
 *    This function launch the actual application.
 *
 ****************************************************************************/

static ULONG LaunchApp(PCSZ pcszDisplay, 
		       PCSZ pcszApplication,
		       PCSZ pcszArguments)
{
  PSZ
    pszEnv = CloneEnvironment(pcszDisplay),
    pszArg = 0;
  
  if (pcszArguments)
    {
      ULONG
	ulApplication = strlen(pcszApplication) + 1,
        ulArguments   = strlen(pcszArguments) + 1,
        ulSize        = ulApplication + ulArguments + 1;
      
      pszArg = new CHAR [ulSize];
      
      strcpy(pszArg, pcszApplication);
      strcpy(&pszArg[ulApplication], pcszArguments);
      
      pszArg[ulSize - 1] = 0;
    }
  
  // Ok, then we can launch the application.
  
  RESULTCODES
    resultCodes;
  
  ULONG
    rc = DosExecPgm(0, 0, EXEC_BACKGROUND, 
 		    pszArg, pszEnv, &resultCodes, pcszApplication);
  
  delete [] pszEnv;
  
  switch (rc)
    {
    case NO_ERROR:
      rc = RMXAPP_STARTED;
      break;

    case ERROR_FILE_NOT_FOUND:
    case ERROR_PATH_NOT_FOUND:
      rc = RMXAPP_NOT_FOUND;
      break;
      
    case ERROR_NOT_ENOUGH_MEMORY:
      rc = RMXAPP_RESOURCE_EXHAUSTED;
      break;

    default:
      rc = RMXAPP_EXEC_FAILED;
    }

  if (rc == RMXAPP_STARTED)
    Print("Started %s on %s.\n", pcszApplication, pcszDisplay);
  else
    Print("Failed to start %s on %s.\n", pcszApplication, pcszDisplay);

  return rc;
}


/****************************************************************************
 *
 * FUNCTION: BOOL ParseRequest(const Request* pRequest,
 *                             PSZ* ppszDisplay,
 *			       PSZ* ppszApplication, 
 *			       PSZ* ppszArguments);
 *
 * INPUT:
 *    pRequest: The request as returned from ReadRequest.
 *
 * OUTPUT:
 *    *ppszDisplay:
 *    *ppszApplication:
 *    *ppszArguments:
 *        Upon successful return these pointer will point to the
 *        correct position in the request.
 *
 * RETURN:
 *    TRUE:  Success
 *    FALSE: Failure.
 *
 * OVERVIEW:
 *    This function reads a request from the connection. It is the
 *    callers responsibility to free the buffer returned in pRequest.
 *
 ****************************************************************************/

static BOOL ParseRequest(const Request* pRequest,
			 PSZ* ppszDisplay, 
			 PSZ* ppszApplication, 
			 PSZ* ppszArguments)
{
  // The request should look like
  //
  //     'display0application00'           or
  //     'display0application0arguments00'
  //
  // i.e., two or three ASCIIZ strings followed by an additional NULL.

  PBYTE
    pbRequest = pRequest->pbRequest,
    pbEnd     = pbRequest + pRequest->ulSize,
    pb        = pbRequest;
  
  *ppszDisplay = pb;

  // Let's look for the NULL of the first string.
  
  while ((pb < pbEnd) && (*pb != 0))
    pb++;

  if ((pb == pbEnd) || (*pb != 0))
    return FALSE;

  pb++; // To get past the NULL.

  *ppszApplication = pb;

  // Then we must find the NULL of the second string.
  
  while ((pb < pbEnd) && (*pb != 0))
    pb++;

  if ((pb == pbEnd) || (*pb != 0))
    return FALSE;

  pb++; // To get past the NULL.
  
  // If *pb is non-NULL then there are arguments.

  if (*pb != 0)
    {
      *ppszArguments = pb;
      
      while ((pb < pbEnd) && (*pb != 0))
	pb++;

      if ((pb == pbEnd) || (*pb != 0))
	return FALSE;

      pb++; // To get past the NULL.
    }
  else
    *ppszArguments = 0;

  if ((pb != pbEnd - 1) || (*pb != 0))
    return FALSE;
  
  return TRUE;
}


/****************************************************************************
 *
 * FUNCTION: VOID Print(pszFormat, ...);
 *
 * INPUT:
 *    pszFormat: The printf format string.
 *
 * OVERVIEW:
 *    This functions provide threadsafe printing.
 *
 ****************************************************************************/

static VOID Print(PCSZ pszFormat, ...)
{
  if (DosRequestMutexSem(hmtx, SEM_INDEFINITE_WAIT))
    // We ignore this one, there is not much we can do about it.
    return;

  printf("[%d, %d]: ", pid, CurrentTid());
  
  va_list 
    arguments;
  
  va_start(arguments, pszFormat);
  
  vprintf(pszFormat, arguments);
  
  va_end(arguments);

  DosReleaseMutexSem(hmtx);
}


/****************************************************************************
 *
 * FUNCTION: BOOL ReadRequest(HCONNECTION hConn, Request* pRequest)
 *
 * INPUT:
 *    hConn: The connection handle.
 *
 * OUPUT:
 *    pRequest: Upon successful return the fields are updated with
 *              proper values.  
 *
 * RETURN:
 *    TRUE:  Success
 *    FALSE: Failure.
 *
 * OVERVIEW:
 *    This function reads a request from the connection. It is the
 *    callers responsibility to free the buffer returned in pRequest.
 *
 ****************************************************************************/

static BOOL ReadRequest(HCONNECTION hConn, Request* pRequest)
{
  ULONG
    rc;
  
  // We loop until the buffer we provide is big enough.
  
  ULONG
    ulSize      = SIZE_BUFFER,
    ulBytesRead = 0;
  BYTE
    *pbRequest = 0;

  do
    {
      pbRequest = new BYTE [ulSize];

      rc = RmxRead(hConn, pbRequest, ulSize, &ulBytesRead);
      
      if (rc == RMXERR_ENLARGE_BUFFER)
	{
	  delete [] pbRequest;
	  ulSize = ulBytesRead;
	}
    }
  while (rc == RMXERR_ENLARGE_BUFFER);

  if (rc == NO_ERROR)
    {
      pRequest->pbRequest = pbRequest;
      pRequest->ulSize    = ulBytesRead;
      
      return TRUE;
    }
  else
    {
      delete [] pbRequest;
      return FALSE;
    }
}


/****************************************************************************
 *
 * FUNCTION: VOID SendResponse(HCONNECTION hConn, ULONG ulResponse)
 *
 * INPUT:
 *    hConn:      The connection to use.
 *    ulResponse: The response to send.
 *
 * OVERVIEW:
 *    This functions sends a response to the starter.
 *
 ****************************************************************************/

static VOID SendResponse(HCONNECTION hConn, ULONG ulResponse)
{
  ULONG
    rc;
  
  rc = RmxWrite(hConn, (BYTE*) &ulResponse, sizeof(ULONG));

  if (rc != NO_ERROR)
    Print("Failed to send response\n");
  else
    // Just to make sure the response reaches the peer
    DosSleep(2000);
}


/****************************************************************************
 *
 * FUNCTION: ULONG StartApp(PCSZ pcszDisplay, 
 *                          PCSZ pcszApplication,
 *                          PCSZ pcszArguments);
 *
 * INPUT:
 *    pcszDisplay:     The display computer to use.
 *    pcszApplication: The application to start.
 *    pcszArguments:   The arguments of the application (can be NULL).
 *
 * RETURN:
 *    NO_ERROR
 *    RMXAPP_NOT_RMXABLE
 *    RMXAPP_NOT_FOUND
 *
 * OVERVIEW:
 *    If the application exists, and it has been patched for RMX we start it.
 *
 ****************************************************************************/

static ULONG StartApp(PCSZ pcszDisplay, 
		      PCSZ pcszApplication,
		      PCSZ pcszArguments)
{
  if (pcszArguments)
    Print("Request to start '%s %s' on %s received.\n", 
	  pcszApplication, pcszArguments, pcszDisplay);
  else
    Print("Request to start '%s' on %s received.\n", 
	  pcszApplication, pcszDisplay);
  
  ULONG
    rc = NO_ERROR;
  CHAR
    achPath[CCHMAXPATH];
  
  if (access(pcszApplication, 0))
    {
      // Ok, so the application is not directly accessible. Let's see if
      // we can find it on the path. 

      rc = DosSearchPath(SEARCH_IGNORENETERRS |
			 SEARCH_ENVIRONMENT   |
			 SEARCH_CUR_DIRECTORY, 
			 "PATH", pcszApplication, achPath, CCHMAXPATH);
      
      if (rc == NO_ERROR)
	pcszApplication = achPath;
    }

  if (rc == NO_ERROR)
    {
      // The file exists. The next step is to check whether it has
      // been patched for RMX. The function returns FALSE also if the
      // file isn't an executable in the first place.
      
      BOOL
	bPatched = TRUE;
	
      //
      // The following function establishes whether the application to
      // be started has been patched or not. I've commented it out now,
      // so that this code will compile/link without the library where that
      // function is defined.
      // 
      //RmxPatchIsPatched(pcszApplication, &bPatched);
      
      if (bPatched)
	rc = LaunchApp(pcszDisplay, pcszApplication, pcszArguments);
      else
	{
	  Print("%s is not marked for RMX.\n", pcszApplication);
	  rc = RMXAPP_NOT_RMXABLE;
	}
    }
  else
    {
      Print("%s is not found.\n", pcszApplication);
      rc = RMXAPP_NOT_FOUND;
    }

  return rc;
}


/****************************************************************************
 *
 * FUNCTION: VOID StarterLoop(PCSZ pcszName);
 *
 * INPUT:
 *    The name to use when creating the starter connection.
 *
 * OVERVIEW:
 *    This is the main loop of the starter. We sit in a loop waiting for
 *    clients to connect, verify the request and start the requestes
 *    application. 
 *
 ****************************************************************************/

static VOID StarterLoop(PCSZ pcszName)
{
  // Sofar we've been in single-threaded context. Now we enter
  // multi-threaded context.

  while (TRUE)
    {
      HCONNECTION
	hConn;
      ULONG
	rc;

      rc = RmxCreate(pcszName, &hConn);
      
      if (rc != NO_ERROR)
	{
	  Print("FATAL: Failed to create connection.\n");
	  
	  // Posting the semaphore will cause the application to be
	  // terminated. 
	  
	  DosPostEventSem(hev);
	  return;
	}

      // The following call blocks and returns only when somebody has
      // opened the connection.

      rc = RmxConnect(hConn);

      if (rc != NO_ERROR)
	{
	  // This should never happen, but in case it does we just
	  // ignore it.
	  
	  Print("Failed to connect, ignoring client.\n");
	  RmxClose(hConn);
	  continue;
	}
      
      // In order to minimize the time the connection is busy, we deal
      // with the the request in another thread.

      Print("Connection opened, spawning handler thread.\n");
      
      int
	tid;
      
      tid = _beginthread(HandlerThread, SIZE_STACK, (VOID*) hConn);

      if (tid == -1)
	{
	  Print("Failed to create handler thread, ignoring request.\n");
	  RmxClose(hConn);
	}
    }
}





