(**************************************************************************)
(*                                                                        *)
(*  DKIM digest                                                           *)
(*  Copyright (C) 2023   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE MyDKIM;

        (********************************************************)
        (*                                                      *)
        (*                    DKIM Signatures                   *)
        (*                                                      *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            25 July 2020                    *)
        (*  Last edited:        9 July 2023                     *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*  The standard governing this code is RFC 6376        *)
        (*                                                      *)
        (********************************************************)


IMPORT Strings, INIData;

FROM SYSTEM IMPORT CARD8, ADR;

FROM DKIM IMPORT
    (* const*)  DKIM_RC_OK, DKIM_RC_ERROR, DKIM_RC_FILEREADERROR,
                DKIM_RC_OUTPUTERROR, DKIM_RC_OUTPUTEXISTS,
                DKIM_RC_INVALIDFORMAT, DKIM_RC_KEYNOTPROVIDED, DKIM_RC_NOSIGNATURE,
                DKIM_RC_INVALIDBODYHASH, DKIM_RC_VERIFICATIONFAILED,
                DKIM_RC_UNSUPPORTEDVERSION,
                DKIM_FL_DOTSTUFF, DKIM_FL_BODYRELAXED, DKIM_FL_HEADERRELAXED,
                DKIM_FL_OVERWRITE, DKIM_ENTIREBODY,
    (* proc *)  _dkimSignatureValidate, _dkimSignatureSet;

FROM DKIMKeys IMPORT
    (* type *)  CharPtr, DataPtr, PDKIMKEY, GetKeyProc,
    (* proc *)  _dkimCreatePubKeyDNSTXT, _dkimLoadPubKey, _dkimLoadPrivKey;

FROM GetDNStxt IMPORT
    (* proc *)  GetFirstTXTrecord;

FROM WINI IMPORT
    (* proc *)  OpenINI, CloseINI;

FROM INIData IMPORT
    (* proc *)  INIValid, INIGet, INIGetString;

FROM Arith64 IMPORT
    (* type *)  CARD64API;

FROM Names IMPORT
    (* type *)  DomainName;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransactionL;

(*
FROM STextIO IMPORT
    (* proc *)  WriteString, WriteLn;
*)

FROM LowLevel IMPORT
    (* proc *)  AddOffset, EVAL;

FROM FileOps IMPORT
    (* type *)  FilenameString;

(************************************************************************)

CONST
    Nul = CHR(0);

VAR
    (* The domain name of the signer. *)

    ourdomain: DomainName;

    (* Public key file and password - used only for testing. *)

    PubKeyFile: FilenameString;
    PubKeyPassword: ARRAY [0..63] OF CHAR;

    (* Pointer to the private key in memory.  *)

    pPrivKey: PDKIMKEY;

    (* Header fields to include in signature.  *)

    HeaderFields: ARRAY [0..511] OF CHAR;

    (* Selector to use in the DNS lookup. *)

    selector: ARRAY [0..63] OF CHAR;

    (* Relaxed option for canonicalisation.  If FALSE we use "simple". *)

    HeaderRelaxed, BodyRelaxed: BOOLEAN;

(************************************************************************)
(*                      SETTING BASIC PARAMETERS                        *)
(************************************************************************)

PROCEDURE SetParameters;

    (* The public key file is needed only for testing.  (Normally the   *)
    (* public key is fetched via dns/txt.)  All other parameters are    *)
    (* loaded from the INI file, see later.                             *)

    BEGIN
        PubKeyFile := "";
        PubKeyPassword := "";
    END SetParameters;

(************************************************************************)
(*                    PRODUCING THE SIGNATURE STRING                    *)
(************************************************************************)

PROCEDURE InsertDKIMSignature (logID: TransactionLogID;  fname: ARRAY OF CHAR);

    (* Inserts a DKIM signature as a header in the message file     *)
    (* fname.  No result is returned to the caller.  Instead, error *)
    (* messages are sent to the log file.                           *)

    VAR rc, flags: CARDINAL;

    CONST length = DKIM_ENTIREBODY;

    BEGIN
        flags := DKIM_FL_OVERWRITE + DKIM_FL_DOTSTUFF;
        IF BodyRelaxed THEN INC (flags, DKIM_FL_BODYRELAXED) END (*IF*);
        IF HeaderRelaxed THEN INC (flags, DKIM_FL_HEADERRELAXED) END (*IF*);

        rc := _dkimSignatureSet (ADR(fname), NIL, pPrivKey, flags,
                                    ADR(HeaderFields), length.low, length.high,
                                      ADR(ourdomain), NIL, ADR(selector));

    (* Insert a DKIM signature into the file.  The parameters are:                  *)
    (*                                                                              *)
    (*  File:       Input file.                                                     *)
    (*  OutFile:    Output file. If pointer is NIL, changes will be made to         *)
    (*                 the input file.                                              *)
    (*  pPrivKey:   Pointer to the private key.                                     *)
    (*  Flags:      DKIM_FL_xxxxx.                                                  *)
    (*  Fields:     List of fields included in the signature. Field names can       *)
    (*                be separated by any of the following characters:              *)
    (*                <SP>/:/,/;/<HTAB>. If a field name starts with a '+', that    *)
    (*                field (without the '+' character) will be included even if    *)
    (*                it is not present in the message. This prevents attackers     *)
    (*                from adding the missing field.                                *)
    (*             If NIL is specified, the default field set will be used.         *)
    (*  BodyLen: The maximum number of bytes that will be used to calculate         *)
    (*                the hash of the message body.  The value DKIM_ENTIREBODY      *)
    (*                means that the entire body of the message should be used.     *)
    (*  SDID:      (tag d) Signing Domain Identifier.                               *)
    (*  AUID:      (tag i) Agent or User Identifier (pointer may be NIL).           *)
    (*  Selector:  (tag s) Selector for Public Key.                                 *)
    (*                                                                              *)
    (*  Returns: DKIM_RC_OK, DKIM_RC_ERROR, DKIM_RC_FILEREADERROR,                  *)
    (*           DKIM_RC_OUTPUTERROR, DKIM_RC_OUTPUTEXISTS, DKIM_RC_INVALIDFORMAT,  *)
    (*           DKIM_RC_KEYNOTPROVIDED.                                            *)
    (*                                                                              *)
    (*  Not that all character strings are passed by address.  Value parameters     *)
    (*  would be more convenient for the caller, but I think that would create      *)
    (*  problems with cross-language conventions.                                   *)

        CASE rc OF
            DKIM_RC_OK:
                LogTransactionL (logID, "DKIM signature added");
          | DKIM_RC_ERROR:
                LogTransactionL (logID, "DKIM signature creation error");
          | DKIM_RC_FILEREADERROR:
                LogTransactionL (logID, "DKIM: can't read input file");
          | DKIM_RC_OUTPUTERROR:
                LogTransactionL (logID, "DKIM: can't write to output file");
          | DKIM_RC_OUTPUTEXISTS:
                LogTransactionL (logID, "DKIM: output file can't be overwritten");
          | DKIM_RC_INVALIDFORMAT:
                LogTransactionL (logID, "DKIM: invalid input file format");
          | DKIM_RC_KEYNOTPROVIDED:
                LogTransactionL (logID, "DKIM: can't read private key file");
        ELSE
            LogTransactionL (logID, "DKIM: unknown error code");
        END (*CASE*);

    END InsertDKIMSignature;

(********************************************************************)
(*              CHECKING A DKIM SIGNATURE FOR VALIDITY              *)
(********************************************************************)

PROCEDURE Copy (A: CharPtr;  VAR (*OUT*) B: ARRAY OF CHAR);

    (* Copies A^ to B.  This is how we can copy a C string          *)
    (* parameter into a Modula-2 string.                            *)

    VAR k: CARDINAL;
    BEGIN
        k := 0;
        WHILE A^ <> Nul DO
            B[k] := A^;  INC(k);
            A := AddOffset (A,1);
        END (*WHILE*);
        B[k] := Nul;
    END Copy;

(********************************************************************)

PROCEDURE ["SysCall"] GetPubKey (pMethods, pSDID, pSelector: CharPtr;
                                            pData: DataPtr): PDKIMKEY;

    (* GetPubKey is a procedure that will provide the public key.   *)
    (* It is declared ["C"] because parameters are passed using the *)
    (* C calling convention.                                        *)

    (* The parameters of a GetKeyProc are                                       *)
    (*  Methods:  The value of the 'q' tag. An empty string means that the tag  *)
    (*            is not specified and the default value "dns/txt" is assumed.  *)
    (*  SDID:     Domain name (tag 'd').                                        *)
    (*  Selector: Selector (tag 's').                                           *)
    (*  Data:    User data if needed.                                           *)

    (* Unfortunately the number and type of parameters is fixed by  *)
    (* the design of DKIM.DLL, se we have no way to log any errors  *)
    (* that occur in this operation.                                *)

    VAR Buf, TXTRec: ARRAY [0..511] OF CHAR;
        qvalue, dvalue, svalue: ARRAY [0..511] OF CHAR;
        method: ARRAY [0..63] OF CHAR;
        k, L, Lm, pos: CARDINAL;
        found: BOOLEAN;

    BEGIN
        (* This first case is not for the production code.  It is   *)
        (* to allow testing without a DNS lookup.                   *)

        IF PubKeyFile[0] <> Nul THEN

            (* Testing option: get public key from file instead of via DNS. *)

            RETURN _dkimLoadPubKey (PubKeyFile, PubKeyPassword);

        END (*IF*);

        IF pData = NIL THEN      (* to suppress a compiler warning *)
        END (*IF*);

        (* Modula-2 string operations will not work on parameters passed    *)
        (* with the C calling convention, so we have to copy the parameters.*)

         Copy (pMethods, qvalue);
         Copy (pSDID, dvalue);
         Copy (pSelector, svalue);

        (* qvalue is a colon-separated list of query methods, but the only *)
        (* one we will accept is dns/txt.                                  *)

        IF qvalue[0] <> Nul THEN
            k := 0;  L := LENGTH (qvalue);
            LOOP
                Strings.FindNext (':', qvalue, k, found, pos);
                IF NOT found THEN pos := L END(*IF*);
                Lm := pos;
                IF pos <= k+1 THEN
                    method := "";  Lm := 0;
                ELSE
                    DEC (Lm, k+1);
                    Strings.Extract  (qvalue, k, Lm, method);
                END (*IF*);

                (* Strip leading and trailing spaces. *);

                WHILE method[0] = ' ' DO
                    Strings.Delete (method, 0, 1);  DEC(Lm);
                END (*WHILE*);
                WHILE (Lm > 0) AND (method[Lm-1] = ' ') DO
                    DEC (Lm);  method[Lm] := Nul;
                END (*WHILE*);

                IF Strings.Equal (method, "dns/txt") THEN
                    found := TRUE;  EXIT (*LOOP*);
                END (*IF*);
                k := pos + 1;
                IF k >= L THEN
                    found := FALSE;  EXIT (*LOOP*);
                END (*IF*);

            END (*LOOP*);

            IF NOT found THEN
                (*
                WriteString( "The 'q' tag of the DKIM-signature field does not contain the "+
                                "required dns/txt method." );
                WriteLn;
                *)
            END (*IF*);
        END (*IF*);

        (* Make the name of the subdomain to look up. *)

        Strings.Assign (svalue, Buf);
        Strings.Append ("._domainkey.", Buf);
        Strings.Append (dvalue, Buf);
        (*WriteString ("Looking up ");  WriteString (Buf);  WriteLn;*)

        IF GetFirstTXTrecord (Buf, "v=DKIM1", TXTRec) <> 0 THEN
            (*WriteString (ID, "DNS TXT lookup failed");  WriteLn; *)
            RETURN NIL;
        END (*IF*);

        RETURN _dkimCreatePubKeyDNSTXT (TXTRec);

    END GetPubKey;

(********************************************************************)

PROCEDURE CheckDKIMSignature (filename: ARRAY OF CHAR;
                                VAR (*OUT*) reason: ARRAY OF CHAR)
                                                         : DKIMresult;

    (* Checks the DKIM-Signature header field, if any, in the file. *)
    (* If there is no such header, we return TEMPFAIL.  The reason  *)
    (* output is a message suitable for logging.                    *)

    VAR result: CARDINAL;

    BEGIN
        result := _dkimSignatureValidate (filename, DKIM_FL_DOTSTUFF,
                                                    GetPubKey, NIL);
        CASE result OF
            DKIM_RC_OK:
                Strings.Assign ("DKIM signature is valid", reason);
                RETURN SUCCESS;
          | DKIM_RC_ERROR:
                Strings.Assign ("DKIM checker reports error", reason);
                RETURN PERMFAIL;
          | DKIM_RC_FILEREADERROR:
                Strings.Assign ("Cannot read message file", reason);
                RETURN TEMPFAIL;
          | DKIM_RC_INVALIDFORMAT:
                Strings.Assign ("Invalid format of DKIM signature", reason);
                RETURN TEMPFAIL;
          | DKIM_RC_KEYNOTPROVIDED:
                Strings.Assign ("DKIM public key not provided", reason);
                RETURN TEMPFAIL;
          | DKIM_RC_NOSIGNATURE:
                Strings.Assign ("There is no DKIM signature", reason);
                RETURN TEMPFAIL;
          | DKIM_RC_INVALIDBODYHASH:
                Strings.Assign ("DKIM failure: invalid body hash", reason);
                RETURN PERMFAIL;
          | DKIM_RC_VERIFICATIONFAILED:
                Strings.Assign ("DKIM signature does not match", reason);
                RETURN PERMFAIL;
          | DKIM_RC_UNSUPPORTEDVERSION:
                Strings.Assign ("Cannot read message file", reason);
                RETURN TEMPFAIL;
        ELSE
            Strings.Assign ("unknown return code from DKIM checker", reason);
            RETURN PERMFAIL;
        END (*CASE*);

    END CheckDKIMSignature;

(********************************************************************)
(*                 LOADING PARAMETERS FROM INI FILE                 *)
(********************************************************************)

PROCEDURE LoadParameters;

    (* Sets several parameters controlling DKIM operation.  *)

    VAR hini: INIData.HINI;
        app: ARRAY [0..7] OF CHAR;
        file: FilenameString;
        pass: ARRAY [0..63] OF CHAR;
        DKIMsr: CARD8;

    BEGIN
        (* Set initial defaults in case of failure. *)

        ourdomain := "localhost";
        selector := "_default";
        file := "keys\id_rsa";
        pass := "";
        HeaderFields := "Received:From:To:Subject:Date:Message-ID";
        BodyRelaxed := FALSE;
        HeaderRelaxed := FALSE;

        (* Now load the values. *)

        hini := OpenINI();
        IF INIData.INIValid (hini) THEN
            app := "$SYS";
            EVAL(INIGetString (hini, app, "DKIMSigning", ourdomain));
            EVAL(INIGetString (hini, app, "DKIMSelector", selector));
            EVAL(INIGetString (hini, app, "DKIMPrivKeyFile", file));
            EVAL(INIGetString (hini, app, "DKIMPrivPass", pass));
            pPrivKey := _dkimLoadPrivKey (file, pass);
            EVAL(INIGetString (hini, app, "DKIMHeaders", HeaderFields));
            EVAL(INIGet (hini, app, "DKIMsr", DKIMsr));
            (*    00 = s/s     01 = s/r    10 = r/s    11 = r/r  *)
            BodyRelaxed := ODD (DKIMsr);
            HeaderRelaxed := ODD (DKIMsr DIV 2);
            CloseINI;
        END (*IF*);
    END LoadParameters;

(********************************************************************)

BEGIN
    SetParameters;
END MyDKIM.


(* NOTES FOR DKIM

Example of a signature:

DKIM-Signature: v=1; a=rsa-sha1; c=relaxed/relaxed; s=1480499783.regnetwork8854; d=regnetwork8854.com;
 h=List-Unsubscribe:Reply-To:Sender:From:To:Subject:Message-ID:Date; i=info@regnetwork8854.com;
 bh=VzMq4BHOo4hvVrp4E1SvRkvbLFU=;
 b=nqj+VKRNM1eP8I5fnSYfONkg1naOseTim8rkHAeMLoLPO2TQEEO2hY2dnQJ+1cbSphjZ1XJq23kD
   Q3dAsAwN1paAW5c2JcJWxvTtpkJT1ffnPGCvdmO9pWmNkhfBFAMLL4KR0n2GDOT6raPLpJr2LvhT
   wpIuzKk1O+HDiPlNh78=

Defined tags in a DKIM-Signature:
    v   version, always 1 so far, default is DKIM1
    a   algorithm, rsa-sha1 and rsa-sha256 must be supported
    b   signature data, base64 encoded
    bh  signature of the canonicalised body part as limited by the "l=" tag
    c   canonicalisation method, value is simple or relaxed or a pair
        a/b where a and b are one of simple and relaxed
    d   domain of the signer
    h   colon-separated list of header fields that will be included in
        the calculation. Order matters. May line-wrap. May include
        non-existent headers, which are treated as null strings. May
        have duplicate names if the message has those duplicates.
    i   responsible agent, default is @domain where the domain is
        defined by the d tag.
    l   decimal, number of body length bytes (after canonicalisation) to include in
        the calculation. Default is entire body.
    q   colon-separated list of query methods used to retrieve the
        public key. Default, and currently the only valid value, is dns/txt
    s   selector subdividing the namespace for the domain defined by d
    t   timestamp, seconds since start of 1 Jan 1970 in UTC time zone.
        Value could be as high as 10^12, but we MAY consider a value
        of more than 12 digits to be infinite. We may ignore timestamps
        that specify a time in the future.
    x   signature expiration, same format as t tag, must be greater than
        t value if both are present.
    z   vertical-bar-separated list of header lines that were present
        when the message was signed. (It need not be a complete list.)
        Probably not something that I need to implement.
Section 3.6.1 gives a different list, which I take to be the way
that public key data are presented.

The DNS query to get public key appears to be for domain <s>._domainkey.<d>.
We are not told what to do if <s> is empty, but I'm guessing that in that
case the period after <s> is also missing.

Section 3.7 of the standard claims to describe how the hashes are
computed, but it's very vague.

I still need to work out how (and why) binary data are base64-encoded.

*)

