(**************************************************************************)
(*                                                                        *)
(*  PMOS library                                                          *)
(*  Copyright (C) 2021   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE VarStrings;

        (********************************************************)
        (*                                                      *)
        (*                Variable-length strings               *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        23 November 2021                *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)


FROM SYSTEM IMPORT ADDRESS, ADR, CARD8, LOC;

FROM BigNum IMPORT
    (* type *)  BN,
    (* proc *)  Nbytes, BNtoBytes;

FROM MiscFuncs IMPORT
    (* proc *)  Swap4;

FROM LowLevel IMPORT
    (* proc *)  Copy, EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM STextIO IMPORT
    (* proc *)  WriteChar, WriteLn;

(************************************************************************)

CONST
    Nul = CHR(0);  LF = CHR(10);  CR = CHR(13);

(************************************************************************)
(*                    WRITE TO STANDARD OUTPUT                          *)
(************************************************************************)

PROCEDURE WriteVarString (VAR (*IN*) str: ARRAY OF CHAR;  N: CARDINAL);

    (* Write character string of length N to standard output.   *)
    (* Convention: we treat LF as the newline character, and    *)
    (* ignore any CR encountered.                               *)

    VAR j: CARDINAL;
        ch: CHAR;

    BEGIN
        j := 0;
        WHILE j < N DO
            ch := str[j];  INC(j);
            IF ch = Nul THEN
                (* Premature termination *)
                j := N;
            ELSIF ch = CR THEN
                (* do nothing *);
            ELSIF ch = LF THEN
                WriteLn;
            ELSE
                WriteChar (ch);
            END (*IF*);
        END (*WHILE*);
    END WriteVarString;

(************************************************************************)
(*                 CREATING AND DESTROYING ByteStr DATA                 *)
(************************************************************************)

PROCEDURE CopyOfBS (src: ByteStr): ByteStr;

    (* Creates and returns a copy of src.  *)

    VAR result: ByteStr;

    BEGIN
        result := src;
        ALLOCATE (result.data, result.allocated);
        Copy (src.data, result.data, src.size);
        RETURN result;
    END CopyOfBS;

(************************************************************************)

PROCEDURE MakeBS (VAR (*INOUT*) A: ByteStr;  N: CARDINAL);

    (* Allocates enough space to hold N bytes.  *)

    BEGIN
        A.allocated := N;
        A.size := N;
        IF N > 0 THEN
            ALLOCATE (A.data, N);
        END (*IF*);
    END MakeBS;

(************************************************************************)

PROCEDURE DiscardBS (VAR (*INOUT*) A: ByteStr);

    (* Discards the data in the string.  *)

    BEGIN
        IF A.allocated > 0 THEN
            DEALLOCATE (A.data, A.allocated);
        END (*IF*);
        A.allocated := 0;
        A.size := 0;
    END DiscardBS;

(************************************************************************)
(*                           TYPE CONVERSIONS                           *)
(************************************************************************)

PROCEDURE GenStrToByteStr (p: ADDRESS;  N: CARDINAL): ByteStr;

    (* Creates a ByteStr from N bytes at address p.   *)

    VAR result: ByteStr;

    BEGIN
        WITH result DO
            size := N;
            allocated := N;
            ALLOCATE (data, N);
            Copy (p, data, N);
        END (*WITH*);
        RETURN result;
    END GenStrToByteStr;

(********************************************************************************)

PROCEDURE CharStrToByteStr (str: ARRAY OF CHAR): ByteStr;

    (* Creates a ByteStr from a character string.   *)

    BEGIN
        RETURN GenStrToByteStr (ADR(str), LENGTH(str));
    END CharStrToByteStr;

(********************************************************************************)

PROCEDURE ByteStrToStr (VAR (*INOUT*) src: ByteStr;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Converts ByteStr to ordinary character string, deallocates the source.   *)

    CONST Nul = CHR(0);

    BEGIN
        IF src.size > 0 THEN
            Copy (src.data, ADR(result), src.size);
        END (*IF*);
        result[src.size] := Nul;
        DiscardBS (src);
    END ByteStrToStr;

(********************************************************************************)

PROCEDURE BNtoByteStr(A: BN): ByteStr;

    (* Converts a Bignum to a ByteStr, using twos complement for negative   *)
    (* numbers, with a leading 00 or FF inserted if needed to ensure that   *)
    (* the high-order bit agrees with the sign.  Does not deallocate the source.*)

    VAR N: CARDINAL;
        result: ByteStr;

    BEGIN
        N := Nbytes (A);
        result.size := N;
        result.allocated := N;
        ALLOCATE (result.data, N);
        EVAL (BNtoBytes (A, result.data^));
        RETURN result;
    END BNtoByteStr;

(********************************************************************************)
(*                           SUBSTRING DELETION                                 *)
(********************************************************************************)

PROCEDURE BSDelete (VAR (*INOUT*) bs: ByteStr;  from, N: CARDINAL);

    (* Deletes a substring of length N, starting at bs.data^[from]. *)

    VAR newsize: CARDINAL;
        p: ByteStringPtr;

    BEGIN
        IF N > 0 THEN
            newsize := bs.size - N;
            ALLOCATE (p, newsize);
            IF from > 0 THEN
                Copy (bs.data, p, from);
            END (*IF*);
            IF newsize > from THEN
                Copy (ADR(bs.data^[from+N]), ADR(p^[from]), newsize - from);
            END (*IF*);
            IF bs.allocated > 0 THEN
                DEALLOCATE (bs.data, bs.allocated);
            END (*IF*);
            bs.size := newsize;
            bs.allocated := newsize;
            bs.data := p;
        END (*IF*);
    END BSDelete;

(************************************************************************)
(*                     COPYING A FIELD FROM A PACKET                    *)
(************************************************************************)

PROCEDURE GetCard (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL): CARDINAL;

    (* Returns 4-byte number at position pos, updates pos.   *)

    VAR N: CARDINAL;

    BEGIN
        N := 0;
        Copy (ADR(pkt.data^[pos]), ADR(N), 4);
        INC (pos, 4);
        RETURN Swap4(N);
    END GetCard;

(************************************************************************)

PROCEDURE GetByteChunk (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;
                              VAR (*OUT*) str: ARRAY OF LOC;  N: CARDINAL);

    (* Extracts a string of N bytes without preceding count, updates pos. *)

    BEGIN
        Copy (ADR(pkt.data^[pos]), ADR(str), N);
        INC (pos, N);
    END GetByteChunk;

(************************************************************************)

PROCEDURE GetByteString (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;
                              VAR (*OUT*) str: ARRAY OF CARD8): CARDINAL;

    (* Extracts string at position pos, updates pos.    *)
    (* Returns the string length.                       *)

    VAR N: CARDINAL;

    BEGIN
        N := 0;
        Copy (ADR(pkt.data^[pos]), ADR(N), 4);
        N := Swap4(N);
        INC (pos, 4);
        Copy (ADR(pkt.data^[pos]), ADR(str), N);
        INC (pos, N);
        RETURN N;
    END GetByteString;

(************************************************************************)

PROCEDURE GetCharString (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;
                                        VAR (*OUT*) str: ARRAY OF CHAR);

    (* Extracts string at position pos, updates pos.    *)
    (* Returns the string length.                       *)

    VAR N: CARDINAL;

    BEGIN
        N := 0;
        Copy (ADR(pkt.data^[pos]), ADR(N), 4);
        N := Swap4(N);
        INC (pos, 4);
        Copy (ADR(pkt.data^[pos]), ADR(str), N);
        INC (pos, N);
        IF N <= HIGH(str) THEN
            str[N] := CHR(0);
        END (*IF*);
    END GetCharString;

(************************************************************************)
(*                   PUTTING A FIELD INTO A PACKET                      *)
(*    We assume that the packet already has enough space allocated      *)
(************************************************************************)

PROCEDURE AddByte (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;  value: CARD8);

    (* Stores 1-byte number (uint32) at position pos, updates pos.  *)

    BEGIN
        pkt.data^[pos] := value;
        INC (pos);
    END AddByte;

(************************************************************************)

PROCEDURE AddCard (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;  value: CARDINAL);

    (* Stores 4-byte number (uint32) at position pos, updates pos.  *)

    VAR M: CARDINAL;

    BEGIN
        M := Swap4(value);
        Copy (ADR(M), ADR(pkt.data^[pos]), 4);
        INC (pos, 4);
    END AddCard;

(************************************************************************)

PROCEDURE AddByteChunk (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;
                              VAR (*IN*) str: ARRAY OF LOC;  N: CARDINAL);

    (* Stores N bytes of data at position pos, updates pos.  The        *)
    (* difference between this and AddByteString, below, is that no     *)
    (* length field is put ahead of the chunk.                          *)

    BEGIN
        IF N > 0 THEN
            Copy (ADR(str), ADR(pkt.data^[pos]), N);
            INC (pos, N);
        END (*IF*);
    END AddByteChunk;

(************************************************************************)

PROCEDURE AddByteString (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;
                              VAR (*IN*) str: ARRAY OF LOC;  N: CARDINAL);

    (* Stores N-byte string at position pos, updates pos.  *)

    VAR M: CARDINAL;

    BEGIN
        M := Swap4(N);
        Copy (ADR(M), ADR(pkt.data^[pos]), 4);
        INC (pos, 4);
        IF N > 0 THEN
            Copy (ADR(str), ADR(pkt.data^[pos]), N);
            INC (pos, N);
        END (*IF*);
    END AddByteString;

(********************************************************************************)

PROCEDURE AddCharString (pkt: ByteStr;  VAR (*INOUT*) pos: CARDINAL;
                                                    str: ARRAY OF CHAR);

    (* Stores character string at position pos, updates pos.  *)

    BEGIN
        AddByteString (pkt, pos, str, LENGTH(str));
    END AddCharString;

(********************************************************************************)

END VarStrings.

