/****************************************************************/
/* Alters the FtpServer INI file to change the IP address of    */
/* the firewall that is specified in the 'behind firewall'      */
/* rules.  You can do the same job by running Setup.exe, but    */
/* running Setup might not be acceptable to someone who wants   */
/* to automate the job.                                         */
/*                                                              */
/* Run this from the same directory as FTPD.INI or FTPD.TNI     */
/*                                                              */
/*       Programmer:      P. Moylan                             */
/*       Last modified:   17 November 2020                      */
/*                                                              */
/* Prerequisite: for this to work you must have RXU.DLL in      */
/* your LIBPATH.  RXU, a Rexx library written by Dave Boll, is  */
/* available from Hobbes with the name rxu1a.zip.               */
/*                                                              */
/*       Usage:    tools\newfwip addr                           */
/*             where 'addr' is the IP address of the firewall,  */
/*             in dotted IP notation.                           */
/*       Example:  newfwip 188.99.54.3                          */
/*                                                              */
/* If 'addr' is missing then the address is instead taken from  */
/* the current queue.  That means that you can pipe it in from  */
/* another program.  If, for example, you have the program      */
/* MyIP installed, you can run                                  */
/*                                                              */
/*          MyIP | tools\newfwip                                */
/*                                                              */
/* which can be useful if you have a dynamic IP address.  The   */
/* MyIP program can be found on the Hobbes repository.          */
/*                                                              */
/****************************************************************/

call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
call SysLoadFuncs
CALL CheckPrerequisites rxu SelectTNI INI_put

/* Get the address either from the argument or the input queue. */

PARSE arg addr
IF addr = '' THEN PARSE PULL addr

/* Convert IP address to binary (not entirely obvious in Rexx) */

PARSE VAR addr n1'.'n2'.'n3'.'n4
NewAddress = D2C(n1)||D2C(n2)||D2C(n3)||D2C(n4)

/* Update the appropriate INI file entries. */

IF SelectTNI("FTPD") THEN extension = "TNI"
ELSE extension = "INI"
INIfile = 'FTPD.'extension
Call INI_put INIfile, '$SYS', 'BehindFirewall', '01'X
Call INI_put INIfile, '$SYS', 'FirewallIPAddr', NewAddress

/* Tell FtpServer to re-read its INI file. */

SemName = "\SEM32\FTPSERVER\UPDATED"
if RxOpenEventSem(hev, SemName) \= 0 then
    rc = RxCreateEventSem( hev ,'Shared', SemName, 'Reset')
call RxPostEventSem hev
call RxResetEventSem hev

call RxCloseEventSem hev

return 0

/****************************************************************/
/*                      CHECKING PREREQUISITES                  */
/****************************************************************/

CheckPrerequisites: PROCEDURE

    /* The argument is a space-separated list of prerequisite   */
    /* functions, for example                                   */
    /*      CALL CheckPrerequisites rxu SelectTNI INI_get       */
    /* where (at least in this version) each list item is       */
    /* either 'rxu' or a function from my TNItools package.     */
    /* If any is missing then we exit with an error message.    */

    PARSE UPPER ARG funclist
    funclist = STRIP(funclist)
    needrxu = 0
    needtools = 0
    DO WHILE funclist \= ''
        PARSE VAR funclist func funclist
        funclist = STRIP(funclist)
        IF func = 'RXU' THEN DO

            /* Initialise RXU if not already available, fail if */
            /* the RxFuncAdd operation fails.  We must          */
            /* RxFuncQuery RxuTerm because RxuTerm does not     */
            /* deregister RxuInit.  The RxFuncDrop is needed    */
            /* because RxFuncAdd seems to report failure if the */
            /* function is already registered.                  */

            IF RxFuncQuery('RxuTerm') THEN DO
                CALL RxFuncDrop('RxuInit')
                CALL RxFuncAdd 'RxuInit','RXU','RxuInit'
                IF result THEN DO
                    SAY 'Cannot load RXU'
                    needrxu = 1
                END
                ELSE CALL RxuInit
            END
        END
        ELSE DO
            func = func||'.CMD'
            IF SysSearchPath('PATH', func) = '' THEN DO
                SAY 'ERROR: 'func' must be in your PATH'
                needtools = 1
            END
        END
    END
    IF needrxu THEN SAY 'You can find RXU1a.zip at Hobbes'
    IF needtools THEN SAY 'Please install the GenINI package'
    IF needrxu | needtools THEN EXIT 1
    RETURN

/****************************************************************/

