/* state and event objects
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: state.h,v 1.1 2000/11/18 00:30:23 bj Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <time.h>
#include <gmp.h>    /* GNU MP library */

/* Message ID mechanism.
 *
 * Each Phase 2 / Quick Mode exchange has a Message ID that is
 * different from those of all other exchanges between these two peers.
 */

typedef u_int32_t msgid_t;  /* Network order! */

extern bool reserve_msgid(struct in_addr peer, msgid_t msgid);
extern msgid_t generate_msgid(struct in_addr peer);


/* Oakley (Phase 1 / Main Mode) transform and attributes
 * This is a flattened/decoded version of what is represented
 * in the Transaction Payload.
 * Names are chosen to match corresponding names in state.
 */
struct oakley_trans_attrs {
    u_int16_t encrypt;    /* Encryption algorithm */
    const struct encrypt_desc *encrypter; /* package of encryption routines */
    u_int16_t hash;   /* Hash algorithm */
    const struct hash_desc *hasher; /* package of hashing routines */
    u_int16_t auth;   /* Authentication method */
    const struct oakley_group_desc *group;  /* Oakley group */
    time_t life_seconds;  /* When this SA expires (seconds) */
    u_int32_t life_kilobytes; /* When this SA is exhausted (kilobytes) */
#if 0 /* not yet */
    u_int16_t prf;    /* Pseudo Random Function */
#endif
};

/* IPsec (Phase 2 / Quick Mode) transform and attributes
 * This is a flattened/decoded version of what is represented
 * by a Transaction Payload.  There may be one for AH and one
 * for ESP.
 */
struct ipsec_trans_attrs {
    u_int8_t transid; /* transform id */
    ipsec_spi_t spi;
    time_t life_seconds;    /* When this SA expires */
    u_int32_t life_kilobytes; /* When this SA expires */
    u_int16_t encapsulation;
    u_int16_t auth;
    u_int16_t key_len;
    u_int16_t key_rounds;
#if 0 /* not implemented yet */
    u_int16_t cmprs_dict_sz;
    u_int32_t cmprs_alg;
#endif
};

/* IPsec per protocol state information */
struct ipsec_proto_info {
    bool present; /* was this transform specified? */
    struct ipsec_trans_attrs attrs;
    ipsec_spi_t our_spi;
    u_int16_t keymat_len; /* same for both */
    u_char *our_keymat;
    u_char *peer_keymat;
};

/* state object: record the state of a (possibly nascent) SA
 *
 * Invariants (violated only during short transitions):
 * - each state object will be in statetable exactly once.
 * - each state object will always have a pending event.
 *   This prevents leaks.
 */
struct state
{
    so_serial_t        st_serialno;            /* serial number (for seniority) */

    struct connection *st_connection;          /* connection for this SA */
    struct state      *st_connection_next;     /* siblings sharing connection */

    int                st_whack_sock;          /* fd for our Whack TCP socket.
                                                * Single copy: close when freeing struct.
                                                */
    struct oakley_trans_attrs st_oakley;

    struct ipsec_proto_info st_ah;
    struct ipsec_proto_info st_esp;
#ifdef KLIPS
    ipsec_spi_t        st_tunnel_spi;          /* KLUDGE */
#endif

    const struct oakley_group_desc *st_pfs_group; /* group for Phase 2 PFS */

    u_int32_t          st_doi;                 /* Domain of Interpretation */
    u_int32_t          st_situation;

    bool               st_pending_quick;       /* need to build IPsec SA after ISAKMP SA */
    lset_t             st_policy;              /* policy for IPsec SA */

    msgid_t            st_msgid;               /* MSG-ID from header.  Network Order! */

    /* symmetric stuff */

    /* initiator stuff */
    chunk_t            st_gi;                  /* Initiator public value */
    u_int8_t           st_icookie[COOKIE_SIZE];/* Initiator Cookie */
    chunk_t            st_ni;                  /* Ni nonce */

    /* responder stuff */
    chunk_t            st_gr;                  /* Responder public value */
    u_int8_t           st_rcookie[COOKIE_SIZE];/* Responder Cookie */
    chunk_t            st_nr;                  /* Nr nonce */

    /* my stuff */

    chunk_t            st_tpacket;             /* Transmitted packet */

    /* phase 1 ID payload info: */
    chunk_t            st_myidentity;          /* identity (network order) (struct in_addr) */
    u_int8_t           st_myidentity_type;     /* Type of identity */

    u_int8_t           st_myuserprotoid;       /* IDcx.protoid */
    u_int16_t          st_myuserport;

    /* his stuff */

    chunk_t            st_rpacket;             /* Received packet */

    /* phase 1 ID payload info: */
    chunk_t            st_peeridentity;        /* identity (network order) (struct in_addr) */
    u_int8_t           st_peeridentity_type;   /* Type of identity */

    u_int8_t           st_peeruserprotoid;     /* IDcx.protoid */
    u_int16_t          st_peeruserport;

    /* end of symmetric stuff */

    u_int8_t           st_sec_in_use;          /* bool: does st_sec hold a value */
    MP_INT             st_sec;                 /* Our local secret value */

    chunk_t            st_shared;              /* Derived shared secret
                                                * Note: during Quick Mode,
                                                * presence indicates PFS
                                                * selected.
                                                */

    u_int8_t           st_peeridentity_protocol;
    u_int16_t          st_peeridentity_port;

    u_int16_t          st_state;               /* State of exchange */
    u_int8_t           st_retransmit;          /* Number of retransmits */
    unsigned long      st_klives;              /* remaining of retries for rekeying */
    time_t             st_margin;              /* life after EVENT_SA_REPLACE */
    chunk_t            st_p1isa;               /* Phase 1 initiator SA (Payload) for HASH */
    chunk_t            st_skeyid;              /* Key material */
    chunk_t            st_skeyid_d;            /* KM for non-ISAKMP key derivation */
    chunk_t            st_skeyid_a;            /* KM for ISAKMP authentication */
    chunk_t            st_skeyid_e;            /* KM for ISAKMP encryption */
    u_char             st_iv[MAX_DIGEST_LEN];  /* IV for encryption */
    u_char             st_new_iv[MAX_DIGEST_LEN];
    unsigned int       st_iv_len;
    unsigned int       st_new_iv_len;
    chunk_t            st_enc_key;             /* Oakley Encryption key */

    struct event      *st_event;               /* backpointer for certain events */
    struct state      *st_hashchain_next;      /* Next in list */
    struct state      *st_hashchain_prev;      /* Previous in list */
    chunk_t           st_peerpreshared;	
    u_char            st_was_new_iv[MAX_DIGEST_LEN];
    u_int8_t	      reserved[1024-8-MAX_DIGEST_LEN];	       /* dev@fx.dk */    	
};

/* global variables */

extern u_int16_t pluto_port;  /* Pluto's port */
extern struct sockaddr_in mask32; /* 255.255.255.255 */
#ifdef ROAD_WARRIOR_FUDGE
extern struct sockaddr_in mask0;  /* 0.0.0.0 */
#endif /* ROAD_WARRIOR_FUDGE */

/* state functions */

extern struct state *new_state(void);
extern void init_states(void);
extern void insert_state(struct state *st);
extern void unhash_state(struct state *st);
extern void release_whack(struct state *st);
extern void delete_state(struct state *st);
struct connection;  /* forward declaration of tag */
extern void delete_states_by_connection(struct connection *c);

extern struct state
    *duplicate_state(const struct state *st),
    *find_state(
  const u_char *icookie,
  const u_char *rcookie,
  const struct in_addr peer,
  msgid_t msgid),
    *find_phase1_state(const struct in_addr peer);

extern struct state *
find_quick_state(const struct in_addr peer);

extern void show_states_status(int wfd);
