/* tinyproxy.c - main tinyproxy source. */
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sysexits.h>
#include <pwd.h>
#include "utils.h"
#include "log.h"
#include "sock.h"
#include "tinyproxy.h"
#include "conns.h"

struct config_s config =
{NULL, DEFAULT_LOG, DEFAULT_CUTOFFLOAD, DEFAULT_PORT, DEFAULT_STATHOST, False,
 DEFAULT_USER};
struct conn_s connections = {0, NULL, 0, NULL, PlaceHolder, False,
#ifdef ANON
 False,
#endif
 NULL, NULL, NULL, 0, 0};
struct stat_s stats = {0, 0, 0, 0, 0, 0, 0, 0};
float load=0.00;
#ifdef ANON
struct allowedhdr_s *allowedhdrs = NULL;
#endif

/* sigh */
void getreqs(void);

/* display an error to the client. */
void httperr(int fd, int err, char *msg) {
   char outbuf[HTTPBUF];
   char *premsg = \
"HTTP/1.0 %d %s\r\n\
Content-type: text/html\r\n\r\n\
<html><head><title>%d</title></head>\r\n\
<body>\r\n\
<font size=\"2\">Cache Error!</font><br>\r\n\
An error of type %d occurred: %s\r\n\
<hr>\r\n\
<font size=\"-1\"><i>generated by %s</font>\r\n\
</body></html>\r\n";

   sprintf(outbuf, premsg, err, msg, err, err, msg, VERSION);
   write(fd, outbuf, strlen(outbuf));
}

/* dump info to the logfile */
void dumpdebug(void) {
   struct conn_s *connptr = &connections;
   int clients = 0, servers = 0, holders = 0, ignores = 0;
   
   log(config.logf, "SIGUSR1 received, debug dump follows.");
   
   while (connptr) {
      switch (connptr->conntype) {
       case ClientConn:
	 clients++;
	 break;
       case ServerConn:
	 servers++;
	 break;
       case PlaceHolder:
	 holders++;
	 break;
       case IgnoreConn:
	 ignores++;
	 break;
      }
      connptr = connptr->next;
   }
   log(config.logf, "servers: %d, clients: %d, ignores: %d, holders: %d",
       servers, clients, ignores, holders);
   log(config.logf, "total requests handled: %lu", stats.num_reqs);
   log(config.logf, "total connections handled: %lu", stats.num_cons);
   log(config.logf, "total sockets listened: %lu", stats.num_listens);
   log(config.logf, "total sockets opened: %lu", stats.num_opens);
   log(config.logf, "total bad opens: %lu", stats.num_badcons);
   log(config.logf, "total bytes tx+rx: %lu", stats.num_txrx);
   log(config.logf, "connections refused due to load: %lu", stats.num_refused);
   log(config.logf, "garbage collections: %lu", stats.num_garbage);
   log(config.logf, "idle connections killed: %lu", stats.num_idles);
   log(config.logf, "end debug dump.");
}

/* handle a signal */
void takesig(int sig) {
   switch (sig) {
    case SIGUSR1:
      dumpdebug();
      break;
    case SIGHUP:
      log(config.logf, "SIGHUP received, cleaning up...");
      garbcoll();
      conncoll();
      log(config.logf, "Finished cleaning memory/connections.");
      break;
    case SIGTERM:
      config.quit = True;
      break;
    case SIGALRM:
      calcload();
      alarm(LOAD_RECALCTIMER);
   }
   if (sig != SIGTERM)
     signal(sig, takesig);
}

/* display usage to the user on stderr. */
void usagedisp(void) {
   fprintf(stderr, "%s\n", VERSION);
   fprintf(stderr, "this software is GPLed; see the file 'COPYING' included with tinyproxy.\n");
   fprintf(stderr, "mail comments/questions/money to: sdyoung@well.com.\n");
   fprintf(stderr, "usage: tinyproxy [args]\n");
   fprintf(stderr, "possible values for [args]:\n");
   fprintf(stderr, "\t-v\t\tdisplay version.\n");
   fprintf(stderr, "\t-h\t\tdisplay this.\n");
   fprintf(stderr, "\t-d\t\tdon't daemonize.\n");
   fprintf(stderr, "\t-p [port]\tlisten on [port].\n");
   fprintf(stderr, "\t-l [logfile]\tlog to [logfile].\n");
   fprintf(stderr, "\t-w [load]\tstop accepting new conns at [load].  0 disables.\n");
   fprintf(stderr, "\t-s [stathost]\tset stathost to [stathost].\n");
   fprintf(stderr, "\t-u [user]\tchange to [user] after startup.  \"\" disables.\n");
#ifdef ANON
   fprintf(stderr, "\t-a [header]\tallow [header] through the anon block.\n");
#endif
   fprintf(stderr, "default port: %d\n", DEFAULT_PORT);
   fprintf(stderr, "default logfile: %s\n", DEFAULT_LOG);
#ifdef ANON
   fprintf(stderr, "anon mode is compiled in.\n");
#endif
}

int main(int argc, char **argv) {
   char optch;
   extern char *optarg;
   flag usage = False, godaemon = True, changeid = False;
   struct passwd *thisuser;
#ifdef ANON
   struct allowedhdr_s *allowedptr;
#endif
   
   while ((optch = getopt(argc, argv, "vhdp:l:a:w:s:u:")) != EOF) {
      switch (optch) {
       case 'v':
	 fprintf(stderr, "%s\n", VERSION);
	 exit(EX_OK);
       case 'h':
       case '?':
	 usage = True;
	 break;
       case 'p':
	 if (!(config.port = atoi(optarg))) {
	    log(NULL, "bad port on commandline, defaulting to %d", DEFAULT_PORT);
	    config.port = DEFAULT_PORT;
	 }
	 break;
       case 'l':
	 config.logf_name = xstrdup(optarg);
	 break;
       case 'd':
	 godaemon = False;
	 break;
       case 'w':
	 sscanf(optarg, "%f", &config.cutoffload);
	 break;
       case 's':
	 config.stathost=xstrdup(optarg);
	 break;
       case 'u' : 
	 config.changeuser=xstrdup(optarg);
	 break;
#ifdef ANON
       case 'a':
	 if (!allowedhdrs) {
	    allowedhdrs = allowedptr = xmalloc(sizeof(struct allowedhdr_s));
	 } else {
	    allowedptr->next = xmalloc(sizeof(struct allowedhdr_s));
	    allowedptr = allowedptr->next;
	 }
	 allowedptr->hdrname = xstrdup(optarg);
	 allowedptr->next = NULL;
#else
       case 'a':
	 log(NULL, "-a only applies when ANON mode has been compiled in!");
	 exit(EX_USAGE);
#endif
      }
   }
   
   if (usage == True) {
      usagedisp();
      exit(EX_OK);
   }

   if (!(config.logf = fopen(config.logf_name, "a"))) {
      log(NULL, "unable to open logfile for appending!");
      exit(EX_CANTCREAT);
   }

   log(config.logf, "%s starting...", VERSION);
   

   if(strlen(config.changeuser)) {
      if((getuid() != 0) && (geteuid() != 0)) {
	 log(config.logf, "not running as root, therefore not changing uid/gid.");
      } else {
         changeid=True;
         if(!(thisuser=getpwnam(config.changeuser))) {
   	    log(config.logf, "unable to find user \"%s\"!", config.changeuser);
       	    exit(EX_NOUSER);
         }
         log(config.logf, "changing to user \"%s\" (%d/%d).",
             config.changeuser, thisuser->pw_uid, thisuser->pw_gid);
      }
   } 
   
//   if (godaemon == True)
//     daemon(0, 0);
   
   signal(SIGPIPE, SIG_IGN);
   signal(SIGUSR1, takesig);
   signal(SIGTERM, takesig);
   signal(SIGHUP, takesig);
   signal(SIGALRM, takesig);
   alarm(LOAD_RECALCTIMER);
   calcload();
   
   if (init_listen_sock(config.port) < 0) {
      log(config.logf, "unable to bind port %d!", config.port);
      exit(EX_UNAVAILABLE);
   }
   if(changeid == True) {
      setuid(thisuser->pw_uid);
      setgid(thisuser->pw_gid);
   }
   log(config.logf, "now accepting connections.");
         
   while (config.quit == False) {
      getreqs();
   }
   log(config.logf, "shutting down.");
   de_init_listen_sock();
   fclose(config.logf);
   
   exit(EX_OK);
}
