/*---------------------------------------------------------------------------+
 | Titel: CHARSET.C                                                          |
 +-----------------------------------------+---------------------------------+
 | Erstellt von: Michael Hohner            | Am: 04.06.1994                  |
 +-----------------------------------------+---------------------------------+
 | System: OS/2 2.x                                                          |
 +---------------------------------------------------------------------------+
 | Beschreibung:                                                             |
 |                                                                           |
 |   Implementierung von FSC-0054 (CHARSET-Proposal) fuer FleetStreet        |
 |                                                                           |
 +---------------------------------------------------------------------------+*/

#pragma strings(readonly)

/*----------------------------- Header-Dateien ------------------------------*/

#define INCL_DOSNLS
#define INCL_WINCOUNTRY
#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include "charset.h"

/*--------------------------------- Defines ---------------------------------*/

/*---------------------------------- Typen ----------------------------------*/

/*---------------------------- Globale Variablen ----------------------------*/

#define NUM_LEVEL1SETS 12

/* Level 1 Character sets */
static const char *Level1Sets[]={ "DUTCH", "FINNISH", "FRENCH", "CANADIAN",
                                  "GERMAN", "ITALIAN", "NORWEG", "PORTU",
                                  "SPANISH", "SWEDISH", "SWISS", "UK", ""};

#define NUM_LEVEL2SETS  5

/* Level 2 Character sets */
static const char *Level2Sets[]={ "LATIN-1", "ASCII", "IBMPC", "MAC", "VT100", ""};

/* Level 1 Table */
static const unsigned char Level1Table[NUM_LEVEL1SETS][128]=
{
   /* Dutch */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', 0x9c, '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', '@',
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x98, 0xab, '|', '^', '_', '`',
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0xac, 0x9f, '?', '`', 0x7f
   },

   /* Finnish */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', '#', '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', '@',
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x8e, 0x99, 0x8f, 0x9a, '_', 0x82,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x84, 0x94, 0x86, 0x81, 0x7f
   },

   /* French */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', 0x9c, '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x85,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0xf8, 0x87, 0x15, '^', '_', '`',
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x82, 0x97, 0x8a, '?', 0x7f
   },

   /* Canadian */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', '#', '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x85,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x83, 0x87, 0x88, 0x8c, '_', 0x93,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x82, 0x97, 0x8a, 0x96, 0x7f
   },

   /* German */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', '#', '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x15,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x8e, 0x99, 0x9a, '^' , '_', '`' ,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x84, 0x94, 0x81, 0xe1, 0x7f
   },

   /* Italian */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', 0x9c, '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x15,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0xf8, 0x87, 0x82, '^' , '_', 0x97,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x85, 0x95, 0x8a, 0x8d, 0x7f
   },

   /* Norweg */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', '#' , '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', '@' ,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x92, 0x9d, 0x8f, '^' , '_', '`' ,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x91, 0x9b, 0x86, '~' , 0x7f
   },

   /* Portu */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', '#' , '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', '@' ,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      'A' , 0x80, 'O' , '^' , '_', '`' ,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      'a' , 0x87, 'o' , '~' , 0x7f
   },

   /* Spanish */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', 0x9c, '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x15,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0xad, 0xa5, 0xa8, '^' , '_', '`' ,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0xf8, 0xa4, 0x87, '~' , 0x7f
   },

   /* Swedish */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', '#' , '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x90,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x8e, 0x99, 0x8f, 0x9a, '_', 0x82,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x84, 0x94, 0x86, 0x81, 0x7f
   },

   /* Swiss */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', 0x97, '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x85,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      0x82, 0x87, 0x88, 0x8c, 0x8a, 0x93,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      0x84, 0x94, 0x81, 0x96, 0x7f
   },

   /* UK */
   {
       0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
      21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      ' ', '!', '\"', 0x9c, '$', '%', '&', '\'', '(', ')', '*', '+',
      ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      ':', ';', '<', '=', '>', '?', 0x85,
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      '[' , '\\', ']' , '^' , '_' , '`' ,
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      '{' , '|' , '}' , '~' , 0x7f
   }
};

/* Level 2 Table */
static const unsigned char Level2Table[NUM_LEVEL2SETS][128]=
{
   /* LATIN-1 */
   {
      128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140,
      141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153,
      154, 155, 156, 157, 158, 159,
      0xff, 0xad, 0x9b, 0x9c, 0x0f, 0x9d,
      '|', 0x15, '?', 'c',
      0xa6, 0xae, 0xaa,
      '-', 'R', '?',
      0xf8, 0xf1, 0xfd, '3', '\'',
      0xe6, 0x14, 0xfe, ',', '1',
      0xa7, 0xaf, 0xac, 0xab, '?', 0xa8,
      'A', 'A', 'A', 'A',
      0x8e, 0x8f, 0x92, 0x80, 'E', 0x90, 'E', 'e',
      'I', 'I', 'I', 'e',
      'D', 0xa5, 'O', 'O', 'O', 'O', 0x99, 'x', 0x9d, 'U', 'U', 'U', 0x9a,
      'Y', '?', 0xe1, 0x85, 0xa0, 0x83, 'a',
      0x84, 0x86, 0x91, 0x87, 0x8a, 0x82, 0x88, 0x89, 0x8d, 0xa1, 0x8c, 0x8b,
      0xe7, 0xa4, 0x95, 0xa2, 0x93,
      'o', 0x94, 0xf6, 0x9b, 0x97, 0xa3, 0x96, 0x81, 'e', '?', 0x98
   },

   /* ASCII   */
   {
      128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
      143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157,
      158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172,
      173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187,
      188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202,
      203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217,
      218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232,
      233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247,
      248, 249, 250, 251, 252, 253, 254, 255
   },

   /* IBMPC   */
   {
      128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
      143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157,
      158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172,
      173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187,
      188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202,
      203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217,
      218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232,
      233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247,
      248, 249, 250, 251, 252, 253, 254, 255
   },

   /* MAC     */
   {
      0x8E, 0x8F, 0x80, 0x90, 0xA5, 0x94, 0x9A, 0xA0, 0x85, 0x83, 0x84,
      'a',
      0x86, 0x87, 0x82, 0x8A, 0x88, 0x89, 0xA1, 0x8D, 0x8C, 0x8B, 0xA4, 0xA2,
      0x95, 0x96, 0x94, 'o' , 0xA3, 0x97, 0x96, 0x81, '+' , 0xF8, 0x9B, 0x9C,
      0x15, 0xFE, 0x14, 0xE1, 'R' , 'c' , 'M', '?', '#',
      0x92, '0' , 0xEC, 0xF1, 0xF3, 0xF2, 0x9D, 0xE6, 0xEB, 0xE4, '?' , 0xE3,
      '?' , 0xA6, 0xA7, 0xEA, 0x91, 0xED, 0xA8, 0xAD, 0xAA, 0xFB, 0x9F, 0xF7,
      0x7F, 0xAF, 0xAE, '.' , 32  , 'A' , 'A' , 'O' , 'E' , 'e' , 0xFA, '-' ,
      '\"', '\"', '`' , '\'', 0xF6, 0x04, 0x98, '?' , '?' , '?' , '?' , '?' ,
      '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?',
      '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?',
      '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?'
   },

   /* VT100   */
   {
      128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
      143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157,
      158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172,
      173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187,
      188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202,
      203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217,
      218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232,
      233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247,
      248, 249, 250, 251, 252, 253, 254, 255
   }
};

/*----------------------- interne Funktionsprototypen -----------------------*/

static int ConvertFromLevel1(char *pchMsgText, const char *pchCharset);
static int ConvertFromLevel2(char *pchMsgText, const char *pchCharset);

/*---------------------------------------------------------------------------*/
/* Funktionsname: ConvertFromCharset                                         */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Beschreibung: Konvertiert einen Text im angegebenen Zeichensatz in einen  */
/*               Text mit lokalem Zeichensatz                                */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Parameter: pchMsgText: Puffer mit Original-Text, Puffer fuer neuen Text   */
/*            pchChrskludge: Charset-Kludgeline lt. FSC-0054 (ohne CHRS:)    */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Rckgabewerte: 0  OK                                                      */
/*                1  Charset nicht unterstuetzt                              */
/*                2  Level nicht unterstuetzt                                */
/*               -1  Parameterfehler                                         */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Sonstiges:                                                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/

int ConvertFromCharset(char *pchMsgText, const char *pchChrsKludge)
{
   char pchKludgeCopy[81];
   char *pchCharset=NULL;
   char *pchLevel=NULL;
   int  iLevel=0;

   if (!pchMsgText || !pchChrsKludge)
      return -1;     /* von nix kommt nix */

   /* Charset und Level in der Kludge trennen */
   memset(pchKludgeCopy, 0, sizeof(pchKludgeCopy));
   strncpy(pchKludgeCopy, pchChrsKludge, 80);
   pchCharset = strtok(pchKludgeCopy, " ");
   if (pchCharset)
      pchLevel = strtok(NULL, " ");

   if (!pchCharset || !pchLevel)
      return -1;   /* Ein Teil der Kludge fehlt */

   iLevel= atoi(pchLevel);

   if (iLevel == 0)
      return 0;     /* schon fertig */

   if (iLevel > 2)
      return 2;     /* Nur Level 0, 1, 2 unterstuetzt */

   if (iLevel == 1)
      return ConvertFromLevel1(pchMsgText, pchCharset);

   if (iLevel == 2)
      return ConvertFromLevel2(pchMsgText, pchCharset);

   return -1; /* Kann kaum vorkommen */
}

/*---------------------------------------------------------------------------*/
/* Funktionsname: ConvertToCharset                                           */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Beschreibung: Konvertiert einen Text im lokalen Zeichensatz in den        */
/*               IBMPC Zeichensatz (Level 2)                                 */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Parameter: pchMsgText: Puffer mit Original-Text, Puffer fuer neuen Text   */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Rckgabewerte: 0  OK                                                      */
/*                1  Charset nicht unterstuetzt                              */
/*                2  Level nicht unterstuetzt                                */
/*               -1  Parameterfehler                                         */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Sonstiges:                                                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/

int ConvertToCharset(char *pchMsgText)
{
   if (!pchMsgText)
      return -1;     /* von nix kommt nix */

   return 0;
}

/*---------------------------------------------------------------------------*/
/* Funktionsname: ConvertFromLevel1                                          */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Beschreibung: Konvertiert einen Text im angegebenen Zeichensatz in einen  */
/*               Text mit lokalem Zeichensatz (Level 1)                      */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Parameter: pchMsgText: Puffer mit Original-Text, Puffer fuer neuen Text   */
/*            pchCharset: Zeichensatz aus der CHRS-Kludge                    */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Rckgabewerte: 0  OK                                                      */
/*                1  Charset nicht unterstuetzt                              */
/*               -1  Parameterfehler                                         */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Sonstiges:                                                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/

static int ConvertFromLevel1(char *pchMsgText, const char *pchCharset)
{
   int tabnum=0;  /* Nummer der gewaehlten Tabelle */
   char *pchConv=pchMsgText;

   if (!pchMsgText || !pchCharset)
      return -1;

   while (Level1Sets[tabnum][0])   /* bis zum Null-String */
   {
      if (!stricmp(Level1Sets[tabnum], pchCharset))
      {
         /* gefunden */
         break;
      }
      tabnum++;
   }

   if (!Level1Sets[tabnum][0])   /* nicht gefunden */
      return 1;

   /* Text konvertieren */
   while (*pchConv)
   {
      *pchConv = Level1Table[tabnum][*pchConv & 0x7f];
      pchConv++;
   }

   return 0;
}

/*---------------------------------------------------------------------------*/
/* Funktionsname: ConvertFromLevel2                                          */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Beschreibung: Konvertiert einen Text im angegebenen Zeichensatz in einen  */
/*               Text mit lokalem Zeichensatz (Level 2)                      */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Parameter: pchMsgText: Puffer mit Original-Text, Puffer fuer neuen Text   */
/*            pchCharset: Zeichensatz aus der CHRS-Kludge                    */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Rckgabewerte: 0  OK                                                      */
/*                1  Charset nicht unterstuetzt                              */
/*               -1  Parameterfehler                                         */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/* Sonstiges:                                                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/

static int ConvertFromLevel2(char *pchMsgText, const char *pchCharset)
{
   int tabnum=0;  /* Nummer der gewaehlten Tabelle */
   char *pchConv=pchMsgText;

   if (!pchMsgText || !pchCharset)
      return -1;

   while (Level2Sets[tabnum][0])   /* bis zum Null-String */
   {
      if (!stricmp(Level2Sets[tabnum], pchCharset))
      {
         /* gefunden */
         break;
      }
      tabnum++;
   }

   if (!Level2Sets[tabnum][0])   /* nicht gefunden */
      return 1;

   /* Text konvertieren */
   while (*pchConv)
   {
      if (*pchConv >= 128)
         *pchConv = Level2Table[tabnum][(*pchConv)-128];
      pchConv++;
   }

   return 0;
}

/*-------------------------------- Modulende --------------------------------*/
