/*----------------------------------------------------------------------*\
    zipdel.cmd	  -- delete files that were unpacked from zip or tar files

    Wonkoo Kim (wkim+@pitt.edu)

    Use this script when you don't want to update an archive (zip or tar)
    file but want to delete files on disk only if they have the same
    timestamp / size / subdir structure as the files in the archive,
    while keeping modified files (of different time or size) on disk.

    Supported archive files are .zip, .tar, .tar.gz, .tgz (gzipped tar), tar.Z.


    This script requires the following programs in PATH:
	Info-Zip's unzip.exe (for zip files)
	GNU tar.exe (for tar files)
	sort.exe of OS/2 (To use GNU sort, change 'sort /r' to 'sort -r' below.)


    History:
	April 25, 1995	    The first version.
	August 11, 1995     Fixed bug when file name had special symbols.
	August 18, 1995     Fixed bug: compute with size of actual files
			    instead of file size info from .zip contents
	August 25, 1995     Added delete option for Read-Only files.
	December 25, 1995   Now more reliably deletes empty dirs.
	May 4, 1996	    Detects delete error
	July 5, 1996	    Enhanced a bit by using rxqueue,
			    Added GNU tar file (.tar, .tar.gz, .tgz) support.
	March 18, 1997	    Fixed bug when file name had '%' characters.
			    (CMD shell converts double '%%' to single '%'.)
	March 20, 1997	    Make sure rxqueue empty before using.
			    -c (Check-only) option added.
	April 14, 1997	    Fixed: files of hidden or system attributes.
			    Fixed: symbolic linked files of tar file.
\*----------------------------------------------------------------------*/

'@echo off'

Call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
Call SysLoadFuncs

parse arg args

subdir = ''
force = 0
junk = 0
quiet = 0
rmattrib = 0
check_flag = 0

cmdargs = ''
do i=1 to words(args)
    opt = word(args, i)

    select
    when left(opt, 2) = '-?' then
	leave i
    when left(opt, 2) = '-c' then
	check_flag = 1
    when left(opt, 2) = '-f' then
	force = 1
    when left(opt, 2) = '-j' then
	junk = 1
    when left(opt, 2) = '-d' then
	do
	    subdir = substr(opt,3)
	    if subdir = '' then do
		i = i + 1
		subdir = word(args, i)
	    end
	end
    when left(opt, 2) = '-q' then
	do
	    quiet = quiet + 1
	    if substr(opt,3) = 'q' then quiet = quiet + 1
	end
    when left(opt, 2) = '-r' then
	rmattrib = 1
    otherwise
	if cmdargs = '' then
	    cmdargs = opt
	else
	    cmdargs = cmdargs||' '||opt
    end
end

if cmdargs = '' | left(opt, 2) = '-?' then do
    say
    say 'Usage: zipdel [-option] [zip or tar archive files]'
    say
    say 'Options:'
    say '   -?          This help screen'
    say '   -c          Check only without actual deletion'
    say '   -d subdir   files to be deleted are under "subdir" tree'
    say '   -f          force to delete even if size and time stamp are different'
    say '   -j          Junk (don''t use) directory info of archive file'
    say '   -q[q]       Quiet mode (default mode is verbose); -qq means quieter'
    say '   -r          Delete files of Read-only, hidden, or system attributes as well'
    say 'Archive files: *.zip, *.tar, *.tar.gz, *.tgz, *.tar.Z'
    say
    say 'Description:'
    say '   ZIPDEL will delete any file that has the same time stamp & size and'
    say '   that is under the same subdir as the one contained in zip or tar file.'
    /**
    say '   If -j option is used, ZIPDEL will look at current dir for all files'
    say '   by disregarding subdir info of files in zip or tar archive file.'
    **/
    say '   The zip or tar archive file is safe (untouched) from this operation.'
    say
    say "WARNING:  Though the chance of failure is very rare, still it's not perfect."
    say "   This program checks file's time stamp only up to minutes, not seconds,"
    say '   thus can be fooled by such slight time stamp differences.'
    say '   Also file contents are not compared but only size, so be careful!'
    say
    say 'Wonkoo Kim (wkim+@pitt.edu)'
    exit 1
end

if (subdir \= '') & (right(subdir,1) \= ':') then
do
    subdir = translate(subdir,'\','/')
    subdir = strip(subdir, 't', '\')||'\'
end

tmpfile = SysTempFileName('zipdel.???')

call flush	/* make sure rxqueue is empty */

do i=1 to words(cmdargs)
    archive = word(cmdargs, i)
    if exists(archive) then
    do
	if quiet < 2 then do
	    say
	    say 'Source Archive:' archive
	    if check_flag then say ' (Check-only flag turned on)'
	end

	fname = filespec('name', archive)
	fl = lowercase(fname)
	select
	when right(fl,4) = '.zip' | right(fl,4) = '.exe' then do
	    /* to include self-extraction .exe zip file */
	    if zipprog = 'ZIPPROG' then do
		zipprog = SysSearchPath('PATH', 'unzip.exe')
		if zipprog = '' then do
		    say "**ERROR: zip.exe is not found in PATH. (Get Info-Zip's zip.exe.)"
		    exit 1
		end
	    end
	    'unzip -Z' archive '2> nul | rxqueue'
	    zip = 1
	    end
	when (right(fl,7) = '.tar.gz' | right(fl,6) = '.tar.z',
				      | right(fl,4) = '.tgz') then do
	    if tarprog = 'TARPROG' then do
		tarprog = SysSearchPath('PATH', 'tar.exe')
		if tarprog = '' then do
		    say "**ERROR: tar.exe is not found in PATH. (Get GNU tar.)"
		    exit 1
		end
	    end
	    'tar -ztvf' archive '2> nul | rxqueue'
	    zip = 0
	    end
	when right(fl,4) = '.tar' then do
	    if tarprog = 'TARPROG' then do
		tarprog = SysSearchPath('PATH', 'tar.exe')
		if tarprog = '' then do
		    say "**ERROR: tar.exe is not found in PATH. (Get GNU tar.)"
		    exit 1
		end
	    end
	    'tar -tvf' archive '2> nul | rxqueue'
	    zip = 0
	    end
	otherwise
	    /* say ' **Unsupported: ' file */
	    iterate
	end

	deleted_files = 0
	deleted_bytes = 0
	notdel_files = 0
	notdel_bytes = 0
	delerr_files = 0
	delerr_bytes = 0
	not_found = 0
	nf_bytes = 0
	rmdirs = 0
	nrmdirs = 0
	prevpath = ''
	do while queued() <> 0
	    parse pull list
	    file = translate( subword(list, 8+zip), '\', '/')
	    path = strip( filespec("path", file), 'T', '\')
	    if (junk) then do
		file = filespec("name", file)
		path = ''
	    end
	    if left(list, 1) = '-' | (zip = 0 & left(list,1) = 'l') then do
		size = word(list, 3+zip)
		if zip = 1 then do	/* zip file list */
		    date = right( word(list, 7), 9, '0')
		    time = word(list, 8)
		    end
		else do 		/* tar file list */
		    date = right(word(list,5),2,'0')'-'word(list,4)'-'right(word(list,7),2)
		    time = word(list, 6)
		    if left(list,1) = 'l' then file = word(file,1)
		end
		stamp = date||' '||time
		file = subdir||file
		if rmattrib = 1 then
		    call SysFileTree file, 'ff', , , '**---'    /* remove attrib */

		if exists(file) then do
		    fbytes = fsize(file)
		    fstamp = left(fasctime(file),9)||right(fbytes, 9)
		    if ( ( left(fasctime(file),length(stamp)) = stamp &,
							fbytes = size ) | force) then
		    do
			if check_flag then
			    rc = 0
			else do
			    rc = SysFileDelete(file)
			end
			if (rc = 0) then do
			    if quiet=0 then say ' Deleted: ' fstamp file
			    deleted_files = deleted_files + 1
			    deleted_bytes = deleted_bytes + fbytes
			end
			else do
			    if quiet<=1 then say ' ** Delete Error:' fstamp file
			    delerr_files = delerr_files + 1
			    delerr_bytes = delerr_bytes + fbytes
			    notdel_files = notdel_files + 1
			    notdel_bytes = notdel_bytes + fbytes
			end
		    end
		    else do
			if quiet<=1 then say ' ** Not Deleted:' fstamp file
			notdel_files = notdel_files + 1
			notdel_bytes = notdel_bytes + fbytes
		    end
		end
		else do
		    if quiet = 0 then say ' ** Not Found:' file
		    not_found = not_found + 1
		    nf_bytes = nf_bytes + size
		end
	    end
	    /*
	     * save a list of subdir names (to delete empty dirs)
	     */
	    if check_flag = 0 then do
		if path \= prevpath then do
		    rc = lineout(tmpfile, subdir||path)
		    if rc \= 0 then do
			say '**ERROR: lineout()' tmpfile
			call flush
			exit 1
		    end
		    pathtoken = translate(path, ' ', '\')
		    do j = words(pathtoken) to 1 by -1
			k = wordindex(pathtoken, j) - 1
			if k > 0 then k = k - 1
			if left(path, k) == left(prevpath, k) then
			    leave j
			rc = lineout(tmpfile, subdir||left(path,k))
		    end
		    prevpath = path
		end
	    end
	end

	if exists(tmpfile) then do
	    call stream tmpfile, 'C', 'CLOSE'
	    'sort /r <' tmpfile '2> nul | rxqueue'

	    if quiet < 2 & \junk then do
		say
		say ' Purging Empty Directories ...'
		end
	    path = ''
	    do while queued() <> 0
		parse pull input
		if input \= '' & input \= path then do
		    path = input
		    rc = SysRmDir(path)
		    if rc = 0 then do
			say '  Removed:  ' path||'\'
			rmdirs = rmdirs + 1
		    end
		    if rc \= 0 & rc \= 3 & quiet < 2 then do
			say '  Non-empty:' path
			nrmdirs = nrmdirs + 1
		    end
		end
	    end
	    call stream tmpfile, 'C', 'CLOSE'
	    call SysFileDelete tmpfile
	end

	if quiet < 2 | check_flag then do
	    len = max(length(deleted_bytes), length(notdel_bytes), length(nf_bytes))
	    say
	    say ' Result:'
	    if (deleted_files > 0) then
		say '  Deleted:    ' format(deleted_files,4) 'files (',
		    || right(commas(deleted_bytes),len+2) 'bytes)'
	    if (notdel_files > 0) then
		say '  Not Deleted:' format(notdel_files,4) 'files (',
		    || right(commas(notdel_bytes),len+2) 'bytes)'
	    if (delerr_files > 0) then
		say '   (Del Error:' format(delerr_files,4) 'files (',
		    || right(commas(delerr_bytes),len+2) 'bytes))'
	    if (not_found > 0) then
		say '  Not Found:  ' format(not_found,4) 'files (',
		    || right(commas(nf_bytes),len+2) 'bytes)'
	    if (rmdirs > 0) then
		say '  Removed:    ' format(rmdirs,4) 'dirs.'
	    if (nrmdirs > 0) then
		say '  Not Removed:' format(nrmdirs,4) 'dirs.'
	end
	if check_flag then do
	    say
	    say ' ** Check-only flag was turned on.  Files/dirs were not really deleted.'
	end
    end
end

exit 0

/*----------------------------------------------------------------------*/
exists: procedure
    arg fname
/*----------------------------------------------------------------------*/
/* check if a file exists.
*/
    return stream(fname, 'C', 'QUERY EXISTS') \= ''

/*----------------------------------------------------------------------*/
fsize: procedure
    arg fname
/*----------------------------------------------------------------------*/
/* get file size */
    return stream(fname, 'C', 'QUERY SIZE')

/*----------------------------------------------------------------------*/
fasctime: procedure
    arg fname
/*----------------------------------------------------------------------*/
/* convert datetime into more readable date string without time part.
 * input argument format is based on getftime().
 */
    datetime = space( stream(fname, 'C', 'QUERY DATETIME'), 1)
    parse var datetime month'-'day'-'year time
    month.01 = 'Jan'
    month.02 = 'Feb'
    month.03 = 'Mar'
    month.04 = 'Apr'
    month.05 = 'May'
    month.06 = 'Jun'
    month.07 = 'Jul'
    month.08 = 'Aug'
    month.09 = 'Sep'
    month.10 = 'Oct'
    month.11 = 'Nov'
    month.12 = 'Dec'
    return day'-'month.month'-'year time

/*----------------------------------------------------------------------*/
commas: procedure
    arg number
/*----------------------------------------------------------------------*/
/* insert thousand commas in a positive number string */
    str = number
    p = length(trunc(number))
    do while p > 3
	p = p - 3
	str = insert(',', str, p)
    end
    return str

/*----------------------------------------------------------------------*/
lowercase: procedure
/*----------------------------------------------------------------------*/
/* change to lowercase
 */
    arg str
    return translate(str,xrange('a','z'),xrange('A','Z'))

/*----------------------------------------------------------------------*/
flush: procedure
/*----------------------------------------------------------------------*/
/* flush rxqueue if not empty */
    if queued() > 0 then
	do while queued() <> 0
	    parse pull line
	end
    return

/* End of zipdel.cmd */
