/*----------------------------------------------------------
   Deletes a value from an INI or TNI file. Although this can
   be run from the command line, it is primarly intended to
   be called from another script.

   Note that it is not easy for a Rexx program to save state
   information from one invocation to the next. On each call
   the INI or TNI file has to be consulted all over again.
   If you need to store a number of INI values, it might
   be more efficient to do the job in a different language.

           Author:       Peter Moylan (peter@pmoylan.org)
           Last revised: 9 February 2019

   Can't yet handle, or untested:
        <nothing at present>

   Usage:
                CALL INIdel filename, app, key

            where
                filename is the name of an INI or TNI file,
                including a path if it is not in the
                current working directory. The file is
                assumed to be an INI file if the name
                ends with ".INI", and a TNI file otherwise.
                If the file does not exist then we
                return without storing anything.

                app and key are the labels of an
                (application,key) pair within that file.

   Special cases:
            If the specified (app,key) doesn't exist, we
            do nothing.

            If key is null then the whole application is
            deleted.

            If app is null then key must also be null, and
            in that case all applications are deleted.

   Remark: The code in this script is almost exactly the same
           as the code in INIput.cmd, except that we don't
           write back a value when we've found the right key.
           Not sure whether this means we should combine the
           two scripts.

------------------------------------------------------------*/

/****************************************************************/
/*                       MAIN PROGRAM                           */
/****************************************************************/

CALL RxFuncAdd SysLoadFuncs, RexxUtil, SysLoadFuncs
CALL SysLoadFuncs

PARSE ARG INIfile, app, key
INIfile = STRIP(INIfile)
IF STREAM(INIfile, 'C', 'QUERY EXISTS') = '' THEN DO
    SAY "ERROR: File "INIfile" does not exist"
    RETURN
END
IF (app = "") & (key \= "") THEN DO
    SAY "ERROR: if application is null then key must be null"
    RETURN
END

TNIoption = 'T'
pos = LASTPOS('.', INIfile)
if pos > 0 THEN
    DO
        extension = TRANSLATE(RIGHT(INIfile, LENGTH(INIfile) - pos))
        IF extension = "INI" THEN TNIoption = 'I'
    END

CALL INIdestroy INIfile, TNIoption, app, key
RETURN

/****************************************************************/
/*                REMOVING AN INI OR TNI VALUE                  */
/****************************************************************/

INIdestroy: PROCEDURE

    /* Deletes one INI or TNI value, depending on TNIoption. */

    PARSE ARG filename, TNIoption, application, key
    key = STRIP(key)
    IF TNIoption = 'T' THEN DO
        IF application = '' THEN CALL RemoveAllApps filename
        ELSE IF key = '' THEN CALL RemoveApp filename, application
        ELSE CALL RemoveVal filename, application, key
    END
    ELSE DO
        IF application = '' THEN 'DEL 'filename
        ELSE IF key = '' THEN CALL SysIni filename, application, 'DELETE:')
        ELSE CALL SysIni filename, application, key, 'DELETE:')
    END
    RETURN

/****************************************************************/

RemoveVal: PROCEDURE

    /* Deletes a value in a TNI file.  */

    PARSE ARG srcfile,app,key
    srcfile = STRIP(srcfile)
    dstfile = SysTempFileName('testp?????.tmp')
    dummy = STREAM(srcfile, 'C', 'OPEN READ')
    mustclose = MoveToApp( srcfile,dstfile,app )
    oldline = MoveToKey(srcfile,dstfile,app,key)
    IF oldline = '' THEN mustclose = 1

    /* Special case: skip over continuation lines. */

    DO WHILE RIGHT(oldline,1) = '+'
        oldline = LINEIN(srcfile)
    END

    /* We have now read the key, if it exists, from the source  */
    /* file.  We don't write it to the destination file, but    */
    /* we must copy the rest of the source file.                */

    IF mustclose THEN CALL LINEOUT dstfile, '[/'app']'
    CALL CopyRemainder srcfile,dstfile
    CALL STREAM srcfile, 'C', 'CLOSE'
    CALL STREAM dstfile, 'C', 'CLOSE'
    BAKname = srcfile'.BAK'
    '@DEL 'BAKname' 2>NUL'
    '@RENAME 'srcfile' 'BAKname
    '@RENAME 'dstfile' 'srcfile
    RETURN

/****************************************************************/
/*                 REMOVING AN ENTIRE APPLICATION               */
/****************************************************************/

RemoveAllApps: PROCEDURE

    /* Deletes every entry from srcfile.  */

    PARSE ARG srcfile
    srcfile = STRIP(srcfile)
    dstfile = SysTempFileName('testp?????.tmp')
    dummy = STREAM(dstfile, 'C', 'OPEN WRITE')
    CALL STREAM dstfile, 'C', 'CLOSE'
    BAKname = srcfile'.BAK'
    '@DEL 'BAKname' 2>NUL'
    '@RENAME 'srcfile' 'BAKname
    '@RENAME 'dstfile' 'srcfile
    RETURN

/****************************************************************/

RemoveApp: PROCEDURE

    /* Deletes an application and all of its keys.  */

    PARSE ARG srcfile,app
    srcfile = STRIP(srcfile)
    dstfile = SysTempFileName('testp?????.tmp')
    dummy = STREAM(srcfile, 'C', 'OPEN READ')
    CALL CopyOmittingApp srcfile,dstfile,app
    CALL STREAM srcfile, 'C', 'CLOSE'
    CALL STREAM dstfile, 'C', 'CLOSE'
    BAKname = srcfile'.BAK'
    '@DEL 'BAKname' 2>NUL'
    '@RENAME 'srcfile' 'BAKname
    '@RENAME 'dstfile' 'srcfile
    RETURN

/****************************************************************/
/*             FINDING THE CORRECT LINE IN A TNI FILE           */
/****************************************************************/

MoveToApp: PROCEDURE

    /* Copies srcfile to dstfile until we match the given app   */
    /* or reach the end of file.  If no match we create a new   */
    /* app.  On return the srcfile file pointer is at the first */
    /* line after the [app] line.  Normally returns 0, but      */
    /* returns 1 if a [/app] line must be appended.             */

    PARSE ARG srcfile,dstfile,app
    app = STRIP(app)
    DO FOREVER
        IF CHARS(srcfile) = 0 THEN DO
            CALL LINEOUT dstfile, '['app']'
            RETURN 1
        END
        line0 = LINEIN(srcfile)
        line = STRIP(line0)
        IF LENGTH(line) \= 0 THEN DO
            CALL LINEOUT dstfile, line0
            IF LEFT(line,1) \= '[' THEN DO
                 SAY "Unexpected line "line
                 RETURN 0
            END
            ELSE DO
                PARSE VAR line '['thisapp']'
                thisapp = STRIP(thisapp)
                IF thisapp = app THEN RETURN 0
                ELSE CALL SkipSection srcfile dstfile thisapp 0
            END
        END
    END

/****************************************************************/

CopyOmittingApp: PROCEDURE

    /* Copies all of srcfile to dstfile, omitting only the  */
    /* portion from [app] to [/app].                        */

    PARSE ARG srcfile,dstfile,app
    app = STRIP(app)
    skipping = 0
    DO WHILE CHARS(srcfile) > 0
        line0 = LINEIN(srcfile)
        line = STRIP(line0)
        IF LENGTH(line) \= 0 THEN DO
            IF skipping THEN DO
                IF LEFT(line,2) = '[/' THEN DO
                    PARSE VAR line '[/'thisapp']'
                    thisapp = STRIP(thisapp)
                    IF thisapp = app THEN skipping = 0 
                END
            END
            ELSE DO
                IF LEFT(line,1) = '[' THEN DO
                    PARSE VAR line '['thisapp']'
                    thisapp = STRIP(thisapp)
                    IF thisapp = app THEN skipping = 1
                    ELSE CALL LINEOUT dstfile, line0
                END
                ELSE CALL LINEOUT dstfile, line0
            END
        END
    END
    RETURN

/****************************************************************/

MoveToKey: PROCEDURE

    /* Copies srcfile to dstfile until we match the given key   */
    /* or reach the end of the current app.  On a match we      */
    /* return the line that matches (without copying it to      */
    /* dstfile).  If no match we return a null string.          */

    PARSE ARG srcfile,dstfile,app,key
    DO FOREVER
        IF CHARS(srcfile) = 0 THEN RETURN ""
        line0 = LINEIN(srcfile)
        line = STRIP(line0)
        IF LENGTH(line) = 0 THEN
            DO
                /* skip blank line */
            END
        ELSE IF LEFT(line,1) = '[' THEN
            DO
                IF SUBSTR(line,2,1) = '/' THEN
                    RETURN ""
                ELSE
                    DO
                        PARSE VAR line '['thiskey']'
                        thiskey = STRIP(thiskey)
                        IF thiskey = key THEN
                            RETURN line
                        ELSE DO
                            CALL LINEOUT dstfile,line0
                            CALL SkipSection srcfile dstfile thiskey 1
                        END
                    END
            END
        ELSE
            DO
                PARSE VAR line thiskey"="v
                thiskey = STRIP(thiskey)
                IF thiskey = key THEN
                    RETURN line
                ELSE CALL LINEOUT dstfile,line0
            END
    END

/****************************************************************/

SkipSection: PROCEDURE

    /* Copies srcfile to dstfile until we have passed the end   */
    /* of the specified application.                            */

    PARSE ARG srcfile dstfile app nested
    DO FOREVER
        IF CHARS(srcfile) = 0 THEN RETURN
        line = LINEIN(srcfile)
        CALL LineOut dstfile, line
        line = STRIP(line)
        IF LENGTH(line) = 0 THEN
            DO
                /* do nothing */
            END
        ELSE IF LEFT(line,1) = '[' THEN DO
            PARSE VAR line '['label']'
            label = STRIP(label)
            IF LEFT(label,1) = '/' THEN
                DO
                    label = DELSTR(label,1,1)
                    IF label = app THEN RETURN
                    ELSE SAY "Mismatched terminator [/"label"]"
                END
            ELSE IF nested = 0 THEN CALL Skipsection srcfile dstfile label 1
        END
    END

/****************************************************************/
/*                      MOVING TO END OF FILE                   */
/****************************************************************/

CopyRemainder: PROCEDURE

    /* Copies what is left in srcfile to dstfile.  */

    PARSE ARG srcfile,dstfile
    DO WHILE CHARS(srcfile) > 0
        line = LINEIN(srcfile)
        CALL LINEOUT dstfile, line
    END
    RETURN

/****************************************************************/

