/************************************************************************
 * os2md.c - OS/2 Mime decoder
 * Copyright (C) 1994, Richard Curry Consulting: trindflo@rain.org
 * All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file name 'COPYING'; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * SUMMARY:
 *    This file contains the code to decode a MIME encoded mail message.
 *    The logic for this program was taken from debase64.cmd by James L. Dean.
 *
 *    To build: See build.cmd: expects IBM CSET/2 compiler and MS assembler.
 *
 *    To use:  os2md file   <OR>
 *             os2md <file (stdin use)
 *
 * REVISION HISTORY:
 *
 *   Date          Version  By    Purpose of Revision
 * --------------- ------- -----  ---------------------------------------
 * Thu  94-07-28    1.00    RRC   Initial Draft
 *
 ************************************************************************/

/******************************************************************************/
/* Include files                                                              */
/******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <io.h>

/******************************************************************************/
/* Constants and macros                                                       */
/******************************************************************************/
#define TRUE  1
#define FALSE 0

/******************************************************************************/
/* External procedures                                                        */
/******************************************************************************/
int xlatblk(char *bufr, char *xlatab, int count);
int Pack4to3(char *cp, char *dp);

/******************************************************************************/
/* Define general variables                                                   */
/******************************************************************************/
/*----------------------------------------------------------------------------*/
/* dump_predicts: output all active predicts values.                          */
/*----------------------------------------------------------------------------*/
char inp_buf[65536];
char out_buf[65536];

char xlatab[256] = {                    /* Translate from Char to 6 bit value */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0x0? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0x1? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,62, 0, 0, 0,63,  /* 0x2? */
  52,53,54,55,56,57,58,59,60,61, 0, 0, 0, 0, 0, 0,  /* 0x3? */
   0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,  /* 0x4? */
  15,16,17,18,19,20,21,22,23,24,25, 0, 0, 0, 0, 0,  /* 0x5? */
   0,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,  /* 0x6? */
  41,42,43,44,45,46,47,48,49,50,51, 0, 0, 0, 0, 0,  /* 0x7? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0x8? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0x9? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0xa? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0xb? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0xc? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0xd? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 0xe? */
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };/* 0xf? */

char linebuf[1024];
char szInpFileName[1024];
char fFoundBody = 0;
char *cp, *dp;
int retval;
int read_count, write_count;
int last_line;

        /* Define file access structures */
FILE *inp_file;
FILE *out_file;

main(int argc, char *argv[]) {
  register long unsigned ii;

  if (argc == 1)
    inp_file=stdin;
  else {
    inp_file = fopen(argv[1], "r");
    if (inp_file == NULL)
    {
      printf("Can't open source file %s\n", argv[1]);
      exit(3);
    }
  }
  retval=setvbuf(inp_file, inp_buf, _IOFBF, sizeof(inp_buf) );
  if (retval!=0) errex("Unable to set input buffer\n");

  for (;;) {                            /* Until end of file                  */
    if (feof(inp_file)) break;

    if ( fgets(linebuf, sizeof(linebuf)-1, inp_file) == NULL ) {
      if (feof(inp_file)) break;
      else {
        perror("Error reading from input file");
        exit(3);
      }                                         
    }

    read_count = strlen(linebuf)-1;     /* Length of line less final '\n'     */
    if ( linebuf[read_count] != '\n' )
      errex("Too many characters per line in input\n");

    cp = strtok(linebuf, " \t\n\r");    /* Remove all spaces and odd chars    */
    if (cp == NULL) {                   /* If empty string resulted           */
      cp = linebuf;                     /* Make the pointer useful            */
      linebuf[0] = 0;
    }

    if (fFoundBody == 0) {              /* Just scanning 'contents' flags     */

      if (*cp == 0) continue;           /* Ignore empty lines now             */

      if (strstr(cp, "Content-Type:") == NULL) continue;  /* Find ID line     */

      cp = strtok(NULL, " \t\n\r");     /* Second field must have special str */
      if (cp == NULL)                   /* If the string is not there         */
        continue;                       /* Then this is not base64            */

          /* Try to disqualify some Content-types */
          /* Some legal ones in know about are: */
          /* "application/zip" */
          /* "image/jpeg" */
      strlwr(cp);                       /* make case insensitive              */
      if (strstr(cp, "multipart") != NULL) continue;  /* Not base64 (yet)     */
      if (strstr(cp, "mixed") != NULL) continue;
      if (strstr(cp, "text") != NULL) continue;
      if (strstr(cp, "plain") != NULL) continue;

      cp = strtok(NULL, " \t\n\r");     /* Third field might be the name      */
      if (cp != NULL)                   /* If the field exists                */
        cp = strstr(cp, "name=");       /* Locate the filename                */
      if (cp == NULL) {                 /* If the filename does not exist     */
        static char *sNoName="noname";

        cp = sNoName;                   /* Use a default filename             */
      }
      else {                            /* 'name=' identifier found           */
        cp += 5;                        /* Point past "name" string           */
        cp = strtok(cp, " \t\"");       /* Remove all separators and quotes   */
      }
      strcpy(szInpFileName, cp);        /* Make a copy of the filename        */

      fFoundBody++;                     /* Got out filename, next phase       */
      continue;                         /* No more processing of this line    */
    }

        /*  Here begins a state machine to interpret the remainder of the
            header such as it may be.  Note that fFoundBody, the state
            variable, != 0 at this point.                               */

    switch (fFoundBody) {               /* Based on what was in the header    */

      case 1 :
          /* The string 'base64' must be in the header somewhere */
          /* Here I try to avoid 'decoding' (scrambling) non-base64 files */

        if (  (*cp == 0) ||             /* Blank line is end of headers       */
              (strchr(cp, ':') == NULL) ||  /* Header lines all contain ':'   */
              (strstr(cp, "Content-Description:") != NULL)  /* Last header */
           )
        {
          fFoundBody = 0;               /* Not BASE64.  Start over.           */
          continue;                     /* No more processing of this line    */
        }

        for (;cp != NULL;) {            /* While tokens remain on line        */

          strlwr(cp);                   /* Check for our magic word           */
          if (strstr(cp, "base64") != NULL)
            break;                      /* Found the identifier               */

          cp = strtok(NULL, " \t\n\r"); /* Check the next token               */
        }

        if (cp == NULL)                 /* If empty string resulted           */
          continue;                     /* No more processing of this line    */

        fFoundBody++;                   /* Found MIME (base64) indicator      */
        last_line = 0;                  /* Arm line termination counter       */
        continue;                       /* No more processing of this line    */

      case 2 :
          /* Now we know base64 was intended.  The end of the header */
          /* is the begin of the body, standard apparently sloppy here */

        if (  (*cp == 0) ||             /* Blank line is end of headers       */
              (strchr(cp, ':') == NULL) ||  /* Header lines all contain ':'   */
              (strstr(cp, "Content-Description:") != NULL)  /* Last header */
           )
        {
          fFoundBody++;                 /* Found end of headers               */
          last_line = 0;                /* Arm line termination counter       */
        }

        else continue;

        if ( strstr(cp, "Content-Description:") != NULL) {  /* Check this     */

          for (;cp != NULL;) {          /* While tokens remain on line        */

            dp = strchr(cp, '<');       /* Is filename here in '<>' form?     */
            if ( dp != NULL) {
              char *ep;

              ep = strchr(cp, '>');
              if ( (ep > ++dp) && (ep-dp < sizeof(szInpFileName)) ) {
                memcpy(szInpFileName, dp, ep-dp);
                szInpFileName[ep-dp] = 0;
              }
            }
            cp = strtok(NULL, " \t\n\r"); /* Check the next token             */
          }
        }

            /* Now attempt to open the file we think we have */
        if ( access(szInpFileName, 0) != -1 ) { /* If the file already exists */
          printf("File %s already exists -- aborting\n", szInpFileName);
          exit(3);
        }

        out_file = fopen(szInpFileName, "wb");  /* Attempt to open destination*/
        if (out_file == NULL) {
          printf("Can't open destination %s -- aborting\n", szInpFileName);
          exit(3);
        }
        retval=setvbuf(out_file, out_buf, _IOFBF, sizeof(out_buf) );
        if (retval!=0) errex("Unable to set output buffer\n");

        printf("Writing file %s -- ", szInpFileName);

        continue;                       /* No more processing of this line    */

      case 3 :
          /* Now we are in the actual encoded binary.  This is where */
          /* you would strip garbage for multiline files.  */
          /* I currently ignore blank lines. */
          /* End of file is detected by the line which ends in '=' */

        if (*cp == 0) continue;         /* Ignore empty lines in body         */

        read_count = strlen(cp);        /* Length of true data                */
        if (read_count != 0)            /* Check for file end flags           */
          while ( cp[read_count-last_line-1] == '=' )
            last_line++;

        xlatblk(cp, xlatab, read_count);/* Xlate from ASCII-64 to binary      */
        read_count -= last_line;        /* Don't count garbage characters     */
        for (ii=0, dp=linebuf; ii<read_count; ii+=4, cp+=4, dp+=3)
          Pack4to3(cp, dp);             /* Pack 4 6-bit values into 3 8-bit   */

        switch (read_count & 3) {       /* Remove partial 'non' bytes         */
          case 1:                       /* Note: 1st case shouldn't happen    */
            dp--;                       /* 18 bits invalid ==> 3 bytes gone   */

          case 2:
            dp--;                       /* 12 bits missing ==> two bytes gone */

          case 3:
            dp--;                       /* 6 bits missing means one byte gone */

          case 0:                       /* Normal case...all bytes valid      */
            break;
        }
        read_count = dp-linebuf;        /* Compute total # chars generated    */

        write_count = fwrite(linebuf, 1, read_count, out_file);
        if ( write_count != read_count ) {
          perror("Error in write!");
          exit(3);
        }

        if (last_line!=0) {             /* Line termination chars seen        */
          fFoundBody = 0;               /* Back to square 1 -- no CRC check   */
          fclose(out_file);             /* Close file and notify user         */
          printf("File complete\n");
          continue;                     /* No more processing of this line    */
        }
        break;

      default :
        printf("Please report internal logic error # %d\n", fFoundBody);
        exit(3);

    } /* Switch (fFoundBody) */

  } /* Forever (read entire file) */
  fclose(inp_file);
}     /* of main */

/********************************************************/
/*                                                      */
/*  Function: errex                                     */
/*                                                      */
/*   Purpose: Prints an error message to the screen and */
/*              exits.                                  */
/*                                                      */
/********************************************************/

errex(char *mess_ptr) {
  printf(mess_ptr);
  exit(3);
}
