//
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//
//  strings.c : several string functions
//
//
//  norbert heller 1997 - 1998
//
#include <os2.h>
#include <stddef.h>
#include <strings.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>



//
//  Function    :  stralign
//
//  Description :  Copy an input string in an output string
//         with specified alignement (blank padding).
//
//  Decisions   :  If given length < 0 returns an empty string.
//
//  Parameters  :  out  char        *out_str     result
//                 in   char        *in_str      in string
//                 in   align_type  type,    type
//                 in   int         length       length to be copied
//
//  Value       :  type = { align_left, align_center, align_right }
//
//  Return code :   pointer to result.
//
char *stralign( char *out_str, const char *in_str, align_type type, int length )

{ int diff;
  char *ptr = out_str;

  if ( length <= 0 ) { *out_str = '\0';
               return out_str;
             }

  if ( (diff = (length - strlen(in_str)) ) <= 0 )
     return strleft( out_str, in_str,diff );

  switch( type ) { case align_left  : while ( *out_str++ = *in_str++ );
                                  out_str --;
                          for (; diff; diff -- ) *out_str++ = ' ';
                          *out_str++ = '\0';
                      break;

           case align_right : for (; diff; diff -- ) *out_str++ = ' ';
                      while ( *out_str++ = *in_str++ );
                      break;

               case align_center: { int lim = diff / 2;
                            for (; diff > lim; diff -- ) *out_str++ = ' ';
                            while ( *out_str++ = *in_str++ ); out_str --;
                            for (; diff; diff -- ) *out_str++ = ' ';
                            *out_str++ = '\0';
                            break;
                          }
             }

  return ptr;
}




//
//  Function    :  strcase
//
//  Description :  change case of a string.
//             All special characters are translated (...)
//
//  Parameters  :  in   char      *string  string to translate
//                 in   casetype  type     UPPER/LOWER
//
//  Value       :  type = { UPPER, LOWER }
//
//  Decisions   :  If character > 255, no change made.
//
//  Return      :  pointer to result
//
char *strcase( char *string, casetype type )

{ char *ptr;

  for ( ptr = string ; *ptr; ptr++ ) *ptr = chcase( *ptr, type );

  return string;
}





//
//  Function    :  chcase
//
//  Description :  change case of a 2-byte character.
//             All special characters are translated (...)
//
//  Parameters  :  in   char      car      char to translate
//                 in   casetype  type     UPPER/LOWER
//
//  Value       :  type = { UPPER, LOWER }
//
//  Decisions   :  If character > 255, no change made.
//
//  Return      :  code of char translated
//
int chcase( int car, casetype type )

{ if ( car > 255 ) return car;

  switch( car )
    {
      case '':
      case '':
      case '':
      case '':
      case '':
      case '': car = 'a'; break;

      case '':
      case '': car = 'A'; break;

      case '':
      case '':
      case '':
      case '': car = 'e'; break;

      case '': car = 'E'; break;

      case '':
      case '':
      case '':
      case '': car = 'i'; break;

      case '':
      case '':
      case '':
      case '':
      case '': car = 'o'; break;

      case '': car = 'O'; break;

      case '':
      case '':
      case '':
      case '': car = 'u'; break;

      case '': car = 'U'; break;

      case '': car = 'y'; break;

      case '': car = 'c'; break;

      case '': car = 'C'; break;

      case '': car = 'n'; break;

      case '': car = 'N'; break;
    }

  if ( type == UPPER ) return toupper(car);
  if ( type == LOWER ) return tolower(car);
  return car;
}





//
//  Function    :  strcomp
//
//  Description :  Compare two strings case-insensitive
//         All special characters are translated (...)
//
//  Parameters  :  in   char  *str1
//                 in   char  *str2
//
//  Return      :  -1 if str1 <  str2
//          1 if str1 >  str2
//          0 if str1 == str2
//
int strcomp( const char *str1, const char *str2 )
{

 for (; *str1 && *str2; str1++, str2++ )
      if ( chcase(*str1, UPPER) != chcase(*str2, UPPER) ) break;

  if ( chcase(*str1, UPPER) <  chcase(*str2, UPPER) ) return -1;
  if ( chcase(*str1, UPPER) >  chcase(*str2, UPPER) ) return  1;

  return 0;
}




//
// Function   : strdelblk
//
// Description   : Remove all spaces, tab and eol before and after a string.
//
char *strdelblk( char *string )

{ char *ptr = string, *str_begin = string, *str_end = NULL;

  // skip blanks
  ptr += strspn( string, " \t\n" );

  while ( 1 )
            { switch( *string++ = *ptr++ )
                      {
                        case '\0': break;

                        case ' ' :
                        case '\t':
                        case '\n': str_end = NULL; continue;
                        default  : if ( ! str_end ) str_end = string;
                                   continue;
                      }
                break;
              }

  if ( str_end ) *str_end = '\0';

  return str_begin;
}




//
//  Function    :   strelt
//
//  Description :   Return the n-th element (token) in a string.
//                  The tokens are separated by several input character.
//
//  Decisions   :   If tokens are separated by ' ', '\t' or '\n',
//                  the input characters will be " ".
//                  If n-th token does not exist, an empty string is returned.
//
//  Parameters  :   out    char  *out_str      out string
//          in      char  *in_str       in string
//          in      char  *token        string containing token
//          in      int   element       element number
//
//  Return      :   pointer to the n-th token.
//
char *strelt( char *out_str, const char *in_str, char *token, int element )
{
  char *ptr;

  strcpy( out_str, in_str );
  ptr = strtok( out_str, token );

  while ( (element > 1) && (ptr = strtok(NULL, token)) )
    element --;

  if ( ptr )
    strcpy( out_str, ptr );
  else
    *out_str = '\0';

  return out_str;

}




//
//  Function    :  strend
//
//  Description :  Returns a pointer to the '\0' ending a string
//
//  Parameters  :  in   char        *in_str      in string
//
//  Return code :   pointer to the '\0' ending the string..
//
char *strend( char *in_str )
{
#if 0
  while ( *in_str ++ );
  in_str --;
  return in_str;
#endif
  //   Should be quicker with built-in Run-Time Library (if well optimized)
  return in_str + strlen(in_str);
}




//
//  Function    :  hextostr
//
//  Description :  Convert an hexadecimal flow of bytes into
//                 a string containing an hexadecimal representation.
//
//  Parameters  :  out   char *string
//                 in    char *hexa
//                 in    int  length        number of hexadecimal bytes
//
//  Return      :   0 if OK
//                 -1 on error
//
int hextostr( char *string, const char *hexa, int length )
{
  while ( length-- )
        { sprintf( string, "%02X", (unsigned char) *hexa++ );
          string += 2;
        }

  return 0;
}





//
//  Function    :  strinsert
//
//  Description :  Insert a string in another.
//
//  Parameters  :  in   char   *out_str    out string
//                 in   char   *in_str     in string
//                 in   char   *to_insert  in string to insert into the other
//                 in   int    place       place to insert string
//
//  Decisions   :  Does nothing if place specified out of range.
//
//  Return      :  pointer to result
//
char *strinsert( char *out_str, const char *in_str, const char *to_insert, int place )

{ char *out_ptr = out_str, buffer[255], *ptr = buffer;

  if ( place <= 0 ) return ptr;

  if ( in_str == out_str ) { while( -- place ) if ( ! *out_str++ ) return(out_ptr);
                 while ( *ptr++ = *in_str++ );
                                                    // strcpy : used if in_str = out_str
                 ptr = buffer;
               }
              else { while( -- place ) if ( ! (*out_str++ = *in_str++) ) return(out_ptr);
                     ptr = (char *) in_str;
               }

  while( *out_str++ = *to_insert++ );
  out_str --;

  while ( *out_str++ = *ptr++ );

  return out_ptr;
}




//
//  Function    :  strisalnum
//
//  Description :  Tests if a string contains only alphanumerical character
//
//  Parameters  :  in   char   *string
//
//  Return      :  1 if string contains only alphanumerical characters.
//                 0 otherwise
//
int strisalnum( char *string )

{
  while ( isalnum(*string++) );

  return( ! *--string );
}




//
//  Function    :  strisalpha
//
//  Description :  Tests if a string contains only alphabetical character
//
//  Parameters  :  in   char   *string
//
//  Return      :  1 if string contains only alphabetical characters.
//                 0 otherwise
//
int strisalpha( const char *string )

{
  while ( isalpha(*string++) );

  return( ! *--string );
}




//
//  Function    :  stris_alnum
//
//  Description :  Tests if a string contains only alphanumerical characters or '_'.
//
//  Parameters  :  in   char   *string
//
//  Return      :  1 if string contains only alphanumerical characters or '_'.
//                 0 otherwise
//
int stris_alnum( const char *string )

{
  for (; *string; string ++ )
      if ( ! isalnum(*string) && (*string != '_') ) return 0;

  return 1;
}




//
//  Function    :  strleft
//
//  Description :  Copy the first ... characters of a string.
//         Like strncpy but add a '\0' at the end of the output string.
//
//  Decisions   :  If given length > string length : normal strcpy
//         If given length <= 0 returns an empty string.
//
//  Parameters  :  out  char  *out_str     result
//                 in   char  *in_str      in string
//                 in   int   length       length to be copied
//
//  Return code :   pointer to result.
//
char *strleft( char *out_str, const char *in_str, int length )

{ char *ptr = out_str;

  if ( length <= 0 )
        {
        *out_str = '\0';
        return out_str;
        }

  while ( (*out_str++ = *in_str++) && -- length );

  if ( !length )
        *out_str = '\0';

  return ptr;
}





//
//  Function    :  strmid
//
//  Description :  Copy n characters of a string, begining at a given position
//         ( form 1 to ... )
//
//  Decisions   :  Stops at the end of input string if given length
//         is too big or length = 0.
//         If given length or position < 0 returns an empty string.
//
//  Parameters  :  out  char  *out_str     result
//                 in   char  *in_str      in string
//                 in   int   pos          position where begin to copy
//                 in   int   length       length to be copied
//
//  Return code :   pointer to result.
//
char *strmid( char *out_str, const char *in_str, int pos, int length )
{

if ( (length < 0) || (pos <= 0)  ) { *out_str = '\0';
                      return(out_str);
                    }

for (; pos; pos -- ) if ( ! *in_str++ ) { *out_str = '\0';
                        return(out_str);
                      }
in_str --;

return( strleft(out_str,in_str,length) );
}




//
//  Function    :  strmvchr
//
//  Description :  Replace all occurences of a target character
//                 by a new character.
//
//  Parameters  :  in/out   char  *string
//                 in       char  target     target char to replace
//                 in       char  new_char   char to put in place of target
//
//  Return code :  pointer to result.
//
char *strmvchr( char *string, char target, char new_char )

{ char *ptr;

  for ( ptr = string; *ptr ; ptr ++ )
      if ( *ptr == target ) *ptr = new_char;

  return string;
}




//
//  Function    :  strmvstr
//
//  Description :  Copy an input string in an output string
//                 with replacing all occurences of a target string.
//
//  Parameters  :  out      char  *out_str    result
//                 in       char  *in_str     in string
//                 in       char  *target     target string to replace
//                 in       char  *new_str    string to put in place of target
//
//  Return code :  pointer to result.
//
char *strmvstr( char *out_str, const char *in_str, const char *target, const char *new_str )
{
  char *ptr = out_str;
  const char *ptr_new = new_str;

  unsigned count;

  for ( count = 0; *out_str = *in_str; out_str ++, in_str ++ )
      if ( *out_str == *target )
        {
        target ++;
        count ++;
            if ( ! *target )
               {
                out_str -= count - 1;
                target -= count;

                count = 0;

                while ( *out_str++ = *new_str++ );
                    new_str = ptr_new;

                out_str --;
                out_str --;
                }
        }
        else
        {
        target -= count;
        out_str -= count;
        in_str -= count;
        count = 0;
        }

  return ptr;
}




//
//  Function    :  strncomp
//
//  Description :  Compare n characters of two strings case-insensitive
//         All special characters are translated (...)
//
//  Parameters  :  in   char  *str1
//                 in   char  *str2
//
//  Return      :  -1 if str1 <  str2
//          1 if str1 >  str2
//          0 if str1 == str2
//
int strncomp( const char *str1, const char *str2, int length )
{


  for (; length && *str1 && *str2; length --, str1++, str2++ )
      if ( chcase(*str1, UPPER) != chcase(*str2, UPPER) ) break;

  if ( ! length ) return 0;

  if ( chcase(*str1, UPPER) <  chcase(*str2, UPPER) ) return -1;
  if ( chcase(*str1, UPPER) >  chcase(*str2, UPPER) ) return  1;

  return 0;
}




//
//  Function    :  strreduce
//
//  Description :  Parse a string to suppress blanks, tabs and newlines
//                 and transform it in uppercase.
//
//  Decisions   :  - Portions of string between double quotes are unmodified
//                   (quotes are removed).
//
//                 - '\' is used to quote the next character.
//
//                 - Characters with accent are also transformed to uppercase.
//
//  Parameters  :  in/out   char *string
//
//  Return      :  pointer to end of string ( '\0' ).
//
char *strreduce( char *string )
{
  char *ptr;

  //  Suppress quotes and put in uppercase
  for ( ptr = string; *string; string ++ )
      switch( *string )
            {
              case '"' : string ++;
                         while ( *string && *string != '"' ) *ptr++ = *string++;
                     if ( *string == '"' && *(++string) == '"' ) *ptr++ = '"';
                     break;

              case ' ' :
              case '\t':
              case '\n': break;

              case '\\': *ptr++ = *string;

              default  : *ptr++ = chcase( *string, UPPER );
                         break;
            }

  *ptr = '\0';
  return --ptr;
}




//
//  Function    :  strright
//
//  Description :  Copy the last ... characters of a string.
//
//  Decisions   :  If given length > string length : normal strcpy
//         If given length <= 0 returns an empty string.
//
//  Parameters  :  out  char  *out_str     result
//                 in   char  *in_str      in string
//                 in   int   length       length to be copied
//
//  Return code :   pointer to result.
//
char *strright( char *out_str, const char *in_str, int length )

{ const char *ptr = in_str;

  if ( length <= 0 ) { *out_str = '\0';
               return out_str;
             }

  while ( *ptr++ );

  if ( ptr - length - 1 > in_str ) in_str = ptr - length - 1;

  return strcpy( out_str, in_str );
}




//
//  Function    :  strrmchr
//
//  Description :  Removing all occurences of a target character.
//
//  Parameters  :  in/out   char  *string
//                 in       char  target     target char to remove
//
//  Return code :  pointer to result.
//
char *strrmchr( char *string, char target )

{ char *ptr, *str_begin = string;

  for ( ptr = string; *string; string ++ )
      if ( *string != target ) *ptr++ = *string;

  *ptr = '\0';

  return str_begin;
}




//
//  Function    :  strrmstr
//
//  Description :  Removing all occurences of a target string.
//
//  Parameters  :  in/out   char  *string
//                 in       char  *target     target string to remove
//
//  Decisions   :  Same implementation as strmvstr without copying
//         a replacement string.
//         Could also be implemented as
//                        strmvstr( ptr, ptr, target, "" )
//         but should be less efficient.
//
//  Return code :  pointer to result.
//
char *strrmstr( char *string, const char *target )

{ char *ptr, *out;
  unsigned count = 0;

  for ( out = ptr = string; *out = *ptr; out++, ptr ++ )
      if ( *ptr == *target ) { target ++; count ++;
                               if ( ! *target )
                      { out -= count;
                    target -= count;
                  }
                  }
             else { target -= count;
                    ptr -= count;
                out -= count;
                count = 0;
                  }

  return string;
}


//
//  Function    :   strskip
//
//  Description :   Skip 'n' words from a string.
//
//  Decisions   :
//
//  Parameters  :   in    char    *string     string to be matched
//                  in    int     word_nb     number of words to skip
//
//  Return code :   pointer to word.
//
char *strskip( char *string, int word_nb )

{ char *ptr;

  while ( *string == ' ' || *string == '\n' || *string == '\t' ) string++;

  for ( ; word_nb; word_nb-- )
      {
        ptr = strpbrk( string, " \n\t" );
        if ( ptr ) string = ptr + strspn( ptr, " \n\t" );
              else while ( *string ) string++;
      }

return string;
}


//
//  Function    :  strcpyb
//
//  Description :  Like strcpy but truncates trailing blanks and \n
//                 and skip leading blanks.
//
//  Parameters  :  out   char * target
//         in    char * source
//
//  Return code :  like strcpy
//
//  OS/Compiler :  All
//
//
char *strcpyb( char *target, const char *source )

{ int len;

  while ( *source++ == ' ' ); source--;

  len = strlen( source );
  while ( len && (source[len - 1] == ' ' || source[len - 1] == '\n') ) len--;
  strncpy( target, source, len );
  target[len] = '\0';

  return target;
}





//
//  Function    : strncpyb
//
//  Description :  Like strncpy but truncates trailing blanks and \n,
//                 skip leading blanks and adds a '\0'.
//
//  Decisions   :
//
//  Parameters  :  out   char * target
//             in    char * source
//                 in    int    length
//
//  Return code :  like strncpy
//
//  Side-effects:  Pad target string to length with '\0'.
//
//  OS/Compiler :  All
//

char *strncpyb( char *target, const char *source, int length )

{ char *lastnonblank = target - 1, *ptr;
  int offs = 0;

  while ( *source++ == ' ' ) offs++; source--;
  length -= offs;

  for ( ptr = target; (length > 0) && *source; length-- )
      {
        if ( *source != ' ' && *source != '\n' )
           lastnonblank = ptr;
        *ptr++ = *source++;
      }

  memset( lastnonblank + 1, '\0', ptr - lastnonblank + length + offs );

  return target;
}




//
// Holt ein token aus einem Satz
//
// Source   :   "wert1;wert2;wert3;wert4; ... ;wertx"
// Target   :   das gewuenschte Token "wert4"
// Separator:   der Token Separator ";"
// iToken   :   welches Token soll geholt werden ?
//
// bRight   :   TRUE:  Token soll von rechts geholt werden
//              FALSE: Token soll von links geholt werden
//
char *get_token(char *Target, char *Source, char *Separator, int iToken, BOOL bRight)
{
    char *ep, *ap;
    int iSepCounter=0;
    int i;

    CHAR szT[300];
    CHAR szSep[10];

    CHAR szT1[300];


// ---------------------------------------------------

    // return buffer leeren
    strcpy(Target, "");
    strcpy(szT, Source);
    strcpy(szSep, Separator);

//    sprintf(szSep, "%c", Separator);

    /*
    for ( ep = Source; *ep ; ep ++ )
        if ( *ep == Separator )
          iSepCounter++;
    */

    // erst mal suchen, wieviele Trenner drin sind
    ep = strstr(szT, szSep);
    do
    {
    if(ep != NULL)
        {
        ep ++;
        iSepCounter++;
        // und wieder zurueck kopieren
        strcpy(szT, ep);
        ep = strstr(szT, szSep);

        }
    }
    while(ep != NULL);


    // oder von rechts ?? dann von rechts abzahlen
    if(bRight)
        iToken = iSepCounter + 2 - iToken;

    // nun den wert ausgeben
    // wert1;wert2 = separator counter + 1

    // welches token, falls zuviele , dann das letzte ..
    if( iToken > iSepCounter + 1)
        iToken = iSepCounter + 1;

    // zuwenig ??
    if( iToken < 1)
        iToken = 1;

    // string reinkopieren
    strcpy(szT, Source);

    for(i = 1; i < iToken; i++)
        {
        ep = strstr(szT, szSep);
        ep ++;

        // wieder zurueck
        strcpy(szT, ep);
        }

    ep = strstr(szT, szSep);

    // falls nicht das letzte
    if(ep != NULL)
      strleft(Target, szT, strlen(szT) - strlen(ep));
    else
    // doch das letzte ??
     strcpyb(Target, szT);

    strrmstr(Target, szSep);

    return Target;
}



int get_number_of_tokens(char *Source, char *Separator)
{
    char *ep, *ap;
    int iSepCounter=0;

    int i;

    CHAR szT[200];
    CHAR szSep[3];

// ---------------------------------------------------

    // return buffer leeren
    strcpy(szT, Source);
    strcpy(szSep, Separator);


    // length > 0 ??
    if(strlen(szT) > 0)
    {
       // erst mal suchen, wieviele Trenner drin sind
       ep = strstr(szT, szSep);
       do
       {
       if(ep != NULL)
           {
           ep ++;
           iSepCounter++;
           // und wieder zurueck kopieren
           strcpy(szT, ep);
           ep = strstr(szT, szSep);

           }
       }
       while(ep != NULL);


       // one more token than separators
       iSepCounter++;

       return iSepCounter ;
    }
    else
        return 0;

}








//
//  Function    :   strtohex
//
//  Description :   convert a string containing an hexadecimal
//                  representation into an hexadecimal flow of bytes.
//
//  Parameters  :   char *string   in/out
//
//  Return      :   length of hexadecimal flow of bytes.
//                  -1 on error
//
//  OS/Compiler :   All
//

int strtohex( char *string )

{ int len = 0;

  char *ptr = string, buffer[2];

  // if string length is even, first char is a full number
  if ( strlen(string) % 2 ) { if ( sscanf(ptr, "%01x", buffer) != 1 ) return -1;
                              *ptr++ = *buffer;
                              len = 1;
                            }

  while ( *ptr )
        { if ( sscanf( ptr, "%02x", string + len++) != 1 ) return -1;
          ptr += 2;
        }

  return len;
}



//
//  Function: Cut file name extension from specified path, if it exist.
//  Arguments: lpPath = Path to cut.
//  Returns: A pointer to lpPath.
//  Calling : CutExtFromFile (Lines.cIniPath);

char* CutExtFromFile ( char* lpPath )
{
   char* lpPath_;

   for (lpPath_ = lpPath + strlen (lpPath);
        lpPath_ > lpPath && *lpPath_ != '\\';
        lpPath_--)
   {
      if (*lpPath_ == '.')
      {
         if (*(lpPath_ - 1) != '\\' &&
             *(lpPath_ - 1) != '.')
         {
            *lpPath_ = '\0';
            break;
         }
         break;
      }
   }

   return lpPath;
}



//
// copies txt into htm code
//
char *txt2htm(char *Target, char *Source)
{
    CHAR szT[500], szT1[500];

// ---------------------------------------------------

    // return buffer leeren
    strcpy(Target, "");
    strcpy(szT, Source);

    //
    strmvstr(szT1, szT, "&", "&amp;" );
    strcpy(szT, szT1);

    // umlaute ersetzen
    strmvstr(szT1, szT, "", "&auml;" );
    strcpy(szT, szT1);
    strmvstr(szT1, szT, "", "&ouml;" );
    strcpy(szT, szT1);
    strmvstr(szT1, szT, "", "&uuml;" );
    strcpy(szT, szT1);

    strmvstr(szT1, szT, "", "&Auml;" );
    strcpy(szT, szT1);
    strmvstr(szT1, szT, "", "&Ouml;" );
    strcpy(szT, szT1);
    strmvstr(szT1, szT, "", "&Uuml;" );
    strcpy(szT, szT1);

    strmvstr(szT1, szT, "", "&szlig;" );
    strcpy(szT, szT1);

    strmvstr(szT1, szT, "<", "&lt;" );
    strcpy(szT, szT1);

    strmvstr(szT1, szT, ">", "&gt;" );
    strcpy(szT, szT1);

    // und zurueckgeben
    strcpy(Target, szT);
    return Target;
}




