/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          capturem.c
** SYSTEM NAME:        beholder
** MODULE NAME:        capture
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/9/3
** DESCRIPTION:        Packet capture group of the RMON MIB.
**                     each function is preceeded by bufferControl:
**                         ControlIndex, ChannelIndex, FullStatus, 
**                         FullAction, CaptureSliceSize, DownloadSliceSize,
**                         DownloadOffset, MaxOctetsRequested,
**                         MaxOctetsGranted, CapturedPackets, TurnOnTime,
**                         Owner, Status.
**                     each function is preceeded by captureBuffer:
**                          ControlIndex, Index, PacketID, PacketData,
**                          PacketLength, PacketTime, PacketStatus.
*************************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <snmp.h>
#include <mibsup.h>
#include <protocol.h>
#include <message.h>
#include <maxmem.h>

#include "capturee.h"
#include "capturec.h"
#include "capture.h"


#define BUFFERSIZE   1
#define CAPTURESIZE  2

#define CAPTURE_FIRST 1
#define CAPTURE_GET   2
#define CAPTURE_NXT   3


static MIB_LOCAL    *buffer = NULL;


EXPORT WORD         memoryPerc = 75;


static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, void *Elm);

static CAPTURE * CaptureSearch (WORD Request, SNMP_OBJECT *Obj, MIB_LOCAL *Local, WORD IdLen);



BOOLEAN CaptureMInit(VOID)
{
    MessageConfig(CAPTURE_ERROR, "Capture");
    return CaptureConfigInit();
}


/*****************************************************************
** NAME:        RmonNext
** SYNOPSIS:    BOOLEAN RmonNext (SNMP_OBJECT *Obj,
**                                  MIB_LOCAL **Local, WORD IdLen,
**                                  WORD IdSze, VOID **Elm)
** PARAMETERS:  Obj: requested object
**              Local: local datastructure: a collector
**              IdLen: identifier length known to the SNMP
**              IdSze: size part application of identifier
**                     including the collector index !!
**              Elm: pointer to object if tables are used
** DESCRIPTION: application specific RMON NEXT function.
**              only called after MibRmon() for next requests 
**              searches the next object in the collector
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     TRUE: object found
**                    OPTIONAL: Elm -> pointer to object in table
*******************************************************************/

static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, VOID **Elm)
{
    MIB_LOCAL    *local = *Local;

    if (local == NULL)
        return FALSE;

    if (Obj->IdLen == IdLen || local->Index > Obj->Id[IdLen])
    {
        if (IdSze == CAPTURESIZE &&
            (*Elm = CaptureSearch (CAPTURE_FIRST, Obj, local, IdLen)) == NULL)
        {
            *Local = local->Next;
            return (RmonNext(Obj, Local, IdLen, IdSze, Elm));
        }
        Obj->Id[IdLen] = local->Index;
        Obj->IdLen = IdLen + IdSze;
        return TRUE;
    }
    if (IdSze == CAPTURESIZE &&
        (*Elm = CaptureSearch (CAPTURE_NXT, Obj, local, IdLen)) != NULL)
        return TRUE;

    *Local = local->Next;
    return (RmonNext (Obj, Local, IdLen, IdSze, Elm));
}



/*****************************************************************
** NAME:        CaptureSearch
** SYNOPSIS:    CAPTURE * CaptureSearch (WORD Request, SNMP_OBJECT *Obj, MIB_LOCAL *local, WORD IdLen)
** PARAMETERS:  Request: CAPTURE_GET = get index request
**                       CAPTURE_FIRST = get first index request
**                       CAPTURE_NXT = get next index request
**              *Obj: requested object
**              *local: local datastructure: a collector
**              IdLen: identifier length known to the SNMP
** DESCRIPTION: capture get/first/next function.
**              searches dependent on Request the requested, the
**              first, or next object in the capture buffer.
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     pointer to the data of the found object
**              NULL: object was not found
*******************************************************************/

CAPTURE * CaptureSearch (WORD Request, SNMP_OBJECT *Obj, MIB_LOCAL *local, WORD IdLen)
{
    CAPTURE  * ptr = ((BUFFER *)local->Data)->FirstPkt;
    LONG     ind = 1;

    if (ptr == NULL)
        return NULL;

    switch (Request)
    {
        case CAPTURE_GET:
            while (ptr->Len != 0)
            {
                if (ind == Obj->Id[Obj->IdLen - 1])
                    return ptr;
                ptr = (CAPTURE *)((BYTE *) ptr + ptr->Len + sizeof (CAPTURE));
                if (ptr->Len == -1)
                    ptr = (CAPTURE *) ((BUFFER *)local->Data)->Capture;
                ind++;
            }
            break;
        case CAPTURE_FIRST:
            if (ptr->Len != 0)
                Obj->Id[IdLen + 1] = ind;
            else
                ptr = NULL;
            break;
        case CAPTURE_NXT:
            if (ptr->Len != 0)
            {
                if (Obj->IdLen < IdLen + CAPTURESIZE)
                {
                    Obj->IdLen = IdLen + CAPTURESIZE;
                    Obj->Id[IdLen] = local->Index;
                    Obj->Id[IdLen + 1] = ind;
                    return ptr;
                }
                while (ptr->Len != 0)
                {
                    if (ind > Obj->Id[IdLen + 1])
                    {
                        Obj->Id[IdLen] = local->Index;
                        Obj->Id[IdLen + 1] = ind;
                        return ptr;
                    }
                    ptr = (CAPTURE *)((BYTE *) ptr + ptr->Len + sizeof (CAPTURE));
                    if (ptr->Len == -1)
                        ptr = (CAPTURE *) ((BUFFER *)local->Data)->Capture;
                    ind++;
                }
            }
            ptr = NULL;
            break;
    }
    return ptr;
}


WORD bufferControlIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);
    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            Obj->Syntax.LngInt = local->Index;
            return SNMP_NOERROR;
    }
}


WORD bufferControlChannelIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->ChannelIndex;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            data->ChannelIndex = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD bufferControlFullStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->FullStatus;
            return SNMP_NOERROR;
    }
}

WORD bufferControlFullAction (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->FullAction;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            if (Obj->Syntax.LngInt != 1 && Obj->Syntax.LngInt != 2)
                return SNMP_BADVALUE;
            data->FullAction = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD bufferControlCaptureSliceSize (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->CaptureSliceSize;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            data->CaptureSliceSize = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD bufferControlDownloadSliceSize (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->DownloadSliceSize;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            if (Obj->Syntax.LngInt > SNMP_SIZE_BUFCHR)
                return SNMP_BADVALUE;
            data->DownloadSliceSize = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD bufferControlDownloadOffset (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->DownloadOffset;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            data->DownloadOffset = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}


WORD bufferControlMaxOctetsRequested (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    BUFFER      *data;
    BYTE        *junk = NULL;
    CAPTURE     *capturePtr, *junkFirst, *junkLast;
    LONG        octetsCopied;
    BOOLEAN     memoryCheck;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->MaxOctetsRequested;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            if (Obj->Syntax.LngInt == -1)
            {
                octetsCopied = (memoryPerc * MemoryGetAvail()) / 100;
                if (octetsCopied > (LONG) MemoryGetMaxChunk())
                    octetsCopied = (LONG) MemoryGetMaxChunk();
                if (octetsCopied <= data->MaxOctetsAvailable)
                    return SNMP_NOERROR;
                if ((junk = DnpapMalloc (octetsCopied)) == NULL)
                    return SNMP_GENERROR;
                data->MaxOctetsRequested = -1;
                data->MaxOctetsGranted = -1;
                data->MaxOctetsAvailable = octetsCopied;
            }
            else
            {
                if ((octetsCopied = (memoryPerc * MemoryGetAvail()) / 100) > Obj->Syntax.LngInt)
                    octetsCopied = Obj->Syntax.LngInt;
                if (octetsCopied > (LONG) MemoryGetMaxChunk())
                    octetsCopied = (LONG) MemoryGetMaxChunk();
                if ((junk = DnpapMalloc (octetsCopied)) == NULL)
                    return SNMP_GENERROR;
                data->MaxOctetsRequested = Obj->Syntax.LngInt;
                data->MaxOctetsGranted = octetsCopied;
                data->MaxOctetsAvailable = octetsCopied;
            }

            memset (junk, 0, (SHORT) octetsCopied);
            junkFirst = (CAPTURE *) junk;
            junkLast = (CAPTURE *) junk;
            capturePtr = data->FirstPkt;
            data->CapturedPkts = 0;
            data->MaxOctetsFree = data->MaxOctetsAvailable;
            data->FullStatus = 1;

            while (capturePtr->Len != 0)                     /* last packet */
            {
                octetsCopied = capturePtr->Len + sizeof (CAPTURE);
                memoryCheck = TRUE;

                if (data->MaxOctetsFree < octetsCopied + sizeof (SHORT))
                {                                                   /* wrap */
                    data->FullStatus = 2;
                    if (data->MaxOctetsAvailable < octetsCopied + sizeof (SHORT))
                    {
                        memoryCheck = FALSE;
                        data->CapturedPkts = 0;
                        memset (junk, 0, (SHORT) data->MaxOctetsAvailable);
                        junkFirst = (CAPTURE *) junk;
                        junkLast = (CAPTURE *) junk;
                        data->MaxOctetsFree = data->MaxOctetsAvailable;
                    }
                    else
                    {
                        if (junkFirst > junkLast)
                        {
                            while (1)
                            {
                                junkFirst = (CAPTURE *)((BYTE *) junkFirst + junkFirst->Len + sizeof (CAPTURE));
                                data->CapturedPkts--;
                                if (junkFirst->Len == -1)
                                {
                                    junkFirst = (CAPTURE *) junk;
                                    break;
                                }
                            }
                        }
                        junkLast->Len = -1;
                        junkLast = (CAPTURE *) junk;
                        data->MaxOctetsFree = data->MaxOctetsAvailable;
                    }
                }

                if (memoryCheck == TRUE)
                {
                    if (junkFirst->Len != 0 && junkFirst >= junkLast)
                    {
                        while ((BYTE *) junkFirst - (BYTE *) junkLast < (SHORT) octetsCopied + sizeof(SHORT))
                        {
                            junkFirst = (CAPTURE *)((BYTE *) junkFirst + junkFirst->Len + sizeof (CAPTURE));
                            data->CapturedPkts--;
                            if (junkFirst->Len == -1)
                            {
                                junkFirst = (CAPTURE *) junk;
                                break;
                            }
                        }
                    }
                    memcpy (junkLast, capturePtr, (SHORT) octetsCopied);
                    junkLast = (CAPTURE *)((BYTE *) junkLast + octetsCopied);
                    junkLast->Len = 0; 
                    data->CapturedPkts++;
                    data->MaxOctetsFree -= octetsCopied;
                }
                capturePtr = (CAPTURE *)((BYTE *) capturePtr + octetsCopied);
                if (capturePtr->Len == -1)
                    capturePtr = (CAPTURE *) data->Capture;
            }
            DnpapFree (data->Capture);
            data->Capture = junk;
            data->FirstPkt = junkFirst;
            data->LastPkt = junkLast;
            return SNMP_NOERROR;
    }
}

WORD bufferControlMaxOctetsGranted (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->MaxOctetsGranted;
            return SNMP_NOERROR;
    }
}

WORD bufferControlCapturedPackets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->CapturedPkts;
            return SNMP_NOERROR;
    }
}

WORD bufferControlTurnOnTime (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon (Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->TurnOnTime;
            return SNMP_NOERROR;
    }
}

WORD bufferControlOwner (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    if ((local = MibRmon(Obj, buffer, IdLen, BUFFERSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (BUFFER*) local->Data;
            memcpy (Obj->Syntax.BufChr, data->Owner, data->OwnerLen);
            Obj->SyntaxLen = data->OwnerLen;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (BUFFER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            memcpy (data->Owner, Obj->Syntax.BufChr, Obj->SyntaxLen);
            data->OwnerLen = Obj->SyntaxLen;
            return SNMP_NOERROR;
    }
}

WORD bufferControlStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *data;

    local = MibRmon(Obj, buffer, IdLen, BUFFERSIZE);

    switch (Obj->Request)
    {
        case SNMP_PDU_GET:
            if (local == NULL)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, BUFFERSIZE, NULL) == FALSE)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (BUFFER*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            if (local == NULL)
            {
                switch (Obj->Syntax.LngInt)
                {
                    case SNMP_CREATEREQUEST:
                        if ((local = MibInsert(Obj, &buffer, IdLen, BUFFERSIZE)) == NULL)
                            return SNMP_GENERROR;
                        if ((local->Data = DnpapMalloc(sizeof(BUFFER))) == NULL)
                            return SNMP_GENERROR;
                        data = (BUFFER*) local->Data;
                        memset(data, 0, sizeof(BUFFER));
                        if (CaptureInit(data) == TRUE)
                        {
                            data->Status = SNMP_UNDERCREATION;
                            DnpapMessage(DMC_MESSAGE, CAPTURE_CREATE, "capture: collector %ld created", local->Index);
                            return SNMP_NOERROR;
                        }
                        DnpapFree(local->Data);
                        MibRemove (Obj, &buffer, IdLen, BUFFERSIZE);
                        return SNMP_GENERROR;
                    default:
                        return SNMP_NOSUCHNAME;
                }
            }
            data = (BUFFER*) local->Data;
            switch (data->Status)
            {
                case SNMP_UNDERCREATION:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_VALID:
                            if (CaptureStart(data) == TRUE)
                            {
                                data->Status = SNMP_VALID;
                                DnpapMessage(DMC_MESSAGE, CAPTURE_ACTIVE, "capture: collector %ld active", local->Index);
                                return SNMP_NOERROR;
                            }
                            else
                                return SNMP_GENERROR;
                        case SNMP_INVALID:
                            if (CaptureStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, CAPTURE_DESTROY, "capture: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &buffer, IdLen, BUFFERSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
                case SNMP_VALID:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_INVALID:
                            if (CaptureStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, CAPTURE_DESTROY, "capture: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &buffer, IdLen, BUFFERSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        case SNMP_VALID:
                            return SNMP_NOERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
            }
    }
}


WORD captureBufferControlIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    CAPTURE    *data = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);
    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Obj->Syntax.LngInt = local->Index;
            return SNMP_NOERROR;
    }
}

WORD captureBufferIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CAPTURE     *data = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Obj->Syntax.LngInt = Obj->Id[Obj->IdLen-1];
            return SNMP_NOERROR;
    }
}

WORD captureBufferPacketID (SNMP_OBJECT *Obj, WORD IdLen) 
{
    MIB_LOCAL  *local = NULL;
    CAPTURE    *data = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Obj->Syntax.LngInt = data->ID;
            return SNMP_NOERROR;
    }
}

WORD captureBufferPacketData (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    BUFFER     *Buffer;
    CAPTURE    *data = NULL;
    BYTE       *octetsPtr;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Buffer = (BUFFER*) local->Data;
            octetsPtr = (BYTE *) data + sizeof (CAPTURE);

            if (Buffer->DownloadOffset + Buffer->DownloadSliceSize <= (LONG) data->Len)
            {
                Obj->SyntaxLen = (WORD) Buffer->DownloadSliceSize;
                memcpy (Obj->Syntax.BufChr, octetsPtr + Buffer->DownloadOffset, Obj->SyntaxLen);
            }
            else if (Buffer->DownloadOffset <= (LONG) data->Len)
            {
                Obj->SyntaxLen = (WORD) data->Len - (WORD) Buffer->DownloadOffset;
                memcpy (Obj->Syntax.BufChr, octetsPtr + Buffer->DownloadOffset, Obj->SyntaxLen);
            }
            else
                Obj->SyntaxLen = 0;
            return SNMP_NOERROR;
    }
}

WORD captureBufferPacketLength (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    CAPTURE    *data = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Obj->Syntax.LngInt = (LONG) data->Len;
            return SNMP_NOERROR;
    }
}

WORD captureBufferPacketTime (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    CAPTURE     *data = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Obj->Syntax.LngInt = data->Time;
            return SNMP_NOERROR;
    }
}

WORD captureBufferPacketStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL  *local = NULL;
    CAPTURE    *data = NULL;

    if ((local = MibRmon (Obj, buffer, IdLen, CAPTURESIZE)) == NULL)
        return(SNMP_NOSUCHNAME);

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, CAPTURESIZE, &data) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            if (data == NULL)
                if ((data = CaptureSearch (CAPTURE_GET, Obj, local, IdLen)) == NULL)
                    return(SNMP_NOSUCHNAME);
            Obj->Syntax.LngInt = 0;
            return SNMP_NOERROR;
    }
}


/*****************************************************************
** NAME:        CapturePkt
** SYNOPSIS:    VOID CapturePkt (LONG channelIndex,
**                  MAC_FRAME FAR *frame)
** PARAMETERS:  channelIndex: the index of the channel that calls
**                  this function
**              frame: the frame received from the channel
** DESCRIPTION: searches all buffer instances and captures packet.
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     VOID
*******************************************************************/

VOID CapturePkt (LONG channelIndex, PROT_PKT *pkt)
{
    MIB_LOCAL   *local;
    BUFFER      *data;
    LONG        octetsCopied;
    PROT_OBJ    len =   {0, {1,5}};
    PROT_OBJ    time =  {0, {1,6}};
    PROT_OBJ    packet= {0, {1,7}};

    if (ProtGetField(pkt, &len) == FALSE ||
        ProtGetField(pkt, &time) == FALSE ||
        ProtGetField(pkt, &packet) == FALSE) 
    {
        DnpapMessage(DMC_ERROR, CAPTURE_BADPACKET, "capture: bad packet");
        return;
    }

    for (local = buffer; local != NULL; local = local->Next)
    {
        data = (BUFFER*) local->Data;
        if (data->Status == 1 && data->ChannelIndex == channelIndex)
        {
            if (data->FullAction == 1 && data->FullStatus == 2) /* full & lock */
                continue;
            if (data->CaptureSliceSize < len.Syntax.LngInt) /* nr of octets */
                octetsCopied = data->CaptureSliceSize;
            else
                octetsCopied = len.Syntax.LngInt;

            if (data->MaxOctetsFree < octetsCopied + sizeof (CAPTURE) + sizeof (SHORT))
            {                                           /* not enough space */
                data->FullStatus = 2;
                if (data->FullAction == 1)                          /* lock */
                    continue;
                if (data->MaxOctetsAvailable < octetsCopied + sizeof (CAPTURE) + sizeof (SHORT))
                    continue;
                if (data->FirstPkt > data->LastPkt)                 /* wrap */
                {
                    while (1)
                    {
                        data->FirstPkt = (CAPTURE *)((BYTE *) data->FirstPkt + data->FirstPkt->Len + sizeof (CAPTURE));
                        data->CapturedPkts--;
                        if (data->FirstPkt->Len == -1)
                        {
                            data->FirstPkt = (CAPTURE *) data->Capture;
                            break;
                        }
                    }
                }
                data->LastPkt->Len = -1;
                data->LastPkt = (CAPTURE *) data->Capture;
                data->MaxOctetsFree = data->MaxOctetsAvailable;
            }

            if (data->FirstPkt->Len != 0 && data->FirstPkt >= data->LastPkt)
            {
                while ((BYTE *) data->FirstPkt - (BYTE *) data->LastPkt < (SHORT) octetsCopied + sizeof (CAPTURE) + sizeof(SHORT))
                {
                    data->FirstPkt = (CAPTURE *)((BYTE *) data->FirstPkt + data->FirstPkt->Len + sizeof (CAPTURE));
                    data->CapturedPkts--;
                    if (data->FirstPkt->Len == -1)
                    {
                        data->FirstPkt = (CAPTURE *) data->Capture;
                        break;
                    }
                }
            }
           
            data->LastPkt->Len = (SHORT) octetsCopied;
            data->LastPkt->ID = data->LastPktID;
            data->LastPkt->Time = (LONG) time.Syntax.LngInt - data->TurnOnTime;
            memcpy ((BYTE *) data->LastPkt + sizeof (CAPTURE), packet.Syntax.Ptr, (SHORT) octetsCopied);
            data->LastPkt = (CAPTURE *)((BYTE *) data->LastPkt + octetsCopied + sizeof (CAPTURE));
            data->LastPkt->Len = 0;
            data->MaxOctetsFree -= (octetsCopied + sizeof(CAPTURE));
            data->LastPktID++;
            data->LastPktID %= (1L << 31);
            data->CapturedPkts++;
        }
    }

    return;
}
