/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       event_m.c
**     SYSTEM   NAME:       event
**     ORIGINAL AUTHOR(S):  Richard Kooijman
**     VERSION  NUMBER:     0.99
**     CREATION DATE:       1992/9/28
**
** DESCRIPTION: event group of the RMON MIB
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif


#include <dnpap.h>
#include <agent.h>
#include <config.h>
#include <log.h>
#include <snmp.h>
#include <message.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys.h>
#include <trap.h>

#include "event_d.h"
#include "event_e.h"
#include "event_c.h"
#include "event_u.h"
#include "event.h"


static CHAR MODULE[] = "EventM";


static MIB_LOCAL*	RmonEventInstance = NULL;


static SHORT LogMaxNrLogs = 100;


static BOOLEAN  EventMibNext(SNMP_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize);
static BOOLEAN  EventFirstLastLog(RmonEvent *event, LONG *firstlog, LONG *lastlog);
static BOOLEAN  EventLogInRange(LONG firstlog, LONG lastlog, LONG logindex);
static RmonLog* EventLogEntry(SNMP_OBJECT *obj, WORD idlen, MIB_LOCAL *local);



BOOLEAN EventMInit(VOID)
{
    MessageConfig(EVENT_ERROR, "Event");
	ConfigGetShort("beholder.event.maxnrlogs", &LogMaxNrLogs);
	if (LogMaxNrLogs < 2)
    {
        DnpapMessage(DMC_WARNING, EVENT_MAX, "event: beholder.event.maxnrlogs < 2, setting it to 2");
        LogMaxNrLogs = 2;
    }
	
	return TRUE;
}


BOOLEAN EventMibNext(SNMP_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize)
{
LONG firstlog, lastlog;

    if (*local == NULL || indexsize < 1)
        return FALSE;

    if (obj->IdLen < idlen + indexsize)
    {
        if (indexsize == 2)
        {                          
            while (*local != NULL && EventFirstLastLog((RmonEvent*)(*local)->Data, &firstlog, &lastlog) == FALSE)
                *local = (*local)->Next;
            if (*local == NULL)
                return FALSE;
            obj->Id[idlen] = (*local)->Index;
            obj->Id[idlen+1] = firstlog;
            obj->IdLen = idlen + 2;
        }
        else
        if (obj->IdLen == idlen)
        {
            obj->Id[idlen] = (*local)->Index;
            obj->IdLen = idlen + 1;
        }
        return TRUE;
    }
    else
    {
        switch (indexsize)
        {            
        case 2:
            if (EventFirstLastLog((RmonEvent*)(*local)->Data, &firstlog, &lastlog) == TRUE &&
                EventLogInRange(firstlog, lastlog, obj->Id[idlen+1]) == TRUE &&
                obj->Id[idlen+1] != lastlog)
            {
                obj->Id[idlen+1]++;
                if (obj->Id[idlen+1] == 0)
                    obj->Id[idlen+1] = 1;
                return TRUE;
            }
        case 1:
            *local = (*local)->Next;
            if (*local != NULL)
            {
                if (indexsize == 2)
                {                          
                    while (*local != NULL && EventFirstLastLog((RmonEvent*)(*local)->Data, &firstlog, &lastlog) == FALSE)
                        *local = (*local)->Next;
                    if (*local == NULL)
                        return FALSE;
                    obj->Id[idlen] = (*local)->Index;
                    obj->Id[idlen+1] = firstlog;
                    obj->IdLen = idlen + 2;
                }
                else
                {
                    obj->Id[idlen] = (*local)->Index;
                    obj->IdLen = idlen + 1;
                }
                return TRUE;
            }
            return FALSE;
        }
        return FALSE;
    }
    return FALSE;
}


BOOLEAN EventFirstLastLog(RmonEvent *event, LONG *firstlog, LONG *lastlog)
{
	if (event->NrLogs == 0)
	{
		*firstlog = *lastlog = 0;
		return FALSE;
	}
	if (event->LogBounced == TRUE && event->LogPos < event->NrLogs)
		*firstlog = event->Logs[event->LogPos].Index;
	else
		*firstlog = event->Logs[0].Index;
	*lastlog = event->Logs[(event->LogPos > 0) ? (event->LogPos-1) : (event->NrLogs-1)].Index;
	return TRUE;
}


BOOLEAN EventLogInRange(LONG firstlog, LONG lastlog, LONG logindex)
{
	if (firstlog <= lastlog)
		return (logindex >= firstlog && logindex <= lastlog);
	else
		return (logindex >= firstlog || logindex <= lastlog);
}


WORD EventIndex(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = MibRmon(obj, RmonEventInstance, idlen, 1)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
        if (EventMibNext(obj, &local, idlen, 1) == FALSE)
	    return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		obj->Syntax.LngInt = local->Index;
		return SNMP_NOERROR;
	case SNMP_PDU_SET:
		return SNMP_READONLY;
	}
}


WORD EventDescription(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonEvent* data = NULL;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 1)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 1) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		data = (RmonEvent*)local->Data;
		memcpy(obj->Syntax.BufChr, data->Description, obj->SyntaxLen = data->DescrLen);
		return SNMP_NOERROR;
	case SNMP_PDU_SET:
		data = (RmonEvent*)local->Data;
        if (obj->SyntaxLen > RMONEVENT_SZEDESCR)
            return SNMP_BADVALUE;
        memcpy(data->Description, obj->Syntax.BufChr, data->DescrLen = obj->SyntaxLen);
		return SNMP_NOERROR;
	}
}


WORD EventType(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonEvent* data = NULL;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 1)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 1) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		data = (RmonEvent*)local->Data;
        obj->Syntax.LngInt = data->Type;
		return SNMP_NOERROR;
	case SNMP_PDU_SET:
		data = (RmonEvent*)local->Data;
        if (obj->Syntax.LngInt < EVENTTYPE_MIN || obj->Syntax.LngInt > EVENTTYPE_MAX)
            return SNMP_BADVALUE;
        data->Type = obj->Syntax.LngInt;
        return SNMP_NOERROR;
	}
}


WORD EventCommunity(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonEvent* data = NULL;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 1)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 1) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		data = (RmonEvent*)local->Data;
        memcpy(obj->Syntax.BufChr, data->Community, obj->SyntaxLen = data->CommLen);
		return SNMP_NOERROR;
	case SNMP_PDU_SET:
		data = (RmonEvent*)local->Data;
        if (obj->SyntaxLen > RMONEVENT_SZECOMM)
            return SNMP_BADVALUE;
        memcpy(data->Community, obj->Syntax.BufChr, data->CommLen = obj->SyntaxLen);
		return SNMP_NOERROR;
	}
}


WORD EventLastTimeSent(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonEvent* data = NULL;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 1)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 1) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		data = (RmonEvent*)local->Data;
        obj->Syntax.LngUns = data->LastTimeSent;
		return SNMP_NOERROR;
	case SNMP_PDU_SET:
        return SNMP_READONLY;
	}
}


WORD EventOwner(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonEvent* data = NULL;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 1)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 1) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		data = (RmonEvent*)local->Data;
		memcpy(obj->Syntax.BufChr, data->Owner, obj->SyntaxLen = data->OwnerLen);
		return SNMP_NOERROR;
	case SNMP_PDU_SET:
		data = (RmonEvent*)local->Data;
		if (data->Status != SNMP_UNDERCREATION)
			return SNMP_READONLY;
		memcpy(data->Owner, obj->Syntax.BufChr, data->OwnerLen = obj->SyntaxLen);
		return SNMP_NOERROR;
	}
}


WORD EventStatus(SNMP_OBJECT* obj, WORD idlen)
{
MIB_LOCAL *local = NULL;
RmonEvent *data;

    local = MibRmon(obj, RmonEventInstance, idlen, 1);

    switch (obj->Request)
    {
        case SNMP_PDU_GET:
            if (local == NULL)
            {
                obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (RmonEvent*)local->Data;
            obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_NEXT:
            if (EventMibNext(obj, &local, idlen, 1) == FALSE)
            {
                obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (RmonEvent*)local->Data;
            obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            if (local == NULL)
            {
                switch (obj->Syntax.LngInt)
                {
                    case SNMP_CREATEREQUEST:
                        if ((local = MibInsert(obj, &RmonEventInstance, idlen, 1)) == NULL)
                            return SNMP_GENERROR;
                        if ((local->Data = NewRmonEvent(1)) == NULL)
                            return SNMP_GENERROR;
                        data = (RmonEvent*) local->Data;
                        if (EventCInit(data) == TRUE)
                        {
                            data->Status = SNMP_UNDERCREATION;
                            return SNMP_NOERROR;
                        }
                        DelRmonEvent(local->Data);
                        MibRemove(obj, &RmonEventInstance, idlen, 1);
                        return SNMP_GENERROR;
                    default:
                        return SNMP_NOSUCHNAME;
                }
            }
            data = (RmonEvent*)local->Data;
            switch (data->Status)
            {
                case SNMP_UNDERCREATION:
                    switch (obj->Syntax.LngInt)
                    {
                        case SNMP_VALID:
                            if (EventCStart(data) == TRUE)
                            {
                                data->Status = SNMP_VALID;
                                return SNMP_NOERROR;
                            }
                            else
                                return SNMP_GENERROR;
                        case SNMP_INVALID:
                            if (EventCStop(data) == TRUE)
                            {
                                DelRmonEvent(local->Data);
                                if (MibRemove(obj, &RmonEventInstance, idlen, 1) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
                case SNMP_VALID:
                    switch (obj->Syntax.LngInt)
                    {
                        case SNMP_INVALID:
                            if (EventCStop(data) == TRUE)
                            {
                                DelRmonEvent(local->Data);
                                if (MibRemove(obj, &RmonEventInstance, idlen, 1) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        case SNMP_VALID:
                            return SNMP_NOERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
            }
    }
}


WORD EventLogEventIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonLog *log;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 2)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 2) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		log = EventLogEntry(obj, idlen, local);
		return LogEventIndex(obj, idlen, log);
	case SNMP_PDU_SET:
        return SNMP_READONLY;
	}
}


WORD EventLogIndex(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonLog *log;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 2)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 2) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		log = EventLogEntry(obj, idlen, local);
		return LogIndex(obj, idlen, log);
	case SNMP_PDU_SET:
        return SNMP_READONLY;
	}
}


WORD EventLogTime(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonLog *log;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 2)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 2) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		log = EventLogEntry(obj, idlen, local);
		return LogTime(obj, idlen, log);
	case SNMP_PDU_SET:
        return SNMP_READONLY;
	}
}


WORD EventLogDescription(SNMP_OBJECT *obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
RmonLog *log;
			
    if ((local = MibRmon(obj, RmonEventInstance, idlen, 2)) == NULL)
		return SNMP_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case SNMP_PDU_NEXT:
		if (EventMibNext(obj, &local, idlen, 2) == FALSE)
			return SNMP_NOSUCHNAME;
	case SNMP_PDU_GET:
		log = EventLogEntry(obj, idlen, local);
		return LogDescription(obj, idlen, log);
	case SNMP_PDU_SET:
        return SNMP_READONLY;
	}
}


RmonLog* EventLogEntry(SNMP_OBJECT *obj, WORD idlen, MIB_LOCAL *local)
{
LONG firstlog, lastlog, logindex, realindex;
RmonEvent *event = (RmonEvent*)local->Data;

	if (obj->IdLen != idlen + 2 || obj ->Id[idlen] != local->Index)
	{
		DnpapMessage(DMC_FATAL, EVENT_LOGSANITY, "something is very wrong: object index too short or event mismatch");
		DnpapExit(EVENT_LOGSANITY);
	}
	
	logindex = obj->Id[idlen+1];
	
	if (EventFirstLastLog(event, &firstlog, &lastlog) == FALSE || EventLogInRange(firstlog, lastlog, logindex) == FALSE)
	{
		DnpapMessage(DMC_FATAL, EVENT_NOLOGENTRIES, "something is very wrong: no applicable log entries");
		DnpapExit(EVENT_NOLOGENTRIES);
	}
	
	if (logindex >= firstlog)
		realindex = event->LogPos + logindex - firstlog;
	else
		realindex = event->LogPos - lastlog + logindex - 1;
		
	if (realindex >= event->NrLogs)
		realindex -= event->NrLogs;
	else
	if (realindex < 0)
		realindex += event->NrLogs;
		
	return &event->Logs[realindex];
}


/*****************************************************************
** NAME:        GenerateEvent
** SYNOPSIS:    BOOLEAN GenerateEvent(
**                  CHAR *generator,
**                  LONG eventindex,
**                  WORD code,
**                  WORD specific,
**                  SNMP_OBJECT *list,
**                  WORD listlen)
** PARAMETERS:  generator: string to identify the module that
**                         called GenerateEvent()
**              eventindex: the index of the event that is
**                          to be generated
**              code: SNMP-trap code
**              specific: SNMP-trap enterprise specific code
**              list: list of objects that are to be sent with
**                    an SNMP-trap
**              listlen: number of objects in the list
** DESCRIPTION: tries to generate an event with number eventindex.
**              As a result of configuration settings a log entry
**              will be created and/or an SNMP-trap will be sent.
** RETURNS:     TRUE: event exists and was successfully generated.
**              FALSE: event does not exist or was not generated
**                     successfully.
** REMARKS:     see also EventGenerated() in event_u
*******************************************************************/

BOOLEAN GenerateEvent(CHAR *generator, LONG eventindex, WORD code, WORD specific, SNMP_OBJECT *list, WORD listlen)
{
MIB_LOCAL *local;
RmonEvent *event;
RmonLog	  *ptr;

	if (eventindex == 0)
		return FALSE;
		
	for (local = RmonEventInstance; local != NULL; local = local->Next)
	{
		event = (RmonEvent*)local->Data;
		if (event->Status == SNMP_VALID && eventindex == local->Index)
		{
			EventGenerated(eventindex);

            event->LastTimeSent = SysTime();
			
			if (event->Type == EVENT_LOG || event->Type == EVENT_LOG_TRAP)
			{
				if (event->LogBounced == TRUE || event->NrLogs == LogMaxNrLogs ||
				    (ptr = ReNewRmonLog(event->Logs, (WORD)event->NrLogs+1, (WORD)event->NrLogs, (WORD)LogMaxNrLogs)) == NULL)
				{
					event->LogBounced = TRUE;
					if (event->NrLogs == 0)
					{
						DnpapMessage(DMC_ERROR, EVENT_NOLOGS, "event: could not create any log entries");
						return FALSE;
					}
					if (event->LogPos == event->NrLogs)
						event->LogPos = 0;
				}
				else
				{
					event->Logs = ptr;
					event->NrLogs++;
				}
				event->Logs[event->LogPos].Index = event->LogIndex++;
                if (event->Logs[event->LogPos].Index == 0)
                    event->Logs[event->LogPos].Index = event->LogIndex++;
				event->Logs[event->LogPos].TimeTicks = SysTime();
				sprintf(event->Logs[event->LogPos].Description, "event description: %.*s, generator: %s, event index: %ld, trap: %s, specific: %s%n",
                        event->DescrLen, event->Description, generator, eventindex, SnmpTrap[code], TrapSpecific[specific], &event->Logs[event->LogPos].DescrLen);
				event->LogPos++;
			}
			if (event->Type == EVENT_SNMP_TRAP || event->Type == EVENT_LOG_TRAP)
				if (AgentSendTrap(event->Community, event->CommLen, code, specific, list, listlen) == FALSE)
                    DnpapMessage(DMC_WARNING, EVENT_TRAPFAIL, "event: trap send failed");
			
			return TRUE;
		}
	}
	
	return FALSE;
}
