/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          protocol.c
** SYSTEM NAME:        beholder
** MODULE NAME:        protocol
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/13
** DESCRIPTION:        packet disassembler: accesses the fields in the
**                          protocol headers dynamically
*************************************************************************/

#include <stddef.h> 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dnpap.h>
#include <snmp.h>
#include "protocol.h"
#include "prot.h"

#define PROT_TYPE_OTHER               1
#define PROT_TYPE_REGULAR_1822        2
#define PROT_TYPE_HDH_1822            3
#define PROT_TYPE_DDN_X25             4
#define PROT_TYPE_RFC877_X25          5
#define PROT_TYPE_ETHERNET_CSMACD     6
#define PROT_TYPE_88023_CSMACD        7
#define PROT_TYPE_88024_TOKENBUS      8
#define PROT_TYPE_88025_TOKENRING     9
#define PROT_TYPE_88026_MAN           10
#define PROT_TYPE_SOFT_LOOPBACK       24

BYTE *protocolString = NULL;

BYTE *protBeholderString[] =
{
    "TYPE",
    "INTERFACE",
    "ID",
    "SIZE",
    "LEN",
    "TIME"
};

/*****************************************************************
** NAME:        ProtFrame
** SYNOPSIS:    BOOLEAN ProtFrame (PROT_PKT *Pkt, BYTE *frame,
**                  USHORT size, USHORT length, ULONG time,
**                  MAC_IFACE *iface)
** PARAMETER:   *FrmPtr: pointer to packet received from net
** DESCRIPTION: initializes protocolanalyzer in run-time.
** REMARKS:     see also ProtFile
** RETURNS:     TRUE: *Pkt points to *PktPtr and is initialized.
*******************************************************************/

BOOLEAN ProtFrame (PROT_PKT *Pkt, BYTE *frame, USHORT size, USHORT length, ULONG time, USHORT type, USHORT index)
{
    if ((Pkt->Frame = DnpapMalloc (sizeof(PROT_BEHOLDER))) == NULL)
        return FALSE;
    Pkt->Ptr = 0;
    Pkt->Frame->Beholder.Type = type;
    Pkt->Frame->Beholder.IfIndex = index;
    Pkt->Frame->Beholder.ID = 0;
    Pkt->Frame->Beholder.Size = size;
    Pkt->Frame->Beholder.Len = length;
    Pkt->Frame->Beholder.Time =time;
    Pkt->Frame->Beholder.Data = frame;
    Pkt->Child = NULL;
    
    switch (Pkt->Frame->Beholder.Type)
    {
        case PROT_TYPE_ETHERNET_CSMACD:
            Pkt->ChildProt = PROT_PKTETHERNET;
            break;
        default:
            Pkt->ChildProt = PROT_PKTUNKNOWN;
    }
    return TRUE;
}

/*****************************************************************
** NAME:        ProtFile
** SYNOPSIS:    BOOLEAN ProtFile (PROT_PKT *Pkt, BYTE *PktPtr)
** PARAMETER:   *PktPtr: pointer to packet = array of bytes
**                  - Beholder Type (2 bytes) = protocol type
**                  - Beholder interface index (2 bytes)
**                  - Beholder ID (4 bytes) = counter
**                  - Beholder Size (2 bytes) = # bytes on net
**                  - Beholder Len (2 bytes) = # bytes received
**                  - Beholer Time (4 bytes) = time of arrival
**                  - Beholder Data (remaining packet len)
** DESCRIPTION: initializes protocolanalyzer for background.
** REMARKS:     see also ProtFrame;
** RETURNS:     TRUE: *Pkt points to *PktPtr and is initialized.
*******************************************************************/

BOOLEAN ProtFile (PROT_PKT *Pkt, BYTE *PktPtr)
{
    if ((Pkt->Frame = DnpapMalloc (sizeof(PROT_BEHOLDER))) == NULL)
        return (FALSE);
    Pkt->Ptr = PktPtr;
    Pkt->Frame->Beholder.Type = *((WORD *) Pkt->Ptr);
    Pkt->Ptr += 2;
    Pkt->Frame->Beholder.IfIndex = *((WORD *) Pkt->Ptr);
    Pkt->Ptr += 2;
    Pkt->Frame->Beholder.ID = *((LONG *) Pkt->Ptr);
    Pkt->Ptr += 4;
    Pkt->Frame->Beholder.Size = *((WORD *) Pkt->Ptr);
    Pkt->Ptr += 2;
    Pkt->Frame->Beholder.Len = *((WORD *) Pkt->Ptr);
    Pkt->Ptr += 2;
    Pkt->Frame->Beholder.Time = *((LWORD *) Pkt->Ptr);
    Pkt->Ptr += 4;
    Pkt->Frame->Beholder.Data = Pkt->Ptr;
    Pkt->Child = NULL;

    switch (Pkt->Frame->Beholder.Type)
    {
        case PROT_TYPE_ETHERNET_CSMACD:
            Pkt->ChildProt = PROT_PKTETHERNET;
            break;
        default:
            Pkt->ChildProt = PROT_PKTUNKNOWN;
    }
    return (TRUE);
}

/*****************************************************************
** NAME:        ProtGetField
** SYNOPSIS:    BOOLEAN ProtGetField (PROT_PKT *Pkt, PROT_OBJ *Obj)
** PARAMETERS:  *Pkt: initialized with ProtFrame or ProtFile
**              *Obj:
**                  Obj->Level: 0 = beholder
**                              1 = ethernet (etc)
**                  Obj->Id: object identification: see prot.h
**                  Obj->Type: SNMP_NULL: Obj->Syntax.Ptr
**                             SNMP_INTEGER: Obj->Syntax.LngInt
**                             SNMP_OCTETSTR: Obj->Syntax.BufChr
**                             SNMP_OBJECTID: Obj->Syntax.BufInt
**                             SNMP_IPADDR: Obj->Syntax.LngUns
**                             SNMP_COUNTER: Obj->Syntax.LngUns
**                             SNMP_GAUGE: Obj->Syntax.LngUns
**                             SNMP_TIMETICKS: Obj->Syntax.LngUns
**                             SNMP_OPAQUE: Obj->Syntax.BufChr
**                             SNMP_DISPLAYSTR: Obj->Syntax.BufChr
**                  Obj->Syntax: response (depends on Obj->Type)
**                  Obj->SyntaxLen: length of Obj->Syntax
** DESCRIPTION: searches the packet for the protocol and field.
**              Obj->Id[0] = protocol, Obj->Id[1] = field, etc.
**              Obj->Level is used to stop at a certain Level
**              if (Obj->Id[1] == PROT_TYPE && Obj-Level == 0)
**                  then returns: ChildProt
**              last Obj: returns pointer to data after header.
** REMARKS:     see also protocol.h and prot.h for definitions
** RETURNS:     TRUE:   Obj->Id found & Obj->Syntax inserted
**              FALSE:  Obj->Id not found in packet, or
**                      Obj->Level too low, or
**                      Out of memory
*******************************************************************/

BOOLEAN ProtGetField (PROT_PKT *Pkt, PROT_OBJ *Obj)
{
    if (Obj->Level == 0 && Obj->Id[1] == PROT_TYPE)
    {
        memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
        Obj->Type = SNMP_GAUGE;
        Obj->Syntax.LngUns = (LWORD) Pkt->ChildProt;
        return (TRUE);
    }

    if (Obj->Id[0] == PROT_PKTBEHOLDER)
    {
        switch (Obj->Id[1])
        {
            case 1:
                Obj->Type = SNMP_INTEGER;
                Obj->Syntax.LngInt = (LONG) Pkt->Frame->Beholder.Type;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            case 2:
                Obj->Type = SNMP_INTEGER;
                Obj->Syntax.LngInt = (LONG) Pkt->Frame->Beholder.IfIndex;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            case 3:
                Obj->Type = SNMP_INTEGER;
                Obj->Syntax.LngInt = Pkt->Frame->Beholder.ID;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            case 4:
                Obj->Type = SNMP_GAUGE;
                Obj->Syntax.LngUns = (LWORD) Pkt->Frame->Beholder.Size;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            case 5:
                Obj->Type = SNMP_GAUGE;
                Obj->Syntax.LngUns = (LWORD) Pkt->Frame->Beholder.Len;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            case 6:
                Obj->Type = SNMP_TIMETICKS;
                Obj->Syntax.LngUns = Pkt->Frame->Beholder.Time;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            case 7:
                Obj->Type = SNMP_NULL;
                Obj->Syntax.Ptr = Pkt->Frame->Beholder.Data;
                Obj->SyntaxLen = Pkt->Frame->Beholder.Len;
                memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);
                return (TRUE);
            default:
                return (FALSE);
        }
    }

    if (Obj->Level > 0 && Pkt->ChildProt != PROT_PKTUNKNOWN)
    {
        if (Pkt->Child == NULL)
        {
            if ((Pkt->Child = DnpapMalloc (sizeof(PROT_PKT))) == NULL)
                return (FALSE);
            Pkt->Child->Ptr = Pkt->Frame->Beholder.Data;
            Pkt->Child->Child = NULL;
            Pkt->Child->DataLen = Pkt->Frame->Beholder.Len;
            if ((ProtPtr[Pkt->ChildProt].Header(Pkt->Child)) == FALSE)
            {
                DnpapFree(Pkt->Child);
                return (FALSE);
            }
        }
        Obj->Level--;
        return (ProtPtr[Pkt->ChildProt].Field (Pkt->Child, Obj));
    }
    return (FALSE);
}


/*****************************************************************
** NAME:        ProtFree
** SYNOPSIS:    VOID ProtFree (PROT_PKT *Pkt)
** PARAMETER:   *Pkt: see ProtGetField
** DESCRIPTION: frees all space allocated for protocol headers.
** REMARKS:     use only after packet has been entirely processed
** RETURNS:     VOID
*******************************************************************/

VOID ProtFree (PROT_PKT *Pkt)
{
    if (Pkt->Child != NULL)
    {
        ProtFree (Pkt->Child);
        DnpapFree (Pkt->Child);
    }
    DnpapFree (Pkt->Frame);
    return;
}


/**************************************************************
** NAME:        ProtInit
** SYNOPSIS:    VOID ProtInit(VOID)
** PARAMETER:   VOID
** DESCRIPTION: initializes the Protocol library.
** REMARKS:     none
** RETURNS:     VOID
**************************************************************/
BOOLEAN ProtInit(VOID)
{
    DnpapAtExit(ProtExit);
    return TRUE;
}

/**************************************************************
** NAME:        ProtExit
** SYNOPSIS:    VOID ProtExit(VOID)
** PARAMETER:   VOID
** DESCRIPTION: frees all space allocated for the library.
** REMARKS:     none
** RETURNS:     VOID
**************************************************************/
VOID ProtExit(VOID)
{
    if (protocolString != NULL)
        DnpapFree(protocolString);
    return;
}


/*****************************************************************
** NAME:        ProtStringNr
** SYNOPSIS:    BOOLEAN ProtStringNr (PROT_OBJ *Obj)
** PARAMETERS:  *Obj:
**                  Obj->ProtStr: protocol in ascii
**                  Obj->FieldStr: protocol field in ascii
** DESCRIPTION: converts protocol- and field-strings to Obj->Id.
**              blanks in a name are written as _
**              optional: FieldStr.subfieldnr.subfieldnr. (etc)
** REMARKS:     use in command-line; see prot.h for Id-nrs
** RETURNS:     TRUE: Obj->Id inserted
**              FALSE: protocol or field not converted to Obj->Id
*******************************************************************/

BOOLEAN ProtStringNr (PROT_OBJ *Obj)
{
    BOOLEAN Flag = 0;
    WORD Ind, Size;
    BYTE *Ptr1, *Ptr2;

    memset(&Obj->Id[2], 0, SNMP_SIZE_OBJECTID - 4);

    if ((Ptr1 = strchr(Obj->FieldStr,'.')) != NULL)
    {
        *Ptr1 = '\0';
        Size = strlen(Ptr1 + 1);
        for (Ind = 0; Size > 0; Ind++)
        {
            Ptr1++;
            if ((Ptr2 = strchr(Ptr1,'.')) != NULL)
            {
                *Ptr2 = '\0';
                Obj->Id[2+Ind] = (WORD) atoi(Ptr1);
                Size = strlen(Ptr2 + 1);
                Ptr1 = Ptr2;
            }
            else
            {
                Obj->Id[2+Ind] = (WORD) atoi(Ptr1);
                Size = 0;
            }
        }
    }

    for (Ind = 1; Ind < sizeof(ProtPtr)/sizeof(PROT_PTR); Ind++)
    {
        if (!strcmp(ProtPtr[Ind].Name, Obj->ProtStr))
        {
            Obj->Id[0] = Ind;
            Flag = 1;
        }
    }
    if (Flag)
    {
        for (Ind = 0; Ind < ProtPtr[Obj->Id[0]].StringLen; Ind++)
        {
            if (!strcmp(ProtPtr[Obj->Id[0]].String[Ind], Obj->FieldStr))
            {
                Obj->Id[1] = Ind + 1;
                return (TRUE);
            }
        }
    }
    return (FALSE);
}

/*****************************************************************
** NAME:        ProtPrint   
** SYNOPSIS:    BOOLEAN ProtPrint (PROT_OBJ *Obj, BYTE **StrPtr)
** PARAMETER:   *Obj: see ProtGetField
**              **StrPtr: pointer to string
** DESCRIPTION: converts Obj->Syntax to a printable string.
** REMARKS:     use after ProtGetField has returned TRUE
** RETURNS:     TRUE: Obj->Syntax converted to string
**              FALSE: out of memory or Obj->Id not found
*******************************************************************/

BOOLEAN ProtPrint (PROT_OBJ *Obj, BYTE **StrPtr)
{
    if (protocolString == NULL)
    {
        if ((protocolString = DnpapMalloc(SNMP_SIZE_BUFCHR)) == NULL) /* worst case */
            return (FALSE);
    }
    if (Obj->Id[0] < sizeof(ProtPtr)/sizeof(PROT_PTR))
        return (ProtPtr[Obj->Id[0]].Print (Obj, StrPtr));
}


BOOLEAN ProtBeholderPrint (PROT_OBJ *Obj, BYTE **StrPtr)
{
    switch (Obj->Id[1])
    {
        case 1:                                 /* type */
        case 2:                                 /* IfIndex */
        case 4:                                 /* size */
        case 5:                                 /* len */
            sprintf(protocolString, "%4lu", Obj->Syntax.LngUns);
            *StrPtr = protocolString;
            return (TRUE);
        case 3:                                 /* ID */
            sprintf(protocolString, "%6ld", Obj->Syntax.LngInt);
            *StrPtr = protocolString;
            return (TRUE);
        case 6:                                 /* time */
            sprintf(protocolString, "%u:%03u:%03u",
                (WORD) ((Obj->Syntax.LngUns / 1000000L) % 1000),
                (WORD) ((Obj->Syntax.LngUns / 1000L) % 1000),
                (WORD) (Obj->Syntax.LngUns % 1000));
            *StrPtr = protocolString;
            return (TRUE);
        default:
            return (FALSE);
    }
}




