/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          sys.c
** SYSTEM NAME:        beholder
** MODULE NAME:        sys
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/13
** DESCRIPTION:        MIB-support for system group:
**                          sysDescr, sysObjectId, sysUpTime, sysContact
**                          sysName, sysLocation, sysServices
*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dnpap.h>
#include <config.h>
#include <mibsup.h>
#include <mibrqs.h>
#include <timer.h>

static struct SYS
{
    BYTE    Descr[SNMP_SIZE_BUFCHR];
    LONG    ObjectID[SNMP_SIZE_BUFINT];
    WORD    ObjectIDLen;
    BYTE    Contact[SNMP_SIZE_BUFCHR];
    WORD    ContactLen;
    BYTE    Name[SNMP_SIZE_BUFCHR];
    WORD    NameLen;
    BYTE    Location[SNMP_SIZE_BUFCHR];
    LONG    Services;
} sys = {0};


/*****************************************************************
** NAME:        SysInit
** SYNOPSIS:    VOID SysInit (VOID)
** PARAMETERS:  VOID
** DESCRIPTION: initializes the system collector.
** REMARKS:     ONLY FOR BEHOLDER & MIB must be initialized !
** RETURNS:     VOID
*******************************************************************/

BOOLEAN SysInit (VOID)
{
    BYTE *p, *t;
    LONG l;

    if (!ConfigGetString("beholder.sys.description", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="RMON-based network monitor";
    strcpy(sys.Descr, p);
    
    if (!ConfigGetString("beholder.sys.contact", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="DNPAP <btng@dnpap.et.tudelft.nl>";
    strcpy(sys.Contact, p);
    sys.ContactLen = strlen(p);

    if (!ConfigGetString("beholder.sys.name", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="beholder - the next generation";
    strcpy(sys.Name, p);
    sys.NameLen = strlen(p);
    
    if (!ConfigGetString("beholder.sys.location", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="Delft University of Technology";
    strcpy(sys.Location, p);
    
    if (!ConfigGetLong("beholder.sys.services", &l))
        sys.Services = 127L;
    else
        sys.Services = l;

    if (!ConfigGetString("beholder.sys.objectid", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="1.3.6.1.4.1.464.1";
    sys.ObjectIDLen = 0;
    if ((t = strtok(p, ".")) != NULL)
    {
        sys.ObjectID[sys.ObjectIDLen++] = atol(t);
        while ((t = strtok (NULL, ".")) != NULL)
            sys.ObjectID[sys.ObjectIDLen++] = atol(t);
    }
    return TRUE;
}

/*****************************************************************
** NAME:        SysTime
** SYNOPSIS:    LWORD SysTime (VOID)
** PARAMETERS:  VOID 
** DESCRIPTION: gives the time since the system was started.
** REMARKS:     ONLY FOR BEHOLDER & MIB must be initialized !
** RETURNS:     time in 1/100 sec.
*******************************************************************/

LWORD SysTime (VOID)
{
    SNMP_OBJECT time = {SNMP_PDU_GET,  {1,3,6,1,2,1,1,3,0}, 9};

    if (MibRequest ("collector", 9, &time) != SNMP_NOERROR)
        return 0;
    else
        return time.Syntax.LngUns;
}

WORD sysDescr (SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    strcpy (Obj->Syntax.BufChr, sys.Descr);
    Obj->SyntaxLen = strlen (sys.Descr);
    return (SNMP_NOERROR);
}
                                                
WORD sysObjectID (SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    memcpy (Obj->Syntax.BufInt, sys.ObjectID, sys.ObjectIDLen * sizeof(LONG));
    Obj->SyntaxLen = sys.ObjectIDLen;
    return (SNMP_NOERROR);
}

WORD sysUpTime (SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    Obj->Syntax.LngInt = TimerNow()/10;
    return (SNMP_NOERROR);
}

WORD sysContact(SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    if (Obj->Request == SNMP_PDU_SET)
    {
        memcpy (sys.Contact, Obj->Syntax.BufChr, Obj->SyntaxLen);
        sys.ContactLen = Obj->SyntaxLen;
    }
    else
    {
        memcpy (Obj->Syntax.BufChr, sys.Contact, sys.ContactLen);
        Obj->SyntaxLen = sys.ContactLen;
    }
    return (SNMP_NOERROR);
}

WORD sysName(SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    if (Obj->Request == SNMP_PDU_SET)
    {
        memcpy (sys.Name, Obj->Syntax.BufChr, Obj->SyntaxLen);
        sys.NameLen = Obj->SyntaxLen;
    }
    else
    {
        memcpy (Obj->Syntax.BufChr, sys.Name, sys.NameLen);
        Obj->SyntaxLen = sys.NameLen;
    }
    return (SNMP_NOERROR);
}

WORD sysLocation(SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    strcpy (Obj->Syntax.BufChr, sys.Location);
    Obj->SyntaxLen = strlen (sys.Location);
    return (SNMP_NOERROR);
}

WORD sysServices(SNMP_OBJECT *Obj, WORD IdLen)
{
    if (MibSimple(Obj, IdLen) == FALSE)
        return (SNMP_NOSUCHNAME);
    Obj->Syntax.LngInt = sys.Services;
    return (SNMP_NOERROR);
}



