/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          mgmt.c
** SYSTEM NAME:        beholder
** MODULE NAME:        
** ORIGINAL AUTHOR(S): Dirk Wisse
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/9/28
** DESCRIPTION:        
*************************************************************************/

#ifdef OS2
#define PIPE
#endif

#ifdef UNIX
#define UDP
#endif

#ifdef MSC
#define UDP
#endif

#include        <stdio.h>
#include        <string.h>
#include        <dnpap.h>
#include        <snmp.h>
#include        <config.h>
#include        "mgmt.h"
#ifdef PIPE
#include        "pipe.h"
#endif

#ifdef UDP
#include        "udp.h"
#ifdef MSC
void sock_init(void);
#endif
#endif




#define MGMT_SIZE_BUFFER      2048
#define MGMT_SIZE_COMMUNITY   255

#ifdef PIPE
int    mgmtMedium = MGMT_TYPE_PIPE;
#else
int    mgmtMedium = MGMT_TYPE_UDP;
#endif

BYTE   mgmtAgent[255] = "beholder";
BYTE   mgmtComm[255] = "public";
USHORT mgmtCommLen = 6;
ULONG  mgmtTimeout = 5000000;
USHORT mgmtRetries = 3;


BOOLEAN MgmtInit(void)
{
    BYTE            *medium;
    BYTE            *p;
    USHORT          n;
    static BOOLEAN  init = FALSE;
    
    if (!init)
    {
#ifdef  UDP
#ifdef MSC
        sock_init();
#endif
#endif

        if (ConfigGetString("mgmt.medium", &medium))
        {
            if (stricmp(medium, "UDP")==0)
                mgmtMedium = MGMT_TYPE_UDP;
            else
            {
                if (stricmp(medium, "PIPE")==0)
                    mgmtMedium = MGMT_TYPE_PIPE;
            }
        }
        if (ConfigGetString("mgmt.agent", &p))
        {
            strcpy(mgmtAgent, p);
        }
        if (ConfigGetBytes("mgmt.community", &p, &n))
        {
            memcpy(mgmtComm, p, n);
            mgmtCommLen = n;
        }
        ConfigGetLong("mgmt.timeout", &mgmtTimeout);
        ConfigGetShort("mgmt.retries", &mgmtRetries);
        init = TRUE;
    }

    return init;
}



/*****************************************************************
** NAME:        MgmtOpen        
** SYNOPSIS:    MGMT *MgmtOpen(char *agent, int medium)
** PARAMETERS:  - char *agent
**              MGMT_TYPE_PIPE -> OS/2 named pipe \pipe\<agent>.
**              MGMT_TYPE_UDP  -> UDP/IP connection to host <agent>.
**              - int medium
**              Transport medium:
**              MGMT_TYPE_DEFAULT
**              MGMT_TYPE_UDP
**              MGMT_TYPE_PIPE
** DESCRIPTION: Opens an SNMP connection over a certain medium.
**              The connection can be used to query an SNMP agent
**              by using MgmtRequest.
** RETURNS:     MGMT * pointer to management descriptor.
**              NULL -> error!
**              Can be used by MgmtRequest and MgmtClose.
**              Do not change the descriptor!
** SEE ALSO:    MgmtClose, MgmtRequest.
*******************************************************************/
MGMT *MgmtOpen(char *agent, int medium)
{
    MGMT    *mgmt;

    if (agent==0)
        agent=mgmtAgent;

    if (medium==MGMT_TYPE_DEFAULT)
        medium=mgmtMedium;
    
    mgmt = DnpapMalloc(sizeof(MGMT));
    if (mgmt == NULL)
    {
	    DnpapMessage(DMC_WARNING, MGMT_ERROR+2, "Error malloc mgmt handle\n");
        return NULL;
    }
    switch (medium)
    {
#ifdef PIPE
        case MGMT_TYPE_PIPE:
            mgmt->type = MGMT_TYPE_PIPE;
            if (PipeOpen(mgmt, agent))
                return mgmt;
            break;
#endif
#ifdef UDP
        case MGMT_TYPE_UDP:
            mgmt->type = MGMT_TYPE_UDP;
            if (UdpOpen(mgmt, agent))
                return mgmt;
            break;
#endif
        default:
            break;
    }

    DnpapMessage(DMC_WARNING, MGMT_ERROR+3, "Error opening mgmt handle\n");
    DnpapFree(mgmt);
    return NULL;
}

/*****************************************************************
** NAME:        MgmtClose        
** SYNOPSIS:    BOOLEAN MgmtClose(MGMT *mgmt)
** PARAMETERS:  - MGMT *mgmt
**              Descriptor obtained by MgmtOpen.
** DESCRIPTION: Closes an SNMP connection.
** RETURNS:     BOOLEAN success.
** SEE ALSO:    MgmtOpen, MgmtRequest.
*******************************************************************/
BOOLEAN MgmtClose(MGMT *mgmt)
{
    switch(mgmt->type)
    {
#ifdef PIPE
        case MGMT_TYPE_PIPE:
            PipeClose(mgmt);
            break;
#endif
#ifdef UDP
        case MGMT_TYPE_UDP:
            UdpClose(mgmt);
            break;
#endif
        default:
            return FALSE;
    }
    DnpapFree(mgmt);
    return TRUE;
}



/*****************************************************************
** NAME:        MgmtRequest       
** SYNOPSIS:    BOOLEAN MgmtRequest(MGMT        *mgmt,
**                                  BYTE        *comm,
**                                  WORD        commLen,
**                                  WORD        type,
**                                  SNMP_OBJECT  *list,
**                                  WORD        listLen,
**                                  WORD        *status,
**                                  WORD        *index,
**                                  ULONG       timeout,
**                                  USHORT      retries)
** PARAMETERS:  - MGMT *mgmt
**              Pointer to descriptor obtained by MgmtOpen.
**              - BYTE *comm
**              SNMP community for request.
**              - WORD commLen
**              Length (in bytes) of community.
**              - WORD type
**              Type of request.
**              SNMP_PDU_GET get request. 
**              SNMP_PDU_NEXT get next request. 
**              SNMP_PDU_SET set request.
**              - SNMP_OBJECT *list
**              List of desired MIB objects from the SNMP agents.
**
**              typedef struct _SNMP_OBJECT
**              {
**                  WORD            Request;
**                  LONG            Id [SNMP_SIZE_OBJECTID];
**                  WORD            IdLen;
**                  WORD            Type;
**                  SNMP_SYNTAX      Syntax;
**                  WORD            SyntaxLen;
**              } SNMP_OBJECT;
**
**              typedef union _SNMP_SYNTAX
**              {
**                  LONG            LngInt;
**                  LWORD           LngUns;
**                  BYTE            BufChr [SNMP_SIZE_BUFCHR];
**                  LONG            BufInt [SNMP_SIZE_BUFINT];
**                  VOID            *Ptr;
**              } SNMP_SYNTAX;
**
**              The following types can be used in combination
**              with the specified syntax.
**
**              <Type>          ->   <Syntax>
**              SNMP_NULL        ->     - 
**              SNMP_INTEGER     ->   LngInt 
**              SNMP_OCTETSTR    ->   BufChr 
**              SNMP_OBJECTID    ->   BufInt 
**              SNMP_IPADDR      ->   LngUns 
**              SNMP_COUNTER     ->   LngUns 
**              SNMP_GAUGE       ->   LngUns 
**              SNMP_TIMETICKS   ->   LngUns 
**              SNMP_OPAQUE      ->   BufChr 
**              SNMP_DISPLAYSTR  ->   BufChr 
**
**              - WORD listLen
**              Number of objects in list. This number should not
**              exceed 16.
**
**              - WORD *status
**              Pointer to a WORD that receives the status of
**              the performed SNMP request.
**              SNMP_NOERROR    
**              SNMP_TOOBIG     
**              SNMP_NOSUCHNAME 
**              SNMP_BADVALUE   
**              SNMP_READONLY   
**              SNMP_GENERROR
**
**              - WORD *index
**              If status != SNMP_NOERROR this WORD recieves the
**              index of the object that is in error.
**
** DESCRIPTION: Performes an SNMP request.
** RETURNS:     BOOLEAN success.
** SEE ALSO:    MgmtOpen, MgmtClose.    
*******************************************************************/
BOOLEAN MgmtRequest(MGMT        *mgmt,
                    BYTE        *comm,
                    WORD        commLen,
                    WORD        type,
                    SNMP_OBJECT  *list,
                    WORD        listLen,
                    WORD        *status,
                    WORD        *index,
                    ULONG       timeout,
                    USHORT      retries)
{
    STATIC ULONG    id=0;
    SNMP_PDU        pdu;
    BYTE            *rqs;
    unsigned        rqsLen;
    BYTE            *rsp;
    unsigned        rspLen;
    BYTE            commRsp[MGMT_SIZE_COMMUNITY];
    unsigned        commLenRsp;
    unsigned        listLenRsp;
    BOOLEAN         success;
    BYTE            buffer[MGMT_SIZE_BUFFER];


    if (comm==0 || commLen==0)
    {
        comm    = mgmtComm;
        commLen = mgmtCommLen;
    }
    
    if (timeout==0)
        timeout = mgmtTimeout;

    if (retries==0)
        retries = mgmtRetries;
    
    success = FALSE;
    
    rqs     = buffer;
    rqsLen  = MGMT_SIZE_BUFFER;
    
    pdu.Type                = type;
    pdu.Request.Id          = ++id;
    pdu.Request.ErrorStatus = SNMP_NOERROR;
    pdu.Request.ErrorIndex  = 0;

    if (SnmpEnc(&rqs,
                &rqsLen,
                &pdu,
                comm,
                commLen,
                list,
                (unsigned) listLen))
    {
        while (success == FALSE && retries-- > 0)
        {
            if (MgmtSend(mgmt, rqs, rqsLen))
            {
                while (success == FALSE && MgmtRecv(mgmt, &rsp, &rspLen, timeout))
                {
                    if (SnmpDec(rsp,
                                (unsigned) rspLen,
                                &pdu,
                                commRsp,
                                sizeof(commRsp),
                                &commLenRsp,
                                list,
                                (unsigned) listLen,
                                &listLenRsp))
                    {
                        if (pdu.Request.Id == id)
                        {
                            *status = pdu.Request.ErrorStatus;
                            *index  = pdu.Request.ErrorIndex;
                            success = TRUE;
                        }
                    }
                }
            }
        }
    }
    
    return success;
}



BOOLEAN MgmtSend(MGMT *mgmt, BYTE *buffer, WORD length)
{
    switch(mgmt->type)
    {
#ifdef PIPE
        case MGMT_TYPE_PIPE:
            return PipeSend(mgmt, buffer, length);
            break;
#endif
#ifdef UDP
        case MGMT_TYPE_UDP:
            return UdpSend(mgmt, buffer, length);
            break;
#endif
        default:
            return FALSE;
    }
}


BOOLEAN MgmtRecv(MGMT *mgmt, BYTE **buffer, WORD *length, ULONG timeout)
{
    switch(mgmt->type)
    {
#ifdef PIPE
        case MGMT_TYPE_PIPE:
            return PipeRecv(mgmt, buffer, length, timeout);
            break;
#endif
#ifdef UDP
        case MGMT_TYPE_UDP:
            return UdpRecv(mgmt, buffer, length, timeout);
            break;
#endif
        default:
            return FALSE;
    }
}
