package spook;
#**********************************************************************
#* NAME:	spook.pl
#* Description:	This module contains the subroutines
#*		to start and interrogate the spook
#*		snmp requestor.
#**********************************************************************

#**********************************************************************
#* Name:	snmp_open
#* Synopsis:	void snmp_open( int nr, char *host, char *community,
#*				char *medium);
#
#* Description:	This fuction opens snmp connection <nr> to the
#*		host <host>. For the communication, medium <medium>
#*		will be used (u/p) with community <community>.
#* Returns:	none
#**********************************************************************
sub main'snmp_open
{
	local($nr,$host,$comm,$medium) = @_;
	local($resp,$w1,$r1,$w2,$r2);
	$w1 = 'wh10' , $nr;
	$r1 = 'rh10' , $nr;
	$w2 = 'wh20' , $nr;
	$r2 = 'rh20' , $nr;
	pipe($r1,$w1) || die "can't pipe";
	pipe($r2,$w2) || die "can't pipe";
	if ($pid = fork())
	{
		# parent code  
		close($r1);		# parent should
		close($w2);		# parent should 
		select($w1); $| = 1;	# make unbuffered
		select(STDOUT); $|=1;
		sleep(1);		# give deamon time to start
		print $w1 "= host $host\n";
		$resp = <$r2>;
		print $w1 "= community $comm\n";
		$resp = <$r2>;
		print $w1 "= medium $medium\n";
		$resp = <$r2>;
		return 1;
	} 
	elsif (defined $pid)
	{
		# child code
		close($r2);
		close($w1);
		select($w2); $| = 1;
		open(STDOUT,">&$w2") || die "can't redirect STDOUT";
		open(STDIN,"<&$r1")  || die "can't redirect STDIN";
		select(STDOUT);	$| = 1;
		exec 'spook' || die "can't exec spook";
		exit(1);
	}
	elsif ($! =~ /o more process/)
	{
		# nor more processes, try again later
		sleep 5;
		redo FORK;
	}
	else
	{
		# real error
		die "Can't fork: $!\n";
	}
}

#**********************************************************************
#* Name:	snmp_close
#* Synopsis:	void snmp_close( int nr )
#
#* Description:	This fuction closes snmp connection <nr> that
#*		was opened using &snmp_open.
#* Returns:	none
#**********************************************************************
sub main'snmp_close
{
	local($nr) = @_;
	local($w1,$r2,$resp);
	
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 ".\n";
	$resp = <$r2>;
	close($w1);
	close($r2);
}
#**********************************************************************
#* Name:	snmp_community
#* Synopsis:	void snmp_community( int nr , char *comm)
#
#* Description:	This fuction sets the community string for 
#*		snmp connection <nr>.
#* Returns:	none
#**********************************************************************
sub main'snmp_community
{
	local($nr,$comm) = @_;
	local($w1,$r2,$resp);
	
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 "=community $comm\n";
	$resp = <$r2>;
	if(substr($resp,1,1) eq "-")  { return 0; };
	return 1;
}
#**********************************************************************
#* Name:	snmp_host
#* Synopsis:	void snmp_host( int nr , char *host)
#
#* Description:	This fuction sets the host for 
#*		snmp connection <nr>.
#* Returns:	none
#**********************************************************************
sub main'snmp_host
{
	local($nr,$host) = @_;
	local($w1,$r2,$resp);
	
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 "=host $host\n";
	$resp = <$r2>;
	if(substr($resp,1,1) eq "-")  { return 0; };
	return 1;
}
#**********************************************************************
#* Name:	snmp_medium
#* Synopsis:	void snmp_medium( int nr , char *medium)
#
#* Description:	This fuction sets the medium for 
#*		snmp connection <nr>.
#* Returns:	none
#**********************************************************************
sub main'snmp_medium
{
	local($nr,$medium) = @_;
	local($w1,$r2,$resp);
	
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 "=medium $medium\n";
	$resp = <$r2>;
	if(substr($resp,1,1) eq "-")  { return 0; };
	return 1;
}

#**********************************************************************
#* Name:	snmp_info
#* Synopsis:	void snmp_info( int nr , char *mibvar)
#* Description:	This fuction retrieves MIB information about
#*		MIB variables <mibvar>. 
#* Returns:	It returns a list of four elements:
#*			(char * mibvar,
#*			 char * type,
#*			 char * access,
#*			 char * status)
#*		Where <type> is one of:
#*			TimeTicks
#*			Counter
#*			Aggregate
#*			INTEGER
#*			Gauge
#*			OctetString
#*			DisplayString
#*			ObjectID
#*			IpAddress
#*		And <access> is one of:
#*			read-only
#*			write-only
#*			read-write
#*			not-accessible
#*		And <status> is one of:
#*			mandatory
#*			deprecated
#*			optional
#**********************************************************************
sub main'snmp_info
{
	local($nr,$mibvar) = @_;
	local($w1,$r2,$resp,@ret);
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 "\$$mibvar\n";
	$resp = <$r2>; 
	@ret = $resp =~ /^\$([+-])(\S+) (\S+) (\S+) (\S+).?$/;
}

#**********************************************************************
#* Name:	snmp_rsp2lst
#* Synopsis:	void snmp_rsp2lst( char *resp)
#* Description:	This fuction translates  an (Tricklet) responce
#*		into a list of SNMP varaibles.
#* Returns:	list with values.
#**********************************************************************
sub snmp_rsp2lst
{				  
	local($rsp) = @_;
	split(";",$rsp); 
}
#**********************************************************************
#* Name:	snmp_var2lst
#* Synopsis:	void snmp_var2lst( char *var)
#* Description:	This fuction translates  an (Tricklet) varaible
#*		into a list of the form:
#*			($name , $index , $value)
#* Returns:	list 
#**********************************************************************
sub snmp_var2lst
{				  
	local($var) = @_;
	$var =~ /^(.+)\[(.+)\]=(.+)$/;
}

#**********************************************************************
#* Name:	snmp_trans
#* Synopsis:	void snmp_trans( char *type, char *value)
#* Description:	This fuction translates  an (Tricklet) SNMP type
#*		to a perl usable value. <type> constains the
#*		type of SNMP variable, and <value> its value.
#* Returns:	perl value
#**********************************************************************
sub snmp_trans
{
	local($type,$value) = @_;
	
	if($type eq "OctetString")
	{
		$value =~ /^\"(.*)\"/;
		return $1;
	}
	return $value;
}

#**********************************************************************
#* Name:	snmp_parse
#* Synopsis:	void snmp_parse( char *resp)
#* Description:	This fuction parse a spook answer into
#*		 a perl list. The list consists of pairs of
#*		three strings
#*			(name,index,value)
#*		was opened using &snmp_open.
#* Returns:	none
#**********************************************************************
sub snmp_parse
{
	@ret = $resp =~ /^\?([+-])(.+)\[(.+)\]=(.+)$/;
	@info = &snmp_info($nr,$mibvar);
	&snmp_trans($info[2],$ret[3]);
}

sub snmp_raw_get
{
	local($nr,$request) = @_;
	local($w1,$r2,$resp,@ret,@info);
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 "<$request\n";
	$resp = <$r2>; 
	print $w1 "?\n";
	$resp = <$r2>; # should give return code
	if(substr($resp,1,1) eq "-")  { return 0; };
	return substr($resp,2);
}
sub snmp_raw_set
{
	local($nr,$request) = @_;
	local($w1,$r2,$resp,@ret,@info);
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	print $w1 ">$request\n";
	$resp = <$r2>; 
	print $w1 "?\n";
	$resp = <$r2>; # should give return code
	if(substr($resp,1,1) eq "-") { return 0; };
	return substr($resp,2);
}
#**********************************************************************
#* Name:	snmp_get
#* Synopsis:	void snmp_get( int nr, list *value)
#* Description:	This fuction does one SNMP get command over connection
#*		<nr>. This command will contain the MIB varaibles
#*		given in <list>.
#* Returns:	perl value
#**********************************************************************
sub main'snmp_get
{
	local($nr,@list) = @_;
	local($req,$ans,@rlst,$var,@ret);
	
	if(@list > 1) 
		{$req = join(";",@list);}
	else
		{$req = $list[0];}
	$ans = &snmp_raw_get($nr,$req) || return 0;
	@ret = ();
	foreach $var ( &snmp_rsp2lst($ans))
	{
		@rlst = &snmp_var2lst($var);
		push(@ret,$rlst[2]);
	}
	return @ret;
}
#**********************************************************************
#* Name:	snmp_set
#* Synopsis:	void snmp_set( int nr, list *value)
#* Description:	This fuction does one SNMP set command over connection
#*		<nr>. This command will contain the MIB varaibles
#*		given in <list>.
#* Returns:	perl value
#**********************************************************************
sub main'snmp_set
{
	local($nr,@list) = @_;
	local($req,$ans,@rlst,$var,@ret);
	
	if(@list > 1) 
		{$req = join(";",@list);}
	else
		{$req = $list[0];}
	$ans = &snmp_raw_set($nr,$req) || return 0;
	@rlst = split(";",$ans);	# dangerous, no ';' in answers
	foreach $var ( &snmp_rsp2lst($ans))
	{
		@rlst = &snmp_var2lst($var);
		push(@ret,$rlst[2]);
	}
	return @ret;
}
#**********************************************************************
#* Name:	snmp_tbl
#* Synopsis:	void snmp_tbl( int nr, funct fun , list *value)
#* Description:	This fuction does SNMP table request over connection
#*		<nr>. This command will contain the MIB varaibles
#*		given in <list>.
#*		athe answer will be passed to function <fun>
#*		in the form
#*			(# row , #column , $name, $index, $val)
#* Returns:	none
#**********************************************************************
sub main'snmp_tbl
{
	local($nr,$fun,@list) = @_;
	local($w1,$r2,$resp,$request,$ind,$var,@tmp);
	$w1 = 'wh10' , $nr;
	$r2 = 'rh20' , $nr;
	if(@list > 1) 
		{$request = join(";",@list);}
	else
		{$request = $list[0];}
	print $w1 "*$request\n";
	$resp = <$r2>; 
	if(substr($resp,1,1) eq "-") { return 0; };
	
	# loop until spook indicates that is it all over
		
	$i = 1;
	do
	{
		print $w1 "?\n";
		$resp = <$r2>; # should give return code
		$ind = substr($resp,1,1);
		if($ind eq "-") { return 0; };
		if( $ind eq "+")
		{
			$j = 1;
			$ans = substr($resp,2);
			foreach $var ( &snmp_rsp2lst($ans))
			{
				@rlst = &snmp_var2lst($var);
				@tmp = ($i,$j,@rlst);
				&$fun(@tmp);
				$j++;
			}
			$i++;
		}
	} until $ind eq ".";
}
1;		


