/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipfont.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Module with Font Selection functions.
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#define LIBRARY
#include "vipinc.h"

/* Only if pointsize differs more than 1, use vector font */

#define FITSIZE     1
#define MAXSIZES    16
#define VIP_VECTOR 0x8000

typedef struct _fontlist FONTLIST;
    struct _fontlist
    {
        FONTLIST * next;
        int        attribs;
        int        scrCnt;
        char       face[FACESIZE];
        SHORT      scrSizes[MAXSIZES];
        SHORT      scrMBLE[MAXSIZES];
        LONG       matchId[MAXSIZES];
    };


PRIVAT FONTLIST * vipFonts=NULL;
                  
PRIVAT LONG       xPrintRes, yPrintRes;
PRIVAT LONG       xScreenRes, yScreenRes;

PRIVAT void       VipQueryScreenFonts (void);
PRIVAT BOOLEAN    ScaleVectorFont     (HPS hps, SHORT pointSize);
PRIVAT void       ReleaseFont         (VIPFONT *fp);


EXPORT void
VipInitFonts()
{
    xScreenRes=yScreenRes=72;
    xPrintRes=yPrintRes=0;
    vipFonts=NULL;
    VipQueryScreenFonts();
}


/**************************************************************
** NAME:        VipSelectFont                             [OLD]
** SYNOPSIS:    BOOLEAN VipSelectFont (VIPINFO *wip,
**                  CONST char *face, int size, int attr)
** DESCRIPTION: Selects a font for a window.
**              <face> can be FONTFACE_SYSTEM, FONTFACE_HELV,
**              FONTFACE_TIMES, FONTFACE_COUR. <size> must be
**              given in points, attr is unused yet.
** RETURNS:     TRUE, font installed.
**              FALSE, font not found.
**************************************************************/
EXPORT BOOLEAN
VipSelectFont (VIPINFO *wip, CONST char *face, int size, int attr)
{
    if (!wip) return FALSE;
    if (!face)
        wip->font=NULL;
    else
    {
        wip->font=VipCreateFont(face, size, attr);
        if (!wip->font) return FALSE;
    }
    return TRUE;
}


/**************************************************************
** NAME:        VipSetFont                                [API]
** SYNOPSIS:    BOOLEAN VipSetFont (VIPINFO *wip, VIPFONT *fp);
** DESCRIPTION: Selects a font for a window.
**              <fp> is created by VipCreateFont();
**              If <fp> is NULL, the standard system font will
**              be used.
** RETURNS:     void.
** SEE ALSO:    VipCreateFont
**************************************************************/
EXPORT void
VipSetFont (VIPINFO *wip, VIPFONT *fp)
{
    VIPTEST(wip,return)
    if (fp)
        fp->refCount++;         /* First increment use counter */
    ReleaseFont(wip->font);     /* Then decrement use counter prev. font */
    wip->font=fp;
}


/**************************************************************
** NAME:        VipCreateFont                             [API]
** SYNOPSIS:    VIPFONT * VipCreateFont(
**                    CONST char *face, int size, int attr)
** DESCRIPTION: Creates a font.
** RETURNS:     Pointer to VIPFONT, if font was created.
**              NULL, out of memory, or no match found.
** SEE ALSO:    VipSetFont
**************************************************************/
EXPORT VIPFONT *
VipCreateFont(CONST char *face, int size, int attr)
{
    FONTLIST *bestFont;
    FONTLIST *flp;
    VIPFONT *fp;
    int minScore;
    int bestSize;

    if (!vipFonts)
        VipInitFonts();

    fp=VipMalloc(sizeof(VIPFONT));
    if (!fp) return NULL;
#ifdef DEBUG
    DnpapMessage(DMC_TRACE, VIP_ERROR,
            "VipCreateFont(%s, %d, %d)", face, size,attr);
#endif
    fp->lcid=0;
    fp->face=face;
    fp->size=size;
    fp->attr=attr;
    fp->pitch=0;
    fp->match=0;
    fp->refCount=1;

    minScore=32000;
    bestFont=NULL;
    bestSize=8;
    for (flp=vipFonts;flp;flp=flp->next)
    {
        if (strstr(flp->face, face))
        {
            int match, score;
            int k, mindist, dist;

            score=0;
            mindist=10000;
            match=-1;

            if ((flp->attribs&0xFF)!=attr)
                score+=2;
            if (flp->scrCnt)    /* This font has bitmaps */
            {
                for (k=0;k<flp->scrCnt;k++)
                {
                    dist = abs(flp->scrSizes[k]-size);
                    if (dist<mindist)
                    {
                        match=k;
                        mindist=dist;
                    }
                }
            }
            if (mindist<4)
                score+=mindist;
            else if (flp->attribs&VIP_VECTOR)
            {
                match=-1;
                score+=4;
            }
            else    
                score+=mindist;
            if (score<minScore)
            {
                minScore = score;
                bestFont = flp;
                bestSize = match;
            }
        }
    }
    if (bestFont)
    {
        fp->face=bestFont->face;
        if (bestSize!=-1)
        {
            fp->match=bestFont->matchId[bestSize];
            fp->pitch=bestFont->scrMBLE[bestSize];
        }
#ifdef DEBUG
        DnpapMessage(DMC_TRACE, VIP_ERROR,
            "Created => %s, %d, %ld", fp->face, fp->pitch, fp->match);
#endif
        return fp;                      
    }
    VipFree(fp);
    return NULL;
}


/**************************************************************
** NAME:        VipDestroyFont                            [API]
** SYNOPSIS:    void VipDestroyFont(VIPINFO *fp);
** DESCRIPTION: Destroys a font structure. (as soon as no
**              window use the font anymore.)
** RETURNS:     void
**************************************************************/
EXPORT void
VipDestroyFont(VIPFONT *fp)
{
    ReleaseFont(fp);
}


PRIVAT void
ReleaseFont(VIPFONT *fp)
{
    if (fp)
    {
        fp->refCount--;
        if (fp->refCount<=0)
            VipFree(fp);
    }
}

IMPORT BOOLEAN vipPrinting;

EXPORT void
VipUseFont(HPS hps, VIPFONT *fp)
{
    FATTRS fat;
    if (fp)
    {
        fat.usRecordLength=sizeof(FATTRS);
        strcpy(fat.szFacename, fp->face);
        fat.fsSelection=fp->attr;
        fat.lMatch=fp->match; 
        fat.idRegistry=0;
        fat.lAveCharWidth=0;
        fat.fsType=0;
        fat.usCodePage=850;
        if (fp->pitch && !vipPrinting)
        {
            /* Use a bitmap font */
            fat.lMaxBaselineExt=fp->pitch;
            fat.fsFontUse=FATTR_FONTUSE_NOMIX;
        }
        else
        {
            /* Didn't found a bitmap font, select stretchable one */
            fat.lMaxBaselineExt=fp->size;
            fat.fsFontUse=FATTR_FONTUSE_OUTLINE|FATTR_FONTUSE_TRANSFORMABLE;
            if (vipPrinting)
                fat.lMatch=0;
        }
        fp->lcid=1;
        GpiCreateLogFont(hps, NULL, fp->lcid, &fat);
        GpiSetCharSet(hps, fp->lcid);
        if (fat.fsFontUse!=FATTR_FONTUSE_NOMIX)
            ScaleVectorFont(hps, fp->size);
    }
    else
        GpiSetCharSet(hps, 0);
}



EXPORT void
VipReleaseFont(HPS hps, VIPFONT *fp)
{
    if (fp)
    {
        GpiSetCharSet(hps, 0);
        if (fp->lcid) GpiDeleteSetId(hps, fp->lcid);
        fp->lcid=0;
    }
}



PRIVAT BOOLEAN
ScaleVectorFont (HPS hps, SHORT pointSize)
{
    LONG   xRes, yRes ;
    POINTL ptlFont;
    SIZEF  sizfx;

    if (vipPrinting)
    {
        if (xPrintRes==0)
        {
            HDC hdc;
            hdc = GpiQueryDevice (hps) ;
            DevQueryCaps (hdc, CAPS_HORIZONTAL_RESOLUTION, 1L, &xRes) ;
            DevQueryCaps (hdc, CAPS_VERTICAL_RESOLUTION,   1L, &yRes) ;
            xPrintRes=((xRes*300L)/10000L);
            yPrintRes=((yRes*300L)/10000L);
            /* Donno why, but printres must be divided by an extra 10! */
        }
        xRes=xPrintRes;
        yRes=yPrintRes;
    }
    else
    {
        xRes=xScreenRes;
        yRes=yScreenRes;
    }
    ptlFont.x = pointSize * xRes / 72L;
    ptlFont.y = pointSize * yRes / 72L;
    GpiConvert (hps, CVTC_DEVICE, CVTC_PAGE, 1L, &ptlFont) ;
    sizfx.cx = MAKEFIXED (ptlFont.x, 0) ;
    sizfx.cy = MAKEFIXED (ptlFont.y, 0) ;
    return GpiSetCharBox (hps, &sizfx) ;
}


/**************************************************************
** NAME:        VipQueryFontHeight
** SYNOPSIS:    int VipQueryFontHeight(VIPINFO *wip)
** DESCRIPTION: Returns the optimal distance between two
**              text lines.
** RETURNS:     The text height
**************************************************************/
int
VipQueryFontHeight(VIPFONT *fp)
{
    if (!fp) return 15;
    if (fp->pitch)
        return (int)(fp->pitch);
    else
        return ((int)fp->size*12)/10;
}


/**************************************************************
** NAME:        VipQueryFontFaces                         [API]
** SYNOPSIS:    int VipQueryFontFaces(char **face, int size)
** DESCRIPTION: Fills an array with all available font faces.
**              <size> is the number of
**              <char *> available in <face>
**              If <face> it returns the number of available
**              font faces. Otherwise it returns the number
**              of font faces stored in <face>.
** RETURNS:     The number of (available) faces.
**************************************************************/
EXPORT int
VipQueryFontFaces(char **face, int size)
{
    FONTLIST *flp;
    int i=0;

    if (face)
        for (flp=vipFonts;flp;flp=flp->next)
        {
            face[i++]=flp->face;
            if (i==size) break;
        }
    else
        for (flp=vipFonts;flp;flp=flp->next) i++;
    return i;
}


/**************************************************************
** NAME:        VipQueryFontSizes                         [API]
** SYNOPSIS:    int VipQueryFontSizes(CONST char *face,
**                  int *buf, int size)
** DESCRIPTION: Fills an array with all available sizes of
**              a font. If <buf> is NULL, it returns
**              the total number of available (bitmap) sizes.
**              Otherwise it returns the number of sizes
**              copied in <buf>.
** RETURNS:     The number of sizes filled in.
**************************************************************/
EXPORT int
VipQueryFontSizes(CONST char *face, int *buf, int size)
{
    FONTLIST *flp;
    int i, n=0;

    for (flp=vipFonts;flp;flp=flp->next)
        if (strstr(flp->face, face))
            for (i=0;i<flp->scrCnt;i++)
            {
                if (buf && n<size)
                    buf[n]=flp->scrSizes[i];
                n++;
            }
    return n;
}


#define MAXSTEP (65000/sizeof(FONTMETRICS))


PRIVAT void
VipQueryScreenFonts()
{
    FONTMETRICS *pfm;
    FONTLIST *flp;
    LONG count1;
    char face[FACESIZE];
    HPS hps;
    int j;

    /* Get all names of all available fonts */

    VipSetBusy(TRUE);

    /* Get a Presentation space handle */
    hps=WinGetPS(HWND_DESKTOP);

    /* Count the fonts */
    count1=0;
    count1=GpiQueryFonts(hps, QF_PUBLIC, NULL, &count1, sizeof(FONTMETRICS), NULL);

    if (count1>MAXSTEP)
    {
        count1=MAXSTEP;
        DnpapMessage(DMC_WARNING, VIPERR_FONTS,
            "Too many fonts for memory, will use first %ld fonts\n", count1);
    }

    /* Allocate space for all fontnames */
    pfm=VipMalloc(count1*sizeof(FONTMETRICS));

    /* Get all font metrics */
    GpiQueryFonts(hps, QF_PUBLIC, NULL, &count1, sizeof(FONTMETRICS), pfm);

    /* Release the Presentation space */
    WinReleasePS(hps);

    for (j=0;j<(int)count1;j++)
    {
        strcpy(face, pfm[j].szFacename);

        for (flp=vipFonts;flp;flp=flp->next)
            if (strcmp(face, flp->face)==0)
                break;
        if (!flp) /* Not found... Add it to the list of names... */
        {
            flp=VipMalloc(sizeof(FONTLIST));
            if (flp)
            {
                strcpy(flp->face, face);
                flp->scrCnt=0;
                flp->next=vipFonts;
                flp->attribs=0;
                vipFonts=flp;
            }
        }

        if (pfm[j].fsSelection&FM_SEL_ITALIC)
            flp->attribs|=VIP_ITALIC;
        if (pfm[j].fsSelection&FM_SEL_BOLD)
            flp->attribs|=VIP_BOLD;

        if (strstr(face,"Italic"))
            flp->attribs|=VIP_ITALIC;
        if (strstr(face,"Bold"))
            flp->attribs|=VIP_BOLD;

        /* Now get for this font the screen bitmap sizes */
        if (pfm[j].fsDefn&FM_DEFN_OUTLINE)
            flp->attribs|=VIP_VECTOR;

        if ( (pfm[j].fsDefn&FM_DEFN_GENERIC)
          && (pfm[j].sYDeviceRes==(SHORT)yScreenRes))
        {
            int n, size;
            size = pfm[j].sNominalPointSize/10;
            for (n=0;n<flp->scrCnt;n++)
                if (flp->scrSizes[n]==size)
                    break;
            if (n==flp->scrCnt && n<MAXSIZES) /* not found, add it */
            {
                flp->scrSizes[n] = size;
                flp->scrMBLE[n] = (SHORT)pfm[j].lMaxBaselineExt;
                flp->matchId[n] = pfm[j].lMatch;
                flp->scrCnt = n+1;
            }
        }
    }
    VipFree(pfm);
    VipSetBusy(FALSE);
    for (flp=vipFonts;flp;flp=flp->next)
        DnpapMessage(DMC_TRACE, VIP_ERROR,
            "Face: %32s, attribs=%04x, scrCnt=%3d",
            flp->face,
            flp->attribs,
            flp->scrCnt);
}


