/*------------------------- New Concepts Consulting --------------------------
- Application Module Name : TCPIP
- File Name               : getip.cmd
- Date                    : January 1, 2004
- Author                  : Dariusz Piatkowski
-                                                                              
- Description:                                                                 
- This REXX script will retrieve an IP address assigned to a host either 
- through an external server or through a local router. The address
- is an external IP address which may be dynamically assigned by an ISP.
- By default the script will query a user defined IP echo server and           
- will show the returned IP address. If a 'router' parameter is specified      
- the script will assume the use of a local router and will retrieve the      
- IP through the router built-in status web page. If 
-
- 
- DEFAULT ECHO SERVER: www.whatismyip.com
-
- This code is based on the original DNS update client by Dennis Peterson, 
- modified by Vinicius Beltrao.
-
- Modification History
-
-    REVISON        DATE/TIME         AUTHOR
-
-   Rev 0.1.0  JAN 01 2004 00:00:00   Dariusz Piatkowski
- First alpha version of the code.
-
-   Rev 0.1.1  JAN 09 2004 00:00:00   Dariusz Piatkowski
- Added support for LinkSys BEFSR11/BEFSR41/BEFSRU31 routers and cleaned up the
- output.
-
------------------------------------------------------------------------------*/
                      
/* define some globals that we are going to use */
crlf = "0d0a"x
router_mode = 0
echo_server = "www.whatismyip.com"

/* DEBUG control flag */
debug_on = 0
 
/* if using local router define userid and password if needed */                         
router_IP = ''
router_userid = ''
router_password = ''

/* 
- get the command line parameter, since many router models will eventually be 
- supported, and there are many firmware revisions of a particular router 
- model anyways, the parameter is in the form of:
- 
- ManfuacturerName_ModelNo_FirmwareRevision
-
- for example : LinkSys_BEFW11S4V4_150
*/  
 
PARSE ARG op_mode debug_mode .

/* check whether running in debug mode */
if debug_mode = "1" then do
   debug_on = 1
end

/* echo the command line arguments */
call debug_show "Operating Mode =>" || op_mode, debug_on
call debug_Show "Debug Mode =>" || debug_mode, debug_on

/* process the router specific IP status pages */               
SELECT
   WHEN op_mode = "LinkSys_BEFW11S4V4_1500" THEN
      DO
      server = router_IP
      URL = "GET /RouterStatus.htm  HTTP/1.0" || crlf
      URL = URL || "Authorization: Basic " || Base64(router_userid||':'||router_password) || crlf
      URL = URL || "Host: " || router_userid || ":" || router_password || "@" || server || crlf
      router_mode = 1
      END
   
   WHEN (op_mode = "LinkSys_BEFSR41_1456" | op_mode = "LinkSys_BEFSR11_1456" | op_mode = "LinkSys_BEFSRU31_1456") THEN
      DO
      server = router_IP
      URL = "GET /Status.htm  HTTP/1.0" || crlf
      URL = URL || "Authorization: Basic " || Base64(router_userid||':'||router_password) || crlf
      URL = URL || "Host: " || router_userid || ":" || router_password || "@" || server || crlf
      router_mode = 1
      END

   when op_mode = "echo_server" then
      do
      server = echo_server
      URL = "GET http://"|| echo_server || "/ HTTP/1.0" || crlf || crlf
      end

   OTHERWISE
      DO
      nop
      END
END

call debug_show "URL START ->" || URL || "<- URL END", debug_on
   
/* initialize socket package */
IF RxFuncQuery("SockLoadFuncs") THEN
   DO
   rc = RxFuncAdd("SockLoadFuncs","RxSock","SockLoadFuncs")
   rc = SockLoadFuncs()
   END
   
/* choose port number - http port is normally 80 */
port = 80

/* get server name */
rc = SockGetHostByName(server,"host.!")
IF (rc = 0) THEN
   DO
   SAY "Error" h_errno "calling SockGetHostByName("server")"
   EXIT
   END

server = host.!addr;

/* open socket */
socket  = SockSocket("AF_INET","SOCK_STREAM",0)
IF (socket = -1) THEN
   DO
   SAY "Error on SockSocket:" errno
   EXIT
   END

/* catch breaks */
SIGNAL on halt

/* connect socket */
server.!family = "AF_INET"
server.!port   = port
server.!addr   = server

rc = SockConnect(socket,"server.!")
IF (rc = -1) THEN
   DO
   SAY "Error on SockConnect:" errno
   EXIT
   END

rc = SockSend(socket, URL)
IF (rc = -1) THEN
   DO
   SAY "Error on SockSend URL:" errno
   EXIT
   END

/* receive the result from the server */
DO UNTIL rc = 0
   rc = SockRecv(socket,"newData",512)
   call debug_show "DATA START ->" || newData "<- DATA END", debug_on
   returnData = returnData || newData
END

/* now go ahead and retrieve the IP address info from the data sent back */
IF router_mode THEN 
   SELECT
      WHEN op_mode = "LinkSys_BEFW11S4V4_1500" THEN
         location = lastpos('Internet IP Address:', returnData) + 20
      
      WHEN (op_mode = "LinkSys_BEFSR41_1456" | op_mode = "LinkSys_BEFSR11_1456" | op_mode = "LinkSys_BEFSRU31_1456") THEN
         location = lastpos('IP Address:', returnData) + 11
   
      OTHERWISE
         DO
         nop
         END
   END
ELSE 
   location = pos('Your IP is', returnData) + 11
   
/* 
   return the next 160 bytes to capture the additional HTML formatting along with
   the IP address
 */
ip_address = substr(returnData, location, 160)

CALL debug_show "IP SOURCE START ->" || ip_address "<- IP SOURCE END", debug_on

/* parse the string for the IP address */ 
Parse Var ip_address ip_1 "." ip_2 "." ip_3 "." ip_4 .

/* strip off the leading junk before the first part of the IP address */
do WHILE DataType(ip_1) <> "NUM"
   ip_1 = substr(ip_1,2)
end

/* strip off the trailing junk after the last part of the IP address */
do WHILE DataType(ip_4) <> "NUM"
   ip_4 = substr(ip_4, 1, (length(ip_4)-1))
end

/* now assemble the complete IP address */
final_ip_address = ip_1 || '.' || ip_2 || '.' || ip_3 || '.' || ip_4
   
say final_ip_address

/*----------------------------------------------------------------------------*/

/* close socket (and catch signals) */

halt:

rc = SockSoClose(socket)
if (rc = -1) then
   do
   say "Error on SockSoClose:" errno
   exit
   end

exit
                   
/*----------------------------------------------------------------------------*/

/* Base64 encode the userid & password - code provided by Mike Ruskai */

Base64: procedure
parse arg string

tabout='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz'||,
     '0123456789+/'

tabin=xrange('00'x,'3F'x)
bstring=x2b(c2x(string))

bwords=length(bstring)%6+(length(bstring)//6>0)
bsets=bwords%4
rbsets=bsets+((bwords-(bsets*4))>0)
pad=rbsets*4-bwords

its=length(bstring)%24+(length(bstring)//24>0)
ostring=''
do its
    tstring=''
    parse var bstring otrip +24 bstring
    plen=length(otrip)
    catits=plen%6+(plen//6>0)
    otrip=left(otrip,catits*6,'0')

    do catits
        parse var otrip sext +6 otrip
        tstring=tstring||x2c(b2x(sext))
    end

    ostring=ostring||translate(tstring,tabout,tabin)
end
ostring=ostring||copies('=',pad)

return ostring

/*----------------------------------------------------------------------------*/

/* DEBUG output wrapper function */

debug_show: PROCEDURE
PARSE ARG message, debug_on

IF debug_on THEN 
   SAY message

RETURN

/*----------------------------------------------------------------------------*/
