#if defined(__OS2__) || defined(OS2)
#define INCL_DOS
#define INCL_DOERRORS
#include <os2.h>
#endif
#if defined(WIN32) || defined(_WIN32)
#include <windows.h>
#endif

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static FILE * fo_dbg = stderr;
#ifdef DEBUG
static int optDebug = 1;
#else
static int optDebug = 0;
#endif
FILE * fo_v = stderr;
static int optV = 1;

void
usage(void)
{
  const char szMsg[] = \
    "patch943 (" __DATE__ " " __TIME__ ")\n"
    "usage: patch943 [-ms] [-ibm] IBM943 destfile\n"
    "       patch943 --update\n"
    "    -ms        generate destfile as MS932 from IBM943.\n"
    "    -ibm       generate destfile as IBM943C from IBM943.\n"
    "\n"
    "    --update   update mode\n"
    "               (modify IBM943[N] in ULSPATH and generate IBM943N, MS932)\n"
    ;
  printf("%s\n", szMsg);
}




/*
  IBM943 patcher

  Add/replace optional mappings to IBM943
  proper for Japanese use under Japanese codepage (932 and 943).

  * always preserve path separator (\x5c and \x7e).
  * add some mappings for Windows-31J (Microsoft's variant of Shift_JIS).

  UCS to IBM-943
  --------------
  U+005C -> X'5C'   (revserse solidus/Yen sign '\')
  U+007E -> X'7E'   (tilde/overline '~')
  U+2015 -> X'815C' (Windows-31J: Horizontal Bar)
  U+2225 -> X'8161' (Windows-31J: Parallel To)
  U+FF0D -> X'817C' (Windows-31J: Fullwidth Hyphen-Minus)
  U+FF5E -> X'8160' (Windows-31J: Fullwidth Tilde)
  U+FFE4 -> X'FA55' (Windows-31J: Fullwidth Broken Bar)

  IBM-943 to UCS
  --------------
  X'5C'  -> U+005C  (revserse solidus/Yen sign '\')
  X'7E'  -> U+007E  (tilde/overline '~')


  MS-932 patch
  
  UCS to MS-932
  -------------
  U+005C -> X'5C'   (revserse solidus/Yen sign '\')
  U+007E -> X'7E'   (tilde/overline '~')
  U+007F -> X'7F'
  U+0080 -> X'80'
  U+00A0 -> X'F8F0'
  U+00FD -> X'F8F1'
  U+00FE -> X'F8F2'
  U+00FF -> X'F8F3'
  U+2015 -> X'815C' (Windows-31J: Horizontal Bar)
  U+2225 -> X'8161' (Windows-31J: Parallel To)
  U+FF0D -> X'817C' (Windows-31J: Fullwidth Hyphen-Minus)
  U+FF5E -> X'8160' (Windows-31J: Fullwidth Tilde)
  U+FFE4 -> X'FA55' (Windows-31J: Fullwidth Broken Bar)

  MS-932 to UCS
  -------------
  X'5C'   -> U+005C  (revserse solidus/Yen sign '\')
  X'7E'   -> U+007E  (tilde/overline '~')
  X'7F'   -> U+007F
  X'80'   -> U+0080
  X'815C' -> U+2015  (Windows-31J: Horizontal Bar)
  X'8160' -> U+FF5E  (Windows-31J: Fullwidth Tilde)
  X'8161' -> U+2225  (Windows-31J: Parallel To)
  X'817C' -> U+FF0D  (Windows-31J: Fullwidth Hyphen-Minus)
  X'F8F0' -> U+00A0
  X'F8F1' -> U+00FD
  X'F8F2' -> U+00FE
  X'F8F3' -> U+00FF
  X'FA55' -> U+FFE4  (Windows-31J: Fullwidth Broken Bar)

*/


static
unsigned long
peek_le32(const void *m)
{
  const unsigned char *p = m;
  return (p[0] & 0xff) 
         | ((unsigned long)(p[1] & 0xff) << 8)
         | ((unsigned long)(p[2] & 0xff) << 16)
         | ((unsigned long)(p[3] & 0xff) << 24);
}
static
unsigned short
peek_le16(const void *m)
{
  const unsigned char *p = m;
  return (p[0] & 0xff) | ((unsigned short)(p[1] & 0xff) << 8);
}
static
unsigned long
poke_le32(void *m, unsigned long v)
{
  unsigned char *p = m;
  
  p[0] = v & 0xffU;
  p[1] = (v >> 8) & 0xffU;
  p[2] = (v >> 16) & 0xffU;
  p[3] = (v >> 24) & 0xffU;
  return v;
}
static
unsigned short
poke_le16(void *m, unsigned short v)
{
  unsigned char *p = m;
  p[0] = v & 0xffU;
  p[1] = (v >> 8) & 0xffU;
  return v;
}

static
unsigned short
peek_le16o(const void *mbase, long offset)
{
  return peek_le16((char *)mbase + offset);
}
static
unsigned short
poke_le16o(void *mbase, long offset, unsigned short v)
{
  return poke_le16((char *)mbase + offset, v);
}
static
unsigned long
peek_le32o(const void *mbase, long offset)
{
  return peek_le32((char *)mbase + offset);
}
static
unsigned long
poke_le32o(void *mbase, long offset, unsigned long v)
{
  return poke_le32((char *)mbase + offset, v);
}



static
void *
load_file_to_malloc_mem(const char *filename, size_t *filelength)
{
  char *p;
  FILE *fi;
  fpos_t i, n, n_caller;
  size_t i_once;
  int rc;
  
  if (!filename || !filelength) return NULL;
  n_caller = *filelength;
  
  fi = fopen(filename, "rb");
  if (!fi) {
    *filelength = 0;
    return NULL;
  }
  rc = fseek(fi, 0, SEEK_END);
  if (rc == 0) {
    n = ftell(fi);
    rc = fseek(fi, 0, SEEK_SET);
  }
  if (rc != 0 || n == (fpos_t)-1L) {
    fclose(fi);
    *filelength = 0;
    return NULL;
  }
  *filelength = (size_t)n;
  if (n > n_caller) {
    fclose(fi);
    return NULL;
  }
  p = malloc(n);
  if (!p) {
    fclose(fi);
    return NULL;
  }
  
  for(i=0; i>=0 && i<n;) {
    i_once = fread(p + i, 1, n - i, fi);
    if (i_once==0) break;
    i += i_once;
  }
  
  fclose(fi);
  return p;
}

int
save_mem_to_file(const char *filename, const void *mem, size_t buflength)
{
  int rc = 0;
  FILE *fo;
  const char *p;
  fpos_t i;
  size_t i_once;
  
  if (!filename || !mem || (fpos_t)buflength < 0) return -1;
  fo = fopen(filename, "wb");
  if (!fo) return -1;
  p = (const char *)mem;
  for(i=0; i >= 0 && i<(fpos_t)buflength;) {
    i_once = fwrite(p + i, 1, buflength - i, fo);
    if (i_once == 0) {
      rc = -1;
      break;
    }
    i += i_once;
  }
  fclose(fo);
  
  return rc;
}



void *
search_memblk(const void *mem, size_t memlen, const void *keymem, size_t keysize, int search_lasthit)
{
  void *rc = NULL;
  size_t n, nleft;
  if (!mem || !keymem || !memlen || !keysize || memlen < keysize) return NULL;
  
  for(n=0, nleft=memlen-keysize; n<=nleft; n++) {
    if (memcmp((char *)mem + n, keymem, keysize) == 0) {
      rc = (char *)mem + n;
      if (!search_lasthit) break;
    }
  }
  return rc;
}


static unsigned char uctosj_58to7f[(0x7f-0x58+1) * 2] = {
  0x58,0x00, 0x59,0x00, 0x5a,0x00, 0x5b,0x00,
  0xff,0xff, 0x5d,0x00, 0x5e,0x00, 0x5f,0x00,
  0x60,0x00, 0x61,0x00, 0x62,0x00, 0x63,0x00, 
  0x64,0x00, 0x65,0x00, 0x66,0x00, 0x67,0x00, 
  0x68,0x00, 0x69,0x00, 0x6a,0x00, 0x6b,0x00, 
  0x6c,0x00, 0x6d,0x00, 0x6e,0x00, 0x6f,0x00, 
  0x70,0x00, 0x71,0x00, 0x72,0x00, 0x73,0x00, 
  0x74,0x00, 0x75,0x00, 0x76,0x00, 0x77,0x00, 
  0x78,0x00, 0x79,0x00, 0x7a,0x00, 0x7b,0x00, 
  0x7c,0x00, 0x7d,0x00, 0xff,0xff, 0x1c,0x00
};
static unsigned char uctosj_e8toff[(0xff-0xe8+1) * 2] = {
  0x7f,0x00, 0x7f,0x00, 0x7f,0x00, 0x7f,0x00,
  0x7f,0x00, 0x7f,0x00, 0x7f,0x00, 0x7f,0x00,
  0x7f,0x00, 0x7f,0x00, 0x7f,0x00, 0x7f,0x00,
  0x7f,0x00, 0x7f,0x00, 0x7f,0x00, 0x80,0x81,
  0x7f,0x00, 0x7f,0x00, 0x7f,0x00, 0x7f,0x00,
  0x7f,0x00, 0x7f,0x00, 0x7f,0x00, 0x7f,0x00
};
static unsigned char uctosj_2010to2017[(0x2017-0x2010+1)*2] = {
  0x5d,0x81, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0x5c,0x81, 0xff,0xff, 0x61,0x81, 0xff,0xff
};
static unsigned char uctosj_2220to2227[(0x2227-0x2220+1)*2] = {
  0xda,0x81, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xc8,0x81
};
static unsigned char uctosj_f860tof87f[(0xf87f-0xf860+1)*2] = {
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0x59,0xfa,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff
};
static unsigned char uctosj_ff08toff0f[(0xff0f-0xff08+1)*2] = {
  0x69,0x81, 0x6a,0x81, 0x96,0x81, 0x7b,0x81,
  0x43,0x81, 0xff,0xff, 0x44,0x81, 0x5e,0x81
};
static unsigned char uctosj_ff58toff5f[(0xff5f-0xff58+1)*2] = {
  0x98,0x82, 0x99,0x82, 0x9a,0x82, 0x6f,0x81,
  0x62,0x81, 0x70,0x81, 0xff,0xff, 0xff,0xff
};
static unsigned char uctosj_ffe0toffe7[(0xffe7-0xffe0+1)*2] = {
  0x91,0x81, 0x92,0x81, 0xca,0x81, 0x50,0x81,
  0xff,0xff, 0x8f,0x81, 0xff,0xff, 0xff,0xff
};


static unsigned char sjtouc_58to5f[(0x5f-0x58+1)*2] = {
  0x58,0x00, 0x59,0x00, 0x5a,0x00, 0x5b,0x00,
  0xa5,0x00, 0x5d,0x00, 0x5e,0x00, 0x5f,0x00,
};
static unsigned char sjtouc_78to87[(0x87-0x78+1)*2] = {
  0x78,0x00, 0x79,0x00, 0x7a,0x00, 0x7b,0x00,
  0x7c,0x00, 0x7d,0x00, 0x3e,0x20, 0x1a,0x00,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff,
  0xff,0xff, 0xff,0xff, 0xff,0xff, 0xff,0xff
};
static unsigned char sjtouc_8158to8167[(0x8167-0x8158+1)*2] = {
  0x05,0x30, 0x06,0x30, 0x07,0x30, 0xfc,0x30,
  0x14,0x20, 0x10,0x20, 0x0f,0xff, 0x3c,0xff,
  0x1c,0x30, 0x16,0x20, 0x5c,0xff, 0x26,0x20,
  0x25,0x20, 0x18,0x20, 0x19,0x20, 0x1c,0x20
};
static unsigned char sjtouc_fa50tofa57[(0xfa57-0xfa50+1)*2] = {
  0x66,0x21, 0x67,0x21, 0x68,0x21, 0x69,0x21,
  0xe2,0xff, 0xa6,0x00, 0x07,0xff, 0x02,0xff
};



typedef struct PATCH_RGN {
  int rgn_type;
  unsigned rgn_begin;
  unsigned rgn_end;
  void *key_pattern;
  size_t key_pattern_size;
  void *dest_ptr;
  size_t dest_offset;
} PATCH_RGN;

typedef struct PATCH16_MAP {
  int code_type;
  unsigned code_point;
  unsigned short org_value;
  unsigned short new_value;
} PATCH16_MAP;

enum {
  PATCH_END = 0,
  PATCH_UCTOSJ = 1,
  PATCH_SJTOUC
};

static PATCH_RGN prgn_ibm943[] = {
  { PATCH_UCTOSJ, 0x0058, 0x00a0, uctosj_58to7f, sizeof(uctosj_58to7f), 0, 0 },
  { PATCH_UCTOSJ, 0x00e8, 0x00ff, uctosj_e8toff, sizeof(uctosj_e8toff), 0, 0 },
  { PATCH_UCTOSJ, 0x2010, 0x2017, uctosj_2010to2017, sizeof(uctosj_2010to2017), 0, 0 },
  { PATCH_UCTOSJ, 0x2220, 0x2227, uctosj_2220to2227, sizeof(uctosj_2220to2227), 0, 0 },
  { PATCH_UCTOSJ, 0xf860, 0xf8ff, uctosj_f860tof87f, sizeof(uctosj_f860tof87f), 0, 0 },
  { PATCH_UCTOSJ, 0xff08, 0xff0f, uctosj_ff08toff0f, sizeof(uctosj_ff08toff0f), 0, 0 },
  { PATCH_UCTOSJ, 0xff58, 0xff5f, uctosj_ff58toff5f, sizeof(uctosj_ff58toff5f), 0, 0 },
  { PATCH_UCTOSJ, 0xffe0, 0xffe7, uctosj_ffe0toffe7, sizeof(uctosj_ffe0toffe7), 0, 0 },
  { PATCH_SJTOUC, 0x0058, 0x005f, sjtouc_58to5f, sizeof(sjtouc_58to5f), 0, 0 },
  { PATCH_SJTOUC, 0x0078, 0x00ff, sjtouc_78to87, sizeof(sjtouc_78to87), 0, 0 },
  { PATCH_SJTOUC, 0x8158, 0x817e, sjtouc_8158to8167, sizeof(sjtouc_8158to8167), 0, 0 },
  { PATCH_SJTOUC, 0xfa50, 0xfa57, sjtouc_fa50tofa57, sizeof(sjtouc_fa50tofa57), 0, 0 },
  { PATCH_END, 0, 0 }
};

static PATCH16_MAP patch_ibm943c[] = {
  { PATCH_UCTOSJ, 0x005c, 0xffff, 0x005c },
  { PATCH_UCTOSJ, 0x007e, 0xffff, 0x007e },
  { PATCH_UCTOSJ, 0x2015, 0xffff, 0x815c },
  { PATCH_UCTOSJ, 0x2225, 0xffff, 0x8161 },
  { PATCH_UCTOSJ, 0xff0d, 0xffff, 0x817c },
  { PATCH_UCTOSJ, 0xff5e, 0xffff, 0x8160 },
  { PATCH_UCTOSJ, 0xffe4, 0xffff, 0xfa55 },
  { PATCH_SJTOUC, 0x005c, 0x00a5, 0x005c },
  { PATCH_SJTOUC, 0x007e, 0x203e, 0x007e },
  { PATCH_END }
};
static PATCH16_MAP patch_ms932[] = {
  { PATCH_UCTOSJ, 0x005c, 0xffff, 0x005c },
  { PATCH_UCTOSJ, 0x007e, 0xffff, 0x007e },
  { PATCH_UCTOSJ, 0x007f, 0x001c, 0x001c }, /* nothing to do */
  { PATCH_UCTOSJ, 0x0080, 0x007f, 0x0080 },
  { PATCH_UCTOSJ, 0x2015, 0xffff, 0x815c },
  { PATCH_UCTOSJ, 0x2225, 0xffff, 0x8161 },
  { PATCH_UCTOSJ, 0xf8f0, 0xffff, 0x00a0 },
  { PATCH_UCTOSJ, 0xf8f1, 0xffff, 0x00fd },
  { PATCH_UCTOSJ, 0xf8f2, 0xffff, 0x00fe },
  { PATCH_UCTOSJ, 0xf8f3, 0xffff, 0x00ff },
  { PATCH_UCTOSJ, 0xff0d, 0xffff, 0x817c },
  { PATCH_UCTOSJ, 0xff5e, 0xffff, 0x8160 },
  { PATCH_UCTOSJ, 0xffe4, 0xffff, 0xfa55 },
  { PATCH_SJTOUC, 0x005c, 0x00a5, 0x005c },
  { PATCH_SJTOUC, 0x007e, 0x203e, 0x007e },
  { PATCH_SJTOUC, 0x0080, 0x007f, 0x0080 },
  { PATCH_SJTOUC, 0x00a0, 0xffff, 0xf8f0 },
  { PATCH_SJTOUC, 0x00fd, 0xffff, 0xf8f1 },
  { PATCH_SJTOUC, 0x00fe, 0xffff, 0xf8f2 },
  { PATCH_SJTOUC, 0x00ff, 0xffff, 0xf8f3 },
  { PATCH_SJTOUC, 0x815c, 0x2014, 0x2015 },
  { PATCH_SJTOUC, 0x8160, 0x301c, 0xff5e },
  { PATCH_SJTOUC, 0x8161, 0x2016, 0x2225 },
  { PATCH_SJTOUC, 0x817c, 0x2212, 0xff0d },
  { PATCH_SJTOUC, 0xfa55, 0x00a6, 0xffe4 },
  { PATCH_END }
};



static
int
fetch_patch_region(PATCH_RGN *rgn, const void *mem, size_t memlen)
{
  int rc = 0;
  int i;
  
  if (!rgn || !mem || !memlen) return -1;
  
  for(i=0; rgn[i].rgn_type; i++) {
    void *p, *k;
    unsigned nk;
    
    k = rgn[i].key_pattern; nk = rgn[i].key_pattern_size;
    if (!k || !nk)  continue;
    p = search_memblk(mem, memlen, k, nk, 0);
    if (!p) {
      rc = -1;
      break;
    }
    rgn[i].dest_ptr = p;
    rgn[i].dest_offset = (char *)p - (char *)mem;
    if (optDebug) {
      fprintf(fo_dbg, "type=%d %04X-%04X offset=0x%lx\n", 
              rgn[i].rgn_type,
              rgn[i].rgn_begin, rgn[i].rgn_end,
              rgn[i].dest_offset);
    }
  }
  
  return rc;
}


static
int
patch_mem_tables(void *mem, size_t memlen, PATCH16_MAP *pmap, PATCH_RGN *prgn)
{
  int rc = 0;
  int i, j;
  int do_patch;
  
  
  if (!mem || !memlen) return -1;
  if (!pmap || !prgn) return 0;
  for(i=0; pmap[i].code_type; i++) {
    for(j=0,do_patch=0; do_patch==0; j++) {
      if (prgn[j].rgn_type == PATCH_END) break;
      if (prgn[j].rgn_type != pmap[i].code_type) continue;
      if (prgn[j].rgn_begin <= pmap[i].code_point && pmap[i].code_point <= prgn[j].rgn_end) {
        do_patch = !0;
        break;
      }
    }
    if (do_patch) {
      long ofs;
      unsigned short w;
      ofs = prgn[j].dest_offset + (pmap[i].code_point - prgn[j].rgn_begin) * 2;
      w = peek_le16o(mem, ofs);
      if (optDebug) {
        fprintf(fo_dbg, "patch: offset 0x%08lx %04X->%04X\n"
                      , ofs
                      , w
                      , pmap[i].new_value);
      }
#if defined CHECK_ORIGINAL_VALUE
      if (w != pmap[i].org_value) {
        fprintf(stderr, "incorrect offset 0x%X - %04X\n", ofs, peek_le16o(mem, ofs));
        return -1;
      }
#endif
      poke_le16o(mem, ofs, pmap[i].new_value);
    }
  }
  
  return rc;
}


int
do_chk_ibm943(const void *mem, size_t length)
{
  const unsigned char *p = mem;
  int rc = 0;
  unsigned cp;
  unsigned ofs_comment;
  
  if (length < 0xa0) return -1;
  cp = peek_le32(p + 0x0c);
  ofs_comment = peek_le32(p + 0x24);
  if (optDebug) {
    char s[16];
    memcpy(s, p+8, sizeof(s)); s[sizeof(s)-1] = '\0';
    fprintf(fo_dbg, "file type: `%s'\n", s);
    fprintf(fo_dbg, "codepage: %d (0x%x)\n", cp, cp);
    memcpy(s, p+0x10, sizeof(s)); s[sizeof(s)-1] = '\0';
    fprintf(fo_dbg, "ident: `%s'\n",s );
  }
  if (memcmp("ucv\0", p+8, 4) != 0) return -1;
  if (memcmp("IBM-", p+0x10, 4) != 0) return -1;
  if (ofs_comment >= length) return -1;
  
  switch(cp) {
    case 943:
      rc = cp;
      break;
    default:
      return -1;
  }
  
  return rc;
}


int
gen_ibm943c(void *p, const void *psrc, size_t filelen, const char *dest_fn)
{
  int rc;
  
  memcpy(p, psrc, filelen);
  rc = patch_mem_tables(p, filelen, patch_ibm943c, prgn_ibm943);
  if (rc < 0) {
    fprintf(stderr, "err: patching failure (IBM943C).\n");
  }
  else {
#if 0
    memcpy((char *)p + 0x10, "IBM-943C\0\0\0\0\0\0\0\0", 16);
#endif
    if (optV > 0) {
      fprintf(fo_v, "writing %s.\n", dest_fn);
    }
    rc = save_mem_to_file(dest_fn, p, filelen);
    if (rc < 0) 
      fprintf(stderr, "err: writing file %s.\n", dest_fn);
  }
  
  return rc;
}
int
gen_ms932(void *p, const void *psrc, size_t filelen, const char *dest_fn)
{
  int rc;
  
  memcpy(p, psrc, filelen);
  rc = patch_mem_tables(p, filelen, patch_ms932, prgn_ibm943);
  if (rc < 0) {
    fprintf(stderr, "err: patching failure (MS-932).\n");
  }
  else {
#if 0
    memcpy((char *)p + 0x10, "IBM-943M\0\0\0\0\0\0\0\0", 16);
#endif
    if (optV > 0) {
      fprintf(fo_v, "writing %s.\n", dest_fn);
    }
    rc = save_mem_to_file(dest_fn, p, filelen);
    if (rc < 0) 
      fprintf(stderr, "err: writing file %s.\n", dest_fn);
  }
  
  return rc;
}


static char *szSrc;
static char *szDst;
static int optF;
static int optH;
static int optIBM;
static int optMS;
static int optUpdate;

int
getopt(int argc, char **argv)
{
  int rc = 0;
  char c, *s;
  
  while(argc) {
    s = *argv;
    c = *s;
    if (c == '-' || c == '/') {
      c = *++s;
      if (c == '?' || c == 'H' || c == 'h') optH = 1;
      else if (c == 'F' || c == 'f') optF = 1;
      else if (c == 'V' || c == 'v') optV++;
      else if (c == 'Q' || c == 'q') optV = 0;
      else if (strcmp(s, "ms") == 0 || strcmp(s, "ms932") == 0) {
        optMS = 1; optIBM = !optMS;
      }
      else if (strcmp(s, "ibm") == 0 || strcmp(s, "ibm943") == 0 || strcmp(s, "ibm943c") == 0) {
        optIBM = 1; optMS = !optIBM;
      }
      else if (strcmp(s, "-update") == 0) optUpdate = 1;
      else if (strcmp(s, "-debug") == 0) optDebug = 1;
      else {
        rc = -1;
        break;
      }
    }
    else if (!szSrc) szSrc = s;
    else if (!szDst) szDst = s;
    else {
      rc = -1;
      break;
    }
    --argc;
    ++argv;
    ++rc;
  }
  
  if (optDebug) {
    fprintf(fo_dbg, "args: src=%s dst=%s\n", szSrc, szDst);
  }
  
  return rc;
}


char *
search_ibm943_in_ulspath(const char *envvalname, int *is_ibm943n)
{
  char *ulspath, *s, c;
  size_t n;
  static char u[1024]; /* not thread-safe, not re-entrant */
  int is943n = 0;
  
  ulspath = getenv(envvalname ? envvalname : "ULSPATH");
  if (!ulspath) return NULL;
  
  while(*ulspath) {
    s = strchr(ulspath, ';');
    n = s ? (s - ulspath) : strlen(ulspath);
    if (n < sizeof(u)-24) {
      FILE *fchk;
      memcpy(u, ulspath, n);
      if (u[n-1] == '/') u[n-1] = '\\';
      if (u[n-1] == ':') u[n++] = '.';
      if (n > 0 && u[n-1] != '\\') u[n++] = '\\';
      u[n] = '\0';
      if (optDebug) {
        fprintf(fo_dbg, "search path: %s (len=%d)\n", u, n);
      }
      /* find "IBM943N" at first, and second "IBM943" */
      strcpy(u+n, "IBM943N");
      is943n = 1;
      fchk = fopen(u, "rb");
      if (!fchk) {
        strcpy(u+n, "IBM943");
        is943n = 0;
        fchk = fopen(u, "rb");
      }
      if (!fchk) {
        is943n = 1;
        strcpy(u+n, "CODEPAGE\\IBM943N");
        fchk = fopen(u, "rb");
      }
      if (!fchk) {
        is943n = 0;
        strcpy(u+n, "CODEPAGE\\IBM943");
        fchk = fopen(u, "rb");
      }
      if (fchk) {
        fclose(fchk);
        break;
      }
    }
    ulspath += n + (ulspath[n] == ';');
    u[0] = '\0';
  }
  
  if (u[0] == '\0') return NULL;
  if (is_ibm943n) *is_ibm943n = is943n;
  return u;
}


int
main(int argc, char **argv)
{
  char *p, *p2;
  size_t filelen;
  int is_ibm943n;
  int rc;
  
  rc = getopt(argc - 1, argv + 1);
  if (rc <= 0 || optH || (!optUpdate && !szDst) || (!optUpdate && !optMS && !optIBM)) {
    usage();
    return (rc < 0);
  }
  
  filelen = 0;
  if (optUpdate && !szSrc) {
    szSrc = search_ibm943_in_ulspath(NULL, &is_ibm943n);
    if (!szSrc) {
      fprintf(stderr, "err: not found IBM943[N] in ULSPATH.\n", szSrc);
      return 1;
    }
  }
  if (optV > 0) {
    fprintf(fo_v, "loading %s.\n", szSrc);
  }
  p = load_file_to_malloc_mem(szSrc, &filelen);
  if (filelen > 1024 * 1024) {
    fprintf(stderr, "err: the file %s is too long.\n", szSrc);
    return 1;
  }
  p = load_file_to_malloc_mem(szSrc, &filelen);
  if (!p || !filelen) {
    fprintf(stderr, "err: read failure - %s.\n", szSrc);
    return 1;
  }
  
  rc = do_chk_ibm943(p, filelen);
  if (rc != 943) {
    fprintf(stderr, "err: not IBM943.\n");
    rc = 1;
  }
  else {
    rc = fetch_patch_region(prgn_ibm943, p, filelen);
  }
  if (rc < 0) {
    fprintf(stderr, "err: not supported variant of IBM943.\n");
    rc = 1;
  }
  else {
    p2 = malloc(filelen);
    if (!optUpdate) {
      if (optIBM) rc = gen_ibm943c(p2, p, filelen, szDst);
      if (optMS) rc = gen_ms932(p2, p, filelen, szDst);
    }
    else {
      char *szdn;
      szDst = szSrc;
      szdn = strrchr(szDst, '\\');
      szdn = szdn ? szdn + 1 : szDst;
      rc = 0;
      if (!is_ibm943n) {
        /* copy IBM943 to IBM943N */
#if defined(__OS2__) || defined(OS2)
        char szdo[1024];
        strcpy(szdo, szDst);
        strcpy(szdn, "IBM943N");
        if (optV > 0) {
          fprintf(fo_v, "copying %s.\n", szDst);
        }
        rc = DosCopy(szdo, szDst, 0);
        if (rc != 0) rc = -rc;
#else
        strcpy(szdn, "IBM943N");
        rc = save_mem_to_file(szDst, p, filelen);
#endif
      }
      if (rc >= 0) {
        strcpy(szdn, "IBM943");
        rc = gen_ibm943c(p2, p, filelen, szDst);
      }
      if (rc >= 0) {
        strcpy(szdn, "MS932");
        rc = gen_ms932(p2, p, filelen, szDst);
      }
      if (rc < 0) {
        fprintf(stderr, "err: write error on writing %s.\n", szDst);
        rc = 1;
      }
    }
    if (p2) free(p2);
  }
  if (p) free(p);
  
  return rc;
}

