/*
 * document.c - manipulating printing document
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "nup.h"

/*
 * docDispose - dispose docuemnt record
 */

void    docDispose(DOCPTR doc)
{
    if (doc == NULL) {
        return ;
    }
    if (doc->ulink) {
        unlink(doc->fname) ;
    }
    if (doc->fname) free(doc->fname) ;
    if (doc->title) free(doc->title) ;
    free(doc) ;
    return ;
}

/*
 * docFromFile - create document record for a file
 */

DOCPTR  docFromFile(PSZ fname)
{
    DOCPTR  doc ;
    PUCHAR  pc  ;

    /*
     * check if file exist and readable
     */

    if (access(fname, R_OK) != 0) {
        return NULL ;       /* no such file */
    }

    /*
     * create document record
     */

    if ((doc = (DOCPTR) malloc(sizeof(DOCREC))) == NULL) {
        return NULL ;
    }
    memset(doc, 0, sizeof(DOCREC)) ;

    doc->fname = strdup(fname) ;

    if ((pc = strrchr(fname, '/')) != NULL) {
        doc->title = strdup(&pc[1]) ;
    } else if ((pc = strrchr(fname, '\\')) != NULL) {
        doc->title = strdup(&pc[1]) ;
    } else {
        doc->title = strdup(fname) ;
    }
    if (doc->fname == NULL || doc->title == NULL) {
        docDispose(doc) ;
	return NULL ;
    }
    return doc ;
}

/*
 * docFromStdin - create document record for stdin
 */

DOCPTR  docFromStdin(void)
{
    DOCPTR  doc ;
    ULONG   htype, hflag ;
    PUCHAR  tname ;
    FILE    *fp   ;
    int     ch    ;

    /*
     * check if STDIN was redirected
     */
     
    DosQueryHType(0, &htype, &hflag) ;
    if ((htype & 0x07) == 1) {
        return NULL ;       /* STDIN from character device  */
    }
    
    /*
     * copy STDIN into temporary file
     */
    
    if ((tname = tempnam(getenv("TMP"), "nup")) == NULL) {
        return NULL ;
    }
    if ((fp = fopen(tname, "w")) == NULL) {
        return NULL ;
    }
    while ((ch = getchar()) != EOF) {
        putc(ch, fp) ;
    }
    fclose(fp) ;

    /*
     * create document record for temporary file
     */
     
    if ((doc = docFromFile(tname)) == NULL) {
        unlink(tname) ;
	return NULL   ;
    }

    /*
     * replace document title
     */

    free(doc->title) ;
    doc->title = strdup("[stdin]") ;
    doc->ulink = TRUE ;
    
    if (doc->title == NULL || doc->fname == NULL) {
        docDispose(doc) ;
	return NULL ;
    }
    
    return doc ;
}

/*
 * manipulating document list
 */

void    docListInit(DOCPTR list)
{
    list->prev = list ;
    list->next = list ;
}

void    docListTerm(DOCPTR list)
{
    DOCPTR  doc, prev, next ;
    
    while ((doc = list->next) != list) {
        prev = doc->prev  ;
	next = doc->next  ;
	prev->next = next ;
	next->prev = prev ;
	docDispose(doc) ;
    }
}

void    docAppend(DOCPTR list, DOCPTR doc)
{
    DOCPTR  prev, next ;
    
    next = list       ;
    prev = list->prev ;
    
    doc->prev = prev ;
    doc->next = next ;
    prev->next = doc ;
    next->prev = doc ;
}

void    docDelete(DOCPTR list, DOCPTR doc)
{
    DOCPTR  prev, next ;
    
    prev = doc->prev ;
    next = doc->next ;
    
    prev->next = next ;
    next->prev = prev ;
}

DOCPTR  docFirst(DOCPTR list)
{
    DOCPTR  doc ;
    
    if (list == NULL) {
        return NULL ;
    }
    if ((doc = list->next) != NULL && doc != list) {
        return doc ;
    }
    return NULL ;   
}

DOCPTR  docNext(DOCPTR list, DOCPTR doc)
{
    DOCPTR  nxt ;
    
    if (list == NULL || doc == NULL) {
        return NULL ;
    }
    if ((nxt = doc->next) != NULL && nxt != list) {
        return nxt ;
    }
    return NULL ;
}
