/*
 * draw.c - Drawing to Printer
 */

#include <stdio.h>
#include <stdlib.h>

#include "nup.h"
#include "page.h"

/*
 * Printer Variables
 */

HAB     habPrinter = NULLHANDLE ;
HMQ     hmqPrinter = NULLHANDLE ;
HDC     hdcPrinter = NULLHANDLE ;
HPS     hpsPrinter = NULLHANDLE ;

/*
 * fonts for drawing
 *      Current uses NORM, SPEC, for
 *          NORM for normal characters
 *          SPEC for special characters
 *      some printer font does not have glyph for special characters
 */

#define FID_NORM    10
#define FID_BOLD    11
#define FID_LITE    12
#define FID_ITAL    13
#define FID_ULIN    14
#define FID_HIGH    15
#define FID_BLNK    16
#define FID_REVS    17

#define FID_SPEC    20

/*
 * loadFont - set default font for Printer's Presentation Space
 */
 
static  void    loadFont(LONG fid)
{
    SIZEF           sizef ;

    GpiSetCharSet(hpsPrinter, fid) ;
    sizef.cx = MAKEFIXED(charSize.cx, 0) ;
    sizef.cy = MAKEFIXED(charSize.cy, 0) ;
    GpiSetCharBox(hpsPrinter, &sizef) ;
}

/*
 * createFonts - create fonts for drawing
 *      createNamedFont  - create font with given face name
 *      createDeviceFont - create printer device font with same family
 */

static  LONG    createNamedFont(PUCHAR face, LONG fid)
{
    FATTRS  fattrs ;
    LONG    stat   ;

#ifdef  DEBUG
    printf("Create Named Font - Face [%s] ID %d\n", face, fid) ;
#endif

    /*
     * Create Logical Font with given Face Name
     */

    memset(&fattrs, 0, sizeof(fattrs)) ;
    fattrs.usRecordLength = sizeof(FATTRS) ;
    strcpy(fattrs.szFacename, face) ;

    stat = GpiCreateLogFont(hpsPrinter, (PSTR8) NULL, fid, &fattrs) ;

#ifdef DEBUG
    if (stat == GPI_ERROR) {
        printf("GpiCreateLogFont GPI_ERROR %08x\n", WinGetLastError(habPrinter)) ;
    } else if (stat == FONT_MATCH) {
        printf("GpiCreateLogFont FONT_MATCH\n") ;
    } else if (stat == FONT_DEFAULT) {
        printf("GpiCreateLogFont FONT_DEFAULT\n") ;
    } else {
        printf("GpiCreateLogFont UNKNOWN\n") ;
    }
#endif

    return stat ;
}

static  int     createDeviceFont(PCHAR family, LONG fid)
{
    PFONTMETRICS    pfm ;
    ULONG           nReq, nCnt, nRem, i ;
    LONG            stat = GPI_ERROR ;

#ifdef  DEBUG
    printf("Create Device Font - Family [%s]\n", family) ;
#endif

    /*
     * Listup Printer Fonts
     */
     
    nReq = 0 ;
    nCnt = GpiQueryFonts(hpsPrinter, QF_PUBLIC | QF_NO_GENERIC,
                    NULL, &nReq, sizeof(FONTMETRICS), NULL) ;
#ifdef DEBUG
    printf("Found %d Printer Fonts\n", nCnt) ;
#endif
    if ((pfm = (PFONTMETRICS) malloc(sizeof(FONTMETRICS) * nCnt)) == NULL) {
        return stat ;
    }
    nRem = GpiQueryFonts(hpsPrinter, QF_PUBLIC | QF_NO_GENERIC,
                    NULL, &nCnt, sizeof(FONTMETRICS), pfm) ;

#ifdef  DEBUG
    for (i = 0 ; i < nCnt ; i++) {
        printf("%3d %s [%s]\n", i, pfm[i].szFamilyname, pfm[i].szFacename) ;
    }
#endif

    /*
     * Look for matcing font
     */

    stat = GPI_ERROR ;
    
    for (i = 0 ; i < nCnt ; i++) {
	if ((pfm[i].fsDefn & FM_DEFN_OUTLINE) == 0) {
	    continue ;
	}
	if ((pfm[i].fsType & FM_TYPE_FIXED) == 0) {
	    continue ;
	}
	if ((pfm[i].fsType & FM_TYPE_MBCS) == 0) {
	    continue ;
	}
	if (stricmp(pfm[i].szFamilyname, family) != 0) {
	    continue ;
	}
	if ((stat = createNamedFont(pfm[i].szFacename, fid)) == FONT_MATCH) {
	    break ;
	}
    }
    
    free(pfm) ;
    
    return stat ;
}

static  int     createFonts(LAYPTR lay)
{
    PFONTMETRICS    pfm  ;
    LONG            stat ;

    if ((pfm = fontRefer(lay->fname)) == NULL) {
        return -1 ;
    }
#ifdef  DEBUG
    printf("Create Logical Font for - Family [%s] Face [%s]\n", 
                        pfm->szFamilyname, pfm->szFacename) ;
#endif

    stat = GPI_ERROR ;
    
    if (lay->ppf) {
        stat = createDeviceFont(pfm->szFamilyname, FID_NORM) ;
    }
    if (stat != FONT_MATCH) {
        stat = createNamedFont(pfm->szFacename, FID_NORM) ;
    }
    if (stat != FONT_MATCH) {
        return -1 ;
    }
    if ((stat = createNamedFont(pfm->szFacename, FID_SPEC)) != FONT_MATCH) {
        return -1 ;
    }
    return 0 ;
}

/*
 * drawing control varibales
 */

static  int     curPag ;
static  int     curLin ;
static  int     curCol ;
static  POINTL  drawBase ;

/*
 * drawOpen - open printer's drawing area (presentation space)
 */

int     drawOpen(PPRQINFO3 pprq, LAYPTR lay, DOCPTR doc)
{
    DEVOPENSTRUC    dos ;
    SIZEL           siz ;
    UCHAR   szDriverName[128] ;

    if (pprq == NULL || lay == NULL) {
        return -1 ;
    }

    /*
     * open printer's device context
     */

    strcpy(szDriverName, pprq->pszDriverName) ;
    szDriverName[strcspn(szDriverName, ".")] = '\0' ;

    memset(&dos, 0, sizeof(dos)) ;
    dos.pszLogAddress = pprq->pszName     ;
    dos.pszDriverName = szDriverName      ;
    dos.pdriv         = pprq->pDriverData ;
    dos.pszDataType   = "PM_Q_STD"        ;
    dos.pszComment    = ProgramName       ;

    hdcPrinter = DevOpenDC(
                    habPrinter,
                    OD_QUEUED,
                    "*",
                    5,
                    (PVOID) &dos,
                    NULLHANDLE) ;
    if (hdcPrinter == DEV_ERROR) {
        return -2 ;
    }

    /*
     * create printer's presentation space
     */

    siz.cx = siz.cy = 0 ;
    hpsPrinter = GpiCreatePS(
                    habPrinter,
                    hdcPrinter,
                    &siz,
                    PU_TWIPS | GPIF_LONG | GPIT_NORMAL | GPIA_ASSOC) ;
    if (hpsPrinter == GPI_ERROR) {
        DevCloseDC(hdcPrinter) ;
        return -3 ;
    }

    /*
     * calculate page control variables
     */

    pageLayout(lay) ;
    
    /*
     * Marks start of document
     */

    DevEscape(hdcPrinter, DEVESC_STARTDOC,
            (LONG) strlen(doc->title), doc->title, 0, NULL) ;
 
    /*
     * create & load fonts into printer's presentation space
     */

    if (createFonts(lay) != 0) {
        DevEscape(hdcPrinter, DEVESC_ABORTDOC, 0, NULL, 0, NULL) ;
        GpiDestroyPS(hpsPrinter) ;
	DevCloseDC(hdcPrinter)   ;
        return -4 ;
    }
    
    loadFont(FID_NORM) ;        /* set default font */

    curPag = curLin = curCol = 0 ;
    
    return 0 ;
}

/*
 * drawClose - close printer's drawing area
 */

void    drawClose(BOOL ok)
{
    LONG    n  ;
    USHORT  id ;

    if (hdcPrinter != NULLHANDLE) {
        if (ok) {
	    n = sizeof(id) ;
	    DevEscape(hdcPrinter, DEVESC_ENDDOC, 0, NULL, &n, (PBYTE) &id) ;
        } else {
	    DevEscape(hdcPrinter, DEVESC_ABORTDOC, 0, NULL, 0, NULL) ;
        }
    }

    if (hpsPrinter != NULLHANDLE) {
        GpiDestroyPS(hpsPrinter) ;
    }
    hpsPrinter = NULLHANDLE ;

    if (hdcPrinter != NULLHANDLE) {
        DevCloseDC(hdcPrinter) ;
    }
    hdcPrinter = NULLHANDLE ;    

    curPag = curLin = curCol = 0 ;
}

/*
 * drawChar is finite state machine to handle character sequences
 */

#define SQPLAIN     0   /* not in control sequence      */
#define SQESCS      1   /* prefixed by ESC character    */
#define SQESCM      2   /* Multi-byte escape sequnce    */
#define SQCTRL      3   /* control sequence             */
#define SQKANJI     4   /* KANJI Output sequence        */
#define SQKANA      5   /* KANA  Output sequence        */
#define SQSHIFT     6   /* SHIFTED to KANA (KBD-CTRL)   */

static  int     drawStat = SQPLAIN ;
static  UCHAR   drawBuff[32]       ;

static  int     drawInterF = 0 ;
static  int     drawInterC = 0 ;
static  int     drawInter[32]  ;

static  int     drawParamF = 0 ;
static  int     drawParamC = 0 ;
static  int     drawParamQ = 0 ;
static  int     drawParam[32]  ;

static  int     prevLin, prevCol ;

/*
 * doANK - ANK (Ascii/Numeric/Kana) characters
 */

static  void    doANK(PUCHAR cp)
{
    POINTL  pt ;
    
    if ((curCol + 1) > pageCols) {
        curCol += 1 ;
        return ;
    }

    pt.y = drawBase.y - cellSize1.cy * curLin ;
    pt.x = drawBase.x + cellSize1.cx * curCol ;
    GpiMove(hpsPrinter, &pt) ;
    GpiCharString(hpsPrinter, 1, cp) ;

    curCol += 1 ;
}

/*
 * doKANJI - Kanji Characters
 */

static  BOOL    isSpecial(PUCHAR cp)
{
    ULONG   kc ;

    kc = ((cp[0] << 8) & 0xff00) + (cp[1] & 0xff) ;

    if (kc >= 0x849f && kc <= 0x84bd) {         /* JIS Keisen */
        return TRUE ;
    }else if (kc >= 0x81b8 && kc <= 0x81fc) {   /* JIS Kigou  */
        return TRUE ;
    } else if (kc >= 0xfa40 && kc <= 0xfa53) {  /* Non JIS    */
        return TRUE ;
    } else if (kc >= 0xea9f && kc <= 0xeaa2) {  /* JIS Kanji  */
        return TRUE ;
    }
    return FALSE ;
}

static  void    doKANJI(PUCHAR cp)
{
    POINTL  pt ;
    
    if ((curCol + 2) > pageCols) {
        curCol += 2 ;
        return ;
    }
    
    pt.y = drawBase.y - cellSize1.cy * curLin ;
    pt.x = drawBase.x + cellSize1.cx * curCol ;
    GpiMove(hpsPrinter, &pt) ;

    if (isSpecial(cp)) {
        loadFont(FID_SPEC) ;
        GpiCharString(hpsPrinter, 2, cp) ;
	loadFont(FID_NORM) ;
    } else {
        GpiCharString(hpsPrinter, 2, cp) ;
    }
    curCol += 2 ;
}

/*
 * Control Characters
 */

static  void    doBS(void)      /* Back Space   */
{
    if (curCol > 0) {
        curCol -= 1 ;
    }
}

static  void    doHT(void)      /* Horizontal Tab   */
{
    while (1) {
        curCol += 1 ;
	if ((curCol % 8) == 0) {
	    break ;
	}
    }    
}

static  void    doLF(void)      /* Line Feed    */
{
    curLin += 1 ;
}

static  void    doVT(void)      /* Vertical Tab, a.k.a Form Feed    */
{
    curLin = pageLins ;
}

static  void    doCR(void)      /* Carrige Return   */
{
    curCol = 0 ;
}

static  void    doNL(void)      /* Newline (CR + LF)    */
{
    curLin += 1 ;
    curCol  = 0 ;
}

static  void    doINDEX(void)   /* Index Cursor */
{
    curLin += 1 ;
}

static  void    doREVIN(void)   /* Reverse Index    */
{
    curLin -= 1 ;
}

static  doCURUP(void)           /* cursor up        */
{
    int     cnt ;
    
    cnt = drawParam[0] ? drawParam[0] : 1 ;
    curLin -= cnt ;
}

static  doCURDN(void)           /* cursor down      */
{
    int     cnt ;
    
    cnt = drawParam[0] ? drawParam[0] : 1 ;
    curLin += cnt ;
}

static  doCURFW(void)           /* cursor forward   */
{
    int     cnt ;
    
    cnt = drawParam[0] ? drawParam[0] : 1 ;
    curCol += cnt ;
}

static  doCURBK(void)           /* cursor backward  */
{
    int     cnt ;
    
    cnt = drawParam[0] ? drawParam[0] : 1 ;
    curCol -= cnt ;
}

static  doCURPOS(void)          /* cursor position  */
{
    int     row, col ;
    
    row = drawParam[0] ? drawParam[0] : 1 ;
    col = drawParam[1] ? drawParam[1] : 1 ;
    curLin = row - 1 ;
    curCol = row - 1 ;
}

static  doSAVE(void)            /* Save Cursor Position */
{
    prevLin  = curLin  ;
    prevCol  = curCol  ;
}

static  doRESTORE(void)         /* Restore Cursor Position  */
{
    curLin = prevLin ;
    curCol = prevCol ;
}

/*
 * multiin - start of multibyte escape sequence
 */

static  void    multiin(USHORT start)
{
    drawInterF = 1 ;
    drawInterC = 0 ;
    memset(drawInter, 0, sizeof(drawInter)) ;
    
    drawInter[drawInterC++] = start ;
    drawStat = SQESCM ;
}

/*
 * ctrlin - start of control sequence
 */

static  void    ctrlin(void)
{
    drawParamF = drawParamC = drawParamQ = 0 ;
    drawInterF = drawInterC = 0 ;
    
    drawStat = SQCTRL ;
}

/*
 * c0ctrl - Control 0 Characters
 */

static  void    c0ctrl(USHORT ch)
{
    switch (ch) {
        case 0x08 : doBS()            ; break ;    /* Back Space       */
	case 0x09 : doHT()            ; break ;    /* Horizontal Tab   */
	case 0x0a : doNL()            ; break ;    /* Line Feed        */
        case 0x0c : doVT()            ; break ;    /* Vertical Tab     */
	case 0x0d : doCR()            ; break ;    /* Carridge Return  */
	case 0x1b : drawStat = SQESCS ; break ;
    }
    if (drawStat != SQESCS) {
        drawStat = SQPLAIN ;
    }
}

/*
 * c1ctrl - Control 1 Characters
 */

static  void    c1ctrl(USHORT ch)
{
    ch -= 0x40 ;
    
    switch (ch) {
        case '[' :  ctrlin() ; break ;
	case 'D' : 
    }
    if (drawStat != SQCTRL) {
        drawStat = SQPLAIN ;
    }
}

/*
 * escsfp  -  Dispatch to ESC Fp Controls
 *      Here terminator is in range 30..3f
 */

static  void    escsfp(USHORT ch)
{
    switch (ch) {
        case '7' : doSAVE()    ; break ;
        case '8' : doRESTORE() ; break ;
        default  : break ;
    }
    drawStat = SQPLAIN ;
}

/*
 * escsfs  -  Dispatch to ESC Fs Controls
 *      Here, terminator is in range 60..7f
 */

static  void    escsfs(USHORT ch)
{
    drawStat = SQPLAIN ;
}

/*
 * escmlt  -  Dispatch to Multi-byte ESC controls
 */

static  void    escmlt(USHORT c)
{
    drawStat  = SQPLAIN ;
}

/*
 * e s c c s i  -  Dispatch to CSI Controls
 *      Here, terminator is in range 40..7f
 */

static  void    esccsi(USHORT ch)
{
    switch(ch) {
        case 'A' : doCURUP()   ; break ;
        case 'B' : doCURDN()   ; break ;
        case 'C' : doCURFW()   ; break ;
        case 'D' : doCURBK()   ; break ;
        case 'f' :
        case 'H' : doCURPOS()  ; break ;
        case 'S' : doSAVE()    ; break ;
        case 'u' : doRESTORE() ; break ;
        default  : break ;
    }
    drawStat = SQPLAIN ;
}

/*
 * seqesc - ESC sequence
 */

static  void    seqesc(USHORT ch)
{
    if (ch < 0x20) {
        c0ctrl(ch) ;
	return ;
    }

    /*
     * ESC followed by 20..2f introduces multi-byte escape sequence
     */

    if (ch < 0x30) {
        multiin(ch) ;
        return ;
    }

    /*
     * ESC followed by 30..3f terminates ESC Fp sequence.
     *  It request function determinated with terminator.
     */

    if (ch < 0x40) {
        escsfp(ch) ;
        return ;
    }

    /*
     * ESC followed by 40..5f terminate ESCs Fe sequence.
     *  It request controls defined in C1 Control Characters.
     */

    if (ch < 0x60) {
        c1ctrl(ch + 0x40) ;
        return ;
    }


    /*
     * ESC followed by 40..5f terminate ESCs Fe sequence.
     *  It request controls defined in C1 Control Characters.
     */

    if (ch < 0x60) {
        c1ctrl(ch + 0x40) ;
        return ;
    }

    /*
     * ESC followed by 60..7e terminates ESC Fs sequnce.
     *  It request function determinated with terminator.
     */

    if (ch < 0x7f) {
        escsfs(ch) ;
        return ;
    }

    /*
     * return to PLAIN sequence for undefined characters
     */

    drawStat = SQPLAIN ;
}

/*
 * seqmlt - characters in Multibyte Escape Sequence
 */

static  void    seqmlt(USHORT ch)
{
    if (ch < 0x20) {
        c0ctrl(ch) ;
        return ;
    }

    /*
     * 20..2f is intermediate character
     */

    if (ch < 0x30) {
        drawInter[drawInterC++] = ch ;
        drawInterF = 1 ;
        return ;
    }

    /*
     * Characters in range 30..7f terminate this sequence
     */

    if (ch < 0x80) {
        escmlt(ch) ;
        return ;
    }

    /*
     * Return to plain for undefined characters
     */

    drawStat = SQPLAIN ;
}

/*
 * seqctl - characters in Control Sequence
 */

static  void    seqctl(USHORT ch)
{
    if (ch < 0x20) {
        c0ctrl(ch) ;
        return ;
    }

    /*
     * 20..2f is intermediate character
     */

    if (ch < 0x30) {
        drawInter[drawInterC++] = ch ;
        drawInterF = 1 ;
        return ;
    }

    /*
     * Characters in range 30..3f is paramater string
     */

    if (ch < 0x40) {
        drawParamF = 1 ;
        if (ch <= 0x39) {
            drawParam[drawParamC] *= 10 ;
            drawParam[drawParamC] += (ch - '0') ;
        } else if (ch == 0x3b) {
	    drawParamC += 1 ;
        } else if (ch == 0x3f) {
            drawParamQ = 1 ;
        }
        return ;
    }

    /*
     * Characters in range 40..7f is terminator
     */

    if (ch < 0x80) {
        if (drawParamF) {
	    drawParamC += 1 ;
        }
        esccsi(ch) ;
        return ;
    }

    /*
     * return to PLAIN for undefined characters
     */

    drawStat = SQPLAIN ;
}

/*
 * drawMarker - draw center marker
 */

static  void    drawMarker(void)
{
    POINTL  pt1, pt2 ;

    if (formSize.cy > formSize.cx) {
        pt1.y = pt2.y = (formSize.cy / 2) - formClip.yBottom ;
        pt1.x = 0 ;
        pt2.x = pt1.x + widMarker ;
        GpiMove(hpsPrinter, &pt1) ;
        GpiLine(hpsPrinter, &pt2) ;
        pt1.x = formClip.xRight - formClip.xLeft ;
        pt2.x = pt1.x - widMarker ;
        GpiMove(hpsPrinter, &pt1) ;
        GpiLine(hpsPrinter, &pt2) ;
    } else {
        pt1.x = pt2.x = (formSize.cx / 2) - formClip.xLeft ;
	pt1.y = formClip.yTop - formClip.yBottom ;
	pt2.y = pt1.y - widMarker ;
        GpiMove(hpsPrinter, &pt1) ;
        GpiLine(hpsPrinter, &pt2) ;
	pt1.y = 0 ;
	pt2.y = pt1.y + widMarker ;
        GpiMove(hpsPrinter, &pt1) ;
        GpiLine(hpsPrinter, &pt2) ;
    }
}

/*
 * drawFrame - draw page frame
 */

static  void    drawFrame(int index)
{
    POINTL  pt1, pt2 ;

    pt1.x = pageRect[index].xLeft   - widFrame ;
    pt1.y = pageRect[index].yBottom - widFrame ;
    pt2.x = pageRect[index].xRight  + widFrame ;
    pt2.y = pageRect[index].yTop    + widFrame ;
    GpiMove(hpsPrinter, &pt1) ;
    GpiBox(hpsPrinter, DRO_OUTLINE, &pt2, 0, 0) ;
}

/*
 * drawChar - draw a character into printer's presentation space
 *
 *      returns next page number for upper layer page control
 */

int     drawChar(int page, int ch)
{
    int     index    ;
    LONG    ocnt     ;
    BOOL    newPage = FALSE ;

    /*
     * Control Pages
     */
     
    if (curPag == 0) {          /* first time in document   */
        curPag = 1 ;            /* start first page here    */
	curLin = curCol = 0 ;
	newPage = TRUE ;
    }
    if (curLin >= pageLins) {   /* need form feed for new page  */
        curPag += 1 ;
	curLin = curCol = 0 ;
	newPage = TRUE ;
    }
    if (newPage) {

        /*
	 * set page location
	 */

        index = (curPag - 1) % pagePerForm ;
        drawBase.x = pageRect[index].xLeft   ;
        drawBase.y = pageRect[index].yTop - cellSize1.cy ;
#ifdef DEBUG
        printf("Page %d at x %d - %d, y %d - %d\n", page,
            pageRect[index].xLeft, pageRect[index].xRight,
	    pageRect[index].yBottom, pageRect[index].yTop) ;
        fflush(stdout) ;
#endif
        /*
	 * form feed
	 */

        if (index == 0 && curPag >= 2) {
            DevEscape(hdcPrinter, DEVESC_NEWFRAME, 0, NULL, &ocnt, NULL) ;
        }

        /*
	 * mark page center if required
	 */
	
	if (pageMarker && index == 0) {
            drawMarker() ;
	}
	
        /*
	 * mark page frame
	 */
	
	if (pageFrame) {
	    drawFrame(index) ;
        }
    }
    
    /*
     * draw a Character on Page
     */
     
    switch (drawStat) {

    case SQPLAIN :
        drawBuff[0] = ch ;
        if ((ch >= 0x20) && (ch <= 0x7e)) {
	    doANK(drawBuff) ;
	} else if ((ch < 0x20) || (ch == 0x7f) || (ch == 0x80)) {
	    c0ctrl(ch) ;
	} else if (ch <= 0xa0) {
	    drawStat = SQKANJI ;
	} else if (ch <= 0xdf) {
	    doANK(drawBuff) ;
	} else if (ch < 0xfc) {
	    drawStat = SQKANJI ;
	}
	break ;

    case SQKANJI :
        drawBuff[1] = ch  ;
        doKANJI(drawBuff) ;
	drawStat = SQPLAIN ;
	break ;

    case SQESCS  :          /* Prefixed by ESC              */
        seqesc(ch) ;
        break ;

    case SQESCM   :         /* In Multi-byte ESC Seq.       */
        seqmlt(ch) ;
        break ;

    case SQCTRL   :         /* Control sequence             */
        seqctl(ch) ;
        break ;
    default :
        break ;
    }
    
    return curPag ;
}
