/*
 * kermit - supports KERMIT file transfer protocol
 *
 *	Kermit Protocol was origranlly developped by Frank da Cruz,
 *	in Columbia University.  This implentation is based on
 *	programs written in Kermit Protocol Manual, fifth edition.
 *	Many part of this implementation was directory taken from
 *	that program.
 */

#define	INCL_BASE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <os2.h>
#include "shifty.h"

/*
 * Common Constants
 */

#define	NUL	0	/* Characters on ASCII	*/
#define	SOH	1
#define	LF	10
#define	CR	13
#define	SP	32
#define	DEL	127

#define	KKDEFPS	80		/* default packet size		*/
#define	KKMINPS	16		/* minimum packet size		*/
#define	KKMAXPS	94		/* maximum packet size		*/
#define	KKDEFSOH	SOH	/* default packet marker	*/
#define	KKDEFEOL	CR	/* Default packet terminator	*/
#define	KKPCKCHK	1	/* always single char check sum	*/

#define	KKMYTO	30		/* time out me on 30 seconds	*/
#define	KKINITO	60		/* initial time out for you	*/
#define	KKDEFTO	6		/* defualt time out for you	*/
#define	KKMINTO	2		/* minimum time out for you	*/
#define	KKMAXTO	60		/* maximum time out for you	*/

#define	KKDEFQCTL	'#'	/* Default control quote	*/
#define	KKDEFEBQ	'&'	/* default 8-th bit quote	*/
#define	KKDEFREP	'~'	/* default repeat prefix	*/

/*
 * M A C R O  -  for character encoding/decoding
 *
 * tochar: converts a control character to a printable one by adding a space.
 *
 * unchar: undoes tochar.
 *
 * ctl:    converts between control characters and printable characters by
 *         toggling the control bit (ie. ^A becomes A and A becomes ^A).
 */

#define tochar(ch)  ((ch) + ' ')
#define unchar(ch)  ((ch) - ' ')
#define ctl(ch)     ((ch) ^ 64 )

/*
 * M A C R O  -  for packet sequencing
 *
 * incseq: increment packet's sequence numebr.
 *
 * prvseq: get previous packet's sequence numebr.
 */

#define	incseq(x)	((((x) + 1) % 64))
#define	prvseq(x)	((((x) - 1) < 0) ? 63 : ((x) - 1))

/*
 * Kermit Control Variables
 */
	/* expanded command line arguments (in local.c)	*/

extern	USHORT	Xargc    ;
extern	UCHAR	*Xargv[] ;

	/* Protocol Parameters		*/

static	int	KKiowait = 10      ;	/* Retry Count adj. to 1 sec	*/
static	char	KKmypsiz = KKDEFPS ;	/* packet length to receive	*/
static	char	KKurpsiz = KKDEFPS ;	/* packet length to send	*/
static	char	KKmysoh  = KKDEFSOH ;	/* SOH of receive packet	*/
static	char	KKursoh  = KKDEFSOH ;	/* SOH of sending packet	*/
static	char	KKmynpad = 0 ;		/* padding for me		*/
static	char	KKurnpad = 0 ;		/* padding for you		*/
static	char	KKmycpad = NUL ;	/* pad character for me		*/
static	char	KKurcpad = NUL ;	/* pad character for you	*/
static	char	KKmytout = KKMYTO  ;	/* time out for me		*/
static	char	KKurtout = KKINITO ;	/* time out for you		*/
static	char	KKmyeol = CR ;		/* packet terminator for me	*/
static	char	KKureol = CR ;		/* packet terminator for you	*/
static	char	KKmyqctl = KKDEFQCTL ;	/* control quote for me		*/
static	char	KKurqctl = KKDEFQCTL ;	/* control quote for you	*/
static	char	KKebqflg = FALSE ;	/* TRUE if need 8-th bit quote	*/
static	char	KKebq = KKDEFEBQ ;	/* 8-th bit quote character	*/
static	char	KKrepflg = FALSE ;	/* TRUE if does repeat perfix	*/
static	char	KKrep = KKDEFREP ;	/* Repeat prefix character	*/

	/* files to be send to host	*/

static	char	**filelist = { 0 } ;	/* List of Files	*/
static	char	filecount  =   0   ;	/* Number of Files	*/

	/* file now receiving	*/

static	FILE	*sendfp    = { 0 } ;	/* Sending   File Pointer	*/
static	FILE	*recvfp    = { 0 } ;	/* Receiving File Pointer	*/
static	char	ofname[64] = { 0 } ;	/* Original  File Name		*/
static	char	cfname[64] = { 0 } ;	/* Canonical File Name		*/

	/* Packet Data, Send/Receive	*/

static	int	pksize = 0 ;	/* Length of Packet Data to Send	*/
static	char	spacket[KKMAXPS] = { 0 } ;	/* Send   Packet	*/
static	char	rpacket[KKMAXPS] = { 0 } ;	/* Receive Packet	*/

	/* Protocol Control Variables	*/

#define	MAXTRY_INIT	100

static	int	MAXTRY = MAXTRY_INIT ;	/* Times to retry a packet */
static	int	OLDTRY = MAXTRY_INIT ;

static	char	status = 0 ;	/* Protocol Status		*/
static	int	numseq = 0 ;	/* Packet's sequence number	*/
static	int	numtry = 0 ;	/* Retry count on new packet	*/
static	int	numold = 0 ;	/* Retry count on old packet	*/
static	char	*error = "" ;	/* Reason why FTP failed	*/

#define	tune_retry(x)	if (x > (MAXTRY >> 1)) {\
				OLDTRY = MAXTRY ;\
				if (MAXTRY < 0x4000) {\
					MAXTRY <<= 1 ;\
				}\
			}

/*
 * Kermit Status Display
 */

static	long	pktnum = 0 ;
static	char	pktmod = 0 ;
static	char	*pstat = NULL ;
static	char	pfile[32]  ;
static	char	mbuff[128] ;

/*
 * Error Messages (for reduce memory usage)
 */

static	char	err_too_many_retries[] 	= "too many retires" ;
static	char	err_error_packet[] 	= "error packet"     ;
static	char	err_recv_failed[] 	= "recv failed"      ;
static	char	err_unknown_pkt[] 	= "unknown packet"   ;
static	char	err_ack_lost[]		= "ACK lost"         ;
static	char	err_seq_error[] 	= "SEQ error"        ;
static	char	err_file_open[] 	= "File Open Error"  ;
static	char	err_from_host[] 	= "Error from HOST"  ;
static	char	err_user_abort[] 	= "User Abort"       ;

/*
 * K K p i n i t
 *	Initialize Screen Management
 */

static void KKpinit(char mode)
{
    pktnum = 0 ;
    pktmod = 0 ;
    pstat  = "Initialize" ;
    strcpy(pfile, "*****") ;
    sprintf(mbuff, "%-10s <%-12s> Packets <%8ld> ",
				pstat, pfile, pktnum) ;
    prompt(mbuff) ;
}
 
/*
 * K K p s t a r t
 *	Print message for transaction start
 */

static void KKpstart(char mode, char *fname)
{
    pktnum = 0 ;
    pktmod = 0 ;
    pstat = (mode == 'r') ? "Receiving" : "Sending" ;
    strcpy(pfile, fname) ;
    sprintf(mbuff, "%-10s <%-12s> Packets <%8ld> ",
				pstat, pfile, pktnum) ;
    prompt(mbuff) ;
}

/*
 * K K p p a c k
 *	Print Packet Transaction
 */

static void KKppack(void)
{
    pktnum += 1 ;

    sprintf(mbuff, "%-10s <%-12s> Packets <%8ld> ",
					pstat, pfile, pktnum) ;
    prompt(mbuff) ;
}

/*
 * K K p e n d
 *	Print File Transfer End
 */

static void KKpend(int ok, char *e)
{
    if (ok) {
	sprintf(mbuff, "%-10s <%-12s> Packets <%8ld> Complete",
					pstat, pfile, pktnum) ;
    } else {
	printf("\rFailed") ;
	sprintf(mbuff, "%-10s <%-12s> Packets <%8ld> Failed by %s",
					pstat, pfile, pktnum, e) ;
    }
    prompt(mbuff) ;

    pktnum = 0 ;
    pktmod = 0 ;
    pstat  = "Initialize" ;
    strcpy(pfile, "*****") ;
}
		
/*
 * K K p r e p c k
 *	Print message in Error Packet
 */

static void KKprepck(char *msg, int len)
{
    msg[len] = '\0' ;
    sprintf(mbuff, "Host Abort by %s", msg) ;
    prompt(mbuff) ;
}

/*
 * Kermit Packet Operations
 */

/*
 * K K p r e s e t  -  Set Kermit Parameters to default.
 */

static	void KKpreset(void)
{
    KKmypsiz = KKurpsiz = KKDEFPS ;
    KKmytout = KKMYTO  ;
    KKurtout = KKINITO ;
    KKmynpad = KKurnpad = 0   ;
    KKmycpad = KKurcpad = NUL ;
    KKmyeol  = KKureol  = CR  ;
    KKmyqctl = KKurqctl = KKDEFQCTL ;

    KKebqflg = FALSE ; KKebq = KKDEFEBQ ;
    KKrepflg = FALSE ; KKrep = KKDEFREP ;
}

/*
 * K K p e n c  -  Encode Parameters for packet data.
 */

static	int KKpenc(char *data)
{
    data[0] = (UCHAR) tochar(KKmypsiz) ;
    data[1] = (UCHAR) tochar(KKmytout) ;
    data[2] = (UCHAR) tochar(KKmynpad) ;
    data[3] = (UCHAR) ctl(KKmycpad)    ;
    data[4] = (UCHAR) tochar(KKmyeol)  ;
    data[5] = (UCHAR) KKmyqctl         ;
    return(6) ;
}

/*
 * K K p d e c  -  Decode parameters in packet.
 */

static void KKpdec(char *data, int len)
{
    KKurpsiz = (UCHAR) ((len-- > 0) ? unchar(data[0]) : KKDEFPS) ;
    KKurtout = (UCHAR) ((len-- > 0) ? unchar(data[1]) : KKDEFTO) ;
    KKurnpad = (UCHAR) ((len-- > 0) ? unchar(data[2]) : 0) ;
    KKurcpad = (UCHAR) ((len-- > 0) ? ctl(data[3]) : NUL)  ;
    KKureol  = (UCHAR) ((len-- > 0) ? unchar(data[4]) : KKDEFEOL) ;
    KKurqctl = (UCHAR) ((len-- > 0) ? data[5] : KKDEFQCTL) ;
    KKebqflg = (UCHAR) (KKrepflg = FALSE) ;
}
	
/*
 * K K s e n c  -  Encode string and generate packet data.
 *	char	*str 		NULL terminate string
 *	char	*pkt 		Packet data buffer
 */

static int  KKsenc(char *str, char *pkt)
{
    int	    c7, c8, i ;

    i = 0 ;
    while ((c8 = *str++) != '\0') {
	c7 = c8 & 0177 ;
	if ((c7 < SP) || (c7 == DEL) || (c7 == KKurqctl)) {
	    pkt[i++] = KKurqctl ;
	    if (c7 != KKurqctl) {
		c8 = ctl(c8) ;
		c7 = ctl(c7) ;
	    }
	}
	pkt[i++] = (char) c8 ;

	if (i >= (KKurpsiz - 8)) {
	    break ;
	}
    }
    return(i) ;
}

/*
 * K K f e n c  -  Encode file data and generate packet data.
 */

static int KKfenc(FILE *fp, char *pkt)
{
    int	    c7, c8, i ;

    i = 0 ;
    while ((c8 = getc(fp)) != EOF) {
	c7 = c8 & 0177 ;
	if ((c7 < SP) || (c7 == DEL) || (c7 == KKurqctl)) {
	    pkt[i++] = KKurqctl ;
	    if (c7 != KKurqctl) {
		c8 = ctl(c8) ;
		c7 = ctl(c7) ;
	    }
	}
	pkt[i++] = (char) c8 ;

	if (i >= (KKurpsiz - 8)) {
	    break ;
	}
    }
    return(i) ;
}

/*
 * K K s d e c  -  Accept packet data and decode to string.
 *	char	*str 		Genrating string
 *	char	*pkt 		encoded packet data
 *	int	len  		length of packet data
 */

static void KKsdec(char *str, char *pkt, int len)
{
    int	    c, i ;

    for (i = 0 ; i < len ; i++) {
	if ((c = pkt[i]) == KKmyqctl) {
	    c = pkt[++i] ;
	    if ((c & 0177) != KKmyqctl) {
		c = ctl(c) ;
	    }
	}
	*str++ = (char) c ;
    }
    *str = '\0' ;
}

/*
 * K K f d e c  -  Accept packet data and decode to file.
 *	FILE	*fp  	Output File Pointer
 *	char	*pkt 	encoded packet data
 *	int	len  	length of packet data
 */

static void KKfdec(FILE *fp, char *pkt, int len)
{
    int	    c, i ;

    for (i = 0 ; i < len ; i++) {
	if ((c = pkt[i]) == KKmyqctl) {
	    c = pkt[++i] ;
	    if ((c & 0177) != KKmyqctl) {
		c = ctl(c) ;
	    }
	}
	putc(c, fp) ;
    }
}

/*
 * K K i f l u s h  -  Flush communication port.
 */

static	void KKiflush(void)
{
	/* nothing to do */
}

/*
 *  K K s p a c k  -  Send Kermit Packet to the serial line.
 *	Padding, mark, and put packet, with checksum and EOL.
 *
 *	int		typ   		Packet Type
 *	int		num   		Sequence Number
 *	int		len   		Packet Data Length
 *	unsigned char	*data 		Packet Data
 */

static	UCHAR	sbuff[128] ;

static  void    KKspack(int typ, int num, int len, unsigned char *data)
{
    UCHAR   *p ;
    int	    i  ;
    USHORT  slen, scnt, chksum ;

    p = sbuff ; slen = 0 ; chksum = 0 ;

    for (i = 0 ; i < KKurnpad ; i++) {
	*p++ = KKurcpad ; slen++ ;
    }
    *p++ = (UCHAR) KKursoh       ; slen++ ;
    *p++ = (UCHAR) tochar(len+3) ; slen++ ; chksum += tochar(len+3) ;
    *p++ = (UCHAR) tochar(num)   ; slen++ ; chksum += tochar(num) ;
    *p++ = (UCHAR) typ           ; slen++ ; chksum += typ ;

    for (i = 0 ; i < len ; i++) {
	*p++ = data[i] ; slen++ ; chksum += data[i] ;
    }
    chksum = (((chksum & 0300) >> 6) + chksum) & 077 ;
	
    *p++ = (UCHAR) tochar(chksum) ; slen++ ;
    *p++ = (UCHAR) KKureol        ; slen++ ;

    scnt = (*comDevice->comSend) (sbuff, slen) ;
    if (slen != scnt) {
	prompt("write failed") ;
    }
}

/*
 *  K K r p a c k
 *	Receive Kermit Packet from Communication Port.
 *	Strip off padding and trailers, and check block.
 *	then extract length, sequnce number, type and data.
 *
 *	int	*len  		Packet Data Length
 *	int	*num  		Sequence Number
 *	char	*data 		Packet Data
 */

static	UCHAR	rbuff[128] ;
static	UCHAR	*rbptr     ;
static	USHORT	rbytes = 0 ;

static	BOOL    lgetc(char *c)
{
    int	    i ;

    if (rbytes > 0) {
	*c = *rbptr++ ; rbytes -= 1 ;
	return(TRUE) ;
    }
    for (i = 0 ; ; i++) {
	if ((rbytes = queGet(rbuff, 128)) > 0) {
	    break ;
	}
	if (i >= KKiowait) {
	    return(FALSE) ;
	}
    }
    rbptr = rbuff ;
    *c = *rbptr++ ; rbytes -= 1 ;
    return(TRUE) ;
}

static	int KKrpack(int *len, int *num, char *data)
{
    char    c, typ  ;
    int	    i, done ;
    int	    cchksum ;
    int	    rchksum ;

    do {                        /* Wait Packet Header	*/
	if (! lgetc(&c)) {
	    return(FALSE) ;
	}
	c &= 0177 ;
    } while (c != KKmysoh) ;

again:
    done = FALSE  ;
    while (! done) {
	if (! lgetc(&c))   return(FALSE) ;
	if (c == KKmysoh)  goto again    ;
	cchksum = c ;
	*len = unchar(c) - 3 ;

	if (! lgetc(&c))   return(FALSE) ;
	if (c == KKmysoh)  goto again    ;
	cchksum += c ;
	*num = unchar(c) ;

	if (! lgetc(&c))   return(FALSE) ;
	if (c == KKmysoh)  goto again    ;
	cchksum += c ;
	typ = c ;

	for (i = 0 ; i < *len ; i++) {
	    if (! lgetc(&c))   return(FALSE) ;
	    if (c == KKmysoh)  goto again    ;
	    cchksum += c ;
	    data[i] = c  ;
	}
	data[*len] = '\0' ;

	if (! lgetc(&c))   return(FALSE) ;
	rchksum = unchar(c) ;
	if (! lgetc(&c))   return(FALSE) ;
	if (c == KKmysoh)  goto again    ;
	done = TRUE ;
    }

    cchksum = (((cchksum & 0300) >> 6) + cchksum) & 077 ;
    return((cchksum != rchksum) ? FALSE : typ) ;
}

/*
 * Kermit File Receiving Part
 */

/*
 * f s a v e  -  Decode and Save File Name.
 */

static	void    fsave(char *pkt, int len)
{
    KKsdec(ofname, pkt, len) ;
    strcpy(cfname, ofname)   ;
}

/*
 *  r i n i t  -  Process Recv-Init Status
 *	Accept Send-Init and transit to Recv-File.
 */

static	char    rinit(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {	/* Too many retries,	*/
	error = err_too_many_retries ;
	return('A') ;		/*    Abort transfer	*/
    }

    switch(KKrpack(&len, &num, rpacket)) {

    case 'S' :				/* Send-Init		*/
	KKpdec(rpacket, len) ;		/* Get other side param	*/
	len = KKpenc(spacket) ;		/* and send back mine.	*/
	KKspack('Y', numseq, len, spacket) ;
	tune_retry(numtry) ;
	numold = numtry ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	return('F') ;

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	KKspack('N', numseq, 0, 0) ;	/*   Back NAK and	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  r f i l e  -  Process Recv-File Status
 *	Accept Send-File and transit to Recv-Data.
 */

static	char	rfile(void)
{
    int	    num, len  ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too many retries,	*/
	return('A') ;			/*    Abort Transfer	*/
    }

    switch(KKrpack(&len, &num, rpacket)) {

    case 'S' :				/* Send-Init		*/
	if (numold++ > OLDTRY) {
	    error = err_ack_lost ;	/* Check if ACK was	*/
	    return('A') ;		/* lost.		*/
	} else if (num != prvseq(numseq)) {
	    error = err_seq_error ;
	    return('A') ;
	} else {			/* ACK on previous	*/
	    len = KKpenc(spacket) ;	/* packet was lost.	*/
	    KKspack('Y', num, len, spacket) ;
	    numtry = 0 ;
	    return(status) ;
    	}

    case 'Z' :				/* Send-EOF		*/
	if (numold++ > OLDTRY) {
	    error = err_too_many_retries ;
	    return('A') ;
	} else if (num != prvseq(numseq)) {
	    error = err_seq_error ;
	    return('A') ;
	} else {
	    KKspack('Y', num, 0, 0) ;
	    numtry = 0 ;
	    return(status) ;
	}

    case 'F' :				/* Got File-Header	*/
	if (num != numseq) {
	    error = err_seq_error ;	/* sequence should be	*/
	    return('A') ;		/* match, here.		*/
	}
	KKspack('Y', numseq, 0, 0) ;
	tune_retry(numtry) ;
	numold = numtry ;
	numtry = 0 ;
	numseq = incseq(numseq) ;

	fsave(rpacket, len) ;		/* Save File Name	*/
		
	if ((recvfp = fopen(cfname, "wb")) == NULL) {
	    error = err_file_open ;
	    return('A') ;
	}
	KKpstart('r', cfname) ;
	return('D') ;

    case 'B' :			/* Break Transmission (EOT)	*/
    	if (numseq != num) {	/* Need corrent packet, here.	*/
	    error = err_seq_error ;
	    return('A') ;
	}
	KKspack('Y', numseq, 0, 0) ;
	return('C') ;		/* Receive Complete !!		*/
    
    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	KKspack('N', numseq, 0, 0) ;	/*   Back NAK and	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  r d a t a  -  Process Recv-Data Status
 *	Accept Send-Data to continue receive.
 *	Accept Send-EOF  to transit to Recv-File.
 */

static	char	rdata(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too many reties,	*/
	return('A') ;			/*    Abort Transfer.	*/
    }

    switch(KKrpack(&len, &num, rpacket)) {

    case 'D' :			        /* Received Data	*/
	if (num != numseq) {
	    if (numold++ > OLDTRY) {
		error = err_too_many_retries ;
		return('A') ;
	    } else if (num != prvseq(numseq)) {
		error = err_seq_error ;
		return('A') ;
	    } else {
		KKspack('Y', num, 0, 0) ;
		numtry = 0 ;
		return(status) ;
	    }
	}
	KKfdec(recvfp, rpacket, len) ;
	KKspack('Y', numseq, 0, 0)   ;
	KKppack() ;
	tune_retry(numtry) ;
	numold = numtry ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	return('D') ;

    case 'F' :			/* File Header, ACK lost	*/
	if (numold++ > OLDTRY) {
	    error = err_ack_lost ;
	    return('A') ;
	} else if (num != prvseq(numseq)) {
	    error = err_seq_error ;
	    return('A') ;
	} else {		/* Lost ACK, send again.	*/
	    KKspack('Y', num, 0, 0) ;
	    numtry = 0 ;
	    return(status) ;
	}

    case 'Z' :			/* End-Of-File			*/
    	if (num != numseq) {
	    error = err_seq_error ;
	    return('A') ;
	}
	KKspack('Y', numseq, 0, 0) ;
	tune_retry(numtry) ;
	numold = numtry ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	fclose(recvfp) ;
	recvfp = NULL  ;
	KKpend(TRUE, NULL) ;
	return('F') ;

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	error = err_from_host ;		/*   Abort transfer	*/
	return('A') ;

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	KKspack('N', numseq, 0, 0) ;	/*   Back NAK and	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  r e c s w 
 *	Status dispatcher for File Receive Protocol.
 */

static	int  recsw(void)
{
    /*
     * Start Receive Protocol
     */
    status = 'R' ;			/* Start from Recv-Init	*/
    numseq = 0   ;
    numold = 0   ;
    numtry = 0   ;

    KKpinit('r') ;

    while (1) {
	if (keycheck()) {
	    error = err_user_abort ;
	    KKspack('E', numseq, strlen(error), error) ;
	    return(FALSE) ;
	}
	switch(status) {
	case 'R' : status = rinit() ; break ;	/* Recv-Init	*/
	case 'F' : status = rfile() ; break ;	/* Recv-File	*/
	case 'D' : status = rdata() ; break ;	/* Recv-Data	*/
	case 'C' : return(TRUE) ;		/* Complete	*/
	case 'A' :
	default  :
	    KKspack('E', numseq, strlen(error), error) ;
	    return(FALSE) ;
	}
    }
}

/*
 * f i l e r e c v  -  Receive Files
 */

static	filerecv()
{
    int	    stat ;

    filelist  = &Xargv[1] ;
    filecount = (char) (Xargc - 1) ;

    KKpreset() ;

    recvfp = NULL ;	/* File is not opened Yet	*/
    stat = recsw() ;	/* Start Receive Protocol	*/

    if (recvfp) {
	fclose(recvfp) ;
    }
    if (stat == TRUE) {
	DosBeep(880, 50) ;
    } else {
	DosBeep(440, 50) ;
	KKpend(FALSE, error) ;
    }
    return(stat) ;
}

/*
 * Kermit File Sending Part
 */

/*
 * f n e x t   -  Extract File in the List
 */

static	BOOL    fnext(void)
{
    char    *cp, *np ;

    if (filecount <= 0) {
	return(FALSE) ;
    }

    strcpy(ofname, *filelist) ;	/* Extract File Name from List	*/
    filelist++    ;
    filecount--   ;

    np = cp = ofname ;		/* Stript leading directories	*/
    while (*cp) {
	if (*cp++ == '\\') {
	    np = cp ;
	}
    }
    strcpy(cfname, np) ;
    return(TRUE) ;
}

/*
 *  s i n i t	-  Process Send-Init Status.
 *	Send this kermit's parameter to another and
 *	get other side parameters.
 */

static	char	sinit(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too Many Retries	*/
	return('A') ;			/*  Abort Transfer	*/
    }

    len = KKpenc(spacket) ;		/* Encode Parameters		*/
    KKiflush() ;			/* Flush inputs on Comm.Line	*/
    KKspack('S', numseq, len, spacket) ;

    switch(KKrpack(&len, &num, rpacket)) {

    case 'N' : 				/* NAK, try again	*/
	return(status) ;

    case 'Y' :				/* ACK			*/
       	if (numseq != num) {		/* Wrong ACK, stay here	*/
       	    return(status) ;
       	}
       	KKpdec(rpacket, len) ;		/* Decode Parameters	*/
	tune_retry(numtry) ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	return('F') ;			/* Switch to Send-File	*/

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  s f i l e  -  Process Send-File (Header) Status.
 *	Send File-Header Packet, wait its respond.
 */

static	char	sfile(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too Many Retries	*/
	return('A') ;			/*   Abort Transffer	*/
    }

    if (sendfp == NULL) {		/* Open file to SEND		*/
	if ((sendfp = fopen(ofname, "rb")) == NULL) {
	    error = err_file_open ;
	    return('A') ;
	}
    }

    len = KKsenc(cfname, spacket) ;		/* Encode File Name	*/
    KKspack('F', numseq, len, spacket) ;

    switch(KKrpack(&len, &num, rpacket)) {

    case 'N' :				/* NAK, stay here	*/
	num = prvseq(num) ;		/* But is NAK for next	*/
	if (numseq != num) {		/* packet, assume to be	*/
	    return(status) ;	        /* ACKed current one.	*/
	}				/* Not Break Here.	*/

    case 'Y' :				/* ACKed		*/
	if (numseq != num) {		/* If ACK for another,	*/
	    return(status) ;	        /* packet, stay here.	*/
	}
	tune_retry(numtry) ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	KKpstart('s', cfname) ;
	pksize = KKfenc(sendfp, spacket) ;
	return((UCHAR) ((pksize > 0) ? 'D' : 'Z')) ;

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;		/* Receive Failed	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  s d a t a  -  Process Send-Data Status
 *	Send File Data Packet to Host and wait respond.
 */

static	char	sdata(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too Many Retries	*/
	return('A') ;			/*   Abort Transfer	*/
    }

    KKspack('D', numseq, pksize, spacket) ;

    switch(KKrpack(&len, &num, rpacket)) {

    case 'N' :				/* NAK, stay here	*/
	num = prvseq(num) ;		/* But is NAK for next	*/
	if (numseq != num) {		/* packet, assume to be	*/
	    return(status) ;	        /* ACKed current one.	*/
	}				/* Not Break Here.	*/

    case 'Y' :				/* ACKed		*/
	if (numseq != num) {		/* If ACK for another,	*/
	    return(status) ;	        /* packet, stay here.	*/
	}
	tune_retry(numtry) ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	KKppack() ;
	pksize = KKfenc(sendfp, spacket) ;
	return((UCHAR) ((pksize > 0) ? 'D' : 'Z')) ;

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  s e o f  -  Process Send-EOF status
 *	Send Send-EOF packet to Host and wait respond.
 */

static	char	seof(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too many retyies	*/
	return('A') ;			/*    Abort Transfer	*/
    }

    KKspack('Z', numseq, 0, spacket) ;

    switch(KKrpack(&len, &num, rpacket)) {

    case 'N' :				/* NAK, stay here	*/
	num = prvseq(num) ;		/* But is NAK for next	*/
	if (numseq != num) {		/* packet, assume to be	*/
		return(status) ;	/* ACKed current one.	*/
	}				/* Not Break Here.	*/

    case 'Y' :				/* ACKed		*/
	if (numseq != num) {		/* If ACK for another,	*/
		return(status) ;	/* packet, stay here.	*/
	}
	tune_retry(numtry) ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	fclose(sendfp) ;
	sendfp = NULL  ;
	KKpend(TRUE, NULL) ;
	return((UCHAR) (fnext() ? 'F' : 'B')) ;

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  s b r e a k  -  Process Send-Break Status
 *	Send Send-Break *EOT) packet to host and wait respond.
 */

static	char	sbreak(void)
{
    int	    num, len ;

    if (numtry++ > MAXTRY) {
	error = err_too_many_retries ;	/* Too Many reties,	*/
	return('A') ;			/*   Abort transfer	*/
    }

    KKspack('B', numseq, 0, spacket) ;

    switch(KKrpack(&len, &num, rpacket)) {

    case 'N' :				/* NAK, stay here	*/
	num = prvseq(num) ;		/* But is NAK for next	*/
	if (numseq != num) {		/* packet, assume to be	*/
	    return(status) ;	        /* ACKed current one.	*/
	}				/* Not Break Here.	*/

    case 'Y' :				/* ACKed		*/
	if (numseq != num) {		/* If ACK for another,	*/
	    return(status) ;	        /* packet, stay here.	*/
	}
	tune_retry(numtry) ;
	numtry = 0 ;
	numseq = incseq(numseq) ;
	return('C') ;

    case 'E' :
	error = err_error_packet ;	/* Error Packet		*/
	KKprepck(rpacket, len) ;	/*   Print Error Info.	*/
	return('A') ;			/*   Abort Transfer	*/

    case FALSE :
	error = err_recv_failed ;	/* Receive Failed	*/
	return(status) ;		/*   Try Again		*/

    default :
	error = err_unknown_pkt ;	/* Anything Else	*/
	return('A') ;			/*   Abort Transfer	*/
    }
}

/*
 *  s e n d s w
 *	Status dispatcher for the file send protocol.
 */

static	sendsw(void)
{
    /*
     * Start Send Protocol
     */

    status = 'S' ;		/* Start from Send-Init	*/
    numseq = 0   ;
    numtry = 0   ;

    KKpinit('s') ;

    /*
     * Loop Until Protocol Complete or Abort
     */

    while (1) {		/* Transit Protocol Status	*/
	if (keycheck()) {
	    error = err_user_abort ;
	    KKspack('E', numseq, strlen(error), error) ;
	    return(FALSE) ;
	}
    	switch(status) {
	case 'S' : status = sinit()  ; break ;	/* Send-Init	*/
	case 'F' : status = sfile()  ; break ;	/* Send-File	*/
	case 'D' : status = sdata()  ; break ;	/* Send-Data	*/
	case 'Z' : status = seof()   ; break ;	/* Send-EOF	*/
	case 'B' : status = sbreak() ; break ;	/* Send-Break	*/
	case 'C' : return(TRUE)  ;		/* Complete	*/
	case 'A' : 				/* Aborted	*/
	default  :				/* Unknown	*/
	    KKspack('E', numseq, strlen(error), error) ;
	    return(FALSE) ;
	}
    }
}

/*
 * f i l e s e n d  -  Send Files
 */

static	int filesend(void)
{
    int	    stat ;

    filelist  = &Xargv[0] ;
    filecount = (char ) Xargc  ;

    KKpreset() ;

    fnext() ;		/* Extract First file in the List	*/
    sendfp = NULL ;	/* File is not opened Yet		*/
    stat = sendsw() ;	/* Start Send Protocol			*/

    if (sendfp) {
	fclose(sendfp) ;
    }
    if (stat == TRUE) {
	DosBeep(880, 50) ;
    } else {
	DosBeep(440, 50) ;
	KKpend(FALSE, error) ;
    }
    return(stat) ;
}

/*
 * k e r m i t S e n d  /  k e r m i t R e c v
 * 
 *	start kermit file transfer
 */

static	UCHAR	filename[64] ;

void    kermitSend(void)
{
    UCHAR   *fname ;

    scrSave() ;

    /*
     * Accept Sending File Names
     */
    prompt("Kermit Send File Name ? ") ;
	
    if (getlin(filename, 64) == NULL) {
	scrRest() ;
	return ;
    }
    if (*filename == '\0') {
	scrRest() ;
	return ;
    }
    if ((fname = strtok(filename, " \t\r\n")) == NULL) {
	scrRest() ;
	return ;
    }

    arginit() ; argexpand(fname) ;
	
    while ((fname = strtok(NULL, " \t\r\n")) != NULL) {
	argexpand(fname) ;
    }
    if (Xargc == 0) {
	prompt("no such file") ;
	keywait() ;
	scrRest() ;
	return ;
    }

    filesend() ;

    keywait() ;
    scrRest() ;
}

void    kermitRecv(void)
{
    scrSave() ;

    filerecv() ;

    keywait()  ;
    scrRest() ;
}
