/*
 * queue.c - Communication Queue Handling
 */
 
#define	INCL_BASE

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <memory.h>
#include "shifty.h"

/*
 * Character QUEUE for receive data buffer
 */

#define	QUELENG		(1024 * 8)

typedef struct {
	SHORT	leng  ;
	SHORT   used  ;
	SHORT   getp  ;
	SHORT   putp  ;
	UCHAR	base[QUELENG] ;
} QUEUE ;

static QUEUE		QueRec = { 0 } ;

static  HMTX            queMutex ;
static  HEV             queEvent ;

/*
 * queInit - Initialize Communication Queue
 */

BOOL    queInit(VOID)
{
    APIRET  stat1, stat2 ;

    stat1 = DosCreateMutexSem(NULL, &queMutex, 0, FALSE) ;
    stat2 = DosCreateEventSem(NULL, &queEvent, 0, FALSE) ;

    if (stat1 != 0 || stat2 != 0) {
        return FALSE ;
    }

    QueRec.leng  = QUELENG ;
    QueRec.used  = 0 ;
    QueRec.getp  = 0 ;
    QueRec.putp  = 0 ;

    return(TRUE) ;
}

/*
 * quePut - put data into queue
 */

INT     quePut(UCHAR *buf, INT len)
{
    INT     bytes = 0 ;
    INT     space, count ;
    ULONG   evcnt ;

    while (len > 0) {
        /*
	 * Wait if queue is full
	 */
	if (QueRec.used >= QueRec.leng) {
	    DosResetEventSem(queEvent, &evcnt) ;
	    DosWaitEventSem(queEvent, 100L)    ;
	    continue ;
	}
	
	/*
	 * Check contiguous space on queue
	 */

        DosRequestMutexSem(queMutex, -1L) ;

	if (QueRec.putp >= QueRec.getp) {
	    space = QUELENG - QueRec.putp ;
	} else {
	    space = QueRec.getp - QueRec.putp ;
	}
	count = min(len, space) ;
	
	/*
	 * Copy into queue and bump pointers
	 */

        memcpy(&(QueRec.base[QueRec.putp]), buf, count) ;
	
	buf += count ;
	len -= count ;
	
	if ((QueRec.putp += count) >= QUELENG) {
	    QueRec.putp = 0 ;
	}
	QueRec.used += count ;
	bytes += count ;
	
        DosPostEventSem(queEvent)    ;
        DosReleaseMutexSem(queMutex) ;
    }

    return(bytes) ;
}

/*
 * queGet - get data from queue
 */
 
INT     queGet(UCHAR *buf, INT len)
{
    INT     bytes = 0 ;
    INT     space, count ;
    ULONG   evcnt ;

    /*
     * Wait if queue is empty
     */

    if (QueRec.used == 0) {
	DosResetEventSem(queEvent, &evcnt) ;
	DosWaitEventSem(queEvent, 100L)    ;
    }

    while (QueRec.used > 0 && len > 0) {

        DosRequestMutexSem(queMutex, -1L) ;

        /*
	 * Check contiguous data on queue
	 */
	
	if (QueRec.putp > QueRec.getp) {
	    space = QueRec.putp - QueRec.getp ;
	} else {
	    space = QUELENG - QueRec.getp ;
	}
	count = min(len, space) ;
        
	/*
	 * Copy queue data into buffer
	 */

        memcpy(buf, &(QueRec.base[QueRec.getp]), count) ;
	
	buf += count ;
	len -= count ;
	
	if ((QueRec.getp += count) >= QUELENG) {
	    QueRec.getp = 0 ;
	}
	QueRec.used -= count ;
	bytes += count ;

        DosPostEventSem(queEvent)    ;
        DosReleaseMutexSem(queMutex) ;
    }
    
    return(bytes) ;
}



