/*
 * serial.c - serial communication handler for Shifty
 */

#define	INCL_BASE
#define	INCL_DOSDEVICES

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "shifty.h"

/*
 * Commnucation Control Block for Serial Device
 */
 
UCHAR   serName[256] ;
BOOL    serInit(INT ac, CHAR *av[]) ;
VOID    serTerm(VOID) ;
VOID    serStat(VOID) ;
INT     serRecv(PUCHAR buff, INT len) ;
INT     serSend(PUCHAR buff, INT len) ;
VOID    serBreak(VOID) ;
VOID    serFlush(VOID) ;

COMREC	serDevice = {
    serName,
    serInit,
    serTerm,
    serStat,
    serRecv,
    serSend,
    serBreak,
    serFlush
} ;

/*
 * Serial Communication Control Data
 */

static  UCHAR   devName[16] = "COM1" ;      /* as default   */
static  HFILE   devHndl ;

static  ULONG   devSpeed   = 9600    ;
static  BOOL    givenSpeed = FALSE   ;

typedef struct {
	UCHAR	data ;
	UCHAR	pari ;
	UCHAR	stop ;
	UCHAR   brk  ;
} LMODE ;

static	LMODE	devMode = { 8, 0, 0 } ;
static  BOOL    givenMode = FALSE     ;

typedef struct {
    UCHAR   *str ;
    UCHAR   data ;
    UCHAR   pari ;
    UCHAR   stop ;
} PMODE ;

static  PMODE   tabModes[] = {
{	"7E1",	7,	2,	0	} ,
{	"7E2",	7,	2,	2	} ,
{	"7O1",	7,	1,	0	} ,
{	"7O2",	7,	1,	2	} ,
{	"8N1",	8,	0,	0	} ,
{	"8N2",	8,	0,	2	} ,
{	NULL,	0,	0,	0, 	}
} ;

static  VOID    setMode(UCHAR *mode)
{
    PMODE   *p ;
    
    for (p = tabModes ; p->str != NULL ; p++) {
        if (stricmp(p->str, mode) == 0) {
	    devMode.data = p->data ;
	    devMode.pari = p->pari ;
	    devMode.stop = p->stop ;
	    givenMode = TRUE ;
	    return ;
	}
    }
}

typedef struct {
	USHORT	stout  ;
	USHORT	rtout  ;
	UCHAR	flag1  ;
	UCHAR	flag2  ;
	UCHAR	flag3  ;
	UCHAR	errrpl ;
	UCHAR	brkrpl ;
	UCHAR	xonchr ;
	UCHAR	xofchr ;
} LDCB ;

static	LDCB	devDCB ;

static	USHORT	devModem = 0xff03 ;
static	USHORT	devStat  = 0x0000 ;

/*
 * serInit - Initialize Serial Communication Device
 *  
 *      -port <name>        Name of COM port, such as COM1
 *      -speed #            Speed of Serial Communication
 *      -mode <mode>        Communication Mode
 */

BOOL    serInit(INT ac, CHAR *av[])
{
    INT     i ;
    ULONG   action ;
    ULONG   leng   ;
    APIRET  stat   ;
    ULONG   genLong  ;
    USHORT  genShort ;

    /*
     * Parse Command Line Arguments
     */

    for (i = 1 ; i < ac ; i++) {
        if (stricmp(av[i], "-port") == 0) {
	    if (((i + 1) < ac) && (strlen(av[i+1]) < 16)) {
	        strcpy(devName, av[i+=1]) ;
	    }
	}
	if (stricmp(av[i], "-speed") == 0) {
            if ((i + 1) < ac) {
	        devSpeed = atol(av[i+=1]) ;
		givenSpeed = TRUE ;
	    }
	}
	if (stricmp(av[i], "-mode") == 0) {
            if ((i + 1) < ac) {
	        setMode(av[i+=1]) ;
	    }
	}
	        	        
    }
    sprintf(serName, "serial on %s", devName) ;
    
    /*
     * Open and Set Communication Parameters
     */

    stat = DosOpen(devName, &devHndl,
                    &action, 0L, 0, 0x0001, 0x0012, 0L) ;
    if (stat) {
        fprintf(stderr, 
	    "%s : failed to open serial device %s\n", av[0], devName) ;
        return(FALSE) ;
    }

    /*
     * Set Device Control Block Parameters
     *      Query, Modify, and Set DCB.
     *      DCB is temporary, vanish when closed.
     */

    stat = DosDevIOCtl(devHndl, 1, 0x73, 
            NULL, 0, NULL,
            (PVOID) &devDCB, sizeof(devDCB), &leng) ;

    if (stat != 0) {
        fprintf(stderr, "%s : failed to get DBC\n", av[0]) ;
	DosClose(devHndl) ;
	return(FALSE) ;
    }

    /*
     * Device control mode was suggested by tanahasi@sra
     */

    devDCB.stout =   5    ;	/* write time-out 50ms	*/
    devDCB.rtout =   100  ;	/* read  time-out 1sec	*/
    devDCB.flag1 |=  0x01 ;	/* Ena. DTR Control	*/
    devDCB.flag1 &= ~0x02 ;	/* Dis. DTR Handshake	*/
    devDCB.flag1 &= ~0x08 ;	/* Dis. CTS Handshake	*/
    devDCB.flag1 &= ~0x10 ;	/* Dis. DSR Handshake	*/
    devDCB.flag1 &= ~0x20 ;	/* Dis. DCD Handshake	*/
    devDCB.flag1 &= ~0x40 ;	/* Dis. DSR Sensing	*/

    devDCB.flag2 &= ~0x01 ;	/* Dis. TX XON/XOFF	*/
    devDCB.flag2 &= ~0x02 ;	/* Dis. RX XON/XOFF	*/
    devDCB.flag2 &= ~0x04 ;	/* Dis. Error Replace	*/
    devDCB.flag2 &= ~0x08 ;	/* Dis. Null Stripping	*/
    devDCB.flag2 &= ~0x10 ;	/* Dis. BREAK Replace	*/
    devDCB.flag2 |=  0x40 ;	/* Ena. RTS Control	*/
    devDCB.flag2 &= ~0x80 ;	/* Dis. RTS Handshake	*/

    devDCB.flag3 &= 0xf8 ;	/* Clear time-out mode	*/
    devDCB.flag3 |= 0x04 ;	/* Wait-for-somthing	*/

    stat = DosDevIOCtl(devHndl, 1, 0x53, 
            (PVOID) &devDCB, sizeof(devDCB), &leng,
            NULL, 0, NULL) ;

    if (stat != 0) {
        fprintf(stderr, "%s : failed to set DBC\n", av[0]) ;
	DosClose(devHndl) ;
	return(FALSE) ;
    }

    /*
     * Set Communication Parameters
     */
    
    if (givenSpeed && devSpeed < 0x10000L) {
        genShort = devSpeed ;
        stat = DosDevIOCtl(devHndl, 1, 0x41,
                &genShort, sizeof(genShort), &leng,
                NULL, 0, NULL) ;
	if (stat != 0) {
            fprintf(stderr, "%s : failed to set speed\n", av[0]) ;
            DosClose(devHndl) ;
	    return(FALSE) ;
	}
    }

    if (givenMode) {
        stat = DosDevIOCtl(devHndl, 1, 0x42,
	        &devMode, sizeof(devMode), &leng,
		NULL, 0, NULL) ;
	if (stat != 0) {
            fprintf(stderr, "%s : failed to set mode\n", av[0]) ;
            DosClose(devHndl) ;
	    return(FALSE) ;
	}
    }

    if (givenSpeed || givenMode) {
        stat = DosDevIOCtl(devHndl, 1, 0x46,
	        &devModem, sizeof(devModem), &leng,
		&genShort, sizeof(genShort), &leng) ;
	if (stat != 0) {
            fprintf(stderr, "%s : failed to set modem control\n", av[0]) ;
            DosClose(devHndl) ;
	    return(FALSE) ;
	}
    }
    return(TRUE) ;
}

/*
 * serTerm - terminate Serial Communication
 */
 
VOID    serTerm(VOID)
{
    DosClose(devHndl) ;
}

/*
 * serStat - report Status of Serial Communication Device
 */
 
VOID    serStat(VOID)
{
    APIRET  stat  ;
    ULONG   leng  ;
    USHORT  bps   ;
    LMODE   lmode ;
    LDCB    ldcb  ;
    UCHAR   lstat1, lstat2 ;
    UCHAR   mstat1, mstat2 ;
    UCHAR   *on  = "ON"  ;
    UCHAR   *off = "OFF" ;
    UCHAR   *ena = "enabled"  ;
    UCHAR   *dis = "disabled" ;

    /*
     * Get Current Status
     */

    stat = DosDevIOCtl(devHndl, 1, 0x61,
            NULL, 0, NULL, &bps, sizeof(bps), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    stat = DosDevIOCtl(devHndl, 1, 0x62,
            NULL, 0, NULL, &lmode, sizeof(lmode), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    stat = DosDevIOCtl(devHndl, 1, 0x64,
            NULL, 0, NULL, &lstat1, sizeof(lstat1), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    stat = DosDevIOCtl(devHndl, 1, 0x64,
            NULL, 0, NULL, &lstat2, sizeof(lstat2), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    stat = DosDevIOCtl(devHndl, 1, 0x66,
            NULL, 0, NULL, &mstat1, sizeof(mstat1), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    stat = DosDevIOCtl(devHndl, 1, 0x67,
            NULL, 0, NULL, &mstat2, sizeof(mstat2), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    stat = DosDevIOCtl(devHndl, 1, 0x73,
            NULL, 0, NULL, &ldcb, sizeof(ldcb), &leng) ;
    if (stat != 0) {                
    	return ;
    }

    /*
     * Display Current Status
     */

    printf("Current Communication Status for %s\n", devName) ;
    printf("%d bps, %d bits data, ", bps, lmode.data) ;
    printf("parity ") ;

    switch(lmode.pari) {
	case 0x00 : printf("non, ")  ; break ;
	case 0x01 : printf("odd, ")  ; break ;
	case 0x02 : printf("even. ") ; break ;
	default   : printf("???. ")  ; break ;
    }
    switch(lmode.stop) {
	case 0x00 : printf("1 ")   ; break ;
	case 0x01 : printf("1.5 ") ; break ;
	case 0x02 : printf("2 ")   ; break ;
	default   : printf("??? ") ; break ;
    }
    printf("stop bit\n") ;

    if (lstat1 || lstat2) {
	printf("Line Status\n") ;
    }
    if (lstat1 & 0x01) {
	printf("\tTX Waiting CTS\n") ;
    }
    if (lstat1 & 0x02) {
	printf("\tTX Waiting DSR\n") ;
    }
    if (lstat1 & 0x04) {
    	printf("\tTX Waiting DCD\n") ;
    }
    if (lstat1 & 0x80) {
	printf("\tTX Waiting, received XOFF\n") ;
    }
    if (lstat1 & 0x10) {
	printf("\tTX Waiting, send XON\n") ;
    }
    if (lstat1 & 0x20) {
	printf("\tSending BREAK\n") ;
    }
    if (lstat1 & 0x40) {
	printf("\tWaiting Immediate TX\n") ;
    }
    if (lstat1 & 0x80) {
	printf("\tWaiting DSR\n") ;
    }
    if (lstat2 & 0x01) {
    	printf("\tWrite req. in progress\n") ;
    }
    if (lstat2 & 0x02) {
    	printf("\tData in TX Queue\n") ;
    }
    if (lstat2 & 0x04) {
    	printf("\tData in TX Device\n") ;
    }
    if (lstat2 & 0x08) {
    	printf("\tImmdeiate TX Data\n") ;
    }
    if (lstat2 & 0x10) {
    	printf("\tXON Request\n") ;
    }
    if (lstat2 & 0x20) {
    	printf("\tXOFF Request\n") ;
    }

    printf("Modem Control Register\n\t") ;
    printf("DTR %s,",  ((mstat1 & 0x01) ? on : off)) ;
    printf("RTS %s\n", ((mstat1 & 0x02) ? on : off)) ;
	
    printf("Modem Status Register\n\t") ;
    printf("CTS %s,",  ((mstat2 & 0x10) ? on : off)) ;
    printf("DSR %s,",  ((mstat2 & 0x20) ? on : off)) ; 
    printf("RING %s,", ((mstat2 & 0x40) ? on : off)) ; 
    printf("DCD %s\n", ((mstat2 & 0x80) ? on : off)) ; 

    printf("Device Control Block\n") ;
	
    printf("\tTX timeout %d, RX timeout %d\n",
					ldcb.stout, ldcb.rtout) ;
    printf("\tDTR Control         %s\n", ((ldcb.flag1 & 0x01) ? ena : dis)) ;
    printf("\tDTR Handshake       %s\n", ((ldcb.flag1 & 0x02) ? ena : dis)) ;
    printf("\tRTS Control         %s\n", ((ldcb.flag2 & 0x40) ? ena : dis)) ;
    printf("\tRTS Handshake       %s\n", ((ldcb.flag2 & 0x80) ? ena : dis)) ;
    printf("\tCTS Handshake       %s\n", ((ldcb.flag1 & 0x04) ? ena : dis)) ;
    printf("\tDSR Handshake       %s\n", ((ldcb.flag1 & 0x10) ? ena : dis)) ;
    printf("\tDCD Handshake       %s\n", ((ldcb.flag1 & 0x20) ? ena : dis)) ;
    printf("\tDSR Sensing         %s\n", ((ldcb.flag1 & 0x40) ? ena : dis)) ;
    printf("\tTX XON/XOFF Control %s\n", ((ldcb.flag2 & 0x01) ? ena : dis)) ;
    printf("\tRX XON/XOFF Control %s\n", ((ldcb.flag2 & 0x02) ? ena : dis)) ;
    printf("\tError Replacement   %s\n", ((ldcb.flag2 & 0x04) ? ena : dis)) ;
    printf("\tNULL Deletion       %s\n", ((ldcb.flag2 & 0x08) ? ena : dis)) ;
    printf("\tBreak Replacement   %s\n", ((ldcb.flag2 & 0x10) ? ena : dis)) ;
}

/*
 * serRecv - receive from Serial Communication Device
 */
 
typedef struct {
    USHORT  nums ;
    USHORT  size ;
} COMQUERY ;
	
INT     serRecv(PUCHAR buff, INT len)
{
    APIRET      stat  ;
    ULONG       leng  ;
    COMQUERY    query ;
    ULONG       reqst, bytes ;

    stat = DosDevIOCtl(devHndl, 1, 0x68,
            NULL, 0, NULL, &query, sizeof(query), &leng) ;

    if (stat != 0) {
        reqst = 1 ;
    } else if ((reqst = query.nums) == 0) {
        reqst = 1 ;
    } else if (reqst > len) {
        reqst = len ;
    }
    if ((stat = DosRead(devHndl, buff, reqst, &bytes)) != 0) {
    	return(-1) ;
    }
    return(bytes) ;
}

INT     serSend(PUCHAR buff, INT len)
{
    ULONG   bytes, stat ;

    if ((stat = DosWrite(devHndl, buff, len, &bytes)) != 0) {
    	return(-1) ;
    }
    return(bytes) ;
}

VOID    serBreak(VOID)
{
    APIRET  stat ;
    ULONG   leng ;
    USHORT  err  ;

    DosDevIOCtl(devHndl, 1, 0x4b,
            NULL, 0, NULL, &err, sizeof(err), &leng) ;
    DosSleep(100L) ;
    DosDevIOCtl(devHndl, 1, 0x45,
            NULL, 0, NULL, &err, sizeof(err), &leng) ;
}

VOID    serFlush(VOID)
{
    queInit() ;
    (*emuDevice->emuRst) () ;
}
