/* makelist.cmd
*/
info.PROGRAM_STR = 'MAKELIST for OS/2 NISTIME'
info.VERSION_STR = 'Version 1.4, 23-Nov-2003'
info.COPYRGT_STR = 'Copyright (C) 2002, Pieter Bras.  All rights reserved.'
/*
  description:  create a customized version of the "nist-srv.lst" file 
        usage:  see the routine Usage (below)
     requires:  original nist-srv.lst file from NIST, or similar
                RxSock.DLL

       author:  Pieter Bras
       e-mail:  pbras@pobox.com
*/
signal on halt
signal on novalue
signal on syntax

/* initialize OS/2 RexxUtil functions */
if LoadLib( 'RexxUtil', 'SysLoadFuncs') then
   call Fatal 'RexxUtil.dll not found'

/* initialize OS/2 RxSock functions */
if LoadLib( 'RxSock', 'SockLoadFuncs') then
   call Fatal 'RxSock.dll not found'
if SockInit() <> 0 then
   call Fatal 'SockInit() failure'

/* load sort function */
if RxFuncQuery( 'SysStemSort') = 0 then			    /* if available */
   say 'Using SysStemSort()'
else do
   say 'SysStemSort() not available'
   /* note that the name 'rexxlibregister' must be completely lower-case */
   if LoadLib( 'RexxLib', 'rexxlibregister') then	/* load REXXLIB.DLL */
      say 'ArraySort() not available'
   else if RxFuncQuery( 'ArraySort') <> 0 then	   /* check for ArraySort() */
      say 'ArraySort() not available'
   else
      say 'Using ArraySort()'
   end

opts. = ''
env = 'OS2ENVIRONMENT'
TZval = value( 'TZ',, env)
if TZval = '' then
   TZval =  '(not available)'
logfile = 'makelist.log'
tmpfile = 'makelist.tmp'
parse arg commandline
commandline = GetOpts( commandline)
if opts.ifile = '' then
   opts.ifile = 'nist-srv.lst'
if opts.ofile = '' then
   opts.ofile = strip( commandline)
if opts.ofile = '' then
   opts.ofile = 'my-nist-srv.lst'
if opts.go \= '1' then
   signal Usage

call FileDelete logfile
oline = translate( opts.ofile) 'generated on' Date() 'at' Time() '(local time)'
call logger oline
call logger 'TZ =' quote( TZval)

SrvAddr.0 = 0                                      /* count of NIST servers */
if \BuildServerTable( opts.ifile) then
   call Fatal 'Input file: not found.'
if SrvAddr.0 = 0 then
   call Fatal 'Input file: incorrect format.'

/* collect information from servers */
do i = 1 to SrvAddr.0
   say 'Testing:' SrvName.i
   call logger ''
   OKct = 0
   ERct = 0
   nist.delay = '9990'
   nist.offset = 0
   t = Time( 'E')
   rv = GetNistime( '-N' SrvAddr.i)	  /* check for unresponsive servers */
   if rv >= 300 & rv < 400 then do
      t = Time( 'E')
      rv = GetNistime( '-N' SrvAddr.i)		       /* try one more time */
      end
   if Time( 'E') - t < 15 then do until OKct = 3 | ERct = 5
      rv = GetNistime( '-N' SrvAddr.i)
      if rv = 1 then do
         OKct = OKct + 1
         nist.delay = 10000 + 10*nist.delay
         nist.delay = substr( nist.delay, 2, 4)
         results.OKct = nist.delay ' ' nist.offset
         /* format: dddd...oooooooooo
                    12345678901234567 */
         call logger SrvName.i ' delay =' nist.delay ' offset =' nist.offset
         end   /* if rv = 1 */
      else do
         /* analyze errors more carefully? */
         ERct = ERct + 1
         call logger SrvName.i ' error =' rv
         end   /* else do */
      end   /* if ... then do until ... */

   /* process results for this server */
   if OKct = 0 & ERct = 5 then      /* if all errors or connections refused */
      nist.delay = '9980'
   j = (OKct + 1) % 2                                       /* integer part */
   if OKct > 0 then do                                /* sort by delay time */
      results.0 = OKct
      if RxFuncQuery( 'SysStemSort') = 0 then do
         if SysStemSort( 'results.', 'A', 'I', 1, results.0, 1, 4) \= 0 then
            call Fatal 'SysStemSort failed'
         end
      else
   if RxFuncQuery( 'ArraySort') = 0 then do
         if ArraySort( 'results', 1, results.0, 1, 4, 'A', 'I') \= 1 then
            call Fatal 'ArraySort failed'
         end
      parse var results.j nist.delay nist.offset .	    /* median delay */
      end
   SrvResult.i = right( i + 100, 2) ' ' OKct ' ' ERct ' ' nist.delay ' ' nist.offset
   /* format: ii...c...c...dddd...oooooooooo
              123456789012345678901234567890 */
   call logger SrvName.i SrvResult.i                      /* output results */
   end   /* do i = ... */

/* sort the server results by delay time */
SrvResult.0 = SrvAddr.0
if RxFuncQuery( 'SysStemSort') = 0 then do
   if SysStemSort( 'SrvResult.', 'A', 'I', 1, SrvAddr.0, 14, 17) \= 0 then
      call Fatal 'SysStemSort failed'
   end
else if RxFuncQuery( 'ArraySort') = 0 then do
   if ArraySort( 'SrvResult', 1, SrvAddr.0, 14, 4, 'A', 'I') \= 1 then
      call Fatal 'ArraySort failed'
   end

/* write output file */
/* rename old output file to *.BAK ??? */
call FileDelete opts.ofile                               /* delete old list */
call owrite oline                        /* heading with starting timestamp */
call owrite 'TZ =' quote( TZval)
call owrite '$'
do i = 1 to SrvAddr.0
   parse var SrvResult.i j . . nist.delay .
   j = j + 0                                     /* strip leading 0, if any */
   nist.delay = nist.delay/10
   if pos( '.', nist.delay) = 0 then
      nist.delay = nist.delay || '.0'
   nist.delay = right( nist.delay, 5)
   call owrite left( SrvName.j, 30) left( SrvAddr.j, 18) left( SrvNote.j, 5) left( SrvLoc.j, 16) nist.delay
   end
call owrite '$'
call owrite 'Statistics:'
call owrite left( 'Server Name', 35) 'Reads   Errs    Delay     Diff'
do i = 1 to SrvAddr.0                        /* write out server statistics */
   parse var SrvResult.i j OKct ERct nist.delay nist.offset .
   if left( nist.offset, 1) \= '-' then
      nist.offset = ' 'nist.offset
   j = j + 0                                     /* strip leading 0, if any */
   nist.delay = nist.delay/10
   if pos( '.', nist.delay) = 0 then
      nist.delay = nist.delay || '.0'
   nist.delay = right( nist.delay, 5)
   oline = right( i + 100, 2) ' ' left( SrvName.j, 32) left( OKct, 6) left( ERct, 6)
   if nist.delay < 998 then
      oline = oline left( nist.delay, 8) nist.offset
   else
      oline = oline 'unreachable'
   call owrite oline
   call logger oline
   end
call owrite ''
call owrite 'Explanation:'
call owrite 'Reads: Number of successful attempts to contact server (max = 3).'
call owrite 'Errs:  Number of failed attempts to contact server (max = 5).'
call owrite 'Delay: Measured round-trip time, milliseconds (lower is better).'
call owrite 'Diff:  Measured clock error, seconds (Local Clock - NIST).'
call owrite ''
oline = 'File generation completed on' Date() 'at' Time()
call owrite oline
call logger oline

call FileClose opts.ofile
if opts.log = 1 then
   call FileClose logfile
call FileDelete tmpfile
say 'Done.'
exit 0

/* write a line to the log file */
logger:
   if opts.log = 1 then
      call lineout logfile, arg( 1)
   return

/* write a line to output file */
owrite:
   call lineout opts.ofile, arg( 1)
   return

/**************************************/
/* get info from a NIST server
      input:   arg( 1) = IP address of server
      return:  1 = success; otherwise error
      if success: leaves results in nist.offset, nist.delay, nist.error
*/
GetNistime: procedure expose nist. tmpfile
   call SysSleep 1                                      /* delay one second */
   call FileDelete tmpfile
   '@nistime.exe -m1 -s0 -d500' arg( 1) '>'tmpfile '2>&1'
   nist.error = RC
   if nist.error >= 100 then                          /* nistime error code */
      return nist.error
   q.0 = 0
   do while lines( tmpfile) > 0                   /* read redirected output */
      ii = q.0 + 1
      q.ii = linein( tmpfile)
      q.0 = ii
      end   /* do while ... */
   call FileClose tmpfile
   do ii = q.0 to 1 by -1
      if Pos( 'Local Clock - NIST =', q.ii) > 0 then
         parse var q.ii 'Local Clock - NIST =' nist.offset .
      else if Pos( 'ound-trip delay', q.ii) > 0 then do
         parse var q.ii 'delay =' nist.delay .
         return 1
         end
      end   /* do ii = ... */
   return 0   /* shouldn't get here */

/**************************************/
/* Read input file and build a table of NIST time servers.
      input:   arg( 1) = input file name
      return:  0 = error, 1 = success
      if success: builds arrays in expose list below
                  SrvAddr.0 = count of servers
*/
BuildServerTable: procedure expose SrvName. SrvAddr. SrvNote. SrvLoc.
   parse arg srvlist
   HTAB  = '09'X
   SPACE = '20'X
   host.! = ''
   if \FileExists( srvlist) then
      return 0
   if \FileOpen( srvlist) then
      return 0
   i = SrvAddr.0
/* scan until a line starting with '$' is found */
   do while lines( srvlist) > 0
      if left( linein( srvlist), 1) = '$' then
         leave
      end /* do while */
/* parse the lines until another line starting with '$' is found */
   do while lines( srvlist) > 0
      server = translate( linein( srvlist), SPACE, HTAB)
      if left( server, 1) = '$' then
         leave
      parse var server hostname IPaddr notes location .
      if hostname = '' then				     /* blank line? */
         iterate
      if notes = '' then
         notes = '0'
      if pos( '3', notes) > 0 then
         iterate                                 /* test server, do not use */
      if location = '' then
         location = '(unknown)'

      /* fix up IP addr -- for the benefit of the Win32 client */
      if IPaddr = '' then do			       /* if IPaddr missing */
         /* if the hostname field looks like an IP address... */
         if Verify( hostname, '0123456789.') = 0 then
            IPaddr = hostname
         else do			      /* if hostname not an IP addr */
            if SockGetHostByName( hostname, 'host.!' ) then
               IPaddr = host.!addr
            else do
               say "Can't resolve host name:" hostname ' (ignored)'
               iterate
               end
            end
         end
      /* we have IP address, now fix up host name for Win32 client */
      if Verify( hostname, '0123456789.') = 0 then do
         hostname = 'nameless.time.server'	/* default if can't resolve */
         if SockGetHostByAddr( IPaddr, 'host.!' ) then 
            hostname = host.!name
         end
      if Pos( '.', hostname) = 0 then do	       /* if no '.' in name */
         if SockGetHostByAddr( IPaddr, 'host.!' ) then 
            hostname = host.!name
         if Pos( '.', hostname) = 0 then	 /* if still no '.' in name */
            hostname = strip( hostname) || '.local'
         end

      i = i + 1
      SrvName.i = hostname
      SrvAddr.i = IPaddr
      SrvNote.i = notes
      SrvLoc.i = location
      end /* do while */
   SrvAddr.0 = i
   call FileClose srvlist
   return 1

/**************************************/
/* miscellaneous utility routines */

/* return a string with quotes around it */
quote:
   return '"' || arg( 1) || '"'

/**************************************/
/* file utility routines */

/* rv = FileExists( fileName) */         /* TRUE if exists, FALSE otherwise */
FileExists:
   return (stream( arg( 1), 'C', 'QUERY EXISTS') \= '')

/* rv = FileOpen( fileName, fileMode) */     /* TRUE/FALSE if success/error */
FileOpen:
   return abbrev( 'READY:', stream( arg( 1), 'C', 'OPEN' arg( 2)))

/* rv = FileClose( fileName) */              /* TRUE/FALSE if success/error */
FileClose:
   return abbrev( 'READY:', stream( arg( 1), 'C', 'CLOSE'))

FileDelete:                                  /* TRUE/FALSE if success/error */
   return (SysFileDelete( arg( 1)) = 0)

/* Conditionally load a DLL and register its functions.
   returns: 1 if error, 0 otherwise
*/
LoadLib: procedure
   parse arg lib, proc
   if RxFuncQuery( proc) <> 0 then do
      call RxFuncAdd proc, lib, proc
      str = 'call' proc
      signal on syntax name LoadLibFail
      interpret str
      end
   return 0
      
LoadLibFail:
   call RxFuncDrop proc
   return 1

/**************************************/
/* the user interface */

GetOpts: procedure expose info. opts.
   parse arg optlist
   if abbrev( optlist, '?') then
      signal Usage
   do while abbrev( optlist, '-')
      call GetOption
      if opt = '-?' | opt = '-H' then
         signal Usage
      else if opt = '-G' then
         opts.go = 1
      else if opt = '-L' then
         opts.log = 1
      else if opt = '-I' then do
         if val = '' then		     /* if space after option char. */
            val = GetOption()
         opts.ifile = val
         end
      else if opt = '-O' then do
         if val = '' then
            val = GetOption()
         opts.ofile = val
         end
      else
         call Fatal 'Invalid option' quote( option)
      end
   return strip( optlist)

GetOption:
   parse var optlist option optlist
   opt = translate( left( option, 2))
   val = substr( option, 3)
   return option

Usage:
   say info.PROGRAM_STR '   ' info.VERSION_STR
   say ''
   say '   Generates a list of NIST time servers in order of increasing'
   say '   network distance from the user.'
   say ''
   say 'Usage: makelist  [ options ]  [ my-list-name ]'
   say ''
   say '   where "my-list-name" is the name of the output file'
   say '   default output file name = "my-nist-srv.lst"'
   say ''
   say '   default input file name = "nist-srv.lst"'
   say ''
   say 'Options:'
   say '   -g   generate the list (requires several minutes)'
   say '   -i   specify input file name (default: nist-srv.lst)'
   say ''
   say 'Example:  makelist -i my-nist-srv.lst -g'
   say '   Creates a new my-nist-srv.lst based on the current my-nist-srv.lst'
   exit 0

/**************************************/
/* and just in case... */

Fatal:
   say 'ERROR --' arg( 1)
   exit 1

halt:
   say 'Program halted by operator'
   exit 1

novalue:
   say 'Uninitialized variable on line:' sigl
   say sourceline( sigl)
   exit 1

syntax:
   say 'Syntax error on line:' sigl  ' error code:' rc
   say sourceline( sigl)
   
   exit 1

/* <eof> */
