/* WALLCHANGE.CMD
 * (C) 2021,2023 Alex Taylor
 *
 * Change the desktop background to a random image selected from a list.
 *
 * The list of images is built from the command line arguments, with each
 * argument parsed as a filespec representing one of the following:
 *  - image: a single file containing a valid JPEG, PNG, BMP or GIF image
 *  - directory: a directory name (other than the desktop directory)
 *  - list: a specially formatted text file whose filename ends in .LST
 * These may be either relative or fully-qualified filenames.  Any that are
 * unreadable or don't exist will be skipped.  Each valid image will be
 * added to the list of available backgrounds.
 *
 * For a directory, WALLCHANGE will read it recursively, attempting to
 * interpret each of its contents as one of the three types listed above.
 *
 * A list must be a plain text file of formatted entries, one per line.
 * Blank lines, or lines starting with a semicolon (';'), are ignored.
 *
 * Each entry in a list file takes the following format:
 *   file<tab>display-mode<tab>text-colour<tab>active-time
 * (the display-mode, text-colour, and active-time fields are all optional)
 * where:
 *   filespec is one of the three defined types (image, directory, or list),
 *      and will be handled as noted above.
 *   display-mode is one of:
 *      N     (show image at its native size)
 *      T     (tile image at its native size)
 *      S[,#] (tile image, scaled to show # instances in each
 *             direction; # is a number from 1 to 20, defaulting to 1
 *             if not specified)
 *   text-colour is the desktop icon text colour to apply when using
 *      this particular image, if not specified, a default
 *      default text colour will be used.
 *   active-time defines time periods (based on hour and weekday) when the
 *      image may be chosen for the background. It is a comma-separated
 *      string containing one or more sequences of the form:
 *          <hour1>-<hour2>:<weekday1>-<weekday2>
 *      where hour1 is the period's starting hour (0-23, inclusive),
 *      hour2 is the period's ending hour (1-24, exclusive),
 *      weekday1 is the starting day of the week, and weekday 2 is the ending
 *      day of the week (1-7, where Monday is 1 and Sunday is 7).
 *      Each of these variables is optional, but the '-' hyphens are required.
 * For a directory or list entry, the display-mode, text-colour and
 * active-time specifiers are interpreted as defaults to be applied to every
 * image within that directory or list, unless overidden by an individual
 * list entry inside it.
 *
 * WPTOOLS.DLL is required, and so is the Unix 'file' utility.
 */

SIGNAL ON NOVALUE

CALL RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
CALL SysLoadFuncs
CALL RxFuncAdd 'WPToolsLoadFuncs', 'WPTOOLS', 'WPToolsLoadFuncs'
CALL WPToolsLoadFuncs
/*
CALL RXFuncAdd 'rxMIOLoadFuncs', 'RXMMIO', 'rxMIOLoadFuncs'
CALL rxMIOLoadFuncs
*/

file_exe = SysSearchPath('.', 'file.exe')
IF file_exe == '' THEN
    file_exe = SysSearchPath('PATH', 'file.exe')

/* Get arguments */
PARSE ARG args
argc = _ParseParams( args )
IF argc < 1 THEN DO
    PARSE SOURCE . . me .
    myname = FILESPEC('NAME', me )
    PARSE UPPER VAR myname mycommand '.' .
    SAY 'Syntax:' mycommand '<file or directory names>'
    RETURN 0
END

/* Query the current desktop wallpaper */
SAY 'Getting current desktop settings.'
ok = WPToolsQueryObject('<WP_DESKTOP>', "szClass", "szTitle", "szSetupString", "szLocation")
IF ( ok == 0 ) | ( szClass <> 'WPDesktop') THEN DO
    SAY 'Unable to query desktop folder.'
    RETURN 3
END
PARSE VAR szSetupString . 'BACKGROUND='bg_file','bg_options';' .
desktop_folder = TRANSLATE( szLocation || szTitle )

/* Get the current weekday (Monday=1, Sunday=7) */
current_wday = WORDPOS( DATE('W'), 'Monday Tuesday Wednesday Thursday Friday Saturday Sunday')
/* Get the current hour of the day */
current_hour = TIME('H')
SAY 'Current hour =' current_hour
SAY 'Current weekday =' current_wday

/* Build a list of source images from the arguments */
SAY 'Getting list of images.'
count = GetImages()
IF count < 1 THEN DO
    SAY 'No images found!'
    RETURN 2
END

SAY 'Current wallpaper: ' bg_file
SAY 'Background options:' bg_options

next_id = CycleImage( bg_file )
IF next_id == 0 THEN RETURN 0

next_bg = STREAM( source.next_id, 'C', 'QUERY EXISTS')
text_ss = ''

IF SYMBOL('source.next_id.!scale') == 'VAR' THEN DO
    IF source.next_id.!scale == 'N,' THEN
        bg_options = source.next_id.!scale || '1,I'
    ELSE
        bg_options = source.next_id.!scale || ',I'
END
IF SYMBOL('source.next_id.!textclr') == 'VAR' THEN DO
    text_ss = 'ICONTEXTCOLOR='source.next_id.!textclr';'
END

SAY 'Setting background:' next_bg
SAY 'Background options:' bg_options

bg_ss = 'BACKGROUND='next_bg','bg_options';'text_ss
/*
oo_exe = SysSearchPath('.', 'oo.exe')
IF oo_exe == '' THEN
    oo_exe = SysSearchPath('PATH', 'oo.exe')
*/
oo_exe = ''
IF oo_exe == '' THEN
    CALL SysSetObjectData '<WP_DESKTOP>', bg_ss
ELSE
    ADDRESS CMD oo_exe '/As "<WP_DESKTOP>" "'bg_ss'"'

RETURN 0


/* ------------------------------------------------------------------------- *
 * _ParseParams                                                              *
 *                                                                           *
 * Parse program parameters from the command line, including handling quoted *
 * values. Mostly based on code from 'REXX Tips & Tricks' by Bernd Schemmer. *
 * ------------------------------------------------------------------------- */
_ParseParams: PROCEDURE EXPOSE argv.
    PARSE ARG thisArgs

    argv. = ''
    argv.0 = 0
    DO WHILE thisArgs <> ''
        PARSE VALUE STRIP( thisArgs, "B") WITH curArg thisArgs
        PARSE VAR curArg tc1 +1 .
        IF tc1 = '"' | tc1 = "'" THEN
            PARSE VALUE curArg thisArgs WITH (tc1) curArg (tc1) ThisArgs
        i = argv.0 + 1
        argv.i = STRIP( curArg )
        argv.0 = i
    END

RETURN argv.0


/* ------------------------------------------------------------------------- *
 * GetImages                                                                 *
 *                                                                           *
 * Build a list of all images defined by the program arguments.              *
 * ------------------------------------------------------------------------- */
GetImages: PROCEDURE EXPOSE argv. source. desktop_folder current_hour current_wday
    count = 0
    def_mode = 'N'
    def_text = '255 255 255'
    def_interval = ''
    DO i = 1 TO argv.0
        /* Skip the desktop directory (for when called from XWP menu folder) */
        IF TRANSLATE( argv.i ) == desktop_folder THEN ITERATE

        /* See if it's a file */
        image = STREAM( argv.i, 'C', 'QUERY EXISTS')
        IF image <> '' THEN DO
            /* See if it's a list file */
            ext = ''
            ldot = LASTPOS('.', image )
            IF ldot > 0 THEN ext = TRANSLATE( SUBSTR( image, ldot ))
            IF ext == '.LST' THEN
                CALL ReadListFile image

            /* Any other file, just assume it's an image for now */
            ELSE DO
                count = count + 1
                source.count = image
                SAY 'Added' image
            END

            ITERATE
        END

        /* See if it's a directory containing files */
        ok = SysFileTree( argv.i'\*', 'images.', 'O')
        IF ok <> 0 THEN ITERATE
        DO j = 1 TO images.0
            count = count + 1
            source.count = images.j
        END

        IF images.0 > 0 THEN
            SAY 'Added' images.0 'images from' argv.i
    END
    source.0 = count

RETURN count


/* ------------------------------------------------------------------------- *
 * IsActiveTime                                                              *
 *                                                                           *
 * See if the current time falls within the given active-times definition.   *
 * A definition is a string consisting of any number of comma-separated      *
 * intervals of the form                                                     *
 *    <hour1>-<hour2>:<weekday1>-<weekday2>                                  *
 * where                                                                     *
 *  - hour1 is the 24 hour starting hour (inclusive)                         *
 *  - hour2 is the 24-hour ending hour (exclusive)                           *
 *  - weekday1 is the starting weekday number (1=Mon, 7=Sun)                 *
 *  - weekday2 is the ending weekday number                                  *
 * The variables are optional but the '-' hyphens are required.              *
 *                                                                           *
 * The weekday portion (including ':') is optional but must include a '-'    *
 * if defined at all. To specify a single day of the week, use e.g. '1-1'.   *
 *                                                                           *
 * Example:                                                                  *
 *   0-9:1-5,18-24:1-5,0-24:6-7                                              *
 * means the picture is valid during these times:                            *
 *  - before 9am on Monday-Friday                                            *
 *  - after 6pm on Monday-Friday                                             *
 *  - at any time on Saturday or Sunday                                      *
 * The above could also be written as                                        *
 *   -9:1-5,18-:1-5,-:6-7                                                    *
 *                                                                           *
 * Returns 1 if it does, 0 if it does not.                                   *
 * ------------------------------------------------------------------------- */
IsActiveTime: PROCEDURE EXPOSE current_hour current_wday
    PARSE ARG definition

    active = 0
    CALL StringTokenize definition, ',', 'intervals.'
    DO i = 1 TO intervals.0
        PARSE VAR intervals.i hour1'-'hour2':'day1'-'day2
        IF ( hour1 <> '') & ( VERIFY( hour1, '0123456789') > 0 ) THEN hour1 = ''
        IF ( hour2 <> '') & ( VERIFY( hour2, '0123456789') > 0 ) THEN hour2 = ''
        IF ( day1 <> '') & ( VERIFY( day1, '1234567') > 0 ) THEN day1 = ''
        IF ( day2 <> '') & ( VERIFY( day2, '1234567') > 0 ) THEN day2 = ''

        /* Make sure current weekday is within range, if defined */
        IF ( day1 <> '') & ( current_wday < day1 ) THEN
            ITERATE
        IF ( day2 <> '') & ( current_wday > day2 ) THEN
            ITERATE

        /* Make sure current hour is within range, if defined */
        IF ( hour1 <> '') & ( current_hour < hour1 ) THEN
            ITERATE
        IF ( hour2 <> '') & ( current_hour >= hour2 ) THEN
            ITERATE

        /* Made it this far, so the interval is active */
        active = 1
        LEAVE
    END

RETURN active


/* ------------------------------------------------------------------------- *
 * ReadListFile                                                              *
 *                                                                           *
 * Parse a list file and add the images defined therein.  (May be called     *
 * recursively if the list file defines other list files.)                   *
 * ------------------------------------------------------------------------- */
ReadListFile: PROCEDURE EXPOSE argv. source. desktop_folder count current_hour current_wday def_mode def_text def_interval
    PARSE ARG list_file
    SAY 'Reading list file' list_file '...'
    listpath = FILESPEC('DRIVE', list_file ) || FILESPEC('PATH', list_file )
    listpath = STRIP( listpath, 'T', '\')
    difference = count
    CALL LINEIN list_file, 1, 0
    DO WHILE LINES( list_file ) > 0
        /* Read the next line */
        PARSE VALUE STRIP( LINEIN( list_file )) WITH _val';'_comment
        /* Try to parse into filespec, scaling option, text colour and interval spec */
        textclr = ''
        s_mode = ''
        s_factor = 1
        intspec = ''
        PARSE VAR _val next '09'x scale '09'x textclr '09'x intspec

        /* If interval(s) are defined, make sure the current time/day matches */
        IF intspec == '' THEN intspec = def_interval
        IF intspec <> '' THEN DO
            IF \IsActiveTime( intspec ) THEN DO
                SAY next 'is not active now, skipping'
                ITERATE
            END
        END

        IF scale <> '' THEN DO
            PARSE UPPER VAR scale s_mode','s_factor',' .
            IF VERIFY( s_factor, '0123456789') == 0 THEN DO
                IF s_factor > 20 THEN s_factor = '1'
            END
            ELSE s_factor = '1'
            IF ( POS( s_mode, 'NST') == 0 ) THEN
                s_mode = ''
        END
        IF s_mode == '' THEN s_mode = def_mode

        IF textclr <> '' THEN DO
            textclr = STRIP( textclr )
            IF VERIFY( textclr, ' 0123456789') <> 0 THEN
                textclr = ''
        END
        IF textclr == '' THEN textclr = def_text

        IF next == '' THEN ITERATE
        IF FILESPEC('PATH', next ) == '' & FILESPEC('DRIVE', next ) == '' THEN
            next = listpath'\'next

        /* See if the specified item exists as a file */
        exists = STREAM( next, 'C', 'QUERY EXISTS')
        IF exists <> '' THEN DO
            /* See if it's another list file */
            ext = ''
            ldot = LASTPOS('.', exists )
            IF ldot > 0 THEN ext = TRANSLATE( SUBSTR( exists, ldot ))
            IF ext == '.LST' THEN
                CALL ReadListFile exists

            /* Any other file, just assume it's an image for now */
            ELSE DO
                count = count + 1
                source.count = exists
                IF s_mode <> '' THEN
                    source.count.!scale = s_mode','s_factor
                IF textclr <> '' THEN
                    source.count.!textclr = textclr
            END
        END

        /* If not, see if it's a directory containing files */
        ELSE DO
            ok = SysFileTree( next'\*', 'dircontents.', 'DFO')
            IF ok <> 0 THEN ITERATE
            IF dircontents.0 == 0 THEN ITERATE

            /* Process each file in the directory */
            SAY 'Reading directory' next '...'
            DO i = 1 TO dircontents.0
                /* See if it's another list file */
                ext = ''
                ldot = LASTPOS('.', dircontents.i )
                IF ldot > 0 THEN ext = TRANSLATE( SUBSTR( dircontents.i, ldot ))
                IF ext == '.LST' THEN
                    CALL ReadListFile dircontents.i

                /* Any other file, just assume it's an image for now */
                ELSE DO
                    count = count + 1
                    source.count = dircontents.i
                    IF s_mode <> '' THEN
                        source.count.!scale = s_mode','s_factor
                    IF textclr <> '' THEN
                        source.count.!textclr = textclr
                END
            END
        END
    END
    CALL STREAM list_file, 'C', 'CLOSE'
    difference = count - difference
    SAY 'Added' difference 'images from' FILESPEC('NAME', list_file )
RETURN


/* ------------------------------------------------------------------------- *
 * IsSupportedImage                                                          *
 *                                                                           *
 * Return 1 if the file is recognized as a supported image format, 0 if not. *
 * ------------------------------------------------------------------------- */
IsSupportedImage: PROCEDURE EXPOSE file_exe
    PARSE ARG filename

    /* [2023-03-12] RxMMIO seems to hang sometimes; replaced with file command]
    info = rxMIOImgInfo( filename, -1 )
    PARSE VAR info rc resx resy bits . . type
    IF rc >= 100 THEN RETURN 0
    */

    IF file_exe == '' THEN DO
        /* Fallback logic to assume image based on filename extension
         */
        _lastdot = LASTPOS('.', filename )
        IF _lastdot < 1 THEN RETURN 0
        PARSE UPPER VAR filename . =(_lastdot) +1 _extension
        IF WORDPOS( _extension, 'JPG JPEG PNG BMP GIF') > 0 THEN DO
            SAY ' -' _extension
            RETURN 1
        END
        ELSE
            RETURN 0
    END

    /* Use the Posix 'file' utility to read the file signature
     */
    supported_formats ='JPEG PNG GIF X-MS-BMP'

    _type = ''
    nq = RXQUEUE('create')
    oq = RXQUEUE('set', nq )
    '@ file -i' filename '|RXQUEUE' nq
    IF QUEUED() THEN DO
        PARSE PULL _line
        PARSE UPPER VAR _line 'IMAGE/' _type ';' .
    END
    IF _type <> '' THEN
        SAY ' -' _type
    CALL RXQUEUE 'delete', nq
    IF _type <> '' & WORDPOS( _type, supported_formats ) > 0 THEN
        ok = 1
    ELSE
        ok = 0
RETURN ok


/* ------------------------------------------------------------------------- *
 * CycleImage                                                                *
 *                                                                           *
 * Randomly select a new image from the list of candidates.                  *
 * ------------------------------------------------------------------------- */
CycleImage: PROCEDURE EXPOSE source. count file_exe
    ARG previous_bg

    SAY count 'images defined.'

    got_image = 0
    tries = 0
    DO UNTIL got_image > 0
        /* Sanity check */
        tries = tries + 1
        IF tries > count THEN LEAVE

        index = RANDOM( 1, count )
        img_file = source.index

        SAY 'Trying:' img_file

        /* Try not to select the same image twice in a row */
        IF TRANSLATE( img_file ) == previous_bg THEN DO
            SAY ' - Same as current background.'
            ITERATE
        END
        SAY ' - New background'

        /* Make sure the file exists */
        IF STREAM( img_file, 'C', 'QUERY EXISTS') == '' THEN DO
            SAY ' - File not found.'
            ITERATE
        END
        SAY ' - File exists'

        /* Make sure it's readable */
        read_state = STREAM( img_file, 'C', 'OPEN READ')
        IF read_state <> 'READY:' THEN DO
            SAY ' - File cannot be opened:' read_state
            ITERATE
        END
        ELSE
            CALL STREAM img_file, 'C', 'CLOSE'

        /* Make sure it's actually an image file */
        CALL CHAROUT, ' - File is readable, checking image type'
        IF \IsSupportedImage( img_file ) THEN DO
            SAY ' - not a valid image file'
            ITERATE
        END
        SAY ' - Image OK'

        got_image = index
    END
    IF got_image > 0 THEN
        SAY 'Selecting' source.got_image

RETURN got_image


/* ------------------------------------------------------------------------- *
 * StringTokenize                                                            *
 *                                                                           *
 * Utility function to tokenize a string using the given separator.          *
 * ------------------------------------------------------------------------- */
StringTokenize:
    PARSE ARG string, separator, __stem
    CALL __StringTokenize string, separator, __stem
    DROP __stem
RETURN

__StringTokenize: PROCEDURE EXPOSE (__stem)
    PARSE ARG string, separator, tokens

    IF ( string = '') THEN RETURN string
    IF ( separator = '') THEN separator = ' '

    i = 0
    CALL VALUE tokens || '0', i
    string = STRIP( string, 'B', separator )
    DO WHILE LENGTH( string ) > 0
        x = 1
        y = POS( separator, string, x )
        IF y > 0 THEN DO
            current = SUBSTR( string, 1, y-1 )
            x = y + 1
            i = i + 1
            CALL VALUE tokens || 'i', current
        END
        ELSE DO
            current = STRIP( string, 'B', separator )
            i = i + 1
            CALL VALUE tokens || 'i', current
            x = LENGTH( string ) + 1
        END
        string = SUBSTR( string, x )
    END
    CALL VALUE tokens || '0', i

RETURN


/* -------------------------------------------------------------------------- *
 * CONDITION HANDLERS                                                         *
 * -------------------------------------------------------------------------- */
NOVALUE:
    SAY
    CALL LINEOUT 'STDERR:', RIGHT( sigl, 6 ) '+++' STRIP( SOURCELINE( sigl ))
    CALL LINEOUT 'STDERR:', RIGHT( sigl, 6 ) '+++ Non-initialized variable.'
    SAY
EXIT sigl

